/*
 * Copyright (c) 1993             by the University of Southern California
 *
 * For copying and distribution information, please see the file
 * <usc-copyr.h>.
 *
 * ppw was written by Prasad Upasani
 */

#include <usc-copyr.h>

#include <stdio.h>
#include <stdlib.h>
#include <ardp.h>
#include <psite.h>	/* For #define P_P_PASSWORD */
#include <pfs.h>
#include <pprot.h>
#include <termios.h>
#include <perrno.h>
#include <errno.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#ifdef P_P_PASSWORD

static void  	abort_pwread();
static void 	abort_prog();
static int  	read_password();
static int 	write_to_file();
static int	delete_entry();
static VLINK 	p_get_vsdesc();
static TOKEN 	get_vsdesc_principals();
static int 	usage();
static char 	*canonicalize();
static char 	*uppercase();
static char 	*get_passwd_filename();
static int 	read_and_set_password();
static int 	delete_passwd_frm_entries();
static int 	set_princ_with_no_passwd();
static char     *addr_to_name();
static char     *get_host_from_user();

int  pfs_debug = 0;


main(int argc, char *argv[])
{
    RREQ	req;
    char	*dirhst = NULL;
    int		sflag = 0, mflag = 0, dflag = 0, rflag = 0;
    TOKEN 	princp, princ_list = NULL;
    char 	old_password[255], new_password[255];
    PAUTH 	authinfo;    /* Structure containing authentication info */
    PAUTH 	authp;       /* To iterate through authinfo list */
    char 	prompt[255];

    argc--;argv++;

    while (argc > 0 && **argv == '-') {

        switch (*(argv[0]+1)) {
        case 'D':
            pfs_debug = 1; /* Default debug level */
            sscanf(argv[0],"-D%d",&pfs_debug);
            break;

        case 'N':  /* Priority (nice) */
            ardp_priority = ARDP_MAX_PRI; /* Use this if no # */
            sscanf(argv[0],"-N%d",&ardp_priority);
            if(ardp_priority > ARDP_MAX_SPRI) 
                ardp_priority = ARDP_MAX_PRI;
            if(ardp_priority < ARDP_MIN_PRI) 
                ardp_priority = ARDP_MIN_PRI;
            break;

	case 's':	
	    if (dflag || rflag) {
		fprintf(stderr, "Only one of -s, -r and -d allowed!\n");
		usage();
	    }
	    sflag = 1;	/* Set principal's password in remote server's */
	    break;      /* password file */

	case 'm':
	    if (dflag || rflag) {
		fprintf(stderr, "-m not allowed with -d or -r!\n");
		usage();
	    }
	    mflag = 1;	/* Maintainer, old passwd not required */
	    break;

	case 'd':
	    if (sflag || rflag) {
		fprintf(stderr, "Only one of -s, -r and -d allowed!\n");
		usage();
	    }
	    dflag = 1;  /* Delete all passwords from file entries */
	    break;

	case 'r':
	    if (sflag || dflag) {
		fprintf(stderr, "Only one of -s, -r and -d allowed!\n");
		usage();
	    }
	    rflag = 1;  /* Remove principal's password entry */
	    break;

	case 'p':	/* principal specified */
	    argc--; argv++;
	    princ_list = tkappend(argv[0], princ_list);
	    break;

        default:
	    usage();
        }
        argc--; argv++;
    }

    if(argc > 1)
	usage();

    if(argc > 0)
        dirhst = stcopyr(argv[0],dirhst);

    if (sflag) {
	if (!princ_list) {
	    /* Get default principals from instances of PRINCIPAL */
	    /* attribute of VS-DESCRIPTION */
	    if (!(princ_list = get_vsdesc_principals())) {
		fprintf(stderr, "No principals found!\n");
		usage();
	    }
	}
	if (!dirhst) {
	    if (!(dirhst = get_host_from_user(princ_list))) {
		dirhst = stalloc(255);
		gethostname(dirhst, stsize(dirhst));
	    }
	}
	for (princp = princ_list; princp; princp = princp->next) {
	    if (mflag)
		req = p__start_req(dirhst);
	    else {
		if (!(req = ardp_rqalloc()))
		    out_of_memory();
		p__add_req(req, "VERSION %d %s\n", 
			   VFPROT_VNO, PFS_SW_ID);
		qsprintf(prompt, sizeof(prompt), 
			 "Old password for %s on host %s",
			 princp->token, dirhst);
		read_password(prompt, old_password, sizeof(old_password));
		p__add_req(req, "AUTHENTICATE OPTIONAL P_PASSWORD %'s %'s\n",
			   old_password, princp->token);
	    }
	    qsprintf(prompt, sizeof(prompt), 
		     "New password for %s on host %s", 
		     princp->token, dirhst);
	    read_password(prompt, new_password, sizeof(new_password));
	    p__add_req(req, "PARAMETER SET PASSWORD %'s %'s\n",
		       princp->token, new_password);

	    if (ardp_send(req,dirhst,0,ARDP_WAIT_TILL_TO)) {
		perrmesg("ppw failed: ", 0, NULL);
		break;
	    }
	    if(pwarn) pwarnmesg("WARNING: ",0,NULL);
	
	    /* Print response */
	    while(req->inpkt) {
		printf("%s",req->inpkt->text);
		req->inpkt = req->inpkt->next;
	    }
	    
	    ardp_rqfree(req);
	}
    }
    else if (dflag) {
	if (delete_passwd_frm_entries(dirhst, princ_list))
	    fprintf(stderr, 
		    "Could not delete password from entry!\n");
    }
    else if (rflag) {
	if (!princ_list) {
	    fprintf(stderr, "ERROR: No principals specified!\n");
	    usage();
	}
	for (princp = princ_list; princp; princp = princp->next)
	    if (delete_entry(dirhst, princp->token)) {
		fprintf(stderr, "Could not delete entry!\n");
		break;
	    }
    }
    else {
	if (!dirhst)
	    dirhst = stcopy("*");
	if (!princ_list) {
	    /* Prompt for passwords for all instances of PRINCIPAL */
	    /* attribute of VS-DESCRIPTION and for all entries without */
	    /* passwords in the password file */
	    princ_list = get_vsdesc_principals();
	    if (read_and_set_password(dirhst, princ_list))
		fprintf(stderr, "Could not set password!\n");
	    else
		set_princ_with_no_passwd(princ_list);
	}
	else 
	    if (read_and_set_password(dirhst, princ_list))
		fprintf(stderr, "Could not set password!\n");
    }

    tklfree(princ_list);
    if (dirhst)
	stfree(dirhst);
    return 0;
}


/* Display command usage and exit. */
static int usage()
{
    fprintf(stderr,  
	    "Usage: ppw [-s [-m]] [-r] [-d] [-D[#]] [-N[#]]\n");
    fprintf(stderr,
	    "           [-p <principal>] [-p <principal>] .. [hostname]\n");
    exit(1);
}


/* Write entry into user's password file using information supplied 
 * If entry already exists for specified host and principal, the old 
 * password is overwritten woth the new one. 
 */
static int write_to_file(char *hostname, char *principal, char *password)
{
    char in_line[255], out_line[255];
    char server[255], princ_name[255], passwd[255];
    char *hname = stcopy(hostname);
    long pos;
    char *passwd_filename = get_passwd_filename();
    FILE *passwd_fp;
    char *ppw_encode();

    /* Canonicalize host if not wildcarded */
    if (!(hname = canonicalize(hname)))
	return PFAILURE;

    if (!(passwd_fp = fopen(passwd_filename, "r+"))) {
	/* Cannot open password file, so create */
	if (!(passwd_fp = fopen(passwd_filename, "w+"))) {
	    if (pfs_debug)
		fprintf(stderr, "Cannot create file %s\n",
			passwd_filename);
	    return PFAILURE;
	}
	if (chmod(passwd_filename, 0600)) {
	    if (pfs_debug) 
		perror("Could not change permissions of passwd file");
	    return PFAILURE;
	}
	fprintf(passwd_fp, "# Prospero Client Password File.\n");
	fprintf(passwd_fp, "%-25s %-25s %-25s\n\n",
		"# <server>", "<principal>", "<password>");
	fseek(passwd_fp, 0L, 1); /* To allow input after output */
    }

    /* Check if entry exists for principal */
    while (pos = ftell(passwd_fp),      /* Remember position of line */
           fgets(in_line, sizeof(in_line), passwd_fp)) { 
        sscanf(in_line, "%s %s %s\n", 
               server, princ_name, passwd);
        if (strequal(hname, server) && 
	    strequal(principal, princ_name)) { 
            /* Entry found; set file pointer to overwrite */
            fseek(passwd_fp, pos, 0);
            break;
        }
    }

    /* Quote password before writing to file */
    qsprintf(passwd, sizeof(passwd), "%'s", ppw_encode(password));

    sprintf(out_line, "%-25s %-25s %-25s\n", 
	    hname, principal, passwd);
    
    if (feof(passwd_fp) || strlen(in_line) == strlen(out_line))
	fputs(out_line, passwd_fp);
    else {
	/* Cannot insert space; have to create new file	*/
	FILE *tmp_fp;
	int newpos, uid = getuid();
	char tmp_filename[255];
	char line[255];

	qsprintf(tmp_filename, sizeof(tmp_filename), "%s_tmp", 
		 passwd_filename);
	if (!(tmp_fp = fopen(tmp_filename, "w"))) {
	    if (pfs_debug)
		fprintf(stderr, "Cannot create temporary file %s\n",
			tmp_filename);
	    return PFAILURE;
	}

	rewind(passwd_fp);

	while (newpos = ftell(passwd_fp), 
	       fgets(line, sizeof(line), passwd_fp) 
	       && (newpos < pos))
	    fputs(line, tmp_fp);
		
	fputs(out_line, tmp_fp);
	
	while(fgets(line, sizeof(line), passwd_fp))
	    fputs(line, tmp_fp);
	
	(void) fclose(tmp_fp);
	(void) fclose(passwd_fp);
		
	if (rename(tmp_filename, passwd_filename)) {
	    unlink(tmp_filename);
	    if (pfs_debug)
		perror("Could not overwrite password file");
	    return PFAILURE;
	}

	if (chmod(passwd_filename, 0600)) {
	    if (pfs_debug) 
		perror("Could not change permissions of passwd file");
	    return PFAILURE;
	}

	stfree(hname);
	return PSUCCESS;
    }

    stfree(hname);

    (void) fclose(passwd_fp);
    return PSUCCESS;
}


/* Deletes entry for hostname and principal specified from user's */
/* password file. */
/* If hostname is NULL, all entries for the principal are deleted. */
static int delete_entry(char *hostname, char *principal)
{
    char line[255];
    char server[255], princ_name[255], passwd[255];
    char *hname = stcopy(hostname);
    char tmp_filename[255];
    char *passwd_filename = get_passwd_filename();
    FILE *passwd_fp, *tmp_fp;

    if (!principal)
	return PFAILURE;

    /* Canonicalize host if not wildcarded */
    if (hname)
	hname = canonicalize(hname);
    
    if (!(passwd_fp = fopen(passwd_filename, "r"))) {
	fprintf(stderr, "Could not open password file: %s\n",
		passwd_filename);
    	return PFAILURE;
    }

    qsprintf(tmp_filename, sizeof(tmp_filename), "%s_tmp", passwd_filename);
    if (!(tmp_fp = fopen(tmp_filename, "w"))) {
        if (pfs_debug)
            perror("Could not create temporary file");
	return PFAILURE;
    }

    while (fgets(line, sizeof(line), passwd_fp)) { 
        sscanf(line, "%s %s %s\n", 
               server, princ_name, passwd);
	if (hname && strcmp(hname, server) || 
	    strcmp(princ_name, principal))
	    fputs(line, tmp_fp);
    }

    (void) fclose(tmp_fp);
    (void) fclose(passwd_fp);

    if (rename(tmp_filename, passwd_filename)) {
	unlink(tmp_filename);
	if (pfs_debug)
	    perror("Could not overwrite password file");
    	return PFAILURE;
    }
    
    if (chmod(passwd_filename, 0600)) {
	if (pfs_debug) 
	    perror("Could not change permissions of passwd file");
	return PFAILURE;
    }

    stfree(hname);
    return PSUCCESS;
}


/* Read password and writes entry to file for principals in princ_list */
static int read_and_set_password(char *dirhst, TOKEN princ_list)
{
    TOKEN princp;
    char  prompt[255], passwd[255];

    for (princp = princ_list; princp; princp = princp->next) {	
	if(strequal(dirhst,"*")) 
	    qsprintf(prompt, sizeof(prompt), "Password for %s", princp->token);
	else qsprintf(prompt, sizeof(prompt), "Password for %s on host %s",
		 princp->token, dirhst);
	read_password(prompt, passwd, sizeof(passwd));
	if (write_to_file(dirhst, princp->token, passwd))
	    return PFAILURE;
    }

    return PSUCCESS;
}	

/* Adds to princ_list, principals from the password file that do not */
/* have any passwords, and that are not already in princ_list. */
static int set_princ_with_no_passwd()
{
    char line[255], buf[255], prompt[255];
    char server[255], princ_name[255], passwd[255];
    char *passwd_filename = get_passwd_filename();
    char tmp_filename[255];
    FILE *passwd_fp, *tmp_fp;
    int  num;
    unsigned long hostaddr;
    char *hostname = NULL;
    struct hostent *host;
    char *ppw_encode();

    if (!(passwd_fp = fopen(passwd_filename, "r"))) {
	if (pfs_debug)
	    fprintf(stderr, "Could not open password file: %s\n",
		    passwd_filename);
    	return PFAILURE;
    }   

    qsprintf(tmp_filename, sizeof(tmp_filename), "%s_tmp", passwd_filename);
    if (!(tmp_fp = fopen(tmp_filename, "w"))) {
        if (pfs_debug)
            perror("Could not create temporary file");
	return PFAILURE;
    }

    while (fgets(line, sizeof(line), passwd_fp)) { 
	if (line[0] == '#' || line[0] == ' ') {
	    fputs(line, tmp_fp);
	    continue;
	}
        num = sscanf(line, "%s %s %s\n", 
               server, princ_name, passwd);

	if (num == 2) { 	/* No password */
	    if (isdigit(server[0])) {
		hostname = addr_to_name(server);
	    }
	    else
		hostname = stcopyr(server,hostname);
	    if(strequal(hostname,"*")) 
		qsprintf(prompt, sizeof(prompt), "Password for %s", princ_name);
	    else qsprintf(prompt, sizeof(prompt), "Password for %s on host %s",
			  princ_name, hostname);

	    stfree(hostname);
	    read_password(prompt, buf, sizeof(buf));
	    qsprintf(passwd, sizeof(passwd), "%'s", ppw_encode(buf));
	    fprintf(tmp_fp, "%-25s %-25s %-25s\n", 
		    server, princ_name, passwd);
	}
	else
	    fputs(line, tmp_fp);
    }

    (void) fclose(passwd_fp);
    (void) fclose(tmp_fp);

    if (rename(tmp_filename, passwd_filename)) {
	unlink(tmp_filename);
	if (pfs_debug)
	    perror("Could not overwrite password file");
    	return PFAILURE;
    }
    
    if (chmod(passwd_filename, 0600)) {
	if (pfs_debug) 
	    perror("Could not change permissions of passwd file");
	return PFAILURE;
    }

    return PSUCCESS;
}


/* Deletes just the password from the entries specified. */
/* If princ_list is NULL, then passwords are removed for all principal */
/* entries with the specified host. If dirhst is NULL, then entries */
/* are removed for all principals. */
static int delete_passwd_frm_entries(char *dirhst, TOKEN princ_list)
{
    char line[255], buf[255], prompt[255];
    char server[255], princ_name[255], passwd[255];
    char *passwd_filename = get_passwd_filename();
    char tmp_filename[255];
    FILE *passwd_fp, *tmp_fp;
    char *hname = NULL;

    if (dirhst) 
	if (!(hname = canonicalize(dirhst)))
	    return PFAILURE;

    if (!(passwd_fp = fopen(passwd_filename, "r"))) {
	if (pfs_debug)
	    fprintf(stderr, "Could not open password file: %s\n",
		    passwd_filename);
    	return PFAILURE;
    }   

    qsprintf(tmp_filename, sizeof(tmp_filename), "%s_tmp", passwd_filename);
    if (!(tmp_fp = fopen(tmp_filename, "w"))) {
        if (pfs_debug)
            perror("Could not create temporary file");
	return PFAILURE;
    }

    while (fgets(line, sizeof(line), passwd_fp)) { 
	if (line[0] == '#' || line[0] == ' ') {
	    fputs(line, tmp_fp);
	    continue;
	}
        sscanf(line, "%s %s %s\n", 
               server, princ_name, passwd);

	if ((!hname || strequal(hname, server)) &&
	    (!princ_list || member(princ_name, princ_list))) {
	    passwd[0] = '\0';
	    fprintf(tmp_fp, "%-25s %-25s %-25s\n", 
		    server, princ_name, passwd);
	}
	else
	    fputs(line, tmp_fp);
    }

    (void) fclose(passwd_fp);
    (void) fclose(tmp_fp);

    if (rename(tmp_filename, passwd_filename)) {
	unlink(tmp_filename);
	if (pfs_debug)
	    perror("Could not overwrite password file");
    	return PFAILURE;
    }
    
    if (chmod(passwd_filename, 0600)) {
	if (pfs_debug) 
	    perror("Could not change permissions of passwd file");
	return PFAILURE;
    }

    return PSUCCESS;
}


static struct termios	permmodes;


/* Read user's password */
static int read_password(char *prompt, char *pwstr, int pwlen)
{
    struct termios	tempmodes;
    int			tmp;

    if (prompt)
	printf("%s: ", prompt);
    else
	printf("Password: ");
    fflush(stdout);

    if(tcgetattr(fileno(stdin), &permmodes)) return(errno);
    bcopy(&permmodes,&tempmodes,sizeof(struct termios));

    tempmodes.c_lflag &= ~(ECHO|ECHONL);
    
    signal(SIGQUIT,abort_pwread);
    signal(SIGINT,abort_pwread);

    if(tcsetattr(fileno(stdin), TCSANOW, &tempmodes)) return(errno);

    *pwstr = '\0';
    fgets(pwstr, pwlen, stdin);
    pwstr[strlen(pwstr)-1] = '\0';
    printf("\n");
    fflush(stdout);

    if(tcsetattr(fileno(stdin), TCSANOW, &permmodes)) return(errno);
    signal(SIGQUIT,abort_prog);
    signal(SIGINT,abort_prog);
    return(0);
}

static void abort_pwread()
{
    tcsetattr(fileno(stdin), TCSANOW, &permmodes);
    abort_prog();
}

static void abort_prog()
{
    printf("Aborted\n");
    exit(0);
}

/* This may be put into lib/pfs. 
 * Gets a VLINK structure for the VS_DESCRIPTION file.
 */
static VLINK p_get_vsdesc()
{
    VLINK vl = vlalloc();

    if (!pget_dfile()) {
	if (pfs_debug)
	    fprintf(stderr, 
		    "Could not get value of env. var. VSDESC_FILE\n");
	return NULL;
    }

    if (!pget_dhost()) {
	if (pfs_debug)
	    fprintf(stderr, 
		    "Could not get value of env. var. VSDESC_HOST\n");
	return NULL;
    }
	
    vl->hsoname = stcopyr(pget_dfile(), vl->hsoname);
    vl->host    = stcopyr(pget_dhost(), vl->host);

    return vl;
}

/* Gets the value of the PRINCIPAL attribute of the VS-DESCRIPTION */
static TOKEN get_vsdesc_principals()
{
    VLINK   vsdesc_vl;
    PATTRIB attrs;
    TOKEN   princ_list = NULL;

    if (!(vsdesc_vl = p_get_vsdesc()))
	return NULL;
    
    if (!(attrs = pget_at(vsdesc_vl, "PRINCIPAL")))
	return NULL;
    
    for (; attrs; attrs = attrs->next)
	princ_list = tkappend(attrs->value.sequence->next->token,
			      princ_list);

    return princ_list;
}

/* If the hostname is not wildcarded, returns the internet address as */
/* a dotted string. If wildcarded, the hostname is just converted to */
/* uppercase and returned.*/
static char *canonicalize(char *hostname)
{
    struct hostent *host;
    char           *cp;
    struct in_addr hostaddr;
    
    if (!hostname)
	return NULL;

    /* Check if wildcarded */
    for (cp = hostname; *cp; ++cp)
	if (*cp == '*') 	/* Cannot cannicalize */
	    return uppercase(hostname);

    /* Look up server host */
    if ((host = gethostbyname(hostname)) == (struct hostent *) 0) {
	if (pfs_debug) 
	    fprintf(stderr, "canonicalize(): %s: unknown host\n", hostname);
	return NULL;
    }

    bcopy(host->h_addr,&hostaddr,sizeof(hostaddr));
    hostname = stcopyr(inet_ntoa(hostaddr), hostname);

    return hostname;
}

/* Convert a string to uppercase */
static char *uppercase(char *str)
{
    char *cp;

    if (!str)
	return NULL;

    for (cp = str; *cp; cp++)
	if (islower(*cp))
            *cp = toupper(*cp);

    return str;
}

/* Returns name of user's password file. */
static char *get_passwd_filename()
{
    int uid;
    char *home;
    static char *passwd_filename = NULL;

    if (passwd_filename)
	return passwd_filename;

    if (!(passwd_filename = stcopy(getenv("P_PASSWORD")))) {
	passwd_filename = stalloc(255);
	uid = getuid();
	if (!(home = getenv("HOME"))) {
	    if (pfs_debug)
		fprintf(stderr, 
			"Could not get value of HOME\n");
	    return NULL;
	}
	qsprintf(passwd_filename, stsize(passwd_filename),
		 "%s/.ppw_%d", home, uid);
    }

    return passwd_filename;
}

/* Goes sequentially through file, and when principal is a member of */
/* princ_list *and* the hostname is specific, asks user if passwd to */
/* be set for this host. If yes, returns hostname, else repeats with */
/* next matching entry. */ 
static char *get_host_from_user(TOKEN princ_list)
{
    char  *dirhst;
    FILE  *passwd_fp;
    char  *passwd_filename = get_passwd_filename();
    char  line[255], ans[255];
    char  server[255], princ_name[255], passwd[255];
    char  *hostname = NULL;

    if (!(passwd_fp = fopen(passwd_filename, "r+"))) {
	if (pfs_debug)
	    fprintf(stderr, "Could not open password file\n");
	return NULL;
    }

    while (fgets(line, sizeof(line), passwd_fp)) {
	sscanf(line, "%s %s %s\n", 
	       server, princ_name, passwd);
	if (member(princ_name, princ_list) &&
	    isdigit(server[0])) {
	    hostname = addr_to_name(server);
	    printf("Set password on host %s? (y/n): ",
		   hostname);
	    gets(ans);
	    if (strequal(ans, "y"))
		return (hostname);
	}
    }

    return NULL;
}

static char *addr_to_name(char *addr)
{
    unsigned long hostaddr;
    struct hostent *host;
    char *hostname = NULL;

    hostaddr = inet_addr(addr);
    host = gethostbyaddr((char *)&hostaddr,
			 sizeof(hostaddr), AF_INET);
    if(host) 
	hostname = stcopy(host->h_name);
    else 
	hostname = stcopy(addr);
    return hostname;
}

#endif /* P_P_PASSWORD */
