/* tkalloc.c */
/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the file
 * <usc-copyr.h>.
 */

#include <usc-copyr.h>
#include <stdio.h>

#include <pfs.h>

static TOKEN	tokenflist = NULL;
int		token_count = 0;
int		token_max = 0;

/*
 * tkalloc - allocate and initialize token structure
 *
 *    TKALLOC returns a pointer to an initialized structure of type
 *    TOKEN.  If it is unable to allocate such a structure, it
 *    returns NULL.
 */
TOKEN
tkalloc(const char *s)
{
    TOKEN	token_ent;

    if(tokenflist) {
        token_ent = tokenflist;
        tokenflist = tokenflist->next;
    } else {
        token_ent = (TOKEN) malloc(sizeof(TOKEN_ST));
        if (!token_ent) 
            out_of_memory();
        token_max++;
    }

    token_count++;

    /* Initialize and fill in default values */
#ifdef ALLOCATOR_CONSISTENCY_CHECK
    token_ent->consistency = INUSE_PATTERN;
#endif
    if(s) 
        token_ent->token = stcopy(s);
    else
        token_ent->token = NULL;
    token_ent->previous = NULL;
    token_ent->next = NULL;

    return token_ent;
}

#if 0                           /* not currently used, I think. */
/* Prepend the new string to the token list.  The head of the list will have a
   "previous" item that is the tail of the list.  However, the tail of the list
   will have no "next" item.  In the case of a single-element list, the head's
   "previous" will be itself.*/
TOKEN
tkprepend(char *newstr, TOKEN toklist)
{
    TOKEN tok;
    tok = tkalloc(newstr);
    tok->next = toklist;
    if (toklist) {
        tok->previous = toklist->previous;
        toklist->previous = tok;
    } else {
        tok->previous = tok;
    }
    return tok;
}
#endif

/* Append the new string to the token list.  The head of the list will have a
   "previous" item that is the tail of the list.  However, the tail of the list
   will have no "next" item.  Returns the new head of the list. 
   Needed because APPEND_ITEM multiply evaluates its arguments. 
*/
TOKEN
tkappend(const char *newstr, TOKEN toklist)
{
    TOKEN tok;
    tok = tkalloc(newstr);
    tok->next = NULL;
    if (toklist) {
        assert(toklist->previous);
        assert(toklist->previous->next == NULL);
        toklist->previous->next = tok;
        tok->previous = toklist->previous;
        toklist->previous = tok;
        return toklist;
    } else {
        tok->previous = tok;
        return tok;
    }
}


/*
 * tkfree - free an TOKEN structure
 *
 *    TKFREE takes a pointer to an TOKEN structure and adds it to
 *    the free list for later reuse.
 */
tkfree(TOKEN token_ent)
{
#ifdef ALLOCATOR_CONSISTENCY_CHECK
    assert(token_ent->consistency == INUSE_PATTERN);
    token_ent->consistency = FREE_PATTERN;
#endif
    if (token_ent->token) 
        stfree(token_ent->token);
    token_ent->next = tokenflist;
    if(tokenflist) 
        tokenflist->previous = token_ent;
    tokenflist = token_ent;
    token_count--;
    return PSUCCESS;
}

/*
 * tklfree - free an TOKEN structure
 *
 *    TKLFREE takes a pointer to an TOKEN structure frees it and any linked
 *    TOKEN structures.  It is used to free an entrie list of TOKEN
 *    structures.
 */
tklfree(TOKEN token_ent)
{
    TOKEN	nxt;

    while(token_ent != NULL) {
        nxt = token_ent->next;
        tkfree(token_ent);
        token_ent = nxt;
    }
    return PSUCCESS;
}


/*
 * tkcopy()
 * Copy a linked list of token structures.  Return the new list. 
 * Signal out_of_memory() if no more.
 */

TOKEN
tkcopy(TOKEN t)
{
    TOKEN retval = NULL;
    for ( ; t ; t = t->next) {
#ifdef ALLOCATOR_CONSISTENCY_CHECK
        assert(t->consistency == INUSE_PATTERN);
#endif
        retval = tkappend(t->token, retval);
    }
    return retval;
}


/* return TRUE if s is a member of the list t. */
member(const char s[], TOKEN tklist)
{
    for ( ; tklist; tklist = tklist->next) {
        if (strequal(s, tklist->token))
            return TRUE;
    }
    return FALSE;
}


