/*
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the file
 * <usc-copyr.h>.
 */

#include <usc-copyr.h>

/* Definitions of constants */
#define MAX_NUM_LINKS       3000 /* maximum # of directory links to return
                                    before giving up. */

/* Archie server we redirect ftp:hostname@ queries to. */
#define ARCHIE_SERVER       "ARCHIE.SURA.NET"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>
#include <pmachine.h>

#include <pserver.h>
#include <pfs.h>                /* assert, internal_error, prototypes */
#include <psrv.h>
#include <perrno.h>
#include <plog.h>
#include "gopher.h"


extern char hostwport[];

static char * unixerrstr(void);
static GLINK get_menu(char *host, int port, char *selector);
static void glinks_to_dir(GLINK glist, VDIR dir);
static void atput(VLINK vl, char *name, ...);


/* This function is passed an empty directory which has been initialized with
   vdir_init(). It returns a possibly populated directory and
   DSRDIR_NOT_A_DIRECTORY or PSUCCESS.  Note that the directory will need
   to have links within it freed, even if an error code is returned.  */
/* This sets p_warn_string and pwarn if an unrecognized gopher line is
   received.  That is good. */

int 
gopher_gw_dsdb(RREQ    req,           /* Request pointer (unused)           */
            char    *dirname,      /* Name of the directory                 */
            char    **componentsp, /* Next component of name (unused)       */
            TOKEN   *rcompp,       /* Additional components (untouched)     */
            VDIR    dir,           /* Directory to be filled in             */
            int     options,       /* Options to list command 
                                      (we check DSDB_VERIFY) */
            const   char *rattrib, /* Requested attributes  (ignored)       */
            FILTER filters)        /* Filters to be applied (ignored)       */
{
    int tmp;
    char *cp;
    static char *host = NULL;
    int port;
    char type;
    char *selectorcp;           /* a character pointer that might point to the
                                   first part of the selector or possibly the
                                   complete selector. */
    char *selector;             /* the final selector */
    GLINK glist;                /* list of gopher links */

    dir->version = -1;          /* no version of file info format */
    dir->inc_native = VDIN_PSEUDO;
    tmp = qsscanf(dirname, "GOPHER-GW/%&[^(](%d)/%c%r/%r",
              &host, &port, &type, &cp, &selectorcp);
    if (tmp < 4) return DSRDIR_NOT_A_DIRECTORY;

    if (type == '1') {
        if (tmp < 5) return DSRDIR_NOT_A_DIRECTORY;
        selector = selectorcp;
    } else if (type == '7') {
        VLINK r;
        if (tmp < 4) 
            return DSRDIR_NOT_A_DIRECTORY;
        else if (tmp == 4) {
            selectorcp = "";
        } /* if tmp = 5, selectorcp already set. */
        /* Now set the selector. */
        if (!*componentsp || strequal(*componentsp, "")
            || strequal(*componentsp, "*")) /* last test against * might be 
                                               OK  */
            return PSUCCESS;    /* Display no contents for the directory */
        /* Construct a new VLINK and return it. */
        /* This VLINK is a link to a directory that represents the final 
           query. */
        r = vlalloc();
        r->name = stcopy(*componentsp);
        r->host = stcopy(hostwport);
        r->target = stcopyr("DIRECTORY", r->target);
        r->hsoname = qsprintf_stcopyr(r->hsoname,
                                      "GOPHER-GW/%s(%d)/1/%s%s%s", host, port,
                                      selectorcp, (*selectorcp) ? "\t" : "",
                                      *componentsp);
        atput(r, "OBJECT-INTERPRETATION", "DIRECTORY", (char *) 0);
        vl_insert(r, dir, VLI_NOSORT);
        /* We just used up componentsp, so we'd better reset it. */
        if (*rcompp) {
            *componentsp = (*rcompp)->token;
            *rcompp = (*rcompp)->next;
        } else {
            *componentsp = NULL;
        }
        return PSUCCESS;
    } else {                    /* unknown type for gatewaying */
        return DSRDIR_NOT_A_DIRECTORY;
    }
    /* don't bother retrieving the contents if just verifying.  */
    if (options & DSDB_VERIFY) return PSUCCESS; 
    if((glist = get_menu(host, port, selector)) == NULL && perrno)
        return perrno;
    glinks_to_dir(glist, dir);
    gllfree(glist);
    return PSUCCESS;
}




static GLINK next_menu_item(FILE *fp);
#if 0
/* This experiment did not work.  The stdio library did not appear to correctly
   handle writing to a TCP stream that I was then reading from.  If you know 
   why this happened, please tell me. --swa@isi.edu */
static FILE *fopen_stream(char *host, int port);
#endif
static int open_tcp_stream(char host[], int port);

/* Returns NULL & sets perrno on error. */
static GLINK 
get_menu(char *host, int port, char *selector)
{
    GLINK retval = NULL;
    GLINK gl;
    FILE *fp;                    /* used for reading only.  Tried to use it for
                                    reading & writing without success. */
    int fd;
    int tmp;                    /* return value from write() and read()*/
    int nlinks = 0;             /* # of links in directory */
#if 0
    fp = fopen_stream(host, port);
#endif
    fd = open_tcp_stream(host, port);   /* Copied from vcache */
#if 0
    if (!fp) {
#endif
    if (fd < 0) {
        perrno = PFAILURE;   /* work on error reporting */
        return NULL;
    }
    tmp = write (fd, selector, strlen(selector));
    if (tmp != strlen(selector)) {
        close(fd);
        perrno = PFAILURE;
        return NULL;
    }
    tmp = write(fd, "\r\n", 2);
    if (tmp != 2) {
        close(fd);
        perrno = PFAILURE;
        return NULL;
    }
#if 0                           /* this did not work. */
    fprintf(fp, "%s\r\n", selector);
    fflush(fp);
#endif
    fp = fdopen(fd, "r");       /* open just for reading? */
    while(gl = next_menu_item(fp)) {
        if (++nlinks > MAX_NUM_LINKS) {
            perrno = DIRSRV_TOO_MANY;
            return NULL;
        }
        APPEND_ITEM(gl, retval);
    }
    fclose(fp);
    perrno = PSUCCESS;
    return retval;
}

#if 0
/* This client opens a TCP stream to the host/port & returns a FILE pointer. */
/* Returns NULL on failure; sets p_err_string */
FILE *
fopen_stream(char *host, int port)
{
    int fd = open_tcp_stream(host,port);
    if (fd < 0) return NULL;
    return fdopen(fd, "r+");
}
#endif

/* Open a TCP stream from here to the HOST at the PORT. */
/* Return socket descriptor on success, or -1 on failure. */
/* This is mostly swiped from user/vcache/gopherget.c */
static int
open_tcp_stream(host, port)
    char host[];
    int port;
{
    struct sockaddr_in server;  /* server side socket. */
    struct hostent *hp;         /* Remote host we're connecting to. */
    int s;                      /* the socket */

    if ((hp = gethostbyname(host)) == NULL) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, "%s: unknown host\n", host);
        return -1;
    }
    bzero((char *) &server, sizeof server);
    bcopy(hp->h_addr, (char *) &server.sin_addr, hp->h_length);
    server.sin_family = hp->h_addrtype;
    server.sin_port = htons(port);

    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, 
                 "Couldn't reach host %s: Call to socket(AF_INET, SOCK_STREAM,\
 0) failed: errno %d: %s", host, errno, unixerrstr());
        return -1;
    }
    /* Connect will do the bind for us!  Hooray!  That's a relief. */
    if (connect(s, (struct sockaddr *) &server, sizeof server)) {
        qsprintf(p_err_string, P_ERR_STRING_SZ, 
                 "Couldn't reach host %s: Call to connect() failed: errno %d: \
%s", host, errno, unixerrstr());
        return -1;
    }
    return s;
}


/* Reads the next valid menu item from the stream FP.  Returns NULL if 
   no more, or if more than 3 erroneous lines go by. If a line is sent
   longer than 1024 characters, it is definitely illegal according to the
   Gopher spec, which requests no more than 255 chars for the selector and 80
   for the description. */
/* Need to work on the error reporting. */
static GLINK 
next_menu_item(FILE *fp)
{
    GLINK r;
    char buf[1024];
    int buflen;                 /* # of characters in this buffer. */
    int nbad = 0;


 again:
    if (nbad > 3) return NULL;
        
    if(!fgets(buf, sizeof buf, fp)) return NULL;
    buflen = strlen(buf);
    if (sizeof buf - 1 == buflen) { /* line too long; missed CRLF */
        ++nbad;
        goto again;
    }
    r = glalloc();        
    assert(buf[buflen - 1] == '\n');
    buf[buflen - 1] = '\0';
    if (buf[buflen - 2] == '\r') buf[buflen - 2] = '\0';
    /* Check for end of menu */
    if (strequal(buf, ".")) {
        glfree(r);
        return NULL; /* the end has been spotted. */
    }
#if 0
    if (qsscanf(buf, "%c%&[^\t]%*1[\t]%&(^\t)%*1[\t]%&[^\t]%*1[\t]%d",
                &r->type, &r->name, &r->selector, &r->host, &r->port) < 5) {
    /* Note that %&( is not yet implemented, so we're doing it this way
       instead. */
#else
    if ((qsscanf(buf, "%c%&[^\t]%*1[\t]%&[^\t]%*1[\t]%&[^\t]%*1[\t]%d",
                 &r->type, &r->name, &r->selector, &r->host, &r->port) < 5)
        && ((r->selector = stcopyr("", r->selector)) , 
            (qsscanf(buf, "%c%&[^\t]%*1[\t]%*1[\t]%&[^\t]%*1[\t]%d",
                     &r->type, &r->name, &r->host, &r->port) < 4))) {
#endif
        ++nbad;
        qsprintf(p_warn_string, P_ERR_STRING_SZ, 
                 "Encountered unrecognized Gopher message: %s",  buf);
        pwarn = PWARNING;
        glfree(r);
        goto again;
    }
    nbad = 0;
    r->protocol_mesg = stcopy(buf);
    return r;
}

static VLINK gltovl(GLINK gl);
static void add_collation_order(int linknum, VLINK vl);
static long generate_magic(VLINK vl);
static int magic_no_in_list(long magic, VLINK links);

/* It is ok to modify the GLINKs, as long as glist is left at the head of a
   list of stuff to be freed with gllfree() by the caller of this function. 

   A lot of the hair in this function is here because we must make sure that 
   Gopher replicas are returned as Prospero replicas.  For now, the only way to
   make sure two objects are treated as replicas is for them to be links with
   the same positive ID REMOTE value. */
static void    
glinks_to_dir(GLINK glist, VDIR dir)
{
    int menu_entry_number = 0;             /* # of menu entries read */
    int replica_list = 0;       /* nonzero if handling replica list */
    long current_magic_no = 0L; /* set magic starting hash */
    VLINK vl = NULL;                   /* last vlink processed */
    GLINK gl;                   /* index */
    for (gl = glist ; gl; gl = gl->next) {
        /* If we found this link is a replica, put a magic # on the head of the
           list. */
        if (!replica_list && gl->type == '+') {
            if (!vl) continue;  /* no previous gopher link for this to be a
                                   replica of; perhaps we could not convert the
                                   gopher link to a virtual link. */
            if (!current_magic_no) current_magic_no = generate_magic(vl);
            while (magic_no_in_list(++current_magic_no, dir->links)) {
                /* Check for numeric overflow */
                if (current_magic_no <= 0) current_magic_no = 1;
            }
            vl->f_magic_no = ++current_magic_no; /* new magic no */
        }
        if (gl->type != '+') 
            replica_list = 0;
        else {
            assert(gl->previous);
            gl->type = gl->previous->type; /* gets type from head of list. */
        }
        if((vl = gltovl(gl)) == NULL) {
            replica_list = 0;   /* break the chain of replicas */
            continue; /* can't convert it */
        }
        if (replica_list) {
            vl->f_magic_no = current_magic_no;
            add_collation_order(menu_entry_number, vl);
        } else {
            add_collation_order(++menu_entry_number, vl);
        }
        vl_insert(vl, dir, VLI_NOSORT);
    }
}


/* Returns NULL if couldn't convert the gopher link GL to a Prospero link VL.
   Otherwise returns a new VLINK. */ 
static VLINK 
gltovl(GLINK gl)
{
    VLINK r = vlalloc();
    PATTRIB at;
   /* check if something is an AFTP link. If so, handle it appropriately.*/
    /* add AFTP Access method */
    static void ftp_check_am(GLINK gl, VLINK r, char *textorbin);
    static void ftp_check_dir(GLINK gl, VLINK r); /* rewrite vlink to query
                                                     archie directly. */
    static void zap_trailing_spaces(char *host);
    
    r->name = gl->name; gl->name = NULL;
    atput(r, "GOPHER-MENU-ITEM", gl->protocol_mesg, (char *) 0);

    switch(gl->type) {
    case '0':                   /* Text files */
    case 'c':                   /* Calendar Text files (not in RFC1436) */
    case 'e':                   /* Event Text files (not in RFC1436) */
        /* This comes first because, in the current client implementation,
           the server is expected to return access methods in what it believes
           the preferred order is.  We prefer direct AFTP connections to
           overloading the intermediary Gopher gateways. */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "TEXT");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "TEXT", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "DOCUMENT", "TEXT", "ASCII",
              (char *) 0);
        break;
    case '1':                   /* Directories */
        r->host = stcopy(hostwport);
        r->target = stcopyr("DIRECTORY", r->target);
        r->hsoname = qsprintf_stcopyr(r->hsoname,
                                      "GOPHER-GW/%s(%d)/%c/%s", gl->host,
                                      gl->port, gl->type, gl->selector);
        atput(r, "OBJECT-INTERPRETATION", "DIRECTORY", (char *) 0);
        ftp_check_dir(gl, r);
        break;
    case '4':                   /* Macintosh BinHex (.hqx) text file */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "TEXT");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "TEXT", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "EMBEDDED", "BINHEX", "DATA",
              (char *) 0);
        break;
    case '5':                   /* PC-Dos binary files. */
    case '9':                   /* Generic binary files */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "BINARY");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "BINARY",
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "DATA", (char *) 0);
        break;
    case '6':                   /* UNIX UUencoded file. */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "TEXT");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "TEXT", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "EMBEDDED", "UUENCODE", "DATA",
              (char *) 0);
        break;
    case '7':                   /* Gopher searches */
        r->target = stcopyr("DIRECTORY", r->target);
        r->host = stcopy(hostwport);
        r->hsoname = 
            qsprintf_stcopyr(r->hsoname, "GOPHER-GW/%s(%d)/7/%s", 
                             gl->host, gl->port, gl->selector);
        atput(r, "OBJECT-INTERPRETATION", "SEARCH", (char *) 0);
        atput(r, "QUERY-METHOD", "gopher-query(search-words)", 
              "${search-words}", "", (char *) 0);
        atput(r, "QUERY-ARGUMENT", "search-words", 
              "Index word(s) to search for", "mandatory char*", "%s", "",
              (char *) 0);
        atput(r, "QUERY-DOCUMENTATION", "gopher-query()", "This is a Gopher \
protocol query exported through Prospero.  Since it is not a native Prospero \
query, we have very little documentation available for you.", (char *) 0);
        atput(r, "QUERY-DOCUMENTATION", "gopher-string", "This string says \
what you're searching for or what information you're specifying.", 
              "Type any string.  Sometimes SPACEs are treated as implied \
'and' operators.  Sometimes the words 'and', 'or', and 'not' are recognized \
as boolean operators.", (char *) 0);
        break;
    case '8': {                  /* TELNET session */
        char *m = "";                /* message */
        r->target = stcopyr("EXTERNAL", r->target);
        if (*(gl->selector))
            m = qsprintf_stcopyr((char *) NULL, 
                                 "Use the account name \"%s\" to log in",
                                 gl->selector);
        atput(r, "ACCESS-METHOD", "TELNET", "", "", "", "", m, (char *) 0);
        zap_trailing_spaces(gl->host);
        /* Gopher uses a telnet port of 0 to mean the default telnet port. */
        if (gl->port == 0) {
            r->host = gl->host; gl->host = NULL;
        } else {
            r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        }
        atput(r, "OBJECT-INTERPRETATION", "PORTAL", (char *) 0);
        if (*m) stfree(m);
        break;
    }
    case 'I':                   /* Generic Image. */
    case ':':                   /* Gopher+ bitmap image type; not in RFC1436 */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "BINARY");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "BINARY", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "IMAGE", (char *) 0);
        break;
    case 'M':                   /* MIME.  Not in RFC1436. */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "TEXT");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "TEXT", (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "DOCUMENT", "MIME", (char *) 0);
        break;

    case 'T': {                  /* TN3270 session */
        char *m = "";                /* message */
        r->target = stcopyr("EXTERNAL", r->target);
        zap_trailing_spaces(gl->host);
        /* Gopher uses a telnet port of 0 to mean the default telnet port. */
        if (gl->port == 0) {
            r->host = gl->host; gl->host = NULL;
        } else {
            r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        }
        if (*(gl->selector))
            m = qsprintf_stcopyr((char *) NULL, 
                                 "Use the account name \"%s\" to log in",
                                 gl->selector);
        atput(r, "ACCESS-METHOD", "TN3270", "", "", "", "", m, (char *) 0);
        atput(r, "OBJECT-INTERPRETATION", "PORTAL", (char *) 0);
        if (*m) stfree(m);
        break;
    }
    case 'g':                   /* Gif */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "BINARY");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "BINARY", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "IMAGE", "GIF", (char *) 0);
        break;
    case 's':                   /* SOUND.  Not in RFC1436. */
    case '<':                   /* Gopher+ sound type; not in RFC1436 */
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "BINARY");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "BINARY", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "SOUND", (char *) 0);
        break;
    case 'i':                   /* the uncommon i type exists in the University
                                   of Iowa gopher variant, Panda.  */
        r->name = gl->name; gl->name = NULL;
        r->target = stcopyr("NULL", r->target);
        r->hosttype = stcopyr("NULL", r->hosttype);
        r->host = stcopyr("NULL", r->host);
        r->hsonametype = stcopyr("NULL", r->hsonametype);
        r->hsoname = stcopyr("NULL", r->hsoname);
        r->target = stcopyr("NULL", r->target);
        atput(r, "OBJECT-INTERPRETATION", "VOID", (char *) 0);
        break;
    case ';':                   /* Gopher+ MOVIE type.   It is not at all clear
                                   whether this is being used yet.*/
        r->target = stcopyr("EXTERNAL", r->target);
        ftp_check_am(gl, r, "BINARY");
        atput(r, "ACCESS-METHOD", "GOPHER", "", "", "", "", "BINARY", 
              (char *) 0);
        r->host = qsprintf_stcopyr(r->host, "%s(%d)", gl->host, gl->port);
        r->hsoname = gl->selector; gl->selector = NULL;
        atput(r, "OBJECT-INTERPRETATION", "VIDEO", (char *) 0);
        break;
    case '.':                   /* End of directory. */
        break;                  /* go on */
    case '2':                   /* CSO nameserver; Gateway in progress */
    case '3':                   /* unclear how to handle the error type. */
    case 'w':                   /* whois server */ 
    case '-':                   /* error message type? */
    default:                    /* unknown type or type we can't process */
        qsprintf(p_warn_string, P_ERR_STRING_SZ, 
                 "Encountered unrecognized Gopher message: %s", 
                 gl->protocol_mesg);
        pwarn = PWARNING;
        vlfree(r);
        r = NULL;
        break;
    }
    return r;
}


/* Check if we're an AFTP link. If so, add an AFTP access method. */
/* This check should be made before other access methods are added.
   The server is expected to return access methods in what it believes the
   preferred order is.  We prefer direct AFTP connections to overloading the
   intermediary Gopher gateways. */ 
/* Gopher AFTP links to files use the HSONAME format:
   ftp:<aftp-hostname>@/file-path (no trailing slash) */
static void 
ftp_check_am(GLINK gl, VLINK r, char *textorbin)
{
    static char *ftp_hostname = NULL;
    char *ftp_rest;             /* rest of string */
    if (qsscanf(gl->selector, "ftp:%&[^@]@%r", &ftp_hostname, &ftp_rest) == 2) {
        int len = strlen(ftp_rest);
        if (ftp_rest[len - 1] != '/')
            atput(r, "ACCESS-METHOD", "AFTP", "", ftp_hostname, "", ftp_rest,
                  textorbin, (char *) 0);
    }
}
    
/* Check if we're an AFTP link. If so, rewrite the HOST and HSONAME to go
   through the Prospero server at ARCHIE_SERVER (this can be changed). */
/* Gopher AFTP links to directories use the HSONAME format:
   ftp:<aftp-hostname>@/directory.../ */
static void
ftp_check_dir(GLINK gl, VLINK r)
{
    static char *ftp_hostname = NULL;
    int len;
    char *ftp_rest;             /* rest of string */
    if (qsscanf(gl->selector, "ftp:%&[^@]@%r", &ftp_hostname, &ftp_rest) == 2
        && *ftp_rest == '/') {
        int len = strlen(ftp_rest);
        if (ftp_rest[len - 1] == '/')  {
            ftp_rest[len - 1] = '\0';
            r->host = stcopyr(ARCHIE_SERVER, r->host);
            r->hsoname = qsprintf_stcopyr(r->hsoname, "ARCHIE/HOST/%s%s",
                                          ftp_hostname, ftp_rest);
        }
    }
}

static void 
zap_trailing_spaces(char *host)
{
    int len = strlen(host);
    while (--len >= 0) {
        if (host[len] == ' ') host[len] = '\0';
        else break;
    }
}

/* Add a collation-order attribute of the form NUMERIC <linknum> to the link
   VL */ 
static void 
add_collation_order(int linknum, VLINK vl)
{
    PATTRIB ca = atalloc();
    char buf[40];                /* long enough to hold the ASCII
                                    representation of any int for the
                                    foreseeable future. */ 
    ca->precedence = ATR_PREC_LINK;
    ca->nature = ATR_NATURE_APPLICATION;
    ca->avtype = ATR_SEQUENCE;
    ca->aname = stcopy("COLLATION-ORDER");
    ca->value.sequence = tkappend("NUMERIC", ca->value.sequence);
    assert(qsprintf(buf, sizeof buf, "%d", linknum) <= sizeof buf);
    ca->value.sequence = tkappend(buf, ca->value.sequence);
    APPEND_ITEM(ca, vl->lattrib);
}

/* Generate a unique magic number (positive long).  This is based upon hashing
   the vlink VL's HSONAME field. */
long
generate_magic(VLINK vl)
{
    long retval = 0L;
    int n = stsize(vl->hsoname);
    int rvoffset = 0;           /* offset for xoring with the return value. */
    while(n--) {
        rvoffset += 8;
        if (rvoffset >= 8 * sizeof retval) rvoffset = 0;
        retval ^= ((vl->hsoname[n]) << rvoffset);
    }
    if (retval < 0) return ~retval;
    if (retval == 0) return 1;
    return retval;
}

/* Is the magic number MAGIC anywhere in use in the list of vlinks LINKS? */
static int
magic_no_in_list(long magic, VLINK links)
{
    for ( ;links ; links = links->next) {
        if (links->f_magic_no == magic) return TRUE;
    }
    return FALSE;
}


static char *
unixerrstr(void)
{
    extern int sys_nerr, errno;
    extern char *sys_errlist[];
    
    return errno < sys_nerr ? sys_errlist[errno] : "Unprintable Error";
}


static void atput(VLINK vl, char *name, ...)
{
    va_list ap;
    char *s;
    PATTRIB at = atalloc();

    va_start(ap, name);
    at->aname = stcopy(name);
    at->avtype = ATR_SEQUENCE;
    if (strequal(name, "ACCESS-METHOD"))
        at->nature = ATR_NATURE_FIELD;
    else
        at->nature = ATR_NATURE_APPLICATION;
    if (strequal(vl->target, "EXTERNAL"))
        at->precedence = ATR_PREC_REPLACE; /* still not clear what to use */ 
    else
        at->precedence = ATR_PREC_OBJECT;
    while(s = va_arg(ap, char *)) {
        at->value.sequence = 
            tkappend(s, at->value.sequence);
    }
    APPEND_ITEM(at, vl->lattrib);
    va_end(ap);
}


