/*
 * Copyright (c) 1991       by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 *
 * Written  by bcn 1991     as transmit in rdgram.c (Prospero)
 * Modified by bcn 1/93     modularized and incorporated into new ardp library
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdio.h>
#include <ardp.h>
#include <pmachine.h>           /* for bcopy() */
#include <plog.h>

extern RREQ	ardp_doneQ;
extern RREQ	ardp_runQ;
extern int	ardp_srvport;
extern int	ardp_prvport;
extern int	dQlen;
extern int	dQmaxlen;

/*
 * ardp_respond - respond to request
 *
 *    ardp_respond takes a request block whose ->outpkt element is a pointer
 *    to a list of packets that have not yet been returned.  It moves the
 *    packets to the transmission queue of the request (->trns) and assigns a
 *    sequence number to each packet.  If options is ARDP_R_COMPLETE, it
 *    also sets the ->trns_tot field of the request indicating the total 
 *    number of packets in the response.   if the ARDP_R_NOSEND option is 
 *    not specified, the current packet is also sent to the client.
 *    
 *    ardp_respond() expects the text of the packet to have a trailing null
 *    at the end of it, which is transmitted for safety's sake.  
 *    (noted: swa, 8/5/93).  This explains some bugs...
 */
ardp_respond(req, options)
    RREQ		req;
    int			options;
{
    char 		buf[ARDP_PTXT_DSZ];
    int			sent;
    int			retval = ARDP_SUCCESS; /* Return value */
    unsigned short	cid = htons(req->cid);
    unsigned short	nseq = 0;
    unsigned short	ntotal = 0;
    unsigned short	rcvdthru = 0;
    unsigned short	bkoff = 0;
    PTEXT			tpkt; /* Temporary pkt pointer */

    *buf = '\0';

    if(req->ardp_version < 0) cid = 0;

    if(req->status == ARDP_STATUS_FREE) {
	fprintf(stderr,"Attempt to respond to free RREQ\n");
	abort();
	return(ARDP_BAD_REQ);
    }

    if(!req->outpkt) return(ARDP_BAD_REQ);

 more_to_process:

    if(!req->trns) req->outpkt->seq = 1;
    else req->outpkt->seq = req->trns->previous->seq + 1;

    if((options & ARDP_R_COMPLETE) && (req->outpkt->next == NOPKT))
	req->trns_tot = req->outpkt->seq;
    
    nseq = htons((u_short) req->outpkt->seq);
    ntotal = htons((u_short) req->trns_tot);

    if((req->ardp_version < 0)&&(req->ardp_version != -2)) {
	if(req->trns_tot) sprintf(buf, "MULTI-PACKET %d OF %d",
				  req->outpkt->seq, req->trns_tot);
	else sprintf(buf,"MULTI-PACKET %d\n",req->outpkt->seq);
    }

    /* Note that in the following, we don't have to make sure  */
    /* there is room for the header in the packet because we   */
    /* are the only one that moves back the start, and ptalloc */
    /* allocates space for us in all packets it creates        */
    
    /* If a single message and no connection ID to return, */
    /* and no pending server requested wait pending then   */
    /* we can leave the control fields out                 */
    if((req->trns_tot == 1) && (req->svc_rwait == 0)) {
	/* Once V4 clients gone, the next if condition can be removed */
	if((req->ardp_version > -1)||(req->ardp_version == -2)) {
	    if(req->cid == 0) {
		req->outpkt->start -= 1;
		req->outpkt->length += 1;
		*req->outpkt->start = (char) 1;
	    }
	    else {
		req->outpkt->start -= 3;
		req->outpkt->length += 3;
		*req->outpkt->start = (char) 3;
		bcopy(&cid,req->outpkt->start+1,2);     /* Conn ID */
	    }
	}
    }
    /* Fill in the control fields */
    else {	    
	/* if we have to clear a service requested wait */
	if((req->svc_rwait > 5) || (req->svc_rwait_seq > req->prcvd_thru)) {
	    req->outpkt->start -= 12;
	    req->outpkt->length += 12;
	    *req->outpkt->start = (char) 12;

	    rcvdthru = htons((u_short) req->rcvd_thru);
	    bcopy(&rcvdthru,req->outpkt->start+7,2); /* Recvd Through   */

	    /* XXX Note that when all clients V5, new wait will be 0    */
	    /* Assume we should cancel the wait and remember when reset */
	    if(req->svc_rwait > 5) {
		req->svc_rwait = 5;
		req->svc_rwait_seq = req->outpkt->seq;
		req->retries_rem = 4;
		req->wait_till.tv_sec = time(NULL) + req->timeout_adj.tv_sec;
	    }
	    bkoff = htons((u_short) req->svc_rwait);
	    bcopy(&bkoff,req->outpkt->start+9,2);     /* New ttwait  */
	    *(req->outpkt->start+11) = 0x80;          /* Request ack */
	}
	else {
	    req->outpkt->start -= 7;
	    req->outpkt->length += 7;
	    *req->outpkt->start = (char) 7;
	}

	bcopy(&cid,req->outpkt->start+1,2);     /* Conn ID */
	bcopy(&nseq,req->outpkt->start+3,2);     /* Pkt no  */
	bcopy(&ntotal,req->outpkt->start+5,2);   /* Total   */
    }	

    /* Make room for the trailing null */
    req->outpkt->length += 1;

    /* Only send if packet not yet received */
    /* And send set a window of ARDP_WINDOW_SZ packets  */
    if((!(options & ARDP_R_NOSEND)) && 
       (req->outpkt->seq <= (req->prcvd_thru + ARDP_WINDOW_SZ)) &&
       (req->outpkt->seq > req->prcvd_thru)) {
	retval = ardp_snd_pkt(req->outpkt,req);
    }

    /* Add packet to req->trns */
    tpkt = req->outpkt;
    EXTRACT_ITEM(tpkt,req->outpkt);
    APPEND_ITEM(tpkt,req->trns);

    if(req->outpkt) goto more_to_process;

    /* If complete then add request structure to done  */
    /* queue in case we have to retry.                 */
    if(options & ARDP_R_COMPLETE) {
	/* Request has been processed, here you can accumulate */
	/* statistics on system time or service time           */
	plog(L_QUEUE_COMP, req, "Requested service completed"); 

	if(ardp_runQ == req) ardp_runQ = NULL;
	if((req->cid == 0) || (dQmaxlen <= 0)) {
	    req->outpkt = NULL;
	    ardp_rqfree(req);
	}
	else if(!ardp_doneQ) {
	    ardp_doneQ = req; 
	    ardp_doneQ->previous = req;
	    req->next = NULL; 
	    dQlen++;
	}
	else if(dQlen < dQmaxlen) {
	    /* Add to start */
	    req->previous = ardp_doneQ->previous;
	    req->next = ardp_doneQ;
	    ardp_doneQ->previous = req;
	    ardp_doneQ = req;
	    dQlen++;
	}
	else {
	    /* Add to start */
	    req->previous = ardp_doneQ->previous;
	    req->next = ardp_doneQ;
	    ardp_doneQ->previous = req;
	    ardp_doneQ = req;
	    /* If request to remove not acknowldged, log the fact */
	    if(ardp_doneQ->previous->svc_rwait_seq > 
	       ardp_doneQ->previous->prcvd_thru) {
		plog(L_QUEUE_INFO,ardp_doneQ->previous,
		     "Requested ack never received (%d of %d/%d ack)",
		     ardp_doneQ->previous->prcvd_thru, 
		     ardp_doneQ->previous->svc_rwait_seq, 
		     ardp_doneQ->previous->trns_tot, 0);
	    }
	    /* Remove from end - invariant >= 2 in queue here */
	    ardp_doneQ->previous = ardp_doneQ->previous->previous;
	    ardp_rqfree(ardp_doneQ->previous->next);
	    ardp_doneQ->previous->next = NULL;
	}
    }
    return(retval);
}

/*
 * ardp_rwait - sends a message to a client requesting
 *   that it wait for service.  This message indicates that
 *   there may be a delay before the request can be processed.
 *   The recipient should not attempt any retries until the
 *   time specified in this message has elapsed.  Depending on the
 *   protocol version in use by the client, the additional
 *   queue position and system time arguments may be returned.
 */
ardp_rwait(RREQ		req,        /* Request to which this is a reply  */
	   int		timetowait, /* How long client should wait       */
	   short	qpos,       /* Queue position                    */
	   int		stime)      /* System time                       */
{
    PTEXT	r = ardp_ptalloc();
    short	cid = htons(req->cid);
    short	nseq = 0;
    short	zero = 0;
    short	backoff;
    short	stmp;
    int		ltmp;
    int		tmp;
    
    if(req->ardp_version < 0) cid = 0;

    /* Remember that we sent a server requested wait and assume  */
    /* it took effect, though request will be unsequenced        */
    req->svc_rwait = timetowait;
    req->svc_rwait_seq = req->prcvd_thru;

    *r->start = (char) 11;
    r->length = 11;

    backoff = htons((u_short) timetowait);

    bcopy(&cid,r->start+1,2);		/* Connection ID     */
    bcopy(&nseq,r->start+3,2);		/* Packet number     */
    bcopy(&zero,r->start+5,2);		/* Total packets     */
    bcopy(&zero,r->start+7,2);		/* Received through  */
    bcopy(&backoff,r->start+9,2);	/* Backoff           */
    
    if((req->ardp_version >= 0) && (qpos || stime)) {
	r->length = 14;
	bzero(r->start+11,3); 
	*(r->start+12) = (unsigned char) 254;
	if(qpos) *(r->start+13) |= (unsigned char) 1;
	if(stime) *(r->start+13) |= (unsigned char) 2;
	if(qpos) {
	    stmp = htons(qpos);
	    bcopy(&stmp,r->start+r->length,2);
	    r->length += 2;
	}
	if(stime) {
	    ltmp = htonl(stime);
	    bcopy(&ltmp,r->start+r->length,4);
	    r->length += 4;
	    
	}
	*r->start = (char) r->length;
    }

    tmp = ardp_snd_pkt(r,req);
    ardp_ptfree(r);
    return(tmp);
}

/*
 * ardp_refuse - sends a message to the recipient indicating
 *   that its connection attempt has been refused.
 */
ardp_refuse(RREQ req)
{
    PTEXT	r;
    short	cid = htons(req->cid);
    short	zero = 0;
    int		tmp;
    
    /* If old version won't understand refused */
    if(req->ardp_version < 0) return(ARDP_FAILURE);

    r = ardp_ptalloc();
    *r->start = (char) 13;
    r->length = 13;

    bcopy(&cid,r->start+1,2);		/* Connection ID     */
    bcopy(&zero,r->start+3,2);		/* Packet number     */
    bcopy(&zero,r->start+5,2);		/* Total packets     */
    bcopy(&zero,r->start+7,2);		/* Received through  */
    bcopy(&zero,r->start+9,2);		/* Backoff           */
    bzero(r->start+11,2);		/* Flags             */
    *(r->start + 12) = (unsigned char) 1; /* Refused         */
    
    tmp = ardp_snd_pkt(r,req);
    ardp_ptfree(r);
    return(tmp);
}


/*
 * ardp_redirect - sends a message to the recipient indicating that
 * all unacknowledged packets should be sent to a new target destination.
 * The target specifies the host address and the UDP port in network
 * byte order.  For now, redirections should only occur before any
 * request packets have been acknowledged, or response packets sent.
 */
ardp_redirect(RREQ 		     req,     /* Request to be redirected */
	      struct sockaddr_in    *target)  /* Address of new server    */
{
    PTEXT	r;
    short	cid = htons(req->cid);
    int		tmp;

    /* Old versions don't support redirection */
    if(req->ardp_version < 0) return(ARDP_FAILURE);

    r = ardp_ptalloc();

    *r->start = (char) 19;
    r->length = 19;

    bcopy(&cid,r->start+1,2);  	/* Connection ID                     */
    bzero(r->start+3,9);	/* Pktno, total, rcvd, swait, flags1 */
    *(r->start + 12) = '\004';  /* Redirect option                   */
    bcopy(&(target->sin_addr),r->start+13,4);
    bcopy(&(target->sin_port),r->start+17,2);

    tmp = ardp_snd_pkt(r,req);
    ardp_ptfree(r);
    return(tmp);
}


/*
 * ardp_ack - sends an acknowledgment message to the client indicating
 * that all packets through req->rcvd_thru have been recived.  It is
 * only called on receipt of a multi-packet server request.
 */
ardp_acknowledge(RREQ	req)        /* Request to which this is a reply  */
{
    PTEXT	r = ardp_ptalloc();
    short	cid = htons(req->cid);
    short	rcvd = htons(req->rcvd_thru);
    short	zero = 0;
    int		tmp;
    
    *r->start = (char) 9;
    r->length = 9;

    bcopy(&cid,r->start+1,2);		/* Connection ID     */
    bcopy(&zero,r->start+3,2);		/* Packet number     */
    bcopy(&zero,r->start+5,2);		/* Total packets     */
    bcopy(&rcvd,r->start+7,2);		/* Received through  */

    tmp = ardp_snd_pkt(r,req);
    ardp_ptfree(r);
    return(tmp);
}


