/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1992 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdio.h>
#include <strings.h>

#include <ardp.h>
#include <pfs.h>
#include <plog.h>
#include <psrv.h>
#include <perrno.h>
#include <pmachine.h>

#include "dirsrv.h"

/*
 * Modified 6/4/93, swa: It now writes out the child directory first.
 * It only writes out the changed parent directory if the child directory
 * was successfully written.  This means that if we have trouble writing the
 * child directory, we don't end up with a parent directory containing
 * a link to a nonexistent object.
 */

ACL aclcopy(ACL acl);
static VLINK
gensym_child(char client_dir[], char child_linkname[], 
             char child_hsoname[], size_t child_hsonamesz);

/* This was create_directory */
create_object(RREQ req, char **commandp, char **next_wordp, INPUT in, 
              char client_dir[], int dir_magic_no)
{
    char	t_options[ARDP_PTXT_LEN_R]; /* options to command. */
    char 	child_linkname[ARDP_PTXT_LEN_R]; /* link name in parent. */
    char 	child_hsoname[ARDP_PTXT_LEN_R]; /* HSONAME for the child. */
    VLINK	clink;          /* For the child directory. */
    int         tmp;
    int		lpriv;          /* LPRIV option  */
    int		retval;
    VDIR_ST	dir_st;         /* Parent directory contents. */
    VDIR	dir = &dir_st;
    VDIR_ST	new_dir_st;     /* Child directory contents. */
    VDIR	new_dir = &new_dir_st;

    /* still have to read the remainder of the attributes */
    tmp = qsscanf(*next_wordp,"%'s %'s",
                 t_options,child_linkname);

    /* Log and return a better message */
    if(tmp < 2) {
        creply(req,"ERROR too few arguments");
        plog(L_DIR_PERR,req,"Too few arguments: %s",
             *commandp, 0);
        return PFAILURE;
    }

    /* For now, VIRTUAL and DIRECTORY options must be specified */
    if(!sindex(t_options, "VIRTUAL")  
       || !sindex(t_options, "DIRECTORY")) {
        creply(req,"ERROR only VIRTUAL directories implemented\n");
        plog(L_DIR_PERR,req,"Tried to create non-VIRTUAL or
non-directory object: %'s",
             *commandp, 0);
        return PFAILURE;
    }

    if(sindex(t_options,"LPRIV")) lpriv = 1;
    else lpriv = 0;

    plog(L_DIR_UPDATE,req,"%s", *commandp, 0);
    
    vdir_init(dir);

    retval = dsrdir(client_dir,dir_magic_no,dir,NULL,0);
    if(retval == DSRFINFO_FORWARDED) 
        return dforwarded(req, client_dir, dir_magic_no, dir);

    /* If not a directory, say so */
    if(retval == DSRDIR_NOT_A_DIRECTORY) {
        creply(req,"FAILURE NOT-A-DIRECTORY\n");
        plog(L_DIR_ERR,req,"Invalid directory name: %'s", client_dir,0);
        return PFAILURE;
    }

    /* If not authorized, say so */
    if(!srv_check_acl(dir->dacl,NULL,req,"I",SCA_DIRECTORY,client_dir,NULL)) {
        creply(req,"FAILURE NOT-AUTHORIZED\n");
        plog(L_AUTH_ERR,req,"Unauthorized CREATE-DIRECTORY: %s %s",client_dir,
             child_linkname); 
        /* Free the directory links */
        vdir_freelinks(dir);
        return PFAILURE;
    }

    clink = gensym_child(client_dir, child_linkname, child_hsoname, sizeof child_hsoname);
    if (!clink) {
        creply(req, "FAILURE SERVER-FAILED Could not generate a unique \
hsoname for the new directory.\n");
        vdir_freelinks(dir);
        return PFAILURE;
    }
    retval = vl_insert(clink,dir,VLI_NOCONFLICT);
    if(retval == VL_INSERT_ALREADY_THERE) {
        creplyf(req,"FAILURE ALREADY-EXISTS %'s\n",clink->name);
        /* Free the directory links */
        vdir_freelinks(dir);
        return PFAILURE;
    }
    else if(retval == VL_INSERT_CONFLICT) {
        creplyf(req,"FAILURE NAME-CONFLICT %'s\n",clink->name);
        plog(L_DIR_ERR,req,"Conflicting link already exists: %s %s",client_dir,
             clink->name,0);
        /* Free the directory links */
        vdir_freelinks(dir);
        return PFAILURE;
    }
    
    /* Don't write out the parent yet!  */

    /* Initialize the new directory.  The child will get a copy of the parent's
       ACL. */
    vdir_init(new_dir);
    new_dir->inc_native = VDIN_NONATIVE;

    /* Add creator to the ACL */
    if(!lpriv || (!srv_check_acl(dir->dacl,NULL,req,"BIlr",
				 SCA_DIRECTORY,client_dir,NULL))) {
	/* need to copy since EACL_ADD may change it */
	new_dir->dacl = aclcopy(dir->dacl); 

	if(lpriv) srv_add_client_to_acl("AIlr",req,&(new_dir->dacl),
					EACL_DIRECTORY);
	else srv_add_client_to_acl("ALRWDE",req,&(new_dir->dacl),
				   EACL_DIRECTORY);
    }
    else {
        new_dir->dacl = dir->dacl;
    }
    if(!retval) retval = dswdir(child_hsoname,new_dir);
    /* Ok, now we can write out the parent. */
    if(!retval) retval = dswdir(client_dir,dir);

    /* Free the entries */
    if (dir->dacl == new_dir->dacl) new_dir->dacl = NULL; /* don't free twice*/
    vdir_freelinks(dir);
    vdir_freelinks(new_dir);

    /* if successful say so (need to clean this up) */
    if(!retval) reply(req,"SUCCESS\n");
    else creply(req,"FAILURE\n");
    return retval;
}

/* Gensym a vlink for the child directory. */
static VLINK
gensym_child(char client_dir[], char child_linkname[], char child_hsoname[], 
             size_t child_hsonamesz)
{
    VDIR_ST tmp_dir_st;         /* scratch directory */
    VDIR tmp_dir = &tmp_dir_st; /* scratch directory */
    int retval;                 /* value returned by dsrdir() */
    /* Allocate a link for the child directory. */
    VLINK clink = vlalloc();
    clink->name = stcopyr(child_linkname,clink->name);
    clink->target = stcopyr("DIRECTORY",clink->target);
    clink->host = stcopyr(hostwport,clink->host);

    /* come up with an unused hsoname for the new directory. */
    strcpy(child_hsoname,client_dir);
    strcat(child_hsoname,"/");
    strcat(child_hsoname,child_linkname);

    /* Try first a name unadorned with a :# */
    vdir_init(tmp_dir);
    retval = dsrdir(child_hsoname, 0L, tmp_dir, (VLINK) NULL, 0);
    if (retval == PSUCCESS) {
        char template[MAXPATHLEN];
        int i;
        
        strcpy(template, child_hsoname);
        strcat(template, ":%d");
        i = 1;
        for (;;) {
            qsprintf(child_hsoname, child_hsonamesz, template, i);
            vdir_freelinks(tmp_dir);
            retval = dsrdir(child_hsoname, 0L, tmp_dir, (VLINK) NULL, 0);
            if (retval != PSUCCESS)
                /* Either child_hsoname now has an unused name or reading that
                   directory failed for some reason we can't do much about */
                break;          
            ++i;
        }
    }
    vdir_freelinks(tmp_dir);
    if (retval != DSRDIR_NOT_A_DIRECTORY) {
        /* dsrdir() failed for some reason other than because child_hsoname was
           unused.  We can't do much about it here if it wasn't caught when
           reading the parent. */
        vlfree(clink);
        return NULL;
    }
    clink->hsoname = stcopyr(child_hsoname,clink->hsoname);
    return clink;
}

