/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>
#include <stdio.h>

#include <pfs.h>

static VLINK	lfree = NULL;
int		vlink_count = 0;
int		vlink_max = 0;

/*
 * vlalloc - allocate and initialize vlink structure
 *
 *    VLALLOC returns a pointer to an initialized structure of type
 *    VLINK.  If it is unable to allocate such a structure, it
 *    signals out_of_memory().
 */
VLINK
vlalloc()
    {
	VLINK	vl;
	if(lfree) {
	    vl = lfree;
	    lfree = lfree->next;
	}
	else {
	    vl = (VLINK) malloc(sizeof(VLINK_ST));
	    if (!vl) out_of_memory();
	    vlink_max++;
	}

	vlink_count++;

	/* Initialize and fill in default values */
#ifdef ALLOCATOR_CONSISTENCY_CHECK
        vl->consistency = INUSE_PATTERN;
#endif
	vl->dontfree = FALSE;
        vl->flags = 0;          
	vl->name = NULL;
	vl->linktype = 'L';
	vl->expanded = 0;
	vl->target = stcopy("FILE");
	vl->filters = NULL;
	vl->replicas = NULL;
	vl->hosttype = stcopy("INTERNET-D");
	vl->host = NULL;
	vl->hsonametype = stcopy("ASCII");
	vl->hsoname = NULL;
	vl->version = 0;
	vl->f_magic_no = 0;
        vl->oid = NULL;
	vl->acl = NULL;
	vl->dest_exp = 0;
#if 0
	vl->link_exp = 0;
	vl->args = NULL;
#endif
	vl->lattrib = NULL;
	vl->f_info = NULL;
	vl->previous = NULL;
	vl->next = NULL;
	return(vl);
    }

/*
 * vlcopy - allocates a new vlink structure and
 *          initializes it with a copy of another 
 *          vlink, v.
 *
 *          If r is non-zero, successive vlinks will be
 *          iteratively copied.
 *
 *          vl-previous will always be null on the first link, and
 *          will be appropriately filled in for iteratively copied links. 
 *
 *          VLCOPY returns a pointer to the new structure of type
 *          VLINK.  It calls vlalloc(), which signals out_of_memory() if none
 *              is available.
 *
 *          VLCOPY will copy the list of filters associated with a link.
 *          It will not copy the list of replicas.
 */
VLINK
vlcopy(v,r)
	VLINK	v;
	int	r;              /* Currently ignored. */
    {
	VLINK	nl;
	VLINK	snl;  /* Start of the chain of new links */
	VLINK	tl;   /* Temporary link pointer          */

	nl = vlalloc();
	snl = nl;

    copyeach:

	/* Copy v into nl */
#ifdef ALLOCATOR_CONSISTENCY_CHECK
        assert(v->consistency == INUSE_PATTERN);
#endif
	if(v->name) nl->name = stcopyr(v->name,nl->name);
	nl->linktype = v->linktype;
	nl->expanded = v->expanded;
	nl->target = stcopyr(v->target,nl->target);
	nl->hosttype = stcopyr(v->hosttype,nl->hosttype);
	if(v->host) nl->host = stcopyr(v->host,nl->host);
 	nl->hsonametype = stcopyr(v->hsonametype,nl->hsonametype);
	if(v->hsoname) nl->hsoname = stcopyr(v->hsoname,nl->hsoname);
	nl->version = v->version;
	nl->f_magic_no = v->f_magic_no;
        nl->oid = atlcopy(v->oid);
	nl->acl = NULL;
	nl->dest_exp = v->dest_exp;
	if(v->filters) nl->filters = flcopy(v->filters, 1);
	/* Still need to handle lattrib and f_info */
	if(r && v->next) {
	    v = v->next;
	    tl = nl;
	    nl = vlalloc();
	    nl->previous = tl;
	    tl->next = nl;
	    goto copyeach;
	}

	return(snl);
    }

/*
 * vlfree - free a VLINK structure
 *
 *    VLFREE takes a pointer to a VLINK structure and adds it to
 *    the free list for later reuse.
 */
vlfree(vl)
    VLINK	vl;
{
    if(vl->dontfree) return;
#ifdef ALLOCATOR_CONSISTENCY_CHECK
    assert(vl->consistency == INUSE_PATTERN);
    vl->consistency = FREE_PATTERN;
#endif
    if(vl->name) stfree(vl->name);
    stfree(vl->target);
    if(vl->filters) fllfree(vl->filters);
    if(vl->replicas) vllfree(vl->replicas);
    stfree(vl->hosttype);
    if(vl->host) stfree(vl->host);
    stfree(vl->hsonametype);
    if(vl->hsoname) stfree(vl->hsoname);
    if (vl->oid) atlfree(vl->oid);
    if(vl->acl) aclfree(vl->acl);
#if 0
    if(vl->args) stfree(vl->args);
#endif
    if(vl->lattrib) atlfree(vl->lattrib);
    /* No allocation routines for f_info yet */
    if (vl->f_info ) pffree(vl->f_info);
    vl->next = lfree;
    vl->previous = NULL;
    lfree = vl;
    vlink_count--;
}

/*
 * vllfree - free a VLINK structure
 *
 *    VLLFREE takes a pointer to a VLINK structure frees it and any linked
 *    VLINK structures.  It is used to free an entrie list of VLINK
 *    structures.
 */
vllfree(vl)
    VLINK	vl;
{
    VLINK	nxt;

    while((vl != NULL) && !vl->dontfree) {
        nxt = vl->next;
        vlfree(vl);
        vl = nxt;
    }
}

