/* replyf.c */
/*
 * Copyright (c) 1989, 1990 by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdarg.h>

#include <ardp.h>
#include <pfs.h>
#include <plog.h>
#include <pprot.h>


/*
 * Note: If you are looking for the definitions of reply and creply
 *       they are defined as macros that call ardp_reply with
 *       the ARDP_R_COMPLETE or ARDP_R_INCOMPLETE flags.  
 *       This file defines the formatted versions of these
 *       commands, replyf and creplyf.
 */


int
replyf(RREQ req, char *format, ...)
{
    va_list ap;
    int retval;

    va_start(ap, format);
    retval = vreplyf(req, format, ap);
    va_end(ap);
    return retval;
}

/* The ARDP library currently has a limitation that the ardp_reply() function
   it will not accept long buffers.  This limitation will be removed soon, but
   in the mean time it is necessary to accomodate to it by feeing buffers to
   ardp_reply() in bite-sized chunks.  When this limitation is removed, then
   ARDP_REPLY_CHUNKSIZE can be undefined.   */ 

#define ARDP_REPLY_CHUNKSIZE 1024 /* nice number. */

int
vreplyf(RREQ req, char *format, va_list ap)
{
    static char *buf = NULL;
    static int bufsiz = 0;
    int n;                      /* # of characters possibly written.
                                   # of characters in string in buf */
#ifdef ARDP_REPLY_CHUNKSIZE
    int numsent;                /* # of chars already sent. */
    char tempc;                 /*  Temporary character storage. */
    int retval;
#endif

    
 again:
    n = vqsprintf(buf, bufsiz, format, ap);
    if (n > bufsiz) {
        stfree(buf);
        buf = stalloc(bufsiz = n);
        goto again;             /* the test will fail the 2nd time. */
    }
#ifndef ARDP_REPLY_CHUNKSIZE
    /* Perform the reply and pass through any error codes */
    return(ardp_reply(req, ARDP_R_INCOMPLETE, buf));
#else
    for (numsent = 0; numsent < n; numsent += ARDP_REPLY_CHUNKSIZE) {
        /* N always > 0, since qsprintf() always writes a zero byte at the end.
         */ 
        if (numsent + ARDP_REPLY_CHUNKSIZE < n) {
            tempc = buf[numsent + ARDP_REPLY_CHUNKSIZE];
            buf[numsent + ARDP_REPLY_CHUNKSIZE] = '\0';
        }
        retval = ardp_reply(req, ARDP_R_INCOMPLETE, buf + numsent);
        if (retval) return retval;
        if (numsent + ARDP_REPLY_CHUNKSIZE < n) {
            buf[numsent + ARDP_REPLY_CHUNKSIZE] = tempc;
        }
    }
    return retval;
#endif
}


int
creplyf(RREQ req, char *format, ...)
{
    va_list ap;
    int retval;

    va_start(ap, format);
    retval = vcreplyf(req, format, ap);
    va_end(ap);
    return retval;
}


int
vcreplyf(RREQ req, char *format, va_list ap)
{
    static char *buf = NULL;
    static int bufsiz = 0;
    int n;                      /* # of characters possibly written.
                                   # of characters in string in buf */
#ifdef ARDP_REPLY_CHUNKSIZE
    int numsent;                /* # of chars already sent. */
    char tempc;                 /*  Temporary character storage. */
    int retval;
#endif

 again:
    n = vqsprintf(buf, bufsiz, format, ap);
    if (n > bufsiz) {
        stfree(buf);
        buf = stalloc(bufsiz = n);
        if (buf == NULL) out_of_memory();
        goto again;             /* the test will fail the 2nd time. */
    }
#ifndef ARDP_REPLY_CHUNKSIZE
    /* Perform the reply and pass through any error codes */
    return(ardp_reply(req, ARDP_R_COMPLETE, buf));
#else
    for (numsent = 0; numsent < n; numsent += ARDP_REPLY_CHUNKSIZE) {
        /* N always > 0, since qsprintf() always writes a zero byte at the end.
         */ 
        if (numsent + ARDP_REPLY_CHUNKSIZE < n) {
            tempc = buf[numsent + ARDP_REPLY_CHUNKSIZE];
            buf[numsent + ARDP_REPLY_CHUNKSIZE] = '\0';
            retval = ardp_reply(req, ARDP_R_INCOMPLETE, buf + numsent);
        } else {
            retval = ardp_reply(req, ARDP_R_COMPLETE, buf + numsent);
        }
        if (retval) return retval;
        if (numsent + ARDP_REPLY_CHUNKSIZE < n) {
            buf[numsent + ARDP_REPLY_CHUNKSIZE] = tempc;
        }
    }
    return retval;
#endif
}

