/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the file
 * <usc-copyr.h>
 *
 * Written  by swa    1992     to provide common code for starting request
 * Modified by prasad 1992     to add Kerberos support
 * Modified by bcn    1/93     to use new ardp library and rreq structure
 * Modified by swa    6/93     to construct multiple-packet messages.
 */

#include <usc-copyr.h>

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <netdb.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/* For compatability between GCC-2.3.1 running under SunOS 4.1 and the sunOS
 * standard include files (they define size_t and others differently), we MUST 
 * include stddef and stdarg after anything that might include the sun include
 * file <sys/stdtypes.h> (in this case, <pwd.h> includes <sys/stdtypes.h>).
 */ 

#include <stddef.h>
#include <stdarg.h>

#include <ardp.h>
#include <psite.h> /* must precede pfs.h for defs. of P_KERBEROS, */
		   /* P_P_PASSWORD  */
#include <pfs.h>
#include <pprot.h>
#include <perrno.h>
#include <pcompat.h>

#ifdef P_KERBEROS
#include <krb5/krb5.h>
#endif /* P_KERBEROS */

static char	*pfs_sw_id = PFS_SW_ID;

extern int	pfs_debug;
extern int	perrno;
extern char	p_err_string[];

static PAUTH p__get_pauth(int type, const char *hostname);

p_set_sw_id(char *app_sw_id)
{
    pfs_sw_id = qsprintf_stcopyr(NULL,"%s(%s)",app_sw_id,PFS_SW_ID);
    return(PSUCCESS);
}

/* 
 * p__start_req - start a prospero request
 *
 *     p__start_req takes the name of the server (used by Kerberos to 
 *     determine what credentials are needed), it allocates a request
 *     structure, and it fills in the start, including the Prospero
 *     version and software ID, and authentication information.  The
 *     function returns a request of type RREQ, which can be used 
 *     in subsequent calls to p__addreq, and passed to ardp_send.
 */
RREQ
p__start_req(const char server_hostname[])
{
    RREQ req = ardp_rqalloc(); /* The request to fill in         */
    PAUTH authinfo;      /* Structure containing authentication info */
    PAUTH authp;         /* To iterate through authinfo list */
    TOKEN prin;          /* To Iterate through principal names       */

    if (!req)
        internal_error("Out of memory!"); /* XXX this may be an unwise way of
                                             handling an out-of-memory problem.
                                             Get it working now, fix it later.
                                             */ 

    p__add_req(req, "VERSION %d %s", VFPROT_VNO, pfs_sw_id);
    authinfo = p__get_pauth(PFSA_UNAUTHENTICATED, server_hostname);
    p__add_req(req, "\nAUTHENTICATE '' UNAUTHENTICATED %'s",
           authinfo->authenticator);
    /* For the UNAUTHENTICATED and KERBEROS authentication types,  the
       principals are optional additional information for debugging use.  We
       don't worry about them here. */
    for(prin = authinfo->principals; prin; prin = prin->next)
        p__add_req(req, " %'s", prin->token); 
    pafree(authinfo);

#ifdef P_KERBEROS
    authinfo = p__get_pauth(PFSA_KERBEROS, server_hostname);
    for (authp = authinfo; authp; authp = authp->next) {
        p__add_req(req, "\nAUTHENTICATE '' KERBEROS %'s", 
                   authp->authenticator);
        /* For the UNAUTHENTICATED and KERBEROS authentication types,  the
           principals are optional additional information for debugging use.
           We don't worry about them here. */
        for (prin = authp->principals; prin; prin = prin->next)
            p__add_req(req, " %'s", prin->token); 
    }
    if (authinfo)
	palfree(authinfo);
#endif
#ifdef P_P_PASSWORD
    authinfo = p__get_pauth(PFSA_P_PASSWORD, server_hostname);
    for (authp = authinfo; authp; authp = authp->next) {
        p__add_req(req, "\nAUTHENTICATE OPTIONAL P_PASSWORD %'s", 
                   authp->authenticator);
	/* Principal NOT optional here! */
        for (prin = authp->principals; prin; prin = prin->next)
	    p__add_req(req, " %'s", prin->token); 
    }
    if (authinfo)
	palfree(authinfo);
#endif
    p__add_req(req, "\n");
    return(req);
}

/* Adds stuff to a Prospero request. */
int
p__add_req(RREQ req, const char fmt[], ...)
{
    int retval;
    va_list ap;
    va_start(ap,fmt);
    retval = vp__add_req(req, fmt, ap);
    va_end(ap);
    return(retval);
}

int
vp__add_req(RREQ req, const char fmt[], va_list ap)
{
    PTEXT	ptmp;           /* packet being written to. */
    int         mesglen;        /* # of characters in this message, NOT 
                                   including trailing NUL which doesn't need to
                                   be sent. */ 
    static  char *buf = NULL;          /* buffer for long packets */
    int numsent;                /* How many chars. out of buf have been sent so
                                   far? */

    if(!req->outpkt) {
	ptmp = ardp_ptalloc();
	if(!ptmp) return(PFAILURE);
	APPEND_ITEM(ptmp,req->outpkt);
    } else {
        ptmp = req->outpkt->previous; /* last member of list. */
    }

    /* Subtract 1: Trailing NUL character isn't sent,
       since the length of the message is encoded into the message.  Recipient
       will automatically append the NUL byte for convenience of the users. */
    /* Add one to buffer space available, since it includes an MBZ byte to
       catch runaway strings.  vqsprintf() will always null-terminate the
       strings it writes. */
    mesglen = 
        vqsprintf(ptmp->ioptr, ARDP_PTXT_LEN + 1 - ptmp->length, fmt, ap) - 1;
    if (ptmp->length + mesglen <= ARDP_PTXT_LEN) {
        ptmp->length += mesglen;
        ptmp->ioptr = ptmp->start + ptmp->length;
        return(PSUCCESS);
    }
    /* Ran out of room.  Have to write it to a temporary buffer, then copy. */
    /* 1 byte more is necessary than you might think, since vqsprintf() always
       puts a null byte in the last spot. */
    if (stsize(buf) < mesglen + 1) {      /* make sure we have enough room. */
        stfree(buf);
        buf = stalloc(mesglen + 1);
        if (!buf) return PFAILURE;
    }
    /* Write to temporary buffer. */
    assert(mesglen == vqsprintf(buf, stsize(buf), fmt, ap) - 1);
    /* Record what we already sent. */
    numsent = ARDP_PTXT_LEN - ptmp->length;
    ptmp->length = ARDP_PTXT_LEN;
    /* Start this loop with the current ptmp full of as much data as it can
       hold. */ 
    while (numsent < mesglen) {
	ptmp = ardp_ptalloc();
	if(!ptmp) return(PFAILURE);
	APPEND_ITEM(ptmp,req->outpkt);
        ptmp->length = ( mesglen - numsent > ARDP_PTXT_LEN
                        ? ARDP_PTXT_LEN : mesglen - numsent);
        strncpy(ptmp->start, buf + numsent, ptmp->length);
        ptmp->ioptr = ptmp->start + ptmp->length;
        numsent += ptmp->length;
    }
    return PSUCCESS;
}


static 
PAUTH p__get_pauth(int type, const char server_hostname[])
{
#ifdef P_KERBEROS
    krb5_data buf;
    krb5_checksum send_cksum;
    krb5_ccache ccdef;
    krb5_principal server;
    int retval = 0;
#endif
#if defined(KERBEROS) || defined(P_P_PASSWORD)
    char full_hname[255];
    char *cp;
#endif
#ifdef P_P_PASSWORD
    struct hostent *host;
    FILE *passwd_fp;
    char *passwd_filename = NULL;
    PAUTH curr_auth = NULL;
    char line[255];
    char hostname[255], princ_name[255], password[255];
    typedef struct _p_passwd {
	char *hostname;
	char *princ_name;
	char *password;
	struct _p_passwd *next;
    } p_passwd;
    p_passwd *p_list = NULL, *p_ptr;
    char *hostaddr = NULL;
    int num;
#endif    
    PAUTH auth = paalloc();
    struct passwd *whoiampw;
    int uid = 0;

    auth->principals = NULL;    /* not really needed.  Let's be paranoid. */
    switch(auth->ainfo_type = type) {
      case PFSA_UNAUTHENTICATED:
	/* find out who we are */
	DISABLE_PFS(whoiampw = getpwuid(uid = getuid()));
	if (whoiampw == 0) {
	    char tmp_uid_str[100];
	    qsprintf(tmp_uid_str, sizeof tmp_uid_str, "uid#%d", uid);
	    auth->authenticator = stcopy(tmp_uid_str);
	}
	else {
	    auth->authenticator = stcopy(whoiampw->pw_name);
	}
        return(auth);
	break;
	
#ifdef P_KERBEROS                 /* 4/5/93  Made error handling pass errors
                                   to higher layers. */
      case PFSA_KERBEROS:
        qsprintf(full_hname, sizeof full_hname, "%s", server_hostname);
        /* Strip off the trailing (portnum), if any */
        for (cp = full_hname; *cp; ++cp)
            ;
        if (cp > full_hname && *--cp == ')') {
            while (*--cp && cp >= full_hname) {
                if (*cp == '(') {
                    *cp = '\0';
                    break;
                }
                if (!isdigit(*cp)) 
                    break;
            }
        }
	
	/* PREPARE KRB_AP_REQ MESSAGE */
	
	/* compute checksum, using CRC-32 */
	if (!(send_cksum.contents = (krb5_octet *)
	      stalloc(krb5_checksum_size(CKSUMTYPE_CRC32)))) {
            if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Cannot allocate Kerberos checksum\n");
            pafree(auth);
            return NULL;
	}
	
	/* choose some random stuff to compute checksum from */
	if (retval = krb5_calculate_checksum(CKSUMTYPE_CRC32,
					     (char *) server_hostname,
					     strlen(server_hostname),
					     0,
					     0, /* if length is 0, */
						/* crc-32 doesn't use */
						/* the seed */ 
					     &send_cksum)) {
            if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Error while computing Kerberos checksum\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	}
	
        /* Get credentials for server, create krb_mk_req message */
	
	if (retval = krb5_cc_default(&ccdef)) {
            
	    if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error while getting default \
Kerberos  cache\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	}
	
	/* Format service name into a principal structure and */
	/* canonicalizes host name. */
	if (retval = krb5_sname_to_principal(full_hname,
					     KERBEROS_SERVICE, 
					     1,        /* Canonicalize */
					     &server)) {
	    if (pfs_debug)
                fprintf(stderr, "p__get_pauth(): Error while setting up \
Kerberos server principal\n");
            stfree(send_cksum.contents);
            pafree(auth);
            return NULL;
	}
	
	if (retval = krb5_mk_req(server, 0 /* default options */,  &send_cksum,
				 ccdef, &buf)) {
	    if (pfs_debug) fprintf(stderr, "p__get_pauth(): Error while \
preparing Kerberos AP_REQ\n");
            pafree(auth);
            stfree(send_cksum.contents);
            return NULL;
	}
	stfree(send_cksum.contents);
	auth->authenticator = stalloc(AUTHENTICATOR_SZ);
	retval = binencode(buf.data, buf.length, 
			   auth->authenticator, 
			   AUTHENTICATOR_SZ);
	if (retval >= AUTHENTICATOR_SZ) {
	    internal_error("binencode: Buffer too small");
	}

	return(auth);
	break;
#endif /* P_KERBEROS */
#ifdef P_P_PASSWORD
      case PFSA_P_PASSWORD:
	/* Convert hostname into canonical form */
	qsprintf(full_hname, sizeof full_hname, "%s", server_hostname);
        
	/* Strip off the trailing (portnum), if any */
        for (cp = full_hname; *cp; ++cp)
            ;
        if (cp > full_hname && *--cp == ')') {
            while (*--cp && cp >= full_hname) {
                if (*cp == '(') {
                    *cp = '\0';
                    break;
                }
                if (!isdigit(*cp)) 
                    break;
            }
        }

        /* Look up server host */
        if ((host = gethostbyname(full_hname)) == (struct hostent *) 0) {
            if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error in P_PASSWORD \
authentication: %s: unknown host\n", full_hname);
            pafree(auth);
            return NULL;
        }
        qsprintf(full_hname, sizeof full_hname, "%s", host->h_name);
	hostaddr = stcopy(inet_ntoa(host->h_addr));

        /* Upper-case to use in search of password file */
        for (cp = full_hname; *cp; cp++)
          if (islower(*cp))
            *cp = toupper(*cp);

	if (!(passwd_filename = stcopy(getenv("P_PASSWORD")))) {
	    char *home;
	    passwd_filename = stalloc(255);
	    uid = getuid();
	    if (!(home = getenv("HOME"))) {
		if (pfs_debug)
		    fprintf(stderr, 
			    "Could not get value of HOME\n");
		pafree(auth);
		return NULL;
	    }
	    qsprintf(passwd_filename, stsize(passwd_filename),
		     "%s/.ppw_%d", home, uid);
	}	

	if (!(passwd_fp = fopen(passwd_filename, "r"))) {
	    if (pfs_debug) 
                fprintf(stderr, "p__get_pauth(): Error in P_PASSWORD \
authentication: Cannot open password file %s\n", passwd_filename);
	    stfree(passwd_filename);
            pafree(auth);
            return NULL;
	}

	stfree(passwd_filename);
	
	while (fgets(line, sizeof(line), passwd_fp)) {
	    if (line[0] == '#')
		continue;
	    num = qsscanf(line, "%s %s %'!!s\n", 
			  hostname, princ_name, password, sizeof(password));
	    
	    if (num < 3) /* Ignore if no password */
		continue;

	    if (wcmatch(full_hname, hostname) || 
		strequal(hostaddr, hostname)) {
		p_passwd *last = NULL;

		for (p_ptr = p_list; p_ptr; last = p_ptr, p_ptr = p_ptr->next)
		    if (!strcmp(princ_name, p_ptr->princ_name)) {
			if (isdigit(p_ptr->hostname[0]))
			    break; /* Already have most specific */
				   /* address */
			
			if (isdigit(hostname[0]) ||
			    strlen(hostname) > strlen(p_ptr->hostname)) {
			    p_ptr->hostname = stcopyr(hostname, p_ptr->hostname);
			    p_ptr->password = stcopyr(password, p_ptr->password);
			}
			break;
		    }
		
		if (!p_ptr) {
		    if (!last) {
			p_list = (p_passwd *) stalloc(sizeof(p_passwd));
			last = p_list;
		    }
		    else {
			last->next = (p_passwd *) stalloc(sizeof(p_passwd));
			last = last->next;
		    }
		    last->hostname = stcopy(hostname);
		    last->princ_name = stcopy(princ_name);
		    last->password = stcopy(password);
		    last->next = NULL;
		}
	    }
	}
	
	(void) fclose(passwd_fp);

	pafree(auth);
	auth = NULL;

	for (p_ptr = p_list; p_ptr; p_ptr = p_ptr->next) {
	    if (!auth)
		auth = curr_auth = paalloc();
	    else {
		curr_auth->next = paalloc();
		curr_auth = curr_auth->next;
	    }
	    curr_auth->ainfo_type = type;
	    curr_auth->authenticator = stcopy(p_ptr->password);
	    curr_auth->principals = tkalloc(p_ptr->princ_name);
	    curr_auth->next = NULL;  
	}

        return(auth);
	break;
#endif /* P_P_PASSWORD */
      default:
	fprintf(stderr, "p__get_pauth(): Unknown authenticator type: %d\n", type);
        internal_error("Unknown authenticator type!");
        /* NOTREACHED */
    }
}
