/*
 * Copyright (c) 1991 by the University of Washington
 * Copyright (c) 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 *
 * Written  by bcn 1991     as part of the Prospero distribution
 * Modified by bcn 1/93     extended rreq structure
 */

#include <uw-copyright.h>
#include <usc-copyr.h>
#include <stdio.h>
#include <ardp.h>

static ardp_default_timeout = ARDP_DEFAULT_TIMEOUT;
static ardp_default_retry = ARDP_DEFAULT_RETRY;
int    ardp_priority = 0; /* Default priority */

static RREQ	free = NOREQ;
int 		rreq_count = 0;
int		rreq_max = 0;

/*
 * rralloc - allocate and initialize an rreq structure
 *
 *    ardp_rqalloc returns a pointer to an initialized structure of type
 *    RREQ.  If it is unable to allocate such a structure, it
 *    returns NOREQ (NULL).
 */
RREQ
ardp_rqalloc()
{
    RREQ	rq;

    if(free) {
	rq = free;
	free = free->next;
    }
    else {
	rq = (RREQ) malloc(sizeof(RREQ_ST));
	if (!rq) return(NOREQ);
	rreq_max++;
    }
    rreq_count++;

    rq->status = ARDP_STATUS_NOSTART;
    rq->cid = 0;
    rq->priority = ardp_priority;
    rq->pf_priority = 0;
    rq->ardp_version = 0;
    rq->inpkt = NOPKT;
    rq->rcvd_tot = 0;
    rq->rcvd = NOPKT;
    rq->rcvd_thru = 0;
    rq->comp_thru = NOPKT;
    rq->outpkt = NOPKT;
    rq->trns_tot = 0;
    rq->trns = NOPKT;
    rq->prcvd_thru = 0;
    bzero(&(rq->peer),sizeof(rq->peer));
    rq->rcvd_time.tv_sec = 0;
    rq->rcvd_time.tv_usec = 0;
    rq->svc_start_time.tv_sec = 0;
    rq->svc_start_time.tv_usec = 0;
    rq->timeout.tv_sec = ardp_default_timeout;
    rq->timeout.tv_usec = 0;
    rq->timeout_adj.tv_sec = ardp_default_timeout;
    rq->timeout_adj.tv_usec = 0;
    rq->wait_till.tv_sec = 0;
    rq->wait_till.tv_usec = 0;
    rq->retries = ardp_default_retry;
    rq->retries_rem = ardp_default_retry;
    rq->svc_rwait = 0;
    rq->svc_rwait_seq = 0;
    rq->inf_queue_pos = 0;
    rq->inf_sys_time = 0;
#ifdef PROSPERO
    rq->auth_info = NULL;
#endif /* PROSPERO */
    rq->client_name = NULL;
    rq->sw_id = NULL;
    rq->cfunction = NULL;
    rq->cfunction_args = NULL;
    rq->previous = NOREQ;
    rq->next = NOREQ;
    return(rq);
}

/*
 * ardp_rqfree - free a RREQ structure
 *
 *    ardp_rqfree takes a pointer to a RREQ structure and adds it to
 *    the free list for later reuse.
 */
void ardp_rqfree(rq)
    RREQ	rq;
{
    if(rq->status == ARDP_STATUS_FREE) {
	fprintf(stderr,"Attempt to free free RREQ\n");
	abort();
	return;
    }
    rq->status = ARDP_STATUS_FREE;
    /* Don't free inpkt or comp_thru, already on rcvd     */
    if(rq->rcvd) ardp_ptlfree(rq->rcvd);
    /* But outpkt has not been added to trns */
    if(rq->outpkt) ardp_ptlfree(rq->outpkt);
    if(rq->trns) ardp_ptlfree(rq->trns);
#ifdef PROSPERO
    if (rq->auth_info) pafree(rq->auth_info);
#endif /* PROSPERO */
    if (rq->client_name) stfree(rq->client_name);
    if (rq->sw_id) stfree(rq->sw_id);

    rq->next = free;
    rq->previous = NOREQ;
    free = rq;
    rreq_count--;
}

/*
 * ardp_rqlfree - free many RREQ structures
 *
 *    ardp_rqlfree takes a pointer to a RREQ structure frees it and any linked
 *    RREQ structures.  It is used to free an entrie list of RREQ
 *    structures.
 */
void ardp_rqlfree(rq)
    RREQ	rq;
{
    RREQ	nxt;

    while(rq != NOREQ) {
	nxt = rq->next;
	ardp_rqfree(rq);
	rq = nxt;
    }
}


/*
 * ardp_set_retry - change default values for timeout
 *
 *    ardp_set_retry takes a value for timout in seconds and a count
 *    of the number of retries to allow.  It sets static variables in this
 *    file used by pralloc to set the default values in request structures
 *    it allocates.
 */
ardp_set_retry(to,rt)
    int	to;
    int rt;
{
    ardp_default_timeout = to;
    ardp_default_retry = rt;
    return(ARDP_SUCCESS);
}
