/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1993             by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdio.h>
#include <pfs.h>
#include <pserver.h>
#include <errno.h>

#ifdef PSRV_P_PASSWORD

#include <ppasswd.h>

static int write_ppw_to_file();
static char *get_salt(void);

extern int pfs_debug;


/* Get password file entry for principal. Returns pointer to p_passwd */
/* structure if found, else returns NULL. */
p_passwd *
get_ppw_entry(char *principal)
{
    FILE *passwd_fp;
    char line[255];
    char princ_name[255], encrypted_passwd[255];
    p_passwd *p_pwd = NULL;

    if (!(passwd_fp = fopen(PSRV_PW_FILE, "r")))
	return NULL;

    while (fgets(line, sizeof(line), passwd_fp)) {
	sscanf(line, "%s %s\n", 
	       princ_name, encrypted_passwd);
	if (!strcmp(princ_name, principal)) {
	    p_pwd = (p_passwd *) stalloc(sizeof(p_passwd));
	    p_pwd->principal = stcopy(princ_name);
	    p_pwd->encrypted_passwd = stcopy(encrypted_passwd);
	}
    }

    (void) fclose(passwd_fp);
    return p_pwd;
}


/* Write password entry in structure p_pwd to password file, */
/* overwriting old entry if one exists for the principal */
static int write_ppw_to_file(p_passwd *p_pwd)
{
    FILE *passwd_fp;
    char in_line[255], out_line[255];
    char princ_name[255], encrypted_passwd[255];
    long pos;

    if (!(passwd_fp = fopen(PSRV_PW_FILE, "r+"))) {
	/* Cannot open password file, so create */
	if (!(passwd_fp = fopen(PSRV_PW_FILE, "w+"))) {
	    if (pfs_debug)
		fprintf(stderr, "Cannot create file %s\n",
			PSRV_PW_FILE);
	    return PFAILURE;
	}
	if (chmod(PSRV_PW_FILE, 0600)) {
	    if (pfs_debug) 
		perror("Could not change permissions of passwd file");
	    return PFAILURE;
	}
	fprintf(passwd_fp, "# Prospero Server Password File.\n");
	fprintf(passwd_fp, "%-32s %-32s\n\n",
		"# <principal>", "<encrypted password>");
	fseek(passwd_fp, 0L, 1); /* To allow input after output */
    }

    /* Check if entry exists for principal */
    while (pos = ftell(passwd_fp), 	/* Remember position of line */
	   fgets(in_line, sizeof(in_line), passwd_fp)) { 
	if (in_line[0] == '#')
	    continue;
	sscanf(in_line, "%s %s\n", 
	       princ_name, encrypted_passwd);
	if (!strcmp(princ_name, p_pwd->principal)) {
	    /* Entry found; set file pointer to overwrite */
	    fseek(passwd_fp, pos, 0);
	    break;
	}
    }
	    
    /* If entry not found, write new entry to end of file, else overwrite */
    fprintf(passwd_fp, "%-32s %-32s\n", 
	    p_pwd->principal, p_pwd->encrypted_passwd);

    (void) fclose(passwd_fp);
    return PSUCCESS;
}


/* Set password for principal in password file, overwriting old */
/* entry if one exists. */
set_passwd(char *principal, char *passwd)
{
    p_passwd p_pwd;
    char *salt = get_salt();

    p_pwd.principal = stcopy(principal);
    p_pwd.encrypted_passwd = (char *) crypt(passwd, salt);
    if (write_ppw_to_file(&p_pwd))
	return PFAILURE;
    stfree(p_pwd.principal);
    stfree(salt);
    return PSUCCESS;
}


/* Returns TRUE if password specified for the principal matches */
/* password in entry for principal in password file, and FALSE */
/* otherwise */
int passwd_correct(char * principal, char *passwd)
{
    p_passwd *p_pwd;
    char *salt, *p;
    int correct;
    
    /* Get password file entry */
    if (!(p_pwd = get_ppw_entry(principal)))
	return FALSE;
    
    /* Salt stored as first two characters of encrypted password */
    salt = p_pwd->encrypted_passwd;

    /* Encrypt supplied password and compare to encrypted password */
    /* stored in password file */
    p = (char *) crypt(passwd, salt);
    correct = strequal(p, p_pwd->encrypted_passwd);

    stfree(p_pwd);
    return correct;
}


/* Constructs and returns a random two-character salt. */
/* Salt needed by crypt() is two characters from the set [a-zA-Z0-9./] */
static char *get_salt(void)
{
    time_t tim = time(NULL);      /* Use current time to achieve */
				  /* randomness between sessions */
    char *salt;

    salt = stalloc(2);
    
    salt[0] = 'a' + (tim % 26);	  /* Letter between 'a' and 'z' */
    salt[1] = '.' + (tim % 12);	  /* One of '.', '/', and '0'-'9' */

    return salt;
}


/* Deletes principal's password entry from file */
int delete_ppw_entry(char *principal)
{
    FILE *passwd_fp, *tmp_fp;
    char line[255];
    char princ_name[255], encrypted_passwd[255];
    char tmp_filename[255];
    int found = 0;

    if (!get_ppw_entry(principal))
	return PFAILURE;

    if (!(passwd_fp = fopen(PSRV_PW_FILE, "r")))
	return PFAILURE;

    qsprintf(tmp_filename, sizeof(tmp_filename), "%s_tmp", PSRV_PW_FILE);
    if (!(tmp_fp = fopen(tmp_filename, "w"))) {
	if (pfs_debug)
	    fprintf(stderr, "Could not create temporary file %s\n",
		    tmp_filename);
	return PFAILURE;
    }

    while (fgets(line, sizeof(line), passwd_fp)) {
	sscanf(line, "%s %s\n", 
	       princ_name, encrypted_passwd);
	if (strcmp(princ_name, principal)) 
	    fputs(line, tmp_fp);
    }

    (void) fclose(tmp_fp);
    (void) fclose(passwd_fp);
    
    if (rename(tmp_filename, PSRV_PW_FILE)) {
	unlink(tmp_filename);
	if (pfs_debug)
	    perror("Could not overwrite password file");
	return PFAILURE;
    }
	
    if (chmod(PSRV_PW_FILE, 0600)) {
	if (pfs_debug) 
	    perror("Could not change permissions of passwd file");
	return PFAILURE;
    }

    return PSUCCESS;
}


/* Lists the principals in the Prospero password file */
int list_ppw_file(void)
{
    FILE *passwd_fp;
    char line[255];
    char princ_name[255], encrypted_passwd[255]; 
    int  num;
  
    if (!(passwd_fp = fopen(PSRV_PW_FILE, "r")))
	return PFAILURE;

    while (fgets(line, sizeof(line), passwd_fp)) {
	if (line[0] == '#')
	    continue;

	num = sscanf(line, "%s %s\n", 
		     princ_name, encrypted_passwd);
	if (num == 2)
	    printf("%s\n", princ_name);
    }

    (void) fclose(passwd_fp);

    return PSUCCESS;
}

#endif /* PSRV_P_PASSWORD */
