/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <prm-copyr.h>.
 *
 * Modified by srao 6/92 - 6/93
 *
 */

#include <prm-copyr.h>

#include <stdio.h>
#include <signal.h>      
#include <utmp.h>
#include <sys/utsname.h>

#define  MAIN_PROG
#include <comm.h>
#include <nodemngr.h>


char            *loc_bin_dir;     /* Default directory for executables       */
char            *smhostname;      /* Name of host on which sysmngr runs      */
char            *node_archname;   /* Architecture type of this node          */
int             utflag;           /* If set, timer enabled for utmp check    */
int             avail_win_flag;   /* Set if nodemngr is available only 
				     within specified time windows           */
int             node_archtype;    /* Code for node architecture              */
int             win_start;        /* Variables marking the beginning and end */
int             win_end;          /* of "node-available" windows             */
int             win_start_mins;
int             win_end_mins;
u_char          node_status;      /* Status of node (also see nodemngr.h)    */
u_short         next_udp_port;    /* Port number assigned to next child task */
long            timer_intvl[4];   /* Interval in secs, for timer alarm for
	        		     the various node states                 */
u_long          inaddr;           /* Internet address of nodemngr host       */
	        
prm_node_addr   my_node_addr;     /* Address of nodemngr host                */
prm_node_addr   smaddr;           /* Address of sysmngr  host                */
prm_process_t   pid_table;        /* Maps unix process (by pid) to job/task  */
prm_job_t       job_table;        /* Pointer to head of list of jobs whose   */
                                  /* task(s) is running on this node.*/

struct  itimerval timeval;        /* To handle timers , interrupts           */
struct  itimerval otimeval;       


extern  int  unix_create_task();
extern  void NM_sigchld_hndlr();      /* Signal handler for SIGCHLD */
extern  void NM_sigalrm_hndlr();      /* Signal handler for SIGALRM */
extern  void NM_sigterm_hndlr();      /* SIGTERM */

extern  prm_job_t prm_jalloc();


/* Functions to use for creating tasks. Currently tasks can be created only
   through unix fork(). Add function names here as and when support for other 
   methods for task creation are added  */

int     (*os_create[])()= {  unix_create_task   };



main(int argc, char *argv[], char *envp[])
{

  char   buf[128];              
  struct utsname unamebuf;
  struct hostent *hent;

  utflag = FALSE;
  avail_win_flag = FALSE;
  next_udp_port = FIRST_USR_PORT; 

  node_status = NODE_UNUSED;
  timer_intvl[NODE_UNUSED] = 60; 
  timer_intvl[NODE_USER_ON] = 900;
  timer_intvl[NODE_JOB_ON] = 0;
  win_start = 0000;
  win_end = 0600;

  bzero(&smaddr, PRM_AD_SZ);
  loc_bin_dir = LOC_BIN_DIR;
  pid_table = NULL;
  job_table = NULL;

  _progname = argv[0];
  argc--;argv++;
  
  while (argc > 0 && **argv == '-') {

    switch (*(argv[0]+1)) {

    case 'D':
	    sscanf(argv[0],"-D%d", &prm_debug);
	    if (!prm_debug)
	      prm_debug = 1;       /* Default debug level */
            break;

    case 'i':
	    utflag = UT_SET;
	    sscanf(argv[0],"-i%d:%d", &timer_intvl[NODE_UNUSED], 
		   &timer_intvl[NODE_USER_ON]);
	    if(timer_intvl[NODE_UNUSED] <= 0) timer_intvl[NODE_UNUSED] = 60; 
	    if(timer_intvl[NODE_USER_ON] <= 0) timer_intvl[NODE_USER_ON] = 900;
	    break;

   case 'l':
	    argc--; argv++;
	    loc_bin_dir = (char *)malloc(strlen(argv[0]) + 1);
	    sscanf(argv[0], "%s", loc_bin_dir);
	    break;

   	    
   case 's':
	    argc--; argv++;
	    smhostname = (char *)malloc(strlen(argv[0]) + 1);
	    sscanf(argv[0], "%s", smhostname);
	    break;

   case 't':
	    avail_win_flag = TRUE;
	    sscanf(argv[0],"-t%d:%d", &win_start, &win_end);
	    win_start_mins = (win_start/100) * 60 + (win_start%100);
	    win_end_mins  = (win_end/100) * 60 + (win_end%100);
	    break;
	    

    default:
            fprintf(stderr, "Usage: %s [ -D[debug-level] ] [ -i[intvl_unused_secs:intvl_user_secs] ]\n [ -s sysmngr-hostname ] [ -twindow_start:window_end ]\n", 
		    _progname);
            exit(1);
	  }
    argc--; argv++;
  }

  pfs_debug = prm_debug;


  /* If this node is to be made available for jobs only during certain periods,
     a timer is set to signal the end/beginning of such periods. When timer
     expires, the node status is updated. 
     If utflag is set, the timer is set to expire periodically, the period
     specified in the timer_intvl[] array. 
     If avail_win_flag is set, timer is set to expire at the next window 
     boundary. 
     If both are set, timer is set to expire at the time which logically 
     occurs first. 
     */
  
  if( (utflag == UT_SET) ||  (avail_win_flag) ) {
    
    update_node_status();
    
    /* Set next expiry time to the smallest of: seconds given on command line,
       time-to-start of window boundary, time-to-end of window boundary  */
    
    timeval.it_value.tv_sec = timeval.it_interval.tv_sec = 
      smallest_time_to_expiry(timer_intvl[node_status], win_start, win_end);
    
    timeval.it_value.tv_usec = timeval.it_interval.tv_usec = 0;
    setitimer(ITIMER_REAL, &timeval, &otimeval);
    signal(SIGALRM, NM_sigalrm_hndlr);
    utflag |= TM_ENBL;
  }
  
  /* Initialization of hostname and host address */

  gethostname(_my_hostname, sizeof(_my_hostname));
  hent = gethostbyname(_my_hostname);
  bcopy(hent->h_addr, &inaddr, 4);
  bzero((char *)&my_node_addr, PRM_AD_SZ);
  my_node_addr.sin_family = hent->h_addrtype;
  bcopy(hent->h_addr, (char *)&(my_node_addr.sin_addr), hent->h_length);
  my_node_addr.sin_port = htons((u_short) NODEMNGR_PORT);
  

  /* Create a socket and bind a name to it. If running as a privileged 
     process, two sockets are created. One is bound to a privileged port 
     and the other to an unprivileged one. If running as an unprivileged
     process, only the unprivileged socket is created. */

#ifdef ROOT
  my_udp_portnum = ardp_bind_port("prm-nm");
  ardp_set_prvport(ardp_srvport);
#endif
  
  sprintf(buf, "#%d", PRM_NM_UNPRIV_PORT);
  ardp_bind_port(buf);


  signal(SIGCHLD, NM_sigchld_hndlr);
  signal(SIGTERM, NM_sigterm_hndlr);

  if (uname(&unamebuf) == -1) {
    (void)printf("(%s) Could not get host architecture type\n", _progname);
    exit(1);
  }
  else
     calc_nname(&node_archname, unamebuf.machine);

  node_archtype = HOSTTYPE(node_archname);

  /* NODEMNGR'S LOOP  */ 

  for (;;) {   

    NM_Process_Req(envp);

  } 

}  



/* Nodemngr's main loop proc: waits until a  request arrives at the port and 
   processes it. The processing performed depends on the PRM_opcode in the 
   request. */


NM_Process_Req(char **envp)
{
  PTEXT  pkt, rpkt;
  RREQ    current_req;
  

  int             child_pid;
  int             timetowait;
  int             jobfound;
  int             i, taskfound, num_task_ent, arc_stat;
  
  char            filename[100], dirname[MAX_DIR_LINESIZE];
  char            *msg;           /* used as generic position pointers in */
  char            *curplace;      /* message buffers */

  u_char          osname;
  u_char          req_op, req_pnum, signum;
  u_char          tmp_status;       
  u_short         filelength, dirlength, ndeg, np;
  u_short         emsglen, nelen;   /* length of error messages */
  u_long          req_auth, datalen;
  u_long          ntmp1;

  tid_t           req_tid, curtid; 
  jid_t           req_jid;
  prm_job_t       curjob;
  prm_task_t      curtask;
  prm_process_t   cur_pid_ent;
  prm_cache_tag_t tbits;            /* Cache tag */
  prm_node_addr_t dest;             /* Generic destination address          */
  prm_node_addr   h_ad;             /* For other addresses                  */
  struct timeval  task_start_time;
  struct timeval  task_end_time;

  current_req = ardp_get_nxt();
  msg = current_req->rcvd->start;
  
  if (*msg != PRM_PROTO_V) {
    PRM_wrong_proto(current_req);
    return;
  }

  /* Get the opcode */
  req_op = *(msg + PRM_OPCODE_OFF);
  
  /* Get the jobid of the sender from the message. Valid only for jobmngr 
     and tasks. For other entities, we don't care what it is. */ 
  
  bcopy(msg + PRM_JOBID_OFF, &req_jid, LONG_SZ);  
  req_jid = ntohj(req_jid);
  
  switch (req_op) {
    
  case PRM_CREAT_TASK: /* request from a jobmngr for task creation. nodemngr
			  uses the pathname, taskid  and jobid specified by 
			  the jobmngr to fork a child process. On success, pid
			  of task is returned, and job/task info is updated. */
    
    osname = *(msg + PRM_ADINF_OFF);
    
    bcopy(msg + PRM_DLEN_OFF, &dirlength, 2);
    dirlength = ntohs(dirlength);
    bcopy(msg + PRM_FLEN_OFF, &filelength, 2);
    filelength = ntohs(filelength);
    bcopy(msg + PRM_DATA_OFF + dirlength + filelength, &req_auth, LONG_SZ);


    /* Look up job table, ensure that there is an entry for this job. If job
       manager is on a remote node, ensure that it is authorized to use this
       node to run tasks. */

    SEARCH_LIST(job_table,curjob,jid,req_jid);
    
    if (ADDR_REMOTE(&(current_req->peer),&my_node_addr))
      if (check_auth(curjob, req_jid, req_auth, &(current_req->peer))
	  == FAILURE) {
	sprintf(p_err_string, "Job not authorized on node %s",
		inet_ntoa(my_node_addr.sin_addr));
	
	prm_server_respond(current_req, (u_char)PRM_TCREAT_RESP, 
			   (u_char)FAILURE, 0, p_err_string);
	NM_inform_sm_node_status(node_status);
	break;
      }
    else 
      if (!curjob)
	curjob = prm_jalloc(req_jid, (prm_node_addr_t)&(current_req->peer));

    if (dirlength) {
      bcopy(msg + PRM_DATA_OFF, dirname, dirlength);
      sprintf(dirname + dirlength - 1,"/%s", node_archname);
    }
    else
      dirname[0] = '\0';
    
    bcopy(msg + PRM_DATA_OFF + dirlength, filename, filelength);
    
    bcopy(msg + PRM_TINFO_OFF, &req_tid, LONG_SZ);
    req_tid = ntohl(req_tid);
    
    
    /* Inform jobmngr not to retry request for some time */
    
    timetowait = (PRM_FILE_XFER_REQD(dirname) ? 80 : 5);
    if (ardp_rwait(current_req, timetowait, 0, 0) != PSUCCESS)
      fprintf(stderr, "(%s) Could not send backoff pkt to jobmngr\n", 
	      _progname);
    
    if (utflag == (UT_SET | TM_ENBL)){/* utmp checking set and timer not
					 already disabled. */
      timeval.it_value.tv_sec = timeval.it_interval.tv_sec = 0;
      setitimer(ITIMER_REAL, &timeval, &otimeval);
      utflag ^= (TM_ENBL | DIS_TMP);   /* Disable timer and indicate
					  temporarily disabled */ 
    }

    gettimeofday( &task_start_time, (struct timezone *)0);

    child_pid = (*os_create[osname])(dirname, filename, req_tid, req_jid, 
				     next_udp_port, envp);

    if ( (child_pid != -1) && ((emsglen = strlen(p_err_string) + 1) == 1) ) {
			
      rpkt = current_req->outpkt = ardp_ptalloc();
      prm_headers(rpkt, (u_char)PRM_TCREAT_RESP, (u_char)SUCCESS, (u_char)0, 
		  0);

      ntmp1 = htonl(child_pid);
      bcopy(&ntmp1, rpkt->start + PRM_TINFO_OFF, 4);
      nelen = htons(next_udp_port);
      bcopy(&nelen, rpkt->start + PRM_DLEN_OFF, 2);
      rpkt->length = PRM_DLEN_OFF + 2;
      ardp_respond(current_req, ARDP_R_COMPLETE);
    }

    else {     /* Failed to spawn a process */
      prm_server_respond(current_req, (u_char)PRM_TCREAT_RESP, (u_char)FAILURE,
			 0, p_err_string);
      NM_inform_sm_node_status(node_status);
      
      if (utflag == (UT_SET | DIS_TMP)) { /* Timer was recently disabled. re-enable it */
	timeval.it_value.tv_sec = timeval.it_interval.tv_sec = 
	  smallest_time_to_expiry(timer_intvl[node_status], win_start,
				  win_end);
	setitimer(ITIMER_REAL, &timeval, &otimeval);
	utflag ^= (TM_ENBL |DIS_TMP) ;  
      }
      break;
    }

    if (utflag == (UT_SET | DIS_TMP) )
      utflag ^= DIS_TMP;
    
    if (req_tid == 0) { /* This is the fileio task */
      curjob->fioaddr.sin_port = htons(next_udp_port);
      ++next_udp_port;    /* allocate one port to the term_iotask */
      curjob->iotaddr.sin_port = htons(next_udp_port);
      goto task_update_done;
    }

    /* when nodemngr starts caching task to hostaddr mapping, we need to
       add an entry in the job table for the task just created. */
    
    SEARCH_LIST(curjob->taskptr,curtask,tid,req_tid);

    if (!curtask) {
      curtask = (prm_task_t) malloc(sizeof(struct prm_task));
      curtask->tid = req_tid; /* tasks are numbered 1 through ntasks */
      curjob->ctag[i] = 1;
      curtask->ports = (u_short *)calloc(NPORTS, sizeof(u_short));
      curtask->nports = NPORTS;
    }
    curtask->t_status = T_RUNNING;
    curtask->start_time.tv_sec  = task_start_time.tv_sec;
    curtask->start_time.tv_usec = task_start_time.tv_usec;
    *(curtask->ports + 1) = htons(next_udp_port);
    cur_pid_ent = (prm_process_t) malloc(sizeof(struct prm_process));
    curtask->pid_tbl_p = cur_pid_ent;
    cur_pid_ent->pid = child_pid;
    cur_pid_ent->taskptr = curtask;
    cur_pid_ent->jobptr = curjob;
    APPEND_ITEM(cur_pid_ent, pid_table);
    
    node_status = NODE_JOB_ON;
    
  task_update_done:
    ++next_udp_port;
    if (next_udp_port == 0xffff )
      next_udp_port = FIRST_USR_PORT;
    
    break;
    
    
  case PRM_THMAP_UPDT:  /* Job manager has sent the mapping of tasks to nodes.
			   Cache it in my local address space */
    
    bcopy(msg + PRM_DLEN_OFF, &datalen, LONG_SZ);
    datalen = ntohl(datalen);
    bcopy (msg + PRM_DATA_OFF + datalen, &req_auth, LONG_SZ);
    
    /* Check if an entry for this job exists */
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    
    if (ADDR_REMOTE(&(current_req->peer),&my_node_addr)) {
      if (check_auth(curjob, req_jid, req_auth, &(current_req->peer))
	  == FAILURE) {
	sprintf(p_err_string, "Job not authorized on node %s",
		inet_ntoa(my_node_addr.sin_addr));
	
	prm_server_respond(current_req, (u_char)PRM_MPUPD_RESP, 
			   (u_char)FAILURE, 0, p_err_string);
	break;
      }
    }
    else 
      if (!curjob) {
	curjob = prm_jalloc(req_jid, (prm_node_addr_t)(msg + PRM_DATA_OFF +
						       LONG_SZ) );
	curjob->auth_key = req_auth;
      }

    /* Number of task entries to update */
    bcopy(msg + PRM_TINFO_OFF, &num_task_ent, LONG_SZ); 
    num_task_ent = ntohl(num_task_ent);
    
    if (!curjob->ctag) {
      curjob->ctag = (char *)malloc(num_task_ent + 1);
      bzero(curjob->ctag, num_task_ent + 1);
    }    
    tbits = curjob->ctag;
    
    curplace = msg + PRM_DATA_OFF;
    
    for (i = 0; i <= num_task_ent; i++) { 	  
      bcopy(curplace, &curtid, LONG_SZ);  /* get tid of current task entry */
      curtid = ntohl(curtid);
      curplace += LONG_SZ;
      
      if ( tbits[curtid] )  {
	/* An entry for this task exists  in job table */
	/* look for the entry */
	SEARCH_LIST(curjob->taskptr,curtask,tid,curtid);
      }
      else  {
	curtask = (prm_task_t) malloc(sizeof(struct prm_task));
	curtask->tid = curtid; /* tasks are numbered 1 through ntasks */
	tbits[i] = 1;
	curtask->ports = (u_short *)calloc(NPORTS, sizeof(u_short));
	curtask->nports = NPORTS;
	curtask->pid_tbl_p = NULL;
	
	curtask->next = curjob->taskptr;
	curjob->taskptr = curtask;
	if (curtid != 0)  /* This is not an iotask */
	  ++(curjob->numtasks);
      }
      bcopy(curplace, &(curtask->hostaddr), PRM_AD_SZ);
      curtask->hostaddr.sin_family = AF_INET;
      curplace += PRM_AD_SZ;
    }  /* for */
    
    /* Send reply to jobmngr */
    prm_server_respond(current_req, (u_char)PRM_MPUPD_RESP, (u_char)SUCCESS,
		       0, (char *)0);
    
    break;
    
    
  case PRM_ADDR_XLAT:   /* A request from local task for t->h mapping */
    
    bcopy(msg + PRM_TINFO_OFF, &req_tid, LONG_SZ);
    req_tid = ntohl(req_tid);
    req_pnum = *(msg + PRM_ADINF_OFF);
    
    /* Inform task not to retry request for another 15 secs */
    if (ardp_rwait(current_req, 15, 0, 0) != PSUCCESS)
      fprintf(stderr, "(%s) Could not send backoff pkt to task.\n", 
	      _progname);
    
    jobfound = FALSE;
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    
    if(curjob) {
      jobfound = TRUE;
      if (req_tid) {    /* normal task */
	if((curjob->ctag)[req_tid] ) { /* the t->h mapping is cached */
	  SEARCH_LIST(curjob->taskptr,curtask,tid,req_tid);
	  if (curtask) {
	    bcopy(&(curtask->hostaddr), &h_ad, PRM_AD_SZ);
	    if ( (req_pnum > curtask->nports) || 
		((h_ad.sin_port = *(curtask->ports+req_pnum)) == 0)) {
	    
	      if(ADDR_REMOTE(&h_ad,&my_node_addr) && (req_op == PRM_ADDR_XLAT)) {
	      
		get_l2p_from_remote(req_pnum, req_jid, req_tid, &h_ad);
		/* cache it locally */
		if ( req_pnum >= curtask->nports) {
		  curtask->nports = req_pnum + 1;
		  curtask->ports = (u_short *)realloc(curtask->ports,
						      req_pnum * 2);
		}
		*(curtask->ports + req_pnum) = h_ad.sin_port;
	      }
	      else   /* the requested l->p mapping does not exist */
		h_ad.sin_port = 0;
	    }
	  }
	  else
	    h_ad.sin_port = 0;
	}
	else  /* get t->h mapping from jm and l->p mapping from nm */
	  (curjob->ctag)[req_tid] = 1;
      }
      else {             /* JM/iotasks */
	jobfound = TRUE;
	if (req_pnum == 0) /* jobmngr */
	  bcopy(&(curjob->jmaddr), &h_ad, PRM_AD_SZ);
	else
	  if (req_pnum == 1) /* iotask */
	    bcopy(&(curjob->iotaddr), &h_ad, PRM_AD_SZ);
	  else
	    bcopy(&(curjob->fioaddr), &h_ad, PRM_AD_SZ);
	
	/* If port address not known make sure that *I*  am not the host
	   running the jobmngr; then send a query to the remote node
	   on which the jobmngr is executing */
	
	if ( !ADDR_COMPLETE(&h_ad) && ADDR_REMOTE(&h_ad,&my_node_addr)) { 
	  get_l2p_from_remote(req_pnum, req_jid, req_tid, &h_ad);

	  /* cache the port addresses locally */
	  if (req_pnum == 0)
	    curjob->jmaddr.sin_port = h_ad.sin_port;
	  else if (req_pnum == 1)
	    curjob->iotaddr.sin_port = h_ad.sin_port;
	  else 
	    curjob->fioaddr.sin_port = h_ad.sin_port;
	}
      }
    }
    rpkt = current_req->outpkt = ardp_ptalloc();

    if  (curjob == NULL) {
      strcpy(p_err_string, "No such job or task!");
      prm_server_respond(current_req, (u_char)PRM_XLAT_RESP, (u_char)FAILURE, 
			 0, p_err_string);
    }
    else {
      prm_headers(rpkt, (u_char)PRM_XLAT_RESP, (u_char)SUCCESS, 0, 
		  curjob->jid);

      bcopy(&h_ad, rpkt->start + PRM_DATA_OFF, PRM_AD_SZ);
      rpkt->length = PRM_DATA_OFF + PRM_AD_SZ;
      ardp_respond(current_req, ARDP_R_COMPLETE); 
    }
    
    break;
    
    
  case PRM_NSTAT_QRY:   /* sysmngr is requesting the busy status of node */
    
    /* save sysmgr's hostaddr */
    bcopy(&(current_req->peer), &smaddr, PRM_AD_SZ); 
    smaddr.sin_family = AF_INET;
    smaddr.sin_port = htons ((u_short)SYSMNGR_PORT);
    current_req->outpkt = ardp_ptalloc();
    prm_headers(current_req->outpkt, (u_char)PRM_NSTAT_UPDT, (u_char)SUCCESS, 
		node_status, 0);
    current_req->outpkt->length = PRM_ADINF_OFF + 1;
    ardp_respond(current_req, ARDP_R_COMPLETE);

    break;
    
    
  case PRM_ASGN_PORT:  /* Request from task to assign a port addr. Increment a
			  local counter, assign that value, and also store it
			  in the local prm_task structure of the task. */
    
    bcopy(msg + PRM_TINFO_OFF, &req_tid, LONG_SZ);
    req_tid = ntohl(req_tid);
    req_pnum = *(msg + PRM_ADINF_OFF);
    jobfound = FALSE;
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    if(curjob) {
      jobfound = TRUE;
      if (req_tid != 0) {          /* not jobmngr or iotasks */
	SEARCH_LIST(curjob->taskptr,curtask,tid,req_tid);
	if(curtask) {					    
	  if ( req_pnum > curtask->nports) {
	    curtask->nports = req_pnum;
	    curtask->ports = (u_short *)realloc(curtask->ports, 
						req_pnum*2);
	  }
	  *(curtask->ports+req_pnum) = next_udp_port;
	}
      }
      else {  /* jobmngr or tiotask or fiotask */
	if(req_pnum == 0) 
	  curjob->jmaddr.sin_port = htons(next_udp_port);
	else if (req_pnum == 1)
	  curjob->iotaddr.sin_port = htons(next_udp_port);
	else 
	  curjob->fioaddr.sin_port = htons(next_udp_port);
      }
    }
    if (jobfound) {
      prm_server_respond(current_req, (u_char)PRM_PORT_RESP, (u_char)SUCCESS, 
			 htonl((jid_t)next_udp_port), (char *)0);

      ++next_udp_port;
    }
    else {
      sprintf(p_err_string, "(%s) No such authorized job %d\n", _progname,
	      req_jid);
      prm_server_respond(current_req, (u_char)PRM_PORT_RESP, (u_char)FAILURE,
			 (jid_t)0, p_err_string);
      
    }
    if (next_udp_port == 0xffff)
      next_udp_port = FIRST_USR_PORT;
    break;
    
    
  case PRM_AUTH_JOB:  /* Sysmngr has sent an authorizaton code for job */
    
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    
    if (!curjob)              /*  new job */
      curjob = prm_jalloc(req_jid, (prm_node_addr_t)(msg + PRM_DATA_OFF));
    
    bcopy(msg + PRM_DATA_OFF + PRM_AD_SZ, &(curjob->auth_key), LONG_SZ);
    /* Authcode is retained in network byte order */
    
    prm_server_respond(current_req, (u_char)PRM_AUTH_RESP, (u_char)SUCCESS,
		       0, (char *)0);
    
    break;
    
    
  case PRM_NTASKS_JOB:  /* Request for number of tasks in this job domain */
	  
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    if (curjob) 
	ntmp1 = htonl(curjob->numtasks);
    else
      ntmp1 = 0;

    prm_server_respond(current_req, (u_char)PRM_TASK_CNT, (u_char)SUCCESS, 
		       (curjob ? curjob->numtasks : 0), (char *)0);
    
    break;
    
    
  case PRM_JOB_DONE:   /* jobmngr notifies nodemngr that job has completed/
			  terminated. nodemngr kills any residual tasks and 
			  removes entry for this job from job table */
    
    if ((signum = *(msg + PRM_ADINF_OFF)) == 0) 
      signum = SIGTERM;  /* If termination is not due to a signal from 
			    the user, then use SIGTERM to terminate any
			    stray tasks */
    
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    if (!curjob)
      break;
    EXTRACT_ITEM(curjob, job_table);
    
    while(curjob->taskptr) {
      curtask = curjob->taskptr;
      curjob->taskptr = curtask->next;
      cur_pid_ent = curtask->pid_tbl_p;
      /* Check if the task is running on the local node. If so, send
	 signal to that process */
      if (cur_pid_ent)
	if ( (cur_pid_ent->pid > 0) && (curtask->t_status == T_RUNNING) ) {
	  curtask->t_status = T_KILLED;
	  kill(cur_pid_ent->pid, signum);
	}
      free(curtask->ports); /* deallocate storage for all port ids desc.*/
      free(curtask);        /* deallocate storage for current task desc.*/
    }
    
    free(curjob);  /* deallocate storage for job entry */
    
    prm_server_respond(current_req, (u_char)PRM_JDONE_RESP, (u_char)SUCCESS, 
		       0, (char *)0);
    
    break;
    

 /* Change of busy-status of node, requested by external agent */

  case PRM_NSTAT_UPDT:  

    tmp_status = *(msg + PRM_ADINF_OFF);
    prm_server_respond(current_req, (u_char)PRM_NSTAT_QRY, (u_char)SUCCESS,
		       0, (char *)0);
    
    if (tmp_status != node_status) {
      NM_inform_sm_node_status(tmp_status);
      node_status = tmp_status;
    }
    break;
    
    
  case PRM_LOGIN_NTFY: /* Notification from login process */
    
    tmp_status = node_status;
    if (!privileged(&(current_req->peer)) || 
	((node_status = *(msg + PRM_STATUS_OFF)) == 4))
      update_node_status();
    
    prm_server_respond(current_req, (u_char)PRM_LOGIN_RESP, (u_char)SUCCESS,
		       0, (char *)0);

    if (tmp_status != node_status) {
      NM_inform_sm_node_status(node_status);
      if ((tmp_status == NODE_JOB_ON) && (node_status == NODE_USER_ON)) {
/*	suspend_tasks();
	NM_notify_jm_task_status();
*/      }
    }
    break;


  case PRM_TASK_STAT:
    
    SEARCH_LIST(job_table,curjob,jid,req_jid);
    
    if (!curjob) {
      prm_server_respond(current_req, (u_char)PRM_TSTAT_RESP, (u_char)FAILURE,
			 0, 0);
      break;
    }
    bcopy(msg + PRM_TINFO_OFF, req_tid, LONG_SZ);
    req_tid = ntohl(req_tid);
    SEARCH_LIST(curjob->taskptr,curtask,tid,req_tid);

    if(!curtask) {
      prm_server_respond(current_req, (u_char)PRM_TSTAT_RESP, (u_char)FAILURE,
			 0, 0);
      break;
    }

    curtask->t_status = *(msg + PRM_ADINF_OFF); 

    switch (*(msg + PRM_STATUS_OFF)) {

    case T_EXITED:
      gettimeofday(task_end_time, (struct timezone *)0 );
      dest = &(curjob->jmaddr);

      if (!ADDR_COMPLETE(dest) && ADDR_REMOTE(dest,&my_node_addr) )
	get_l2p_from_remote(0, curjob->jid, 0, dest);

      if(ADDR_COMPLETE(dest))
	/* Notify the jobmngr */
	NM_notify_jm_task_status(req_tid,
				 (u_char)T_EXITED, curtask->t_status,
				 0, dest, curtask->start_time, 
				 task_end_time);
      break;
    }
    break;
    
  default:
    break;
  }
}

privileged(prm_node_addr_t x)
{
  return TRUE;
}
