/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <prm-copyr.h>.
 *
 * Written by srao 11/92
 *
 */

#include <prm-copyr.h>


#include <ardp.h>

#include <stdio.h>
#include <sys/socket.h>
#include <signal.h>       /* To handle signals from child tasks, timers */
#include <sys/wait.h>     /* To obtain exit status of child */
#include <utmp.h>
#include <time.h>
#include <netdb.h>

#include <pmachine.h>
#include <nodemngr.h>
#include <pprot.h>
#include <perrno.h>

extern int    prm_debug, utflag, avail_win_flag, pfs_debug, win_start, win_end;
extern int    win_start_mins, win_end_mins;
extern char   _progname, _my_hostname[], *smhostname;
extern prm_process_t pid_table;
extern struct itimerval timeval, otimeval;
extern u_char node_status;
extern long   timer_intvl[];
extern u_long inaddr;
extern struct authlist *authl_head;

extern prm_node_addr my_node_addr;
extern prm_node_addr smaddr;  /* sysmngr's address */


#ifndef NODEMNGR_PORT
#   define NODEMNGR_PORT ARDP_DEFAULT_PORT
#endif


/* NM_sigchld_hndlr: handler for SIGCHILD. Calls wait() to get the exit status/
   signal status of child. Deletes entries in job_table and pid_table 
   corresponding to child. Notifies the concerned jobmngr (if required). 
   Updates node status and informs sysmngr if node status changed.
*/


void
  NM_sigchld_hndlr(sig, code, scp, addr)
int sig, code;
struct sigcontext *scp;
char *addr;
{

  u_char   estatus;             /* exit status of child */
  u_char   sigstatus;           /* signal (if any) that changed child status */
  int           child_pid;
  prm_process_t pid_entry;
  struct timeval child_end_time; 
  prm_node_addr_t dest_addr;


  gettimeofday(&child_end_time, (struct timezone *)0 );
  if( (child_pid = get_child_info(&estatus, &sigstatus) ) <= 0 )
    return;
  
  /* Locate task info through pid. First search for pid entry in pid table */
  SEARCH_LIST(pid_table,pid_entry,pid,child_pid);

  if (pid_entry) {   /* If an entry for this process exists in pid_table */

    if (pid_entry->taskptr->t_status != T_KILLED) {
    /* If task was not killed by jobmngr, nodemngr needs to notify jobmngr */
      
      dest_addr = &(pid_entry->jobptr->jmaddr);

      if (!ADDR_COMPLETE(dest_addr) && ADDR_REMOTE(dest_addr,&my_node_addr) )
	/* If I don't have jobmngr's complete address, and the partial address
	   refers to a remote node, query the nodemngr on the remote node  */
	get_l2p_from_remote(0, pid_entry->jobptr->jid, 0, dest_addr);

      if(ADDR_COMPLETE(dest_addr))
	/* Notify the jobmngr */
	NM_notify_jm_task_status(pid_entry->taskptr->tid, 
				 (u_char)PRM_TASK_STAT,
				 estatus, sigstatus, dest_addr,
				 &(pid_entry->taskptr->start_time), 
				 &child_end_time );
    } 
    pid_entry->taskptr->t_status = T_DEAD;
    
    EXTRACT_ITEM(pid_entry, pid_table);
    free(pid_entry);
    if (pid_table == NULL) {  /* No more tasks running */
      
      if ((utflag == UT_SET ) || avail_win_flag) {
	/* utmp checking is enabled */
	
	update_node_status();
	timeval.it_value.tv_sec = timeval.it_interval.tv_sec =
	  smallest_time_to_expiry(timer_intvl[node_status], win_start, 
				  win_end);
	setitimer(ITIMER_REAL, &timeval, &otimeval);
      }
      else
	node_status = NODE_UNUSED;      
      NM_inform_sm_node_status(node_status);
    }
  }
  signal(SIGCHLD, NM_sigchld_hndlr); /* Reinstate the signal handler */

}


/*  user_logged_on(): Checks /etc/utmp to ascertain whether a user is logged
    into this workstation. Returns TRUE if user is logged on, else returns 
    FALSE. If utmp cannot be opened, TRUE is returned.
*/

int 
  user_logged_on()
{
  struct utmp u1;
  FILE *fp;
  
  int sz;
  if( (fp = fopen("/etc/utmp", "r")) == NULL) 
    return TRUE ;
  
  sz = fread (&u1, sizeof(struct utmp), 1, fp );
  
  while ( (sz == 1) && (u1.ut_name[0] == '\0') )
    sz = fread (&u1, sizeof(struct utmp), 1, fp );
  
  fclose(fp);
  if (u1.ut_name[0] != '\0')
    return TRUE;
  else
    return FALSE;
}


NM_inform_sm_node_status(st)
u_char st;
{
  PTEXT pkt, rpkt;
  RREQ newreq;
  char *msg, sm_hostport[80];

  pkt = ardp_ptalloc();
  msg = pkt->start;
  
  prm_headers(pkt, (u_char)PRM_NSTAT_UPDT, st, 0, my_node_addr.sin_addr.s_addr);
  pkt->length = PRM_JOBID_OFF + LONG_SZ;

  if (smhostname == NULL) {
    smhostname = (char *)malloc(sizeof(DFL_SM)+1);
    strcpy(smhostname, DFL_SM);
  }
  sprintf(sm_hostport, "%s(%d)", smhostname, SYSMNGR_PORT);

  newreq = ardp_rqalloc();
  newreq->outpkt = pkt;
  ardp_send(newreq, sm_hostport, &smaddr, -1);

  if ( (rpkt = newreq->rcvd) == NOPKT )
    fprintf(stderr, 
	    "(%s %s) Could not inform sysmngr %s of status change.\n", 
	    _progname, _my_hostname, inet_ntoa(smaddr.sin_addr) );
}


get_child_info(estatusp,sigstatp)
u_char *estatusp, *sigstatp;
{
  int child_pid, sp;
  if( (child_pid = wait3(&sp, WNOHANG, 0)) == 0)
    return child_pid;
  if (child_pid < 0){   /* No children  */
    perror("wait");
    return child_pid;
  }

  /* IF child exited by calling exit() */
  if ((WIFEXITED(sp)) ) {

  /* If exit status was requested, get the exit status of the child */
    if (estatusp) {
#     ifdef MACH386
      *estatusp = (sp>>16) & 0x000000ff;
#     else
      *estatusp = WEXITSTATUS(sp);
#     endif
    }
    if (sigstatp)
      *sigstatp = 0;
  }
  else 
    if (WIFSIGNALED(sp)) {
      if (sigstatp)
	*sigstatp = WTERMSIG(sp);
      if(estatusp)
	*estatusp = 0;
    }
  return child_pid;
}



void
NM_sigalrm_hndlr(sig, code, scp, addr)  /* signal handler for SIGALRM */
int sig, code;
struct sigcontext *scp;
char *addr;
{
  u_char old_node_status;

  old_node_status = node_status;
  update_node_status();

  if (node_status != old_node_status)  {  /* The status has changed */
    NM_inform_sm_node_status(node_status);
  }
  timeval.it_value.tv_sec = timeval.it_interval.tv_sec = 
    smallest_time_to_expiry(timer_intvl[node_status], win_start, win_end);
  
  setitimer(ITIMER_REAL, &timeval, &otimeval);
}


smallest_time_to_expiry(logon_status_intvl, win_start, win_end)

int logon_status_intvl, win_start, win_end;
{
  int min_intvl, t2start, t2end;
  if (avail_win_flag) {
    t2start = t2expy(win_start);
    t2end   = t2expy(win_end);
    min_intvl = min(t2start, t2end) ;
    if (utflag == UT_SET) 
      min_intvl = min(min_intvl, logon_status_intvl);
  }
  else {
    if (utflag == UT_SET)
      return logon_status_intvl;
  }
  return min_intvl;
}


t2expy(bndry_time)
int bndry_time;
{
  long seconds_now, seconds_bndry, seconds_diff;
  struct tm *tst;
  extern struct tm *current_time();

  tst = current_time();

  /*  seconds since 0000 hrs */
  seconds_now = tst->tm_sec + 60 * tst->tm_min + 3600 * tst->tm_hour;
  seconds_bndry = (bndry_time/100) * 3600 + (bndry_time%100) * 60;

  seconds_diff = seconds_bndry - seconds_now;
  if (seconds_diff < 0)
    seconds_diff += 3600 * 24;

  return seconds_diff;
}


within_window()
{
  long crnt_mins;
  struct tm *crnt;
  extern struct tm *current_time();
 
  crnt = current_time();
  crnt_mins  = crnt->tm_hour * 60 + crnt->tm_min;

  if ( win_start_mins < win_end_mins ) {
    if ( (crnt_mins >= win_start_mins) && (crnt_mins < win_end_mins) ) 
      return TRUE;
    else 
      return FALSE;
  }
  else {
    if ( (crnt_mins >= win_start_mins) 	|| (crnt_mins <  win_end_mins) )
      return TRUE;
    else
      return FALSE;
  }
}


update_node_status()
{
  if (avail_win_flag) {
    if (within_window() ) {
      if ((utflag == UT_SET) && user_logged_on() )
	node_status = NODE_USER_ON;
      else
	node_status = NODE_UNUSED;
    }
    else 
      node_status = NODE_USER_ON;
  }
  else {
    if ((utflag == UT_SET) && user_logged_on() ) 
      node_status = NODE_USER_ON;
    else
      node_status = NODE_UNUSED;
  }

}


struct tm *
  current_time()
{
  struct timeval tp;
  struct timezone tzp;
  struct tm *tst;

  gettimeofday(&tp, &tzp);
  tst = (struct tm *)malloc(sizeof(struct tm));
  tst = localtime(&tp.tv_sec);
  return tst;
}


int 
  check_auth(job_p, req_jid, req_aut, addr) 
prm_job_t job_p;
jid_t req_jid;
u_long req_aut;
prm_node_addr_t addr;
{
  if ((job_p != NULL) && (job_p->jid == req_jid) && 
      (job_p->jmaddr.sin_addr.s_addr == addr->sin_addr.s_addr) && 
      (job_p->auth_key == req_aut)) 
    return SUCCESS;
  else 
    return FAILURE;
} 


NM_notify_jm_task_status(tid, task_status, estatus, sigstatus, 
			 dest_addr, start_time, end_time)
u_long tid; 
u_char task_status, estatus, sigstatus;
prm_node_addr_t dest_addr;
struct timeval *start_time, *end_time;
{
  char *msg, *curposn;
  u_short emsglen, nelen;
  PTEXT pkt, rpkt;
  RREQ newreq;
  u_long nd, ntmp1;
  
  pkt = ardp_ptalloc(); 
  msg = pkt->start;
  prm_headers(pkt, (u_char)PRM_TASK_STAT, task_status, estatus, 0);
  
  nd = htonl(tid);
  bcopy(&nd, msg + PRM_TINFO_OFF,  LONG_SZ);
  *(msg + PRM_DLEN_OFF) = sigstatus;
  
  /* Put in the start and end times */
  
  curposn = msg + PRM_DATA_OFF;
  if (estatus && p_err_string[0]) {  /* There is an error message */
    emsglen = strlen(p_err_string) + 1;
    nelen  = htons(emsglen);
    bcopy(&nelen, msg + PRM_FLEN_OFF, 2);
    bcopy(p_err_string, curposn, emsglen);
    curposn += emsglen;
  }
  nd = htonl(start_time->tv_sec);
  bcopy( &nd, curposn, LONG_SZ);
  curposn += LONG_SZ;
  nd = htonl(start_time->tv_usec);
  bcopy( &nd, curposn, LONG_SZ);
  curposn += LONG_SZ;
  nd = htonl(end_time->tv_sec);
  bcopy( &nd, curposn, LONG_SZ);
  curposn += LONG_SZ;
  nd = htonl(end_time->tv_usec);
  bcopy( &nd, curposn, LONG_SZ);
  pkt->length = (int)(curposn + LONG_SZ - msg);
  
  newreq = ardp_rqalloc();
  newreq->outpkt = pkt;
  ardp_send(newreq, 0, dest_addr, -1);
}


calc_nname(name, aname)
     char **name, *aname;
{
  if (strncmp(aname, "sun4", 4) == 0)
    *name =  "sun4";
  else 
    if (strncmp(aname, "9000/", 5) == 0)
      *name = "hp700" ;
    else
      if (strncmp(aname, "sun3", 4) == 0)
	*name = "sun3";
      else
	*name =  "unknown";
}


void
  NM_sigterm_hndlr(sig, code, scp, addr)
int sig, code;
struct sigcontext *scp;
char *addr;
{
  prm_process_t pid_p;

  pid_p = pid_table;
  while (pid_p) {
    kill(pid_p->pid, SIGTERM);
    pid_p = pid_p->next;
  }
  sleep(2);
  exit(SIGTERM);
}
