/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <prm-copyr.h>.
 *
 * Written by srao 11/92. Modified 1/93, 3/93
 *
 */

#include <prm-copyr.h>


#include <stdio.h>
#include <math.h>

#include <comm.h>
#include <CMMDlib.h>

#define MAXINT 0x7fffffff         /* Maximum integer value                  */
#define MININT -MAXINT            /* Minimum integer value                  */

#define LFLOAT   3.4E+38          /* Maximum float value                    */
#define SFLOAT   (-LFLOAT)        /* Maximum float value                    */

#define LDOUBLE  2.0E+296         /* Maximum double value                   */
#define SDOUBLE  (-LDOUBLE)       /* Maximum double value                   */


static int barrier_cntr[16] = { 0, 0, 0, 0, 0, 0, 0, 0,
				0, 0, 0, 0, 0, 0, 0, 0 };  

/* Initial values for global state variables */

static int    ir_add = 0, 
              ir_max = MININT, 
              ir_min = MAXINT, 
              ir_ior = 0, 
              ir_xor = 0, 
              ir_and = 0xffffffff;

static float  fr_max = SFLOAT,
              fr_min = LFLOAT,
              fr_add = 0.0;

static double dr_max = LDOUBLE, 
              dr_min = SDOUBLE,
              dr_add = 0.0;


/*  perform_global_op: Performs the operation requested by the client, which
    may be one of the reduction operations, or just a call to TIO_procedure,
    the I/O procedure written by the apl. programmer.
    With reduction operations, after all participants have contributed their
    operand, the result of the reduction operation is broadcast to all.
*/


void
perform_global_op(int tid, int msg_tag, char *data, int msg_length)
{
  int combiner_tag, combiner_ival, bc_index;
  float combiner_fval;
  double combiner_dval;
  
  bc_index = BARR_IND(msg_tag);
  
  if (barrier_cntr[bc_index] == 0)      /* Initialize counter */
    barrier_cntr[bc_index] = numtasks();
  
  switch(msg_tag) {
    

  case SYNC_NODES:
    
    --barrier_cntr[0];
    
    if (barrier_cntr[0] == 0 )    /* barrier reached by all */
      
      /* broadcast to all nodes */
      bcast_to_nodes( (char *)0, 0, msg_tag);

    break;


  case IR_ADD:
    
    combiner_ival = *( (int *)data);

    ir_add += combiner_ival;
    --barrier_cntr[bc_index];
    
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_add, sizeof(ir_add), msg_tag );
      ir_add = 0;
    }

    break;


  case IR_MAX:

    combiner_ival = *( (int *)data);

    if (ir_max < combiner_ival) ir_max = combiner_ival;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %d\n", tid, ir_max); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_max, sizeof(int), msg_tag );
      ir_max = MININT;
    }

    break;



  case IR_MIN:

    combiner_ival = *( (int *)data);

    if (ir_min > combiner_ival) ir_min = combiner_ival;
    
    --barrier_cntr[bc_index];
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_min, sizeof(int), msg_tag );
      ir_min = MAXINT;
    }

    break;

  case LR_IOR:

    combiner_ival = *( (int *)data);

    ir_ior |= combiner_ival;
    
    --barrier_cntr[bc_index];
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_ior, sizeof(int), msg_tag );
      ir_ior = 0;
    }

    break;


  case LR_XOR:

    combiner_ival = *( (int *)data);

    ir_xor ^= combiner_ival;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %d\n", tid, ir_xor); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_xor, sizeof(int), msg_tag );
      ir_xor = 0;
    }

    break;


  case LR_AND:

    combiner_ival = *( (int *)data);
    
    ir_and ^= combiner_ival;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %d\n", tid, ir_and); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &ir_and, sizeof(int), msg_tag );
      ir_and = 0;
    }

    break;


  case FR_ADD:

    combiner_fval = *( (float *)data);
    
    fr_add += combiner_fval;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %g\n", tid, fr_add); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &fr_add, sizeof(float), msg_tag );
      fr_add = 0.0;
    }

    break;


  case FR_MIN:

    combiner_fval = *( (float *)data);
    
    if (combiner_fval < fr_min) fr_min = combiner_fval;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %g\n", tid, fr_min); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &fr_min, sizeof(float), msg_tag );
      fr_min = LFLOAT;
    }

    break;


  case FR_MAX:

    combiner_fval = *( (float *)data);
    
    if (combiner_fval > fr_max) fr_max = combiner_fval;
    
    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %g\n", tid, fr_max); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &fr_max, sizeof(float), msg_tag );
      fr_max = SFLOAT;
    }

    break;


  case DR_ADD:

    combiner_dval = *( (double *)data);

    dr_add += combiner_dval;
    
    --barrier_cntr[bc_index];
    /*    printf("task %d, redn value = %g\n", tid, dr_add); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &dr_add, sizeof(double), msg_tag );
      dr_add = 0.0;
    }
    
    break;
    


  case DR_MAX:
    combiner_dval = *((double *)data);

    if (combiner_dval > dr_max) dr_max = combiner_dval;

    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %g\n", tid, dr_max); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &dr_max, sizeof(double), msg_tag );
      dr_max = SDOUBLE;
    }

    break;


  case DR_MIN:
    combiner_dval = *((double *)data);

    if (combiner_dval < dr_min) dr_min = combiner_dval;

    --barrier_cntr[bc_index];
    /* printf("task %d, redn value = %g\n", tid, dr_min); */
    if (barrier_cntr[bc_index] == 0) {
      bcast_to_nodes( (char *) &dr_min, sizeof(double), msg_tag );
      dr_max = LDOUBLE;
    }


  default:
    TIO_procedure(tid, msg_tag, data, msg_length);
    break;
  }
}
