/*
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <prm-copyr.h>.
 *
 * Written by srao 10/92
 */

#include <prm-copyr.h>


#include <ardp.h>

#include <stdio.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h> 
#include <sys/unistd.h>

#if defined(SUNOS)
#include <vfork.h>  /* mach386 system & hpux don't have this header file */
#endif

#define PRM_PROG
#include <comm.h>
#include <pdap.h>

#include <nodemngr.h>

extern int prm_debug, node_archtype;
extern char *loc_bin_dir,  *node_archname;
extern char *strrchr();
extern char *sys_errlist[];
extern prm_job_t job_table;
extern u_long  inaddr;
extern int execle(), execlp();
extern RREQ ardp_pendingQ;

static int rec_level;

/*  unix-create_task: Create a task as a unix process using vfork(). Command
    line arguments passed to the process include the task_id of the task and
    the job_id of job domain within which the task is executing.
    */


unix_create_task(char *dirname, char *filename, tid_t tid, jid_t jid, 
		 u_short udp_port_num, char *envp[])
{
  char *pathname;
  int child_pid, dl, pathlen, remote, bin_fd;
  char tid_str[8];
  char jid_str[16];
  char port_str[8];

  p_err_string[0]='\0';
  
  /* Determine if the executable needs to be retrieved through the fileIO */
  if ((remote = strncasecmp(dirname, "io:", 3)) == 0) {
    pathlen = strlen(loc_bin_dir) + strlen(filename) + 
              strlen(node_archname) + 2;
    pathname = (char *)malloc(pathlen + 1);
    sprintf(pathname, "%s/%s/%s", loc_bin_dir, node_archname, filename);
    if( retrieve_file(dirname, filename, pathname, jid) != SUCCESS) {
      if (p_err_string[0] == '\0')
	sprintf(p_err_string, "Could not access file %s/%s: ", dirname, 
		filename);
      return ERRORCODE;
    }
  }
  else {
    if (dl = strlen(dirname)) {
      pathname = (char *)malloc(dl + strlen(filename) + 2);
      sprintf(pathname, "%s/%s", dirname, filename);
    }
    else
      pathname = filename;
  }

  /* Copy arguments to execle into strings */
  sprintf(tid_str,"%lu", tid);
  sprintf(jid_str,"%lu", jid);
  sprintf(port_str, "%hu", udp_port_num);

  if ((child_pid = vfork()) == -1) {
    sprintf(p_err_string, "%s: %s", pathname, sys_errlist[errno]);
    perror("vfork");
    return (ERRORCODE);    /* Could not create a child process */
  }
  else if (child_pid) {       /* PARENT PROCESS */
    
    if(prm_debug)
      fprintf(stderr,"(%s) Forking task: name=%s, jobid=%lu taskid=%lu, pid=%d.\n", 
	      _progname, pathname, jid, tid, child_pid);
    
    return (child_pid);     
  }
  
  else {      /* CHILD PROCESS */

    /* This is a workaround for EIO on sparcs */
    if ((bin_fd = open(pathname, O_WRONLY|O_CREAT, 0x1B4)) == -1)
      perror("open failed");
    else
      close(bin_fd);
    
    /* Overlay my code with requested program */
    if (dl) {
      if( execle(pathname, filename, tid_str, jid_str, port_str, 
		 (char *)0, envp) == -1)
	sprintf(p_err_string, "%s: %s", pathname, sys_errlist[errno]);
      fprintf(stderr, "(%s) execle %s ", _progname, pathname);
    }
    else {
      if( execlp(pathname, filename, tid_str, jid_str, port_str, 
		 (char *)0, envp) == -1)
	sprintf(p_err_string, "%s: %s", pathname, sys_errlist[errno]);
      fprintf(stderr, "(%s) execlp %s ", _progname, pathname);
    }
  }
  perror("exec");
  /* EXECV error -- return failure indicator */
  _exit (ERRORCODE);
}


retrieve_file(char *dir, char *fname, char *loc_file_path, u_long curjob_id)
{
  PTEXT pkt, rpkt;
  RREQ newreq;
  prm_node_addr_t fiotask_addr;
  u_short rplen, nlen;
  u_long nlong;
  struct stat stat_buf;
  char remote_path[MAXPATHLEN], *msg;
  int bin_fd, ret_code = SUCCESS;
  extern prm_node_addr_t get_fio_addr();

  /* stat the local file */
  
  if (stat(loc_file_path, &stat_buf) == -1)
    stat_buf.st_mtime = 0;

  /* Open loc_file */
  
  if( (bin_fd = open(loc_file_path, O_WRONLY|O_CREAT, 0x1B4)) == -1) {
    sprintf(p_err_string, "%s: %s", loc_file_path, sys_errlist[errno]);
    fprintf(stderr, "(%s) Could not open local file %s", _progname, 
	    p_err_string);
    return FAILURE;
  }

  /* Attempt to lock file. If fails, wait until lock released, periodically
     checking for other client requests. */

  if (lockf(bin_fd, F_TLOCK, 0) == -1) {
    if ((errno == EACCES) || (errno == EAGAIN)) { /* Someone else has 
						     locked it. So wait */
      while(lockf(bin_fd, F_TEST, 0) == -1) {
	ardp_accept();
	if (ardp_pendingQ)
	  NM_Process_Req((char **)0);
      }
      if (fstat(bin_fd, &stat_buf) == -1) {
	sprintf(p_err_string, "%s: %s", loc_file_path, sys_errlist[errno]);
	fprintf(stderr, "(%s) Could not open local file: %s", _progname, 
		p_err_string);
	ret_code = FAILURE;
      }
      else
	if (stat_buf.st_size == 0)  
	  ret_code = FAILURE;
    }
  }

  else {    /* I've opened file and locked it. Now need to perform transfer */
    fiotask_addr = get_fio_addr(curjob_id);
    if (!fiotask_addr || (fiotask_addr->sin_port == 0)) {
      ret_code = FAILURE;
    }
    else {
      sprintf(remote_path, "%s/%s", dir + 3, fname);
      rplen = strlen(remote_path) + 1;
      
      pkt = ardp_ptalloc();
      msg = pkt->start;
      *msg = (u_char)PDAP_VERSION; ++msg;
      *msg = (u_char)PDAP_FILE_XFER; ++msg;
      ENCODE_ASN1_LENGTH(msg,rplen);
      bcopy(remote_path, msg, rplen); msg+= rplen;
      ENCODE_ASN1_LENGTH(msg,stat_buf.st_mtime);
      pkt->length = msg - pkt->start;
      newreq = ardp_rqalloc();
      newreq->outpkt = pkt;
      
      /* Check for other requests (only if we have not gone into 
	 recursion) */
      if (rec_level == 0) {
	++rec_level;
	ardp_accept();
	while (ardp_pendingQ) 
	  NM_Process_Req();
	--rec_level;
      }
      ardp_send(newreq, 0, fiotask_addr, -1);
      ret_code = receive_file(newreq, bin_fd);
      ardp_rqfree(newreq);
    }
    lockf(bin_fd, F_ULOCK, 0);
  }
  
  close(bin_fd);
  return ret_code;
}


receive_file(RREQ req, int bin_fd)
{
  u_short emsglen;
  int i,  st, rlen;
  u_long rcvd_len, bytes_remng;
  PTEXT pkt_list;
  char *msg;

  if ( (pkt_list = req->rcvd) == NOPKT) 
    return FAILURE;
  msg = pkt_list->start + 2;
  DECODE_ASN1_LENGTH(msg,rlen); 
  if (*msg == SUCCESS) {
    ++msg;
    if (*msg == COPY_CRNT)
      return SUCCESS;
    else {
      ++msg;
      DECODE_ASN1_LENGTH(msg,rlen);
      if (PDAP_write_pkts_to_file(pkt_list->next, bin_fd, rlen, 0) == -1) 
	fprintf(stderr, "(%s) Could not write to local file %s", _progname, 
		p_err_string, (char *)0);
      else
	fchmod(bin_fd, 0x1C0);
      return SUCCESS;
    }
  }
  else {
    perrno = *(msg + 2);
    if (rlen > 3) {
      msg += 3;
      bcopy(msg, p_err_string, rlen - 3);
    }
    return FAILURE;
  }
}


prm_node_addr_t
  get_fio_addr(jid_t jid)
{
  prm_job_t curjob;
  prm_node_addr_t fiotask_addr;
  curjob = job_table;
  while (curjob && (curjob->jid != jid))
    curjob = curjob->next;
  
  if (curjob) {
    fiotask_addr = &(curjob->fioaddr);
    if ( (fiotask_addr->sin_port == 0) && (fiotask_addr->sin_addr.s_addr != 
					   inaddr) )
      get_l2p_from_remote(2, jid, 0, fiotask_addr);
    return fiotask_addr;
  }
  else {
    sprintf(p_err_string, "No entry for job %lu", jid);
    return (prm_node_addr_t)0;
  }
}
