/******************************************************************************
 *                                                                            *
 * Copyright (c) David L. Mills 1993                                          *
 *                                                                            *
 * Permission to use, copy, modify, and distribute this software and its      *
 * documentation for any purpose and without fee is hereby granted, provided  *
 * that the above copyright notice appears in all copies and that both the    *
 * copyright notice and this permission notice appear in supporting           *
 * documentation, and that the name University of Delaware not be used in     *
 * advertising or publicity pertaining to distribution of the software        *
 * without specific, written prior permission.  The University of Delaware    *
 * makes no representations about the suitability this software for any       *
 * purpose.  It is provided "as is" without express or implied warranty.      *
 *                                                                            *
 ******************************************************************************/

/*
 * Modification history kern_ntptime.c
 *
 * 28 Nov 93	David L. Mills
 *	Revised frequency scaling to conform with adjusted parameters
 *
 * 17 Sep 93	David L. Mills
 *	Created file
 */
/*
 * ntp_gettime(), ntp_adjtime() - precision time interface for Ultrix
 *
 * These routines consitute the Network Time Protocol (NTP) interfaces
 * for user and daemon application programs. The ntp_gettime() routine
 * provides the time, maximum error (synch distance) and estimated error
 * (dispersion) to client user application programs. The ntp_adjtime()
 * routine is used by the NTP daemon to adjust the system clock to an
 * externally derived time. The time offset and related variables set by
 * this routine are used by hardclock() to adjust the phase and
 * frequency of the phase-lock loop which controls the system clock.
 */

#include <sys/param.h>
#include <sys/user.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/timex.h>

/*
 * The following variables are used by the hardclock() routine in the
 * kern_clock.c module and are described in that module. 
 */
extern long time_offset;	/* time offset (usec) */
extern long time_freq;		/* frequency offset (scaled ppm) */
extern long time_maxerror;	/* maximum time error (usec) */
extern long time_esterror;	/* estimated time error (usec) */
extern int time_status;		/* clock synchronization status */
extern long time_constant;	/* pll time constant */
extern long time_precision;	/* clock precision (usec) */
extern long time_tolerance;	/* frequency tolerance (ppm) */

/*
 * ntp_gettime() - NTP user application interface
 */
ntp_gettime()
{
	register struct a {
		struct ntptimeval *tp;
	} *uap = (struct a *)u.u_ap;
	struct timeval atv;
	struct ntptimeval ntv;

	if (uap->tp) {
		splclock();
		microtime(&atv);
		spl0();
		ntv.time = atv;
		ntv.maxerror = time_maxerror;
		ntv.esterror = time_esterror;
		u.u_error = copyout((caddr_t)&ntv, (caddr_t)uap->tp,
			sizeof (ntv));
	}
	if (!u.u_error)
		u.u_r.r_val1 = time_status;
}

/*
 * ntp_adjtime() - NTP daemon application interface
 */
ntp_adjtime()
{
	register struct a {
		struct timex *tp;
	} *uap = (struct a *)u.u_ap;
	struct timex ntv;
	int s;

	u.u_error = copyin((caddr_t)uap->tp, (caddr_t)&ntv, sizeof(ntv));
	if (u.u_error)
		return;

	/*
	 * Update selected clock variables - only the superuser can
	 * change anything.
	 */
	if (!suser() && ntv.mode != 0)
		return;
	s = splclock();
	smp_lock(&lk_timeout, LK_RETRY);
	if (ntv.mode & ADJ_OFFSET)
		hardupdate(ntv.offset);
	smp_unlock(&lk_timeout);
	(void) splx(s);
	if (ntv.mode & ADJ_FREQUENCY) {
		if (time_freq >= 0)
			time_freq = ntv.frequency >> (SHIFT_USEC - SHIFT_KF);
		else
			time_freq = -(-ntv.frequency >> (SHIFT_USEC - SHIFT_KF));
	}
	if (ntv.mode & ADJ_MAXERROR)
		time_maxerror = ntv.maxerror;
	if (ntv.mode & ADJ_ESTERROR)
		time_esterror = ntv.esterror;
	if (ntv.mode & ADJ_STATUS)
		if (time_status == TIME_OK || ntv.status == TIME_BAD)
			time_status = ntv.status;
	if (ntv.mode & ADJ_TIMECONST)
		time_constant = ntv.time_constant;

	/*
	 * Retrieve all clock variables
	 */
	if (time_offset >= 0)
		ntv.offset = time_offset >> SHIFT_UPDATE;
	else
		ntv.offset = -(-time_offset >> SHIFT_UPDATE);
	ntv.frequency = time_freq << (SHIFT_USEC - SHIFT_KF);
	ntv.maxerror = time_maxerror;
	ntv.esterror = time_esterror;
	ntv.status = time_status;
	ntv.time_constant = time_constant;
	ntv.precision = time_precision;
	ntv.tolerance = time_tolerance;
	u.u_error = copyout((caddr_t)&ntv, (caddr_t)uap->tp, sizeof(ntv));
	if (!u.u_error)
		u.u_r.r_val1 = time_status;
}

