#ifndef lint
static char *sccsid = "@(#)clock.c	4.1      (ULTRIX)  8/9/90";
#endif lint
/************************************************************************
 *									*
 *			Copyright (c) 1989 by				*
 *		Digital Equipment Corporation, Maynard, MA		*
 *			All rights reserved.				*
 *									*
 *   This software is furnished under a license and may be used and	*
 *   copied  only  in accordance with the terms of such license and	*
 *   with the  inclusion  of  the  above  copyright  notice.   This	*
 *   software  or  any  other copies thereof may not be provided or	*
 *   otherwise made available to any other person.  No title to and	*
 *   ownership of the software is hereby transferred.			*
 *									*
 *   The information in this software is subject to change  without	*
 *   notice  and should not be construed as a commitment by Digital	*
 *   Equipment Corporation.						*
 *									*
 *   Digital assumes no responsibility for the use  or  reliability	*
 *   of its software on equipment which is not supplied by Digital.	*
 *									*
 ************************************************************************/

/*
 * Clock routines.
 * Modification History: clock.c
 *
 * 9-Sep-93	David Mills, Univerrsity of Delaware
 *	Added support for microsecond time resolution using the ioasic bus
 *	counter and rewritten microtime() routine. Added microset() routine
 *	so hardclock() can latch the counter at each tick. This code is
 *	enabled by the MICRO define.
 *
 * 27-Mar-90	Randall Brown
 *	Created file to handle generic clock routines.  These routines 
 *	in turn call to system specific clock routines.
 *
 */

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/kernel.h"

#include "../machine/cpu.h"
#include "../../machine/common/cpuconf.h"
#include "../machine/clock.h"

#ifdef MICRO
#include "../io/tc/ioasic.h"
#endif /* MICRO */

extern int cpu;
extern struct cpusw *cpup;
extern struct timeval *timepick;

#define	TEST_PERIOD	500

config_delay()
{
	DELAY(TEST_PERIOD);
}

#ifdef MICRO
/*
 * Tables to map the ioasic bus counter to microseconds. These are
 * useful only with Alpha and those MIPS machines with the ioasic bus
 * counter, including the 5000/240. Therefore, the MICRO define should be
 * used only with these machines.
 *
 * The bus counter increments at a 25-MHz rate. At the alpha timer
 * interrupt rate of 1024 Hz, the tick interval is 24414.0625 cycles (16
 * bits) or 976.5625 microseconds (10 bits), while at the mips timer
 * interrupt rate of 256 Hz, the tick interval is 97656.25 cycles (17
 * bits) or 3906.25 microseconds (12 bits). We include enough bits here
 * to work with either timer interrupt rate. In order to minimize the
 * size of these tables while maintaining conversion accuracy of +-1
 * microsecond, the low-order 5 bits of the bus counter are dropped. In
 * order to avoid extra compare cycles, the high-order table is 16 words
 * longer than needed. These tables are truncated, not rounded.
 *
 *   16  15  14  13  12  11  10  9   8   7   6   5   4   3   2   1   0
 * +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 * | h | h | h | h | h | h | l | l | l | l | l | l | x | x | x | x | x |
 * +---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+---+
 *
 * h = high-order table, l = lor-order table, x = don't care
 */
/*
 * low-order six bits
 */
unsigned int bus2uslo[] = {
	0,    1,    2,    3,    5,    6,    7,    8,	/* 0 */
	10,   11,   12,   14,   15,   16,   17,   19,	/* 256 */
	20,   21,   23,   24,   25,   26,   28,   29,	/* 512 */
	30,   32,   33,   34,   35,   37,   38,   39,	/* 768 */
	40,   42,   43,   44,   46,   47,   48,   49,	/* 1024 */
	51,   52,   53,   55,   56,   57,   58,   60,	/* 1280 */
	61,   62,   64,   65,   66,   67,   69,   70,	/* 1536 */
	71,   72,   74,   75,   76,   78,   79,   80};	/* 1792 */

/*
 * high-order six bits
 */
unsigned int bus2ushi[] = {
	0,   81,  163,  245,  327,  409,  491,  573,	/* 0 */
	655,  737,  819,  901,  983, 1064, 1146, 1228,	/* 16384 */
	1310, 1392, 1474, 1556, 1638, 1720, 1802, 1884,	/* 32768 */
	1966, 2048, 2129, 2211, 2293, 2375, 2457, 2539,	/* 49152 */
	2621, 2703, 2785, 2867, 2949, 3031, 3112, 3194,	/* 65536 */
	3276, 3358, 3440, 3522, 3604, 3686, 3768, 3850,	/* 81920 */
	3932, 4014, 4096, 4177, 4259, 4341, 4423, 4505,	/* 98304 */
	4587, 4669, 4751, 4833, 4915, 4997, 5079, 5160}; /* 114688 */

/*
 * Macro to convert 40-ns bus cycles (bus) to microseconds (us)
 *
 * Requires conversion tables bus2uslo[] and bus2ushi[].
 */
#define BUSTOUS(bus, us) \
	(us) = bus2uslo[(bus >> 5) & 0x3f] + bus2ushi[(bus >> 11) & 0x3f]

long bus_counter = 0;		/* ioasic bus counter */
extern unsigned int scc_ioasic_base; /* variable base address of ioasic */
#endif /* MICRO */

/*
 * Return the best possible estimate of the time in the timeval to which
 * tvp points. The kernel time variable is interpolated between ticks by
 * reading the ioasic counter register to determine the number of 40-ns
 * bus cycles since the last timer tick, then converting the result to
 * microseconds.
 *
 * As the result of clock phase adjustments, the apparent time may
 * decreas by a small amount; therefore, the reported time is adjusted to
 * be greater than the last previously reported time. However, if this
 * routine is called too often, the reported time could recede indefinitely
 * into the future; therefore, the positive adjustment is limited to less
 * than one tick in microseconds. Finally, if the apparent time is more
 * than one second behind the last reported time, it is assumed the time
 * has been changed elsewhere, so the apparent time must be presumed correct.
 */
microtime(tvp)
	register struct timeval *tvp;
{
	static struct timeval lasttime;
	int s;
	long sec, usec;

	s = splclock();
	*tvp = *timepick;
#ifdef MICRO
	IOC_RD(IOC_CTR, usec);
	if (usec >= bus_counter)
		usec = usec - bus_counter;
	else
		usec = bus_counter - usec;
	BUSTOUS(usec, usec);
	tvp->tv_usec += usec;
	if (tvp->tv_usec >= 1000000) {
		tvp->tv_usec -= 1000000;
		tvp->tv_sec++;
	}
#endif /* MICRO */
	sec = lasttime.tv_sec - tvp->tv_sec;
	usec = lasttime.tv_usec - tvp->tv_usec;
	if (usec < 0) {
		usec += 1000000;
		sec--;
	}
	if (sec == 0) {
		if (usec > tick - 2)
			usec = tick - 2;
		tvp->tv_usec += usec + 1;
		if (tvp->tv_usec >= 1000000) {
			tvp->tv_usec -= 1000000;
			tvp->tv_sec++;
		}
	}
	lasttime = *tvp;
	splx (s);
}

#ifdef MICRO
/*
 * This routine is called once each tick by hardclock() to remember the
 * bus counter so microtime() can calculate the microseconds since the
 * last tick.
 */
microset()
{
	IOC_RD(IOC_CTR, bus_counter);
}
#endif /* MICRO */

/*
 * Initialize the time of day register, based on the time base which is
 * from a filesystem.  Base provides the time to within six months.
 */
inittodr(base)
	time_t base;
{
	register u_int todr;
	long deltat;
	int year = YRREF;
	int checktodr();
	extern long savetime;   /* used to save the boot time for errlog */

	/*
	 * Once a day check for clock rollover
	 */
	timeout(checktodr, 0, SECDAY*hz);

	todr = read_todclk();	

	/*
	 * NOTE: 
	 * On pmax & 3max, read_todclk returns seconds from the
	 *    beginning of the year + (1 << 26).
	 * On Mipsfair or Isis (with 10mS VAX time of day clock),
 	 *    read_todclk returns the number of 10mS ticks from the
	 *    beginning of the year + (1 << 28).
	 */
	if (base < 5*SECYR) {
		printf("WARNING: preposterous time in file system");
		time.tv_sec = 6*SECYR + 186*SECDAY + SECDAY/2;
		resettodr();
		goto check;
	}
	/*
	 * cpup->todrzero is base used to detected loss of power to TODCLK
	 */
	if (todr < cpup->todrzero) {
		printf("WARNING: lost battery backup clock");
		time.tv_sec = base;
		/*
		 * Believe the time in the file system for lack of
		 * anything better, resetting the TODR.
		 */
		resettodr();
		goto check;
	}

	/*
	 * Sneak to within 6 month of the time in the filesystem,
	 * by starting with the time of the year suggested by the TODR,
	 * and advancing through succesive years.  Adding the number of
	 * seconds in the current year takes us to the end of the current year
	 * and then around into the next year to the same position.
	 */

	time.tv_sec = (todr - cpup->todrzero) / cpup->rdclk_divider;

	while (time.tv_sec < base-SECYR/2) {
		if (LEAPYEAR(year))
			time.tv_sec += SECDAY;
		year++;
		time.tv_sec += SECYR;
	}

	/*
	 * See if we gained/lost two or more days;
	 * if so, assume something is amiss.
	 */
	deltat = time.tv_sec - base;
	if (deltat < 0)
		deltat = -deltat;
	if (deltat >= 2*SECDAY) {
		printf("WARNING: clock %s %d days",
		time.tv_sec < base ? "lost" : "gained", deltat / SECDAY);
	}
	resettodr();
	savetime = time.tv_sec;
	return;
check:
	savetime = time.tv_sec;
	printf(" -- CHECK AND RESET THE DATE!\n");
	return;
}

/*
 * checktodr -- check for clock rollover and reset if necessary
 */
checktodr()
{
	timeout(checktodr, 0, SECDAY*hz);	/* Check again tomorrow */
	if ((read_todclk()/cpup->rdclk_divider) > SECYR+(2*SECDAY))
	    resettodr();
}
	

/*
 * Reset the TODR based on the time value; used when the TODR
 * has a preposterous value and also when the time is reset
 * by the stime system call.  Also called when the TODR goes past
 * cpup->todrzero + 100*(SECYR+2*SECDAY) (e.g. on Jan 2 just after midnight)
 * to wrap the TODR around.
 */
resettodr()
{
	int year = YRREF;
	u_int secyr;
	u_int yrtime;
	int s;
	
	s = splclock();
	yrtime = time.tv_sec;
	splx(s);

	/*
	 * Whittle the time down to an offset in the current year,
	 * by subtracting off whole years as long as possible.
	 */
	for (;;) {
		secyr = SECYR;
		if (LEAPYEAR(year))
			secyr += SECDAY;
		if (yrtime < secyr)
			break;
		yrtime -= secyr;
		year++;
	}
	write_todclk(yrtime);
}
