/******************************************************************************
 *                                                                            *
 * Copyright (c) David L. Mills 1993                                          *
 *                                                                            *
 * Permission to use, copy, modify, and distribute this software and its      *
 * documentation for any purpose and without fee is hereby granted, provided  *
 * that the above copyright notice appears in all copies and that both the    *
 * copyright notice and this permission notice appear in supporting           *
 * documentation, and that the name University of Delaware not be used in     *
 * advertising or publicity pertaining to distribution of the software        *
 * without specific, written prior permission.  The University of Delaware    *
 * makes no representations about the suitability this software for any       *
 * purpose.  It is provided "as is" without express or implied warranty.      *
 *                                                                            *
 ******************************************************************************/

/*
 * Modification history kern_ntptime.c
 *
 * 28 Nov 93    David L. Mills
 *	Revised frequency scaling to conform with adjusted parameters
 *
 * 17 Sep 93	David L. Mills
 *	Created file
 */
/*
 * ntp_gettime(), ntp_adjtime() - precision time interface for OSF/1
 *
 * These routines consitute the Network Time Protocol (NTP) interfaces
 * for user and daemon application programs. The ntp_gettime() routine
 * provides the time, maximum error (synch distance) and estimated error
 * (dispersion) to client user application programs. The ntp_adjtime()
 * routine is used by the NTP daemon to adjust the system clock to an
 * externally derived time. The time offset and related variables set by
 * this routine are used by hardclock() to adjust the phase and
 * frequency of the phase-lock loop which controls the system clock.
 */

#include <sys/param.h>
#include <sys/user.h>
#include <sys/kernel.h>
#include <sys/proc.h>
#include <sys/timex.h>

/*
 * The following variables are used by the hardclock() routine in the
 * kern_clock.c module and are described in that module. 
 */
extern long time_offset;	/* time offset (usec) */
extern long time_freq;		/* frequency offset (scaled ppm) */
extern long time_maxerror;	/* maximum time error (usec) */
extern long time_esterror;	/* estimated time error (usec) */
extern int time_status;		/* clock synchronization status */
extern long time_constant;	/* pll time constant */
extern long time_precision;	/* clock precision (usec) */
extern long time_tolerance;	/* frequency tolerance (ppm) */

/*
 * ntp_gettime() - NTP user application interface
 */
ntp_gettime(p, args, retval)
	struct proc *p;
	void *args;
	long *retval;
{
	register struct args {
		struct ntptimeval *tp;
        } *uap = (struct args *) args;
	struct timeval atv;
	struct ntptimeval ntv;
	int error = 0;

	if (uap->tp) {
		microtime(&atv);
		ntv.time = atv;
		ntv.maxerror = time_maxerror;
		ntv.esterror = time_esterror;
		if (error = copyout((caddr_t)&ntv, (caddr_t)uap->tp,
		    sizeof (ntv)))
			return (error);
	}
	return (time_status);
}

/*
 * ntp_adjtime() - NTP daemon application interface
 */
ntp_adjtime(p, args, retval)
	struct proc *p;
	void *args;
	long *retval;
{
	register struct args {
		struct timex *tp;
	} *uap = (struct args *) args;
	struct timex ntv;
	int s;
	int error = 0;

	if (uap->tp)
		if (error = copyin((caddr_t)uap->tp, (caddr_t)&ntv,
		    sizeof(ntv)))
			return (error);

	/*
	 * Update selected clock variables - only the superuser can
	 * change anything.
	 */
	if (ntv.status)
		if ((error = suser(u.u_cred, &u.u_acflag)))
			return (error);
	s = splclock();
	if (ntv.mode & ADJ_OFFSET)
		hardupdate(ntv.offset);
	(void) splx(s);
	if (ntv.mode & ADJ_FREQUENCY) {
		if (time_freq >= 0)
			time_freq = ntv.frequency >> (SHIFT_USEC - SHIFT_KF);
		else
			time_freq = -(-ntv.frequency >> (SHIFT_USEC - SHIFT_KF));
	}
	if (ntv.mode & ADJ_MAXERROR)
		time_maxerror = ntv.maxerror;
	if (ntv.mode & ADJ_ESTERROR)
		time_esterror = ntv.esterror;
	if (ntv.mode & ADJ_STATUS)
		if (time_status == TIME_OK || ntv.status == TIME_BAD)
			time_status = ntv.status;
	if (ntv.mode & ADJ_TIMECONST)
		time_constant = ntv.time_constant;

	/*
	 * Retrieve all clock variables
	 */
	if (time_offset >= 0)
		ntv.offset = time_offset >> SHIFT_UPDATE;
	else
		ntv.offset = -(-time_offset >> SHIFT_UPDATE);
	ntv.frequency = time_freq << (SHIFT_USEC - SHIFT_KF);
	ntv.maxerror = time_maxerror;
	ntv.esterror = time_esterror;
	ntv.status = time_status;
	ntv.time_constant = time_constant;
	ntv.precision = time_precision;
	ntv.tolerance = time_tolerance;
	if (error = copyout((caddr_t)&ntv, (caddr_t)uap->tp, sizeof(ntv)))
		return (error);
	return (time_status);
}

