/*==============================================================================
   WINMEM.C

   A Global Memory Manager for Windows 3.0

   Sub Systems, Inc.
   Software License Agreement  (1990)              
   ----------------------------------

   The license agreement allows the purchaser the right to use
   the manual and this software.  The source code
   of WinMem functions may be incorporated into the developer's
   software application.  The distribution of such an
   application software containing WinMem in an executable
   format is royalty free.  The developer may not distribute
   any part of WinMem software in a source format.  Sub
   Systems, Inc. reserves the right to prosecute anybody
   violating aforementioned terms.

===============================================================================*/
#include "windows.h"
#include "stdio.h"
#include "stdlib.h"
#include "ctype.h"
#include "io.h"
#include "fcntl.h"
#include "sys\types.h"
#include "sys\stat.h"
#include "dos.h"
#include "errno.h"
#include "string.h"

/*****************************************************************************
   Compiler specific initialization action
******************************************************************************/
#if defined(__TURBOC__)
   #include "dir.h"
   #include "mem.h"
   #define _makepath fnmerge
   #define _splitpath fnsplit
   extern int _argc;
   extern char  **_argv;
#else
   #include "memory.h"
#endif

#include "winmem.h"

/********** Prototypes of internally used functions ***********************/
int WmToEms(int,WORD,LPSTR,long,long);
int WmFromEms(int,WORD,long,LPSTR,long);
WORD WmEmsFrame(); 
LPSTR WmGetCurDir(LPSTR);

/********************** Define Statements **********************************/
#define WIN_BLOCK_SIZE  8196
#define WIN_MAX_BLOCKS  400
#define WIN_DISCARDED   0x1
#define WIN_NEED_FIX    0x2
#define WIN_EMPTIED     0x4
#define WIN_IN_USE      0x1
#define WIN_LAST_BLOCK  0x2
#define WIN_SIGNATURE   0xD4

#define GetSegment(x)   (WORD)((DWORD)x>>16)
#define GetOffset(x)    (WORD)(((DWORD)x<<16)>>16)
#define SetFlag(x,y)    *x=(*x)|y
#define ResetFlag(x,y)  *x=(*x)&(~y)

/********************* Global Variables **************************************/

struct StrWmGlobMem {              /* structure of each global memory data block */
   HANDLE handle;              /* Global memory handle */
   WORD free;                  /* free memory available */
   WORD FreeOffset;            /* offset of the first free block */
   BYTE flags;                 /* maintains lock status etc. */
   LPSTR pMem;                 /* pointer to memory if locked */
   int   LockCount;            /* current number of lock on this superblock */
   }WmGlobMem[WIN_MAX_BLOCKS];   

int WmGlobMemUsed;

struct StrWmMem {                 /* structure TER memory block header */
   WORD size;                  /* memory block size */
   BYTE signature;             /* an identification of a TER memory block */
   BYTE flags;                 /* in use, last block indicatiors */
   } far *pWmMem;

char       WmSwapFile[128];    /* Disk swap file name */
BOOL       SwapOutCalled=FALSE;/* indicates whether SwapOut process instances has been made once.*/
int        hWmSwapFile;        /* handle to the disk swap file */
LPCATCHBUF sFatalError;        /* exit point in case of a fatal memory error */
WORD       WmAllocFlags;       /* memory allocation flags */
WORD       WmCurBlock;         /* currently locked block */

WORD       WmEmsPercent;       /* percentage of available EMS to utilize */
BOOL       WmEmsUsed;          /* TRUE indicates that EMS is being used */
int        hWmEms;             /* handle to the EMS memory */
WORD       WmEmsCutoff;        /* blocks stored in the EMS memory */
WORD       WmEmsSeg;           /* EMS frame address */
/******************************************************************************
   sInit:
   This routine initializes WinMem package.  This routine must be callled once
   in the beginning of your program.  
   The routine a) registers the Program Exit buffer (long jump on fatal errors)
               b) initialize global memory use count,
               c) opens the disk swap file (in Real and Standard mode only ).
               d) checks for the presense of EMS hardware.
   
   Note: When the 'EmsPercent' argument is a non-zero parameters,  you MUST
   use -L switch on the resource compilation command line.  This switch tells
   Windows' that your application uses EMS memory directly. Example:
            rc -L myappl.res                        
*******************************************************************************/
void sInit(PSTR SwapDir,HANDLE hInst,LPCATCHBUF FatalExit,WORD EmsPercent)
{

    FARPROC lpProcSwap;
    char string[128];

    
    sFatalError=FatalExit; /* Jump to this location to terminate the application
                              if a memory corruption error is detected. */

    WmGlobMemUsed=1;
    WmCurBlock=0;          /* reset currently locked block */
    WmEmsPercent=0;        /* percentage of available EMS memory to use */
    WmEmsUsed=FALSE;
    WmEmsCutoff=0;         /* number of blocks to be stored in EMS memory */

    if (WF_ENHANCED&GetWinFlags()) {  /* Windows in running in the 386 Enhance mode */
        WmAllocFlags=GMEM_MOVEABLE;
    }
    else {                            /* enable disk swapping for Real and Standard modes */
        WmAllocFlags=GMEM_MOVEABLE|GMEM_DISCARDABLE|GMEM_NOTIFY;

        if (strlen(SwapDir)==0)  WmGetCurDir((LPSTR)WmSwapFile);
        else                     strcpy(WmSwapFile,SwapDir);     /* build swap file path name */
        
        if (strlen(WmSwapFile)!=0) {
           if (WmSwapFile[strlen(WmSwapFile)-1]!='\\') strcat(WmSwapFile,"\\");
        }

        strcpy(string,"WMXXXXXX");   /* make a unique file name */
        mktemp(string);
        strcat(WmSwapFile,string);

        if ((hWmSwapFile=open(WmSwapFile,O_BINARY|O_RDWR|O_CREAT,S_IREAD|S_IWRITE))==-1) {
            wsprintf(string,"Could Not Create the Swap File: %s",(LPSTR)WmSwapFile);
            sAbort((PSTR)string,17);
        }
        close(hWmSwapFile);            /* the file needs be opened and closed 
                                          for each swapping incidence. */
              
        /* install a process to handle memory discards */
        if (!SwapOutCalled) { /* GlobalNotify may not be called more than once per instances */
            lpProcSwap = MakeProcInstance(SwapOut, hInst);
            GlobalNotify(lpProcSwap);
            SwapOutCalled=TRUE;
        }
        WmEmsPercent=EmsPercent;                   /* percent of EMS to use */
        
        if (WmEmsPercent && !WmCheckEms()) WmEmsPercent=0;  /* EMS hardware/software not operational */
    }

}
/******************************************************************************
   sAlloc:
   Allocate a block of memory for a text line.  The memory is allocated from
   TER arrays of global buffer memory.  The routine returns a handle (DWORD) 
   to the allocated block.
   The requested block size may not execeed (WIN_BLOCK_SIZE - 4) bytes.
   Logic:
     1.  Scan through WmGlobMem array to find an entry with free memory equal
         to or greater than the request amount of memory.
     2.  If no such global memory entry found, then allocate a new block from
         Windows.
     3.  Break the existing unused block of memory in to two pieces.  The pointer 
         to the first block will be returned as the result of sAlloc.  The second
         block of memory will be marked as 'free' for a later use.
*******************************************************************************/
DWORD sAlloc(WORD bytes)
{
    WORD i,CurBlock,offset,BytesLeft;
    BOOL BlockFound=FALSE,LastBlock,TempLock;
    DWORD result;

    for (i=1;i<WmGlobMemUsed;i++) {  /* find a block with available space */
       if (WmGlobMem[i].free>=(bytes+sizeof(struct StrWmMem))) { 

           TempLock=FALSE;
           if (WmGlobMem[i].flags&WIN_DISCARDED) {
               if (WmGlobMem[i].free<5*(bytes+sizeof(struct StrWmMem))) continue; /* don't do disk read unless ample memory available */
               SwapIn(i);
           }
           
           if (!WmGlobMem[i].LockCount) {  /* lock the memory */
               if  (NULL==(WmGlobMem[i].pMem=GlobalLock(WmGlobMem[i].handle))) {
                   sAbort("Error in obtaining global memory lock(sAlloc)",3);
               }
               TempLock=TRUE;
           }
           if (WmGlobMem[i].FreeOffset<WIN_BLOCK_SIZE) offset=WmGlobMem[i].FreeOffset;else offset=0;
           while (TRUE) {  /* examine each memory block in the chain */
               pWmMem=(struct StrWmMem far *) &(WmGlobMem[i].pMem[offset]);
               if (!(pWmMem->flags&WIN_IN_USE) && pWmMem->size>=bytes) {
                  CurBlock=i;
                  BlockFound=TRUE;
                  break;
               }
               if (pWmMem->flags&WIN_LAST_BLOCK) break; /* end of chain */
               offset=offset+pWmMem->size+sizeof(struct StrWmMem);
           }
           if (BlockFound) break;
           else if (TempLock) { /* remove the temporary lock on this block */
              GlobalUnlock(WmGlobMem[i].handle);
              TempLock=FALSE;
           }
       }
    }
   
    if (!BlockFound) { /* Allocate a new global block */
       
       for (i=1;i<WmGlobMemUsed;i++) {  /* look for an emptied slot */
           if (WmGlobMem[i].flags&WIN_EMPTIED && WmGlobMem[i].free==0) {
              BlockFound=TRUE;
              break;
           }
       }
       if (!BlockFound) {
           i=WmGlobMemUsed;
           WmGlobMemUsed++;
           if (WmGlobMemUsed>WIN_MAX_BLOCKS) {
               sAbort("Ran Out Of Table Space to Hold Global Memory (sAlloc)",2);
           }
       }
       
       WmGlobMem[i].free=WIN_BLOCK_SIZE;
       WmGlobMem[i].LockCount=0;

       if (  (NULL==(WmGlobMem[i].handle=GlobalAlloc(WmAllocFlags,WIN_BLOCK_SIZE)))
          || (NULL==(WmGlobMem[i].pMem=GlobalLock(WmGlobMem[i].handle))) ) {
           sAbort("Ran Out of Memory (sAlloc)",3);
       }

       TempLock=TRUE;
       WmGlobMem[i].flags=0;
       WmGlobMem[i].FreeOffset=WIN_BLOCK_SIZE;    /* free offset not set yet */

       /* initialize the allocated block */
       pWmMem=(struct StrWmMem far *)WmGlobMem[i].pMem;
       pWmMem->size=WmGlobMem[i].free-sizeof(struct StrWmMem);
       pWmMem->signature=0;  /* identifies a TER block */
       pWmMem->flags=0;
       SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
       CurBlock=i;
       offset=0;
    }

    /* break the current block in two pieces */
    if (WmGlobMem[CurBlock].FreeOffset==offset) WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE; /* reset the first free block */
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    BytesLeft=pWmMem->size-bytes;  /* bytes left for the next block */
    pWmMem->size=bytes;
    if (BytesLeft<sizeof(struct StrWmMem)) {
        pWmMem->size+=BytesLeft;
        BytesLeft=0;
    }

    
    SetFlag(&(pWmMem->flags),WIN_IN_USE);
    WmGlobMem[CurBlock].free=WmGlobMem[CurBlock].free-pWmMem->size-sizeof(struct StrWmMem);
    if (pWmMem->flags&WIN_LAST_BLOCK) LastBlock=TRUE;else LastBlock=FALSE;
    if (LastBlock && BytesLeft!=0) ResetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
    pWmMem->signature=WIN_SIGNATURE; 
    

    result=((DWORD)CurBlock<<16)+offset;

    if (BytesLeft>0) { /* initialize the next block */
       offset=offset+pWmMem->size+sizeof(struct StrWmMem);
       pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
       pWmMem->size=BytesLeft-sizeof(struct StrWmMem);
       pWmMem->signature=0;
       pWmMem->flags=0;
       if (LastBlock) SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
       if (WmGlobMem[CurBlock].FreeOffset==WIN_BLOCK_SIZE || WmGlobMem[CurBlock].FreeOffset>offset) 
           WmGlobMem[CurBlock].FreeOffset=offset;   /* set offset to the first free block */
    }
    
    if (TempLock) WmSwitchLock(CurBlock);   /* keep the latest used block in lock */
    
    return result;
}

/******************************************************************************
    WmSwitchLock:
    This routine switches the internal lock to the given block.

    This routine is called only internally within the WinMem package.
*******************************************************************************/
WmSwitchLock(WORD CurBlock)
{
    if (WmCurBlock!=CurBlock) {  
       if (WmCurBlock!=0) {  /* unlock the previous internal lock */
           WmGlobMem[WmCurBlock].LockCount--;
           if (GlobalFlags(WmGlobMem[WmCurBlock].handle)&GMEM_LOCKCOUNT) GlobalUnlock(WmGlobMem[WmCurBlock].handle);
           if (WmGlobMem[WmCurBlock].LockCount<=0) {
               while (GlobalFlags(WmGlobMem[WmCurBlock].handle)&GMEM_LOCKCOUNT) GlobalUnlock(WmGlobMem[WmCurBlock].handle);
               WmGlobMem[WmCurBlock].pMem=NULL;
               WmGlobMem[WmCurBlock].LockCount=0;
           }
       }     
       WmCurBlock=CurBlock;
       WmGlobMem[CurBlock].LockCount++;
    }

    return TRUE;
}

/******************************************************************************
    sReAlloc:
    Reallocate a block of memory for a text line.  The routine reallocates the
    memory for the existing block.  The routine moves the contents of the old
    block to the new block as much as it can fit in.  The routine returns a 
    handle (DWORD) to the allocated memory.
*******************************************************************************/
DWORD sReAlloc(DWORD hMem,WORD NewSize)
{
    WORD CurBlock,offset,NewBlock,NewOffset,OldSize,NextBlockSize,NextOffset;
    LPSTR OldLocation,NewLocation,SaveLocation;
    HANDLE hTempMem;
    BOOL  TempLock=FALSE,LastBlock;
    DWORD hStubMem;
    struct StrWmMem far *pNextMem;


    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);
    
    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to Reallocated (sREAlloc)",37);
    }
    /****** retrieve the old block ******************************/
    if (WmGlobMem[CurBlock].flags&WIN_DISCARDED) SwapIn(CurBlock);
    
    if (!WmGlobMem[CurBlock].LockCount) {    /* lock the current block */
        if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
            sAbort("Error in obtaining global memory lock(sFree)",14);
        }
        TempLock=TRUE;
    }
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    OldSize=pWmMem->size;
    if (pWmMem->flags&WIN_LAST_BLOCK) LastBlock=TRUE;else LastBlock=FALSE;
    OldLocation=&(WmGlobMem[CurBlock].pMem[offset+sizeof(struct StrWmMem)]);

    /******** when reallocating to a smaller block ***************************/
    if (NewSize<=OldSize) {
        if ((NewSize+sizeof(struct StrWmMem))>=OldSize) {  /* difference too small */
           if (TempLock) { /* remove the temporary lock on this block */
              GlobalUnlock(WmGlobMem[CurBlock].handle);
           }
           return hMem;
        }
        
        /********** reduce the size of the current block **************/
        pWmMem->size=NewSize;
        if (LastBlock) ResetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
        
        /********* create a stub block ********************************/
        offset=offset+pWmMem->size+sizeof(struct StrWmMem);
        pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
        pWmMem->size=OldSize-NewSize-sizeof(struct StrWmMem);
        pWmMem->signature=WIN_SIGNATURE;
        pWmMem->flags=0;
        SetFlag(&(pWmMem->flags),WIN_IN_USE);   /* to eliminate fragmentation, freeup later with sFree */
        if (LastBlock) SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
        
        hStubMem=CurBlock;            /* make an handle to the stub block */
        hStubMem=(hStubMem<<16)+offset;

        if (TempLock) {               /* remove the temporary lock on this block */
           GlobalUnlock(WmGlobMem[CurBlock].handle);
        }
        sFree(hStubMem);              /* freeup the stub block */

        return hMem;                  /* return the old handle */
    }
    
    /**************** allocating to a larger block **************************/
    NextBlockSize=0;
    if (!LastBlock) {  /* check if the next available block is a free block */
       NextOffset=offset+pWmMem->size+sizeof(struct StrWmMem);
       pNextMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[NextOffset]);
       if (!(pNextMem->flags&WIN_IN_USE)) {
          NextBlockSize=pNextMem->size+sizeof(struct StrWmMem);
          if (pNextMem->flags&WIN_LAST_BLOCK) LastBlock=TRUE;else LastBlock=FALSE;
       }
    }
    if (NewSize<=(OldSize+NextBlockSize)) {
       
       pWmMem->size=NewSize;      /* extend the size of the current block */
       NextBlockSize=NextBlockSize-(NewSize-OldSize);
       WmGlobMem[CurBlock].free=WmGlobMem[CurBlock].free-(NewSize-OldSize);
       if (WmGlobMem[CurBlock].FreeOffset==NextOffset) WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE;

       if (NextBlockSize<=sizeof(struct StrWmMem)) { /* merge the next block with the current block */
          pWmMem->size=pWmMem->size+NextBlockSize;
          WmGlobMem[CurBlock].free=WmGlobMem[CurBlock].free-NextBlockSize;
          if (LastBlock) SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
       }
       else {                                       /* define a smaller next block */
           NextOffset=offset+pWmMem->size+sizeof(struct StrWmMem);
           pNextMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[NextOffset]);
           pNextMem->size=NextBlockSize-sizeof(struct StrWmMem);
           pNextMem->signature=0;
           pNextMem->flags=0;
           if (LastBlock) SetFlag(&(pNextMem->flags),WIN_LAST_BLOCK);
           
           if (WmGlobMem[CurBlock].FreeOffset==WIN_BLOCK_SIZE) 
               WmGlobMem[CurBlock].FreeOffset=NextOffset;   /* set offset to the first free block */
       }

       if (TempLock) { /* remove the temporary lock on this block */
          GlobalUnlock(WmGlobMem[CurBlock].handle);
       }
       return hMem;
    }

    /***** move the block to a global location ******************/
    if (OldSize!=0) {
       if (  (NULL==(hTempMem=GlobalAlloc(GMEM_MOVEABLE,OldSize)))
          || (NULL==(SaveLocation=GlobalLock(hTempMem))) ) {
           sAbort("Ran Out of Memory (sReAlloc)",15);
       }
       FarMove(OldLocation,SaveLocation,OldSize);
    }
    /******* free the old block and allocated the new block **********/
    if (TempLock) { /* remove the temporary lock on this block */
       GlobalUnlock(WmGlobMem[CurBlock].handle);
    }
    
    sFree(hMem);

    hMem=sAlloc(NewSize);
    NewLocation=sLock(hMem);
    if (min(NewSize,OldSize)>0) FarMove(SaveLocation,NewLocation,min(NewSize,OldSize));
    if (OldSize!=0) {
        GlobalUnlock(hTempMem);
        GlobalFree(hTempMem);
    }
    sUnlock(hMem);

    return hMem;
}

/*****************************************************************************
    sFree:
    Free up a previously allocated memory block .  The freed memory will be 
    consolidated with any free neighbouring blocks.  
    This routine indicates an error if an invalid block of memory is being 
    released.
    The routine returns the size of the blocked being freed.
*****************************************************************************/
WORD sFree(DWORD hMem)
{
    WORD CurBlock,offset,FreeMem,TotalFreeMem,offset1,result;
    BOOL LastBlock,TempLock=FALSE;
    struct StrWmMem far *pWmMem1;
    char msg[80];

    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);

    if (hMem==NULL) return 0;

    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to Free (sFree)",36);
    }
    
    if (WmGlobMem[CurBlock].flags&WIN_EMPTIED) {
        sAbort("Invalid block to Free (sFree 1)",36);
    }

    if (WmGlobMem[CurBlock].flags&WIN_DISCARDED) SwapIn(CurBlock);
    
    if (!WmGlobMem[CurBlock].LockCount) { /* lock the current block */
        if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
            sAbort("Unable to obtain a Global lock (sFree)",4);
        }
        TempLock=TRUE;
    }
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    if (!(pWmMem->flags&WIN_IN_USE) || pWmMem->signature!=WIN_SIGNATURE) {
       sAbort("An attempt to free an illegal memory block. (sFree)",5);
    }
    ResetFlag(&(pWmMem->flags),WIN_IN_USE);
    pWmMem->signature=0;
    result=pWmMem->size;
    
    /********** consolidate free blocks **************************************/
    offset=0;
    TotalFreeMem=0;
    WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE;  /* reset the first free block */
    LastBlock=FALSE;
    while (TRUE) {
       pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
       if (pWmMem->flags&WIN_IN_USE) {  /* check if signature exists */
           if (pWmMem->signature!=WIN_SIGNATURE) {
               sAbort("Memory corruption detected(sFree)",15);
           }
           if (pWmMem->flags&WIN_LAST_BLOCK) LastBlock=TRUE;
       }
       else {   /* free memory, consolidate it */
          FreeMem=0;
          LastBlock=FALSE;
          offset1=offset;
          while (TRUE) {
             pWmMem1=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset1]);
             if (pWmMem1->flags&WIN_IN_USE) break;
             
             FreeMem=FreeMem+pWmMem1->size+sizeof(struct StrWmMem);
             
             if (pWmMem1->flags&WIN_LAST_BLOCK) {LastBlock=TRUE;break;}
             offset1=offset1+pWmMem1->size+sizeof(struct StrWmMem);
          } 
          pWmMem->size=FreeMem-sizeof(struct StrWmMem);
          if (TotalFreeMem==0) WmGlobMem[CurBlock].FreeOffset=offset;  /* offset to the first free block */
          TotalFreeMem=TotalFreeMem+pWmMem->size+sizeof(struct StrWmMem);
          if (LastBlock) SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
       }
       if (LastBlock) break; /* end of the chain */
       offset=offset+pWmMem->size+sizeof(struct StrWmMem);
    }
    
    WmGlobMem[CurBlock].free=TotalFreeMem;

    if (TotalFreeMem==WIN_BLOCK_SIZE) { /* unallocate this block */
       pWmMem=(struct StrWmMem far *)WmGlobMem[CurBlock].pMem;
       if (pWmMem->flags&WIN_IN_USE || !(pWmMem->flags&WIN_LAST_BLOCK)) {
           sAbort("Logic error (sFree)",13);
       }
       
       while (GlobalFlags(WmGlobMem[CurBlock].handle)&GMEM_LOCKCOUNT) {
           GlobalUnlock(WmGlobMem[CurBlock].handle);
       }
       
       GlobalFree(WmGlobMem[CurBlock].handle);
       WmGlobMem[CurBlock].handle=0;
       WmGlobMem[CurBlock].free=0;
       SetFlag(&(WmGlobMem[CurBlock].flags),WIN_EMPTIED);
       WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE;  /* reset the first free block */
       WmGlobMem[CurBlock].LockCount=0;
       if (WmCurBlock==CurBlock) WmCurBlock=0;         /* reset the currently locked block number */
    }
    else if (TempLock) WmSwitchLock(CurBlock);  /* keep the current block locked */
    
    return result;
}

/*****************************************************************************
    sFreeQuick:
    Free up a previously allocated memory block .  The freed memory is NOT 
    consolidated with any free neighbouring blocks.  sFreeQuick may be used
    in place of sFree when a large number of memory handles need to freed up
    at one time.  sFix must be called after a bunch of memory handles are
    freed up using sFreeQuick. Example:
        
        for(i=0;i<100;i++) sFreeQuick(handle[i]);
        sFix();  
    
    This routine indicates an error if an invalid block of memory is being 
    released.
    The routine returns the size of the blocked being freed.
*****************************************************************************/
WORD sFreeQuick(DWORD hMem)
{
    WORD CurBlock,offset,FreeMem,TotalFreeMem,offset1,result;
    BOOL LastBlock,TempLock=FALSE;
    struct StrWmMem far *pWmMem1;

    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);

    if (hMem==NULL) return 0;

    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to Free (sFreeQuick)",36);
    }
    
    if (WmGlobMem[CurBlock].flags&WIN_DISCARDED) SwapIn(CurBlock);
    
    if (!WmGlobMem[CurBlock].LockCount) {     /* lock the current block */
        if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
            sAbort("Error in obtaining global memory lock(sFreeQuick)",4);
        }
        TempLock=TRUE;
    }
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    if (!(pWmMem->flags&WIN_IN_USE) || pWmMem->signature!=WIN_SIGNATURE) {
       sAbort("An attempt to free an illegal memory block (sFreeQuick)",5);
    }
    ResetFlag(&(pWmMem->flags),WIN_IN_USE);
    pWmMem->signature=0;
    SetFlag(&(WmGlobMem[CurBlock].flags),WIN_NEED_FIX);  /* need a fix later on */
    result=pWmMem->size;

    if (TempLock) { /* remove the temporary lock on this block */
       GlobalUnlock(WmGlobMem[CurBlock].handle);
    }

    return result;
}

/*******************************************************************************
    sFix:
    This function consolidates free memory blocks to make bigger blocks.
    This function need to be called only after calling sFreeQuick.
********************************************************************************/        
void sFix()
{
    WORD CurBlock,offset,FreeMem,TotalFreeMem,offset1,result;
    BOOL LastBlock,TempLock=FALSE;
    struct StrWmMem far *pWmMem1;


    for (CurBlock=1;CurBlock<WmGlobMemUsed;CurBlock++) {
        if (WmGlobMem[CurBlock].flags&WIN_EMPTIED) continue;   /* memory already freed up */
        if (!(WmGlobMem[CurBlock].flags&WIN_NEED_FIX)) continue;   /* no fix needed */ 

        /********** consolidate free blocks **************************************/
        TempLock=FALSE;
        if (!WmGlobMem[CurBlock].LockCount) { /* lock the current block */
            if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
                sAbort("Error in obtaining global memory lock(sFix)",4);
            }
            TempLock=TRUE;
        }
        
        offset=0;
        TotalFreeMem=0;
        WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE;  /* reset the first free block */
        LastBlock=FALSE;
        while (TRUE) {
           pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
           if (pWmMem->flags&WIN_IN_USE) {  /* check if signature exists */
               if (pWmMem->signature!=WIN_SIGNATURE) {
                   sAbort("Memory corruption detected(sFix)",15);
               }
               if (pWmMem->flags&WIN_LAST_BLOCK) LastBlock=TRUE;
           }
           else {   /* free memory, consolidate it */
              FreeMem=0;
              LastBlock=FALSE;
              offset1=offset;
              while (TRUE) {
                 pWmMem1=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset1]);
                 if (pWmMem1->flags&WIN_IN_USE) break;
                 
                 FreeMem=FreeMem+pWmMem1->size+sizeof(struct StrWmMem);
                 
                 if (pWmMem1->flags&WIN_LAST_BLOCK) {LastBlock=TRUE;break;}
                 offset1=offset1+pWmMem1->size+sizeof(struct StrWmMem);
              } 
              pWmMem->size=FreeMem-sizeof(struct StrWmMem);
              if (TotalFreeMem==0) WmGlobMem[CurBlock].FreeOffset=offset;  /* offset to the first free block */
              TotalFreeMem=TotalFreeMem+pWmMem->size+sizeof(struct StrWmMem);
              if (LastBlock) SetFlag(&(pWmMem->flags),WIN_LAST_BLOCK);
           }
           if (LastBlock) break; /* end of the chain */
           offset=offset+pWmMem->size+sizeof(struct StrWmMem);
        }
        
        WmGlobMem[CurBlock].free=TotalFreeMem;

        if (TotalFreeMem==WIN_BLOCK_SIZE) { /* unallocate this block */
           pWmMem=(struct StrWmMem far *)WmGlobMem[CurBlock].pMem;
           if (pWmMem->flags&WIN_IN_USE || !(pWmMem->flags&WIN_LAST_BLOCK)) {
               sAbort("Logic error (sFix)",13);
           }
           
           while (GlobalFlags(WmGlobMem[CurBlock].handle)&GMEM_LOCKCOUNT) {
               GlobalUnlock(WmGlobMem[CurBlock].handle);
           }
           
           GlobalFree(WmGlobMem[CurBlock].handle);
           WmGlobMem[CurBlock].free=0;
           WmGlobMem[CurBlock].handle=0;
           SetFlag(&(WmGlobMem[CurBlock].flags),WIN_EMPTIED);
           WmGlobMem[CurBlock].FreeOffset=WIN_BLOCK_SIZE;  /* reset the first free block */
           WmGlobMem[CurBlock].LockCount=0;
           if (WmCurBlock==CurBlock) WmCurBlock=0;         /* reset the currently locked block number */
        }
        else if (TempLock) { /* remove the temporary lock on this block */
           GlobalUnlock(WmGlobMem[CurBlock].handle);
        }
        
        ResetFlag(&(WmGlobMem[CurBlock].flags),WIN_NEED_FIX);

    }
}

/******************************************************************************
    sLock:
    This routine locks the memory to the requested handle and returns a far
    pointer to this block of memory.
*******************************************************************************/
LPSTR sLock(DWORD hMem)
{
    WORD CurBlock,offset;
    DWORD LockPtr;
    struct StrWmMem far *pWmMem;

    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);

    if (hMem==NULL) return NULL;

    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to lock (sLock)",38);
    }

    if (WmGlobMem[CurBlock].flags&WIN_DISCARDED) SwapIn(CurBlock);
    
    if (WmGlobMem[CurBlock].LockCount==0) {
       if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
           sAbort("Error in obtaining global memory lock(sLock)",4);
       }
    }

    WmGlobMem[CurBlock].LockCount++;     /* increment the lock counter */
    
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    if (!(pWmMem->flags&WIN_IN_USE) || pWmMem->signature!=WIN_SIGNATURE) {
       sAbort("Invalid Handle or Memory Corrupt (sLock)",13);
    }

    LockPtr=(DWORD) WmGlobMem[CurBlock].pMem + offset + sizeof(struct StrWmMem);
    return (LPSTR) LockPtr;
}

/******************************************************************************
    sHandle:
    This routine returns the memory handle (DWORD) for a locked block.  The 
    routine accepts the far pointer to block (LPSTR) as the argument.
*******************************************************************************/
DWORD sHandle(LPSTR ptr)
{
    DWORD LockPtr,i;
    WORD CurSeg,CurOff,TempSeg,TempOff;

    LockPtr=(DWORD) ptr;
    CurSeg=LockPtr>>16;
    CurOff=(LockPtr<<16)>>16;

    for(i=1;i<WmGlobMemUsed;i++) {
        if (WmGlobMem[i].pMem!=NULL) {    /* a locked super block */
           LockPtr=(DWORD)WmGlobMem[i].pMem;
           TempSeg=LockPtr>>16;
           TempOff=(LockPtr<<16)>>16;
           if (CurSeg==TempSeg) {         /* same segment */
              if (CurOff>=(TempOff+sizeof(struct StrWmMem)-1) && CurOff<(TempOff+WIN_BLOCK_SIZE)) {
                  CurOff=CurOff-TempOff-sizeof(struct StrWmMem); /* handle found */
                  return ((i<<16)+CurOff);
              }
           }
        }
    }

    return 0;                            /* invalid request */
}

/******************************************************************************
    SwapIn:
    This routine reallocates a discarded block and restores its contents from
    the swap file.
    The input to the routine is the index to global memory array.
*******************************************************************************/
void SwapIn(WORD index)
{

    if (index<1 || index>=WmGlobMemUsed) {
        sAbort("Invalid Swap-in Block",6);
    }
    
    if (NULL!=GlobalFree(WmGlobMem[index].handle)) {
        sAbort("Erron freeing up a discaded block(SwapIn)",5);
    }
    if (  (NULL==(WmGlobMem[index].handle=GlobalAlloc(WmAllocFlags,WIN_BLOCK_SIZE)))
       || (NULL==(WmGlobMem[index].pMem=GlobalLock(WmGlobMem[index].handle))) ) {
        sAbort("Ran Out of Memory (SwapIn)",6);
    }

    /* read back the contents from EMS or the swap file */
    
    if (index<WmEmsCutoff) {       /* swap in from the EMS memory */
        WmFromEms(hWmEms,WmEmsSeg,(long)index*WIN_BLOCK_SIZE,WmGlobMem[index].pMem,WIN_BLOCK_SIZE);
    }
    else {                         /* swap in from the disk file */
        if ((hWmSwapFile=open(WmSwapFile,O_BINARY|O_RDWR))==-1) {    /* open the swap file for swapping */
            sAbort("Could Not Open the Swap File (SwapIn)",42);
        }
                  
        if (_llseek(hWmSwapFile,(long)(index-WmEmsCutoff)*(long)WIN_BLOCK_SIZE,0)==-1L) {
            sAbort("Error seeking the swap file (SwapIn)",7);
        }
        if (_lread(hWmSwapFile,WmGlobMem[index].pMem,WIN_BLOCK_SIZE)!=WIN_BLOCK_SIZE) {
            sAbort("Error reading swap file (SwapIn)",8);
        }

        close(hWmSwapFile);            /* the file will be opened and close for each swapping incidence */
    }

    WmSwitchLock(index);               /* keep the internal lock on this block */

    ResetFlag(&(WmGlobMem[index].flags),WIN_DISCARDED);

}

/******************************************************************************
    SwapOut:
    Windows calls this proceedure when it has to discard a block of global
    memory.  This process would save the contents of the memory to the swap file
    to be later reinstated.
*******************************************************************************/
BOOL FAR PASCAL SwapOut(HANDLE hMem)
{
    WORD i;
    BOOL BlockFound=FALSE;

    for(i=1;i<WmGlobMemUsed;i++) {
        if (hMem==WmGlobMem[i].handle) {BlockFound=TRUE;break;}
    }
    if (!BlockFound) return TRUE;  /* this block belongs to non-WinMem allocation */
    

    if  (NULL==(WmGlobMem[i].pMem=GlobalLock(WmGlobMem[i].handle))) {
        sAbort("Error in obtaining global memory lock(SwapOut)",16);
    }                   

    if (WmEmsPercent>0 && !WmEmsUsed) WmInitEms();              /* initialize the EMS memory */
    
    if (i<WmEmsCutoff) {            /* store in the EMS memory */
        WmToEms(hWmEms,WmEmsSeg,WmGlobMem[i].pMem,(long)i*WIN_BLOCK_SIZE,WIN_BLOCK_SIZE);
    }
    else {                          /* store in the disk file */
        if ((hWmSwapFile=open(WmSwapFile,O_BINARY|O_RDWR))==-1) {    /* open the swap file for writting */
            sAbort("Could Not Open the Swap File (SwapOut)",41);
        }
                  
        if (_llseek(hWmSwapFile,(long)(i-WmEmsCutoff)*(long)WIN_BLOCK_SIZE,0)==-1L) {
            sAbort("Seek error, or Ran out of disk space(MemDiscardNotify)",9);
        }
        if (_lwrite(hWmSwapFile,WmGlobMem[i].pMem,WIN_BLOCK_SIZE)!=WIN_BLOCK_SIZE) {
            sAbort("Error writing to the swap file (MemDiscardNotify)",10);
        }
        
        close(hWmSwapFile);            /* the file will be opened and close for each swapping incidence */
    }

    GlobalUnlock(WmGlobMem[i].handle);
    SetFlag(&(WmGlobMem[i].flags),WIN_DISCARDED);
    WmGlobMem[i].LockCount=0;
    WmGlobMem[i].pMem=NULL;
    if (i==WmCurBlock) WmCurBlock=0;

    return TRUE;
}

/******************************************************************************
    sSize:
    This routine returns the memory block size for a handle
*******************************************************************************/
WORD sSize(DWORD hMem)
{
    WORD CurBlock,offset,FreeMem,TotalFreeMem,offset1,result;
    BOOL LastBlock,TempLock=FALSE;
    struct StrWmMem far *pWmMem1;

    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);

    if (hMem==NULL) return 0;

    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to Size (sSize)",50);
    }
    
    if (WmGlobMem[CurBlock].flags&WIN_DISCARDED) SwapIn(CurBlock);
    
    if (!WmGlobMem[CurBlock].LockCount) { /* lock the current block */
        if  (NULL==(WmGlobMem[CurBlock].pMem=GlobalLock(WmGlobMem[CurBlock].handle))) {
            sAbort("Error in obtaining global memory lock(sSize)",4);
        }
        TempLock=TRUE;
    }
    pWmMem=(struct StrWmMem far *)&(WmGlobMem[CurBlock].pMem[offset]);
    if (!(pWmMem->flags&WIN_IN_USE) || pWmMem->signature!=WIN_SIGNATURE) {
       sAbort("An attempt to size an illegal memory block.",51);
    }
    
    result=pWmMem->size;
    
    if (TempLock) { /* remove the temporary lock on this block */
       GlobalUnlock(WmGlobMem[CurBlock].handle);
    }

    return result;
}

/******************************************************************************
    sUnlock:
    Use this routine to a block of memory.
*******************************************************************************/
void sUnlock(DWORD hMem)
{
    WORD CurBlock,offset;
    char message[80];

    CurBlock=HIWORD(hMem);
    offset=LOWORD(hMem);

    if (hMem==NULL) return;

    if (CurBlock<1 || CurBlock >=WmGlobMemUsed || offset > WIN_BLOCK_SIZE) {
        sAbort("Invalid block to Unlock (sUnlock)",39);
    }

    WmGlobMem[CurBlock].LockCount--;
    
    if (WmGlobMem[CurBlock].LockCount<=0) {
        while (GlobalFlags(WmGlobMem[CurBlock].handle)&GMEM_LOCKCOUNT) GlobalUnlock(WmGlobMem[CurBlock].handle);
        WmGlobMem[CurBlock].pMem=NULL;
        WmGlobMem[CurBlock].LockCount=0;
        if (WmCurBlock==CurBlock) WmCurBlock=0;  /* reset the currently locked block number */
    }

}

/******************************************************************************
    sUnlockAll:
    Use this routine to reset all locks placed by WinMem memory management
    routines.
*******************************************************************************/
void sUnlockAll()
{
    WORD i;
    
    for (i=1;i<WmGlobMemUsed;i++) {
        if (WmGlobMem[i].flags&WIN_EMPTIED) continue;   /* memory already freed up */
        while (GlobalFlags(WmGlobMem[i].handle)&GMEM_LOCKCOUNT) {
           GlobalUnlock(WmGlobMem[i].handle);
        }
        WmGlobMem[i].pMem=NULL;
        WmGlobMem[i].LockCount=0;
    }
    WmCurBlock=0;                /* reset the internally locked block */
}

/******************************************************************************
    sExit:
    Call this routine at the end of your program or when you no longer need to
    call WinMem functions.
    This routine a) unlocks all existing locks,
                 b) frees up any global memory in use,
                 c) closes and deletes the swap file,
                 d) initializes the global memory use count.
*******************************************************************************/
void sExit()
{
    int i;

    sUnlockAll();      /* unlock all locks */

    for (i=1;i<WmGlobMemUsed;i++) {
       if (!(WmGlobMem[i].flags&WIN_EMPTIED) ) GlobalFree(WmGlobMem[i].handle);
       SetFlag(&(WmGlobMem[i].flags),WIN_EMPTIED);
    }

    if (!(WF_ENHANCED&GetWinFlags())) {  /* delete swap file in Real and Standard modes */
        unlink(WmSwapFile);
    }

    if (WmEmsUsed) WmFreeEms(hWmEms);    /* freeup the ems memory */

    WmGlobMemUsed=1;
}

/******************************************************************************
    sAbort:
    This routine is called by WinMem functions when they detect a fatal memory 
    error.  This error may be caused by writing to an invalid or unlocked
    pointer.
    This routine a) displays the error message in a message box, 
                 b) calls an application routine to abort the application.
*******************************************************************************/
void sAbort(PSTR message,WORD code)
{
   MessageBox(NULL,message,"Memory Error",MB_ICONHAND|MB_SYSTEMMODAL);
   PostQuitMessage(code);
   
   Throw(sFatalError,-1);  /* Jump to sFatalError location to end the application. */
}

/******************************************************************************
    Other Memory Management Functions
*******************************************************************************

/******************************************************************************
    Copy a specified number of bytes from the source to the destination location.
    Both the source and destination locations are specified by using far pointers.
    
    When source and destination memory overlaps,  use FarMoveOl function for
    a proper trasfer.
*******************************************************************************/
void FarMove(LPSTR src,LPSTR dest, WORD count)
{
    WORD SrcSeg,DestSeg,SrcOff,DestOff;

    if (count==0) return;                   /* nothing to move */

    SrcSeg=(DWORD)src>>16;                 /* segment and offset of source and destination */
    SrcOff=((DWORD)src<<16)>>16;
    DestSeg=(DWORD)dest>>16;
    DestOff=((DWORD)dest<<16)>>16;

    movedata(SrcSeg,SrcOff,DestSeg,DestOff,count);
    return;
}

/******************************************************************************
    Copy a specified number of bytes from the source to the destination location.
    Both the source and destination locations are specified by using far pointers.
    
    When the source and destination memory segments are nonoverlapping or
    when the destination offset is smaller than the source offset, the 
    regular 'movedata' routine is used. But when the source and destination
    memory segments are identical and the destination offset is greater than
    the source offset, ReverseMemMove assembly routine is used then.  This routine
    starts from the tail end of memory, thereby avoiding premature memory
    overwrite.
*******************************************************************************/
void FarMoveOl(LPSTR src,LPSTR dest, WORD count)
{
    WORD SrcSeg,DestSeg,SrcOff,DestOff;

    if (count==0) return;                       /* nothing to move */

    SrcSeg=(DWORD)src>>16;                      /* segment and offset of source and destination */
    SrcOff=((DWORD)src<<16)>>16;
    DestSeg=(DWORD)dest>>16;
    DestOff=((DWORD)dest<<16)>>16;

    if (SrcSeg!=DestSeg || DestOff<=SrcOff) {   /* treat it as non overlapping segments */
       movedata(SrcSeg,SrcOff,DestSeg,DestOff,count);
       return;
    }
    else FarMoveRev(src,dest,count);            /* move the memory in the reverse order */
}

/******************************************************************************
    FarStringIndex:
    This function returns the position of the second string within the first
    string.  The position is an index from the beginning of the first string.
    If the string is located, the return value will range from 0 to length of
    first string minus the length of the second string.
    If the second string could not be located,  the return value is equal to
    the length of the first string.
******************************************************************************/
WORD FAR FarStringIndex(LPSTR string1,LPSTR string2)
{
    unsigned i,j,k,len1,len2,SearchLen;
    char FirstChar;

    len1=lstrlen(string1);
    len2=lstrlen(string2);
    if (len2>len1) return len1;          /* the second string too big to fit within the first string */

    FirstChar=string2[0];                /* first character of the second string */

    i=0;
    do {
        SearchLen=len1-len2-i+1;         /* length of the string to search */
        j=FarCharIndex(&string1[i],FirstChar,SearchLen);
        if (j==SearchLen) return len1;   /* first character of the second string in the first string not found */
        
        for (k=1;k<len2;k++) if (string1[i+j+k]!=string2[k]) goto CONTINUE_LOOP;
        return i+j;

        CONTINUE_LOOP:
        i=i+j+1;                         /* repeat after the first character match location */
    } while (i<=(len1-len2));

    return len1;                         /* the string was not located */
}

/******************************************************************************
    FarStringRevIndex:
    This function returns the position of the second string within the first
    string.  This function scans the first string in the reverse order.  The
    third argument specifies the length of the string1 to be scanned. Ex. if
    length=10, then the first string will be scanned from position 9 in the
    reverse order until position 0 is scanned.  The function returns the 
    index of the located string from the beginning of the first string.

    Reverse string search is slower than forward string search (FarStringIndex);
*******************************************************************************/
WORD FAR FarStringRevIndex(LPSTR string1,LPSTR string2,WORD len)
{
    int i,len2;
    register int k;
    
    len2=lstrlen(string2);
    if (len2>len) return len;    /* string to big to fit */

    i=len-len2;

    do {
       for (k=0;k<len2;k++) if (string1[i+k]!=string2[k]) break;
       if (k==len2) return i;             /* string located */
       if (i==0) break;else i--;
    } while (TRUE);

    return len;                             /* string not located */
}                

/******************************************************************************
    EMS memory handling routines
*******************************************************************************/

/******************************************************************************
    WmInitEms:
    Initialize the EMS memory for WinMem
******************************************************************************/
WmInitEms()
{
    WORD EmsPages;
    long EmsSize;

    WmEmsSeg=WmEmsFrame();
    
    if ((EmsPages=WmEmsPages()*WmEmsPercent/100)!=0) {  /* use the specified percent of ems space for this purpose */
       EmsSize=(long)EmsPages*(long)0x4000;              /* each page is 16 k */
    }
    else {                                              /* very little or no EMS memory available */
       WmEmsPercent=0;
       return FALSE;
    }

    WmEmsCutoff=EmsSize/WIN_BLOCK_SIZE;                 /* number of superblocks that can go into EMS memory */
    if (WmEmsCutoff>WIN_MAX_BLOCKS) WmEmsCutoff=WIN_MAX_BLOCKS;

    if (WmEmsCutoff==0) {                               /* not even one block fits into EMS */
       WmEmsPercent=0;
       return FALSE;
    }
    
    EmsSize=(long)WmEmsCutoff*(long)WIN_BLOCK_SIZE;
    
    if (!WmAllocEms(&hWmEms,EmsSize)) {
       WmEmsPercent=0;
       return FALSE;
    }

    
    WmEmsUsed=TRUE;                /* ems in use now */

    return TRUE;
}

/*****************************************************************************
    WmCheckEms:
    check if the EMS hardware and software is present.
*****************************************************************************/
WmCheckEms()
{
    char CheckSig[9];

    union  REGS reg;
    union  REGS oreg;
    struct SREGS sreg;


    reg.h.ah=0x35;                 /* get the address of the ems driver */
    reg.h.al=0x67;                 /* ems interrupt */
    segread(&sreg);
    sreg.es=sreg.ds;               /* a valid value for ES */
    int86x(0x21,&reg,&oreg,&sreg);

    if (sreg.es==GetSegment((LPSTR)CheckSig)) {  /* value not changed */
       return FALSE;
    }

    movedata(sreg.es,10,GetSegment((LPSTR) CheckSig),GetOffset((LPSTR)CheckSig),8);
    CheckSig[8]=0;
    
    if (lstrcmp(CheckSig,(LPSTR)"EMMXXXX0")!=0) return FALSE;   /* signature not found */

    reg.h.ah=0x40;                 /* check ems status */
    int86(0x67,&reg,&oreg);
    
    if (oreg.h.ah!=0) return FALSE;/* invalid status */

    return TRUE;
}

/******************************************************************************
     WmEmsFrame:
     Get the EMS frame address.
******************************************************************************/
WORD WmEmsFrame() 
{
    union  REGS reg;
    union  REGS oreg;

    reg.h.ah=0x41;     /* get the address of the ems window */
    int86(0x67,&reg,&oreg);

    return (unsigned) oreg.x.bx;
}

/******************************************************************************
     WmEmsPages:
     Get the available number of EMS pages (each 16K)
*******************************************************************************/
WmEmsPages() 
{
    union  REGS reg;
    union  REGS oreg;

    reg.h.ah=0x42;  /* get unallocated ems pages */
    int86(0x67,&reg,&oreg);

    if (oreg.h.ah==0) return oreg.x.bx;else return 0;
}

/******************************************************************************
    WmAllocEms:
    Allocate a specified amount of EMS memory 
******************************************************************************/
WmAllocEms(int *handle,long bytes)
{
    union  REGS reg;
    union  REGS oreg;

    reg.h.ah=0x43;                       /* get the ems handle */
    reg.x.bx=(unsigned) (bytes/0x4000);  /* how many 16K pages */
    if (bytes>((long)reg.x.bx)*0x4000) reg.x.bx++;
    int86(0x67,&reg,&oreg);
    if (oreg.h.ah!=0) return FALSE;
    *handle=oreg.x.dx;
    return TRUE;
}

/******************************************************************************
   WmFreeEms:
   Free up the EMS memory for an EMS handle
******************************************************************************/
WmFreeEms(int handle)
{
    union  REGS reg;
    union  REGS oreg;

    reg.h.ah=0x45;       /* release ems handle */
    reg.x.dx=handle;
    int86(0x67,&reg,&oreg);
    if (oreg.h.ah!=0) return FALSE;
    return TRUE;
}

/******************************************************************************
    WmToEms:
    Move a block from RAM to EMS Memory
******************************************************************************/
WmToEms(int handle,WORD EmsSeg,LPSTR from,long to,long bytes)
{
    WORD page,PageOff,MoveBytes,PageLength=0x4000,seg,off,SaveOffset;
    long BytesMoved=0;
    union  REGS reg;
    union  REGS oreg;

    page = (unsigned) (to/(long)PageLength);
    PageOff = to - ((long)page*PageLength);
    seg=(unsigned long)from>>16;
    off=((unsigned long)from<<16)>>16;

    while (BytesMoved!=bytes) {
        if (PageOff==0) MoveBytes=PageLength;else MoveBytes=PageLength-PageOff;
        if (BytesMoved+MoveBytes>bytes) MoveBytes=bytes-BytesMoved;

        /* map the logical page to physical page 0 */
        reg.h.ah=0x44;        /* map */
        reg.h.al=0;           /* physical page */
        reg.x.bx=page;        /* logical page */
        reg.x.dx=handle;
        int86(0x67,&reg,&oreg);
        if (oreg.h.ah!=0) return FALSE;

        movedata(seg,off,EmsSeg,PageOff,MoveBytes);
        
        SaveOffset=off;               /* adjust the source pointer for the next move */
        off=off+MoveBytes;
        if (off<SaveOffset)  {
            sAbort("EMS Transfer Error(WmToEms)",43);
        }

        page++;PageOff=0;
        BytesMoved=BytesMoved+MoveBytes;
    }

    return TRUE;
}

/******************************************************************************
    WmFromEms:
    Move a block from EMS memory to RAM
******************************************************************************/
WmFromEms(int handle,WORD EmsSeg,long from,LPSTR to,long bytes)
{
    WORD page,PageOff,MoveBytes,PageLen=0x4000,seg,off,SaveOffset;
    long BytesMoved=0;
    union  REGS reg;
    union  REGS oreg;

    page = (unsigned) (from/PageLen);
    PageOff = from - ((long)page*PageLen);
    seg=(DWORD)to>>16;
    off=((DWORD)to<<16)>>16;

    while (BytesMoved!=bytes) {
        if (PageOff==0) MoveBytes=PageLen;else MoveBytes=PageLen-PageOff;
        if (BytesMoved+MoveBytes>bytes) MoveBytes=bytes-BytesMoved;

        /* map the logical page from physical page 0 */
        reg.h.ah=0x44;        /* map */
        reg.h.al=0;           /* physical page */
        reg.x.bx=page;        /* logical page */
        reg.x.dx=handle;
        int86(0x67,&reg,&oreg);
        if (oreg.h.ah!=0) return 0;

        movedata(EmsSeg,PageOff,seg,off,MoveBytes);

        SaveOffset=off;               /* adjust the source pointer for the next move */
        off=off+MoveBytes;
        if (off<SaveOffset)  {
            sAbort("EMS Transfer Error(WmFromEms)",44);
        }

        page++;PageOff=0;
        BytesMoved=BytesMoved+MoveBytes;
    }

    return TRUE;
}

/******************************************************************************
   WmGetCurDir:
   get full path name of the current working directory 
*******************************************************************************/
LPSTR WmGetCurDir(LPSTR CurDir)
{
    char CurDrv[4],TempDir[64];
    union REGS reg;struct SREGS sreg;
    WORD drive;

    /*********** get default drive ***********/
    segread(&sreg);
    reg.h.ah=0x19;
    int86x(0x21,&reg,&reg,&sreg);
    CurDrv[0]=reg.h.al+'A';
    CurDrv[1]=0;strcat(CurDrv,":\\");

    /* get current sub directory ******/
    reg.h.ah=0x47;reg.h.dl=0;
    sreg.ds=((unsigned long)(LPSTR)TempDir)>>16;
    sreg.es=sreg.ds;                   /* must provide a valid selector in ES */
    reg.x.si=(((unsigned long)(LPSTR)TempDir)<<16)>>16;
    
    int86x(0x21,&reg,&reg,&sreg);
    
    lstrcpy(CurDir,CurDrv);
    lstrcat(CurDir,TempDir);

    return CurDir;
}
