/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

/*
 * PostScript Filter, psf.
 *
 * Handles both PostScript files and text files. Files with the
 * '%!' PostScript header are sent directly to the printer,
 * unmodified. Text files are first converted to PostScript,
 * then sent. Printers may be directly attached or on an AppleTalk
 * network. Other media are possible. Currently, psf invokes
 * pap to send files to AppleTalk-ed printers. Replace the pap*
 * variables to use another program for communication.
 *
 * TODO: Change textps() output so that extra page-start sequence
 * is not output.
 */

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <atalk/paths.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

char		pappath[] = _PATH_PAP;
char		*papargv[] = { "pap", "-sstatus", "-e", 0 };

char		revpath[] = _PATH_PSORDER;
char		*revargv[] = { "psorder", "-d", 0 };

char		*tfargv[] = { "troff2ps", "-Z", "-O-.10", 0 };
char		*dfargv[] = { "dvi2ps", 0 };
struct filter {
    char	*f_name;
    char	*f_path;
    char	**f_argv;
} filters[] = {
    { "tf", _PATH_TROFF2PS, tfargv },
    { "df", "/usr/local/atalk/etc/dvi2ps", dfargv },
};

char		inbuf[ 1024 * 8 ];
int		inlen;

FILE		*acctfile = NULL;
int		literal;
int		width = 80, length = 66, indent = 0;
char		*prog, *name, *host;

main( ac, av ) 
    int		ac;
    char	**av;
{
    extern char		*optarg;
    int			c, rc, i, children = 0;
    union wait		status;

    prog = av[ 0 ];
    while (( c = getopt( ac, av, "x:y:n:h:w:l:i:c" )) != EOF ) {
	switch ( c ) {
	case 'n' :
	    name = optarg;
	    break;

	case 'h' :
	    host = optarg;
	    break;

	case 'w' :
	    if (( width = atoi( optarg )) == 0 ) {
		width = 80;	/* because lpd is bugged */
	    }
	    break;

	case 'l' :
	    length = atoi( optarg );
	    break;

	case 'i' :
	    indent = atoi( optarg );
	    break;

	case 'c' :	/* Print control chars */
	    literal++;
	    break;

	case 'x' :
	case 'y' :
	    break;
	
	default :
	    fprintf( stderr, "%s[%d]: bad option: %c\n", prog, getpid(), c );
	    exit( 1 );		/* reprint */
	}
    }

    fprintf( stderr, "%s[%d]: starting\n", prog, getpid());

restart:
    if (( inlen = read( 0, inbuf, sizeof( inbuf ))) < 0 ) {
	perror( "read" );
	exit( 1 );
    }
    if ( inlen == 0 ) {	/* nothing to be done */
	fprintf( stderr, "%s[%d]: done\n", prog, getpid());
	exit( 0 );
    }

    /*
     * Check prog's name to decide what programs to execute.
     */
    if ( strstr( prog, "pap" ) != NULL ) {
	fprintf( stderr, "%s[%d]: sending to pap\n", prog, getpid());
	if (( c = pexecv( pappath, papargv )) < 0 ) {
	    perror( pappath );
	    exit( 1 );
	}
	children++;
    }

    /*
     * Might be a good idea to have both a "forw" and a "rev", so that
     * reversed documents can be reordered for the printing device.
     */
    if ( strstr( prog, "rev" ) != NULL ) {
	fprintf( stderr, "%s[%d]: sending to rev\n", prog, getpid());
	if (( c = pexecv( revpath, revargv )) < 0 ) {
	    perror( revpath );
	    exit( 1 );
	}
	children++;
    }

    for ( i = 0; i < sizeof( filters ) / sizeof( filters[ 0 ] ); i++ ) {
	if ( strncmp( filters[ i ].f_name, prog,
		strlen( filters[ i ].f_name )) == 0 ) {
	    fprintf( stderr, "%s[%d]: sending to %s\n",
		    prog, getpid(), *filters[ i ].f_argv );
	    if (( c = pexecv( filters[ i ].f_path,
		    filters[ i ].f_argv )) < 0 ) {
		perror( filters[ i ].f_path );
		exit( 1 );
	    }
	    children++;
	    break;
	}
    }

    if ( i < sizeof( filters ) / sizeof( filters[ 0 ] ) ||
	    ( inlen >= 2 && inbuf[ 0 ] == '%' && inbuf[ 1 ] == '!' )) {
	rc = copyio();	/* Other filter, or PostScript */
    } else {
	rc = textps();	/* Straight text */
    }

    if ( children ) {
	close( 1 );
    }
    while ( children ) {
	if (( c = wait3( &status, 0, 0 )) < 0 ) {
	    perror( "wait3" );
	    exit( 2 );
	}
	if ( WIFEXITED( status )) {
#ifndef WEXITSTATUS
#define WEXITSTATUS(x)	((x).w_status)
#endif WEXITSTATUS
	    if ( WEXITSTATUS( status ) != 0 ) {
		fprintf( stderr, "%s[%d]: %d died with %d\n", prog, getpid(),
			c, WEXITSTATUS( status ));
		exit( WEXITSTATUS( status ));
	    } else {
		fprintf( stderr, "%s[%d]: %d done\n", prog, getpid(), c );
		children--;
	    }
	} else {
	    fprintf( stderr, "%s[%d]: %d died badly\n", prog, getpid(), c );
	    exit( 1 );
	}
    }

    if ( rc == 3 ) {
	fprintf( stderr, "%s[%d]: pausing\n", prog, getpid());
	kill( getpid(), SIGSTOP );
	fprintf( stderr, "%s[%d]: restarting\n", prog, getpid());
	goto restart;
    }

    exit( rc );
}

copyio()
{
    do {
	if ( write( 1, inbuf, inlen ) != inlen ) {
	    perror( "write" );
	    return( 1 );
	}
    } while (( inlen = read( 0, inbuf, sizeof( inbuf ))) > 0 );
    if ( inlen < 0 ) {
	perror( "read" );
	return( 1 );
    }
    return( 0 );
}

char		*font = "Courier";
int		point = 11;
float		win = 8.5, hin = 11;

char		pspro[] = "\
/GSV save def						% global VM\n\
/SP {\n\
	/SV save def					% save vmstate\n\
	dup /H exch def					% save font height\n\
	exch findfont exch scalefont setfont		% select font\n\
	( ) stringwidth pop /W exch def			% save font width\n\
	72 mul exch 2 sub H mul add 2 div /CY exch def	% save start Y\n\
	72 mul exch 1 add W mul sub 2 div\n\
		exch W mul 2 div add /CX exch def	% save start X\n\
	CX CY moveto					% make current point\n\
} bind def\n\
/S /show load def\n\
/NL { CX CY H sub dup /CY exch def moveto } bind def\n\
/CR { CX CY moveto } bind def\n\
/B { W neg 0 rmoveto}bind def\n\
/T { W mul 0 rmoveto}bind def\n\
/EP { SV restore showpage } bind def\n\
%%EndProlog\n";

textps()
{
    int		state = 0, line = 0, col = 0, npages = 0, rc;
    char	*p, *end;

#define ST_AVAIL		(1<<0)
#define ST_CONTROL		(1<<1)
#define ST_PAGE			(1<<2)
    /*
     * convert text lines to postscript.
     * A grungy little state machine. If I was more creative, I could
     * probably think of a better way of doing this...
     */
    do {
	p = inbuf;
	end = inbuf + inlen;
	while ( p < end ) {
	    if (( state & ST_PAGE ) == 0 && *p != '\031' && *p != '\001' ) {
		if ( npages == 0 ) {
		    printf( "%%!PS-Adobe-2.0\n%%%%Pages: (atend)\n" );
		    printf( "%%%%DocumentFonts: %s\n", font );
		    fflush( stdout );

		    /* output postscript prologue: */
		    if ( write( 1, pspro, sizeof( pspro ) - 1 ) !=
			    sizeof( pspro ) - 1 ) {
			perror( "write prologue" );
			return( 1 );
		    }
		    if ( name && host ) {
			printf( "statusdict /jobname (%s@%s) put\n", name,
				host );
		    }
		}

		printf( "%%%%Page: ? %d\n", ++npages );
		printf( "%d %d %f %d %f /%s %d SP\n",
			indent, width, win, length, hin, font, point );
		state |= ST_PAGE;
	    }
	    if ( state & ST_CONTROL && *p != '\001' ) {
		if ( !literal ) {
		    fprintf( stderr, "%s[%d]: binary character!!!\n", prog,
			    getpid());
		    return( 2 );	/* Toss job */
		}
		printf( "\\%o", (unsigned char)031 );
		state &= ~ST_CONTROL;
		col++;
	    }

	    switch ( *p ) {
	    case '\n' :		/* end of line */
		if ( state & ST_AVAIL ) {
		    printf( ")S\n" );
		    state &= ~ST_AVAIL;
		}
		printf( "NL\n" );
		line++;
		col = 0;
		if ( line >= length ) {
		    printf( "EP\n" );
		    state &= ~ST_PAGE;
		    line = 0;
		}
		break;

	    case '\r' :		/* carriage return (for overtyping) */
		if ( state & ST_AVAIL ) {
		    printf( ")S CR\n" );
		    state &= ~ST_AVAIL;
		}
		col = 0;
		break;

	    case '\f' :		/* form feed */
		if ( state & ST_AVAIL ) {
		    printf( ")S\n" );
		    state &= ~ST_AVAIL;
		}
		printf( "EP\n" );
		state &= ~ST_PAGE;
		line = 0;
		col = 0;
		break;

	    case '\b' :		/* backspace */
		/* show line, back up one character */
		if ( state & ST_AVAIL ) {
		    printf( ")S\n" );
		    state &= ~ST_AVAIL;
		}
		printf( "B\n" );
		col--;
		break;

	    case '\t' :		/* tab */
		if ( state & ST_AVAIL ) {
		    printf( ")S\n" );
		    state &= ~ST_AVAIL;
		}
		printf( "%d T\n", 8 - ( col % 8 ));
		col += 8 - ( col % 8 );
		break;

	    /*
	     * beginning of lpr control sequence
	     */
	    case '\031' :
		state |= ST_CONTROL;
		break;

	    case '\001' :	/* lpr control sequence */
		if ( state & ST_CONTROL ) {
		    rc = 3;
		    goto out;
		}
		/* FALLTHROUGH */

	    case '\\' :
	    case ')' :
	    case '(' :
		if (( state & ST_AVAIL ) == 0 ) {
		    printf( "(" );
		    state |= ST_AVAIL;
		}
		putchar( '\\' );
		/* FALLTHROUGH */

	    default :
		if (( state & ST_AVAIL ) == 0 ) {
		    printf( "(" );
		    state |= ST_AVAIL;
		}
		if ( !isascii( *p ) || !isprint( *p )) {
		    if ( !literal ) {
			fprintf( stderr, "%s[%d]: binary character!!!\n",
				prog, getpid() );
			return( 2 );	/* Toss job */
		    }
		    printf( "\\%o", (unsigned)*p );
		} else {
		    putchar( *p );
		}
		col++;
		break;
	    }
	p++;
	}
    } while (( inlen = read( 0, inbuf, sizeof( inbuf ))) > 0 );
    if ( inlen < 0 ) {
	perror( "read" );
	return( 1 );
    }
    rc = 0;

out:
    if ( state & ST_AVAIL ) {
	printf( ")S\n" );
	state &= ~ST_AVAIL;
    }

    if ( state & ST_PAGE ) {
	printf( "EP\n" );
	state &= ~ST_PAGE;
    }

    if ( npages > 0 ) {
	printf( "%%%%Trailer\nGSV restore\n%%%%Pages: %d\n", npages );
	fflush( stdout );
    }

    return( rc );
}

/*
 * Interface to pipe and exec, for starting children in pipelines.
 *
 * Manipulates file descriptors 0, 1, and 2, such that the new child
 * is reading from the parent's output.
 */
pexecv( path, argv )
    char	*path, *argv[];
{
    int		fd[ 2 ], c;

    if ( pipe( fd ) < 0 ) {
	return( -1 );
    }

    switch ( c = vfork()) {
    case -1 :
	return( -1 );
	/* NOTREACHED */

    case 0 :
	if ( close( fd[ 1 ] ) < 0 ) {
	    return( -1 );
	}
	if ( dup2( fd[ 0 ], 0 ) < 0 ) {
	    return( -1 );
	}
	if ( close( fd[ 0 ] ) < 0 ) {
	    return( -1 );
	}
	execv( path, argv );
	return( -1 );
	/* NOTREACHED */

    default :
	if ( close( fd[ 0 ] ) < 0 ) {
	    return( -1 );
	}
	if ( dup2( fd[ 1 ], 1 ) < 0 ) {
	    return( -1 );
	}
	if ( close( fd[ 1 ] ) < 0 ) {
	    return( -1 );
	}
	return( c );
    }
}
