/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#ifdef EBUG
#include <stdio.h>
#endif EBUG
#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/uio.h>
#include <sys/signal.h>
#include <netatalk/endian.h>
#include <syslog.h>
#include <ctype.h>
#include <netatalk/at.h>
#include <atalk/nbp.h>
#include <atalk/atp.h>
#include "pap.h"
#include "papd.h"
#include "lw.h"

extern char	*malloc();

extern int	debugflg;
extern int	pipedprinter;
extern char	*procsetdir;
extern int	dffd;
extern char	*printer;
extern char	*jobname;
extern char	*literalname;
extern char	*psetfile;
extern int	jobseqnum;
extern int	acceptprocsets;
extern int	dsc_majorvers;
extern int	dsc_minorvers;

static char		buf[ ATP_MAXDATA * 8 ];
static struct comm	comment[] = {
    PAPEOF,		PAPEOF_STR,		sizeof( PAPEOF_STR )-1,
    BEGINPROCSET,	BEGINPROCSET_STR,	sizeof( BEGINPROCSET_STR )-1,
    INCLUDEPROCSET,	INCLUDEPROCSET_STR,	sizeof( INCLUDEPROCSET_STR )-1,
    STARTJOB,		STARTJOB_STR,		sizeof( STARTJOB_STR )-1,
    ENDJOB,		ENDJOB_STR,		sizeof( ENDJOB_STR )-1,
    TITLE,		TITLE_STR,		sizeof( TITLE_STR )-1,
    USER,		USER_STR,		sizeof( USER_STR )-1,
    BEGINEXITSERVER,	BEGINEXITSERVER_STR,	sizeof( BEGINEXITSERVER_STR )-1,
    0,			NULL,			0,
};


process_job( paph )
    PAP		paph;
{
    int			eof, blen, oblen, rc, len;
    char		*s, *tbuf, *keyword;
    struct query_table	*qt;
    struct comm		*cm;
    struct procset	*psp;
    int			psfd = -1;
    int			ignoring = 1;
    int			jobcomplete = 0;
    int			didwrite = 0;
    int			gottitle = 0;
    int			gotuser = 0;

    if ( initspoolfiles() < 0 ) {
	pap_close_req( paph, "papd: spooling init error" );
	return -1;
    }
    sprintf( buf, "%s processing job %d/%d from ",
      printer, jobseqnum, paph->paph_connid );
    sprintf_addr( buf + strlen( buf ), &paph->paph_saddr );
    setproctitle( buf );
    syslog( LOG_INFO, buf );

    /*
     * wait for initial "send data" request (so we're in sync for pap_writes)
     * if we get close connection here we are in trouble, since we don't check
     */
    while (( rc = pap_get_request( paph )) == PAP_TICKLE );

    while ( 1 ) {
#ifdef notdef
	if ( pap_send_tickle( paph, &paph->paph_saddr ) < 0 ) {
	    pap_logerror( "pap_send_tickle" );
	    return -1;
	}
#endif notdef
	if ( didwrite ) {
	/* if we performed a write last time through, wait for
	 * another "send data" request
	 */
	    didwrite = 0;
	    while (( rc = pap_get_request( paph )) == PAP_TICKLE );
	    if ( rc != 0 ) {
		if ( finishspoolfiles() != 0 ) {
		    return -1;
		}
		if ( !pipedprinter && startdaemon( printer ) != 0 ) {
		    return -1;
		}
		return ( rc );
	    }
	}

	if (( oblen = pap_read( paph, buf, &eof )) < 0 ) {
	    pap_logerror( "pap_read" );
	    pap_close_req( paph, NULL );
	    return -1;
	}

	if ( eof ) {
	    if ( pap_write( paph, NULL, 0, 1 ) < 0 ) {
		pap_logerror( "pap_write" );
		return( -1 );
	    }
	    ++didwrite;
#ifdef EBUG
	    if ( debugflg ) {
		fprintf( stderr, "got PAP EOF\n" );
	    }
#endif EBUG
	}

	tbuf = buf;
	blen = oblen;
	while (( s = getline( &tbuf, &blen )) != NULL ) {
#ifdef EBUG
	    if ( debugflg ) {
		fprintf( stderr, "(in processjob) line is %d bytes:",
			linelen( s ));
		bfprint( stderr, s, linelen( s ));
		fflush( stderr );
	    }
#endif EBUG

	    if ( *s == COMMENTCHAR ) {
	    /* this line is a comment -- check for special stuff */
		if (( qt = find_query( s )) != NULL ) {
		/* found a query -- dispatch the appropriate query handler */
#ifdef EBUG
		    if ( debugflg ) {
			fprintf( stderr, "query comment %s\n",qt->query_text );
		    }
#endif EBUG
		    (*qt->query_handler)( paph, qt, &tbuf, &blen );
		    ++didwrite;
		} else {
		    for ( cm = comment; cm->comm_str != NULL; ++cm ) {
			if ( strncasecmp( cm->comm_str, s, cm->comm_len ) == 0 )
			    break;
		    }
		    if ( cm->comm_str != NULL )
		      switch ( cm->comm_tag ) {
		    case BEGINPROCSET:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "BEGINPROCSET\n" );
#endif EBUG
			if ( psfd >= 0 ) {
			    if (( psp = str2procset( s )) == NULL ) {
				syslog( LOG_ERR, "str2procset fails" );
				pap_close_req( paph, "papd: bad proc set name" );
				unlinkprocset();
				return -1;
			    }
			    if ( !acceptprocsets ) {
				if ( haveprocset( psp ) != 1 ) {
				    syslog( LOG_ERR,
					"not accepting procedure sets" );
				    pap_close_req( paph, "papd: not accepting proc sets" );
				    psp_free( psp );
				    unlinkprocset();
				    return -1;
				}
			    }
			    syslog( LOG_INFO, "storing procedure set %s.%hu.%hu",
			      psp->procset_name, psp->procset_version,
			      psp->procset_release );
			}
			break;

		    case PAPEOF:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "PAPEOF\n" );
#endif EBUG
			if ( psfd >= 0 ) {
			    if ( write( psfd, s, linelen( s )+1 ) < 0 ) {
				syslog( LOG_ERR, "write procset file: %m" );
				pap_close_req( paph, "papd: write proc set error" );
				psp_free( psp );
				return -1;
			    }
				
			    if ( closeprocset( procsetdir, psp, psfd ) < 0 ) {
				syslog( LOG_ERR, "closeprocset: %m" );
				pap_close_req( paph, "papd: close proc set error" );
				psp_free( psp );
				return -1;
			    }
			    psp_free( psp );
			    psfd = -1;
			    syslog( LOG_INFO, "done storing procedure set" );
			}
			break;

		    case BEGINEXITSERVER:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "BEGINEXITSERVER\n" );
#endif EBUG
			if ( ignoring ) {
			    if (( psfd = openprocset( procsetdir, 1,  NULL,
			      O_WRONLY | O_CREAT, 0600 )) < 0 ) {
				syslog( LOG_ERR, "openprocset: %m" );
				pap_close_req( paph, "papd: open proc set error" );
				return -1;
			    }
			}
			break;

		    case INCLUDEPROCSET:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "INCLUDEPROCSET\n" );
#endif EBUG
			if (( psp = str2procset( s )) == NULL ) {
			    syslog( LOG_ERR, "str2procset fails" );
			    pap_close_req( paph, "papd: bad proc set name" );
			    return -1;
			}
			if (( psetfile = malloc( strlen( procsetdir ) +
			  strlen( psp->procset_name ) + 16 )) == NULL ) {
			    syslog( LOG_ERR, "psetfile malloc: %m" );
			    pap_close_req( paph, NULL );
			    return -1;
			}
			sprintf( psetfile, "%s/%s.%hu.%hu", procsetdir,
			  psp->procset_name, psp->procset_version,
			  psp->procset_release );
			psp_free( psp );
			break;

		    case STARTJOB:
			keyword = s + cm->comm_len;
			if ( strncmp( keyword, DSC_CONFORMANT,
				DSC_CONFORMANT_LEN ) == 0 ) {
			    keyword += DSC_CONFORMANT_LEN;
			    dsc_majorvers = atoi( keyword );
			    while ( isdigit( *keyword++ )) {
				;	/* skip digits and the '.' */
			    }
			    dsc_minorvers = atoi( keyword );
			} else {
			    dsc_majorvers = dsc_minorvers = 0;
			}

#ifdef EBUG
			if ( debugflg ) {
			    fprintf( stderr, "DSC version %u.%u\n",
				    dsc_majorvers, dsc_minorvers );
			}
#endif EBUG

			while ( *keyword != EOL && *keyword != NEWLINE
				&& !isspace( *keyword )) {
			    ++keyword;	/* skip rest of first word */
			}
			while ( isspace ( *keyword )) {
			    ++keyword;	/* skip white space */
			}
			if ( *keyword == EOL || *keyword == NEWLINE ) {
#ifdef EBUG
			    if ( debugflg )
				fprintf( stderr, "STARTJOB doc\n" );
#endif EBUG
			    ignoring = 0;	/* assume normal document */
			} else {
			    if ( strncasecmp( keyword, KEY_EPS_STR,
				    KEY_EPS_STR_LEN ) == 0 ) {
#ifdef EBUG
			    if ( debugflg )
				fprintf( stderr, "STARTJOB eps\n" );
#endif EBUG
				ignoring = 1;	/* EPS document */
			    } else if ( strncasecmp( keyword, KEY_QUERY_STR,
				    KEY_QUERY_STR_LEN ) == 0 ) {
#ifdef EBUG
			    if ( debugflg )
				fprintf( stderr, "STARTJOB query\n" );
#endif EBUG
				ignoring = 1;	/* query document */
			    } else if ( strncasecmp( keyword, KEY_EXIT_STR,
				    KEY_EXIT_STR_LEN ) == 0 ) {
#ifdef EBUG
			    if ( debugflg )
				fprintf( stderr, "STARTJOB exitserver\n" );
#endif EBUG
				ignoring = 1;	/* exit server document */
			    } else {
#ifdef EBUG
			    if ( debugflg )
				fprintf( stderr, "STARTJOB doc\n" );
#endif EBUG
				ignoring = 0;	/* assume normal document */
			    }
			}
			break;

		    case ENDJOB:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "ENDJOB\n" );
#endif EBUG
			if ( !ignoring ) {
			    jobcomplete = 1;
			}
			break;

		    case TITLE:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "TITLE\n" );
#endif EBUG
			if ( !ignoring && !gottitle ) {
			    len = linelen( s ) - sizeof(TITLE_STR)-1;
			    if (( jobname = malloc( len+1 )) == NULL ) {
				pap_close_req( paph, NULL );
				return -1;
			    }
			    strncpy( jobname, s + sizeof(TITLE_STR)+1, len );
			    jobname[ len ] = '\0';
			    ++gottitle;
			}
			break;

		    case USER:
#ifdef EBUG
			if ( debugflg ) fprintf( stderr, "USER\n" );
#endif EBUG
			if ( !ignoring && !gotuser ) {
			    len = linelen( s ) - sizeof(USER_STR)-1;
			    if (( literalname = malloc( len+1 )) == NULL ) {
				pap_close_req( paph, NULL );
				return -1;
			    }
			    strncpy( literalname, s + sizeof(USER_STR)+1, len );
			    literalname[ len ] = '\0';
			    ++gotuser;
			}
			break;
		    }
		}
	    }

	    if ( s != NULL ) {
		len = linelen( s )+1;
		if ( tbuf == NULL ) {
		    /* packet breaks line -- omit newline character */
		    --len;
		}
		if ( psfd >= 0 ) {
		    /* proc. set. file is open -- write to it */
		    if ( write( psfd, s, len ) < 0 ) {
			syslog( LOG_ERR, "write proc set file: %m" );
			pap_close_req( paph, "papd: unable to write proc set" );
			psp_free( psp );
			return -1;
		    }
		}
		if ( !ignoring ) {
		    /* in midst of job -- write to data file */
		    if ( write( dffd, s, len ) < 0 ) {
			syslog( LOG_ERR, "write spool file: %m" );
			pap_close_req( paph, "papd: unable to write spool file" );
			psp_free( psp );
			return -1;
		    }
		}
	    }
	}
    }
}


sprintf_addr( s, saddr )
    char		*s;
    struct sockaddr_at	*saddr;
{
    sprintf( s, "%d.%d.%d", ntohs( saddr->sat_addr.s_net ),
	saddr->sat_addr.s_node, saddr->sat_port );
}
