/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <syslog.h>
#include "papd.h"
#include "lp.h"


extern char	*malloc();
extern char	*rindex();
extern int	pgetent();
extern char	*pgetstr();
extern char	*copyarg();

extern char	*host;
extern int	uid;
extern char	*servername;
extern char	*printspooldir;
extern int	debugflg;
extern int	pipedprinter;
extern char	*printer;
char		**printcmdargv;
extern char	*operator;
char		*lplockfile;
char		*tffile = NULL;
int		tffd;
char		*cffile = NULL;
char		*jobname;
char		*literalname;
int		jobseqnum;
int		dffd;
char		*dffile = NULL;
char		*psetfile = NULL;

static char	buf[ 1024 ];
static char	entbuf[ 1024 ];

char		*tempfilename();
char		*spoolfilename();


initspoolfiles()
{
    struct stat	st;
    int		seqfd;
    int		nr;
    
    if ( pipedprinter ) {
    /*
     * piped printer
     */
	dffile = tempfilename();
	if (( dffd = openspoolfile( dffile )) < 0 ) {
	    return( -1 );
	}
	return( 0 );
    }

    /*
     * lpd-style printing
     */
    sprintf( buf, "%s/%s", printspooldir, lplockfile );
    if ( uid != 0 && stat( buf, &st ) == 0 && ( 010 & st.st_mode )) {
	syslog( LOG_ERR, "printer queue disabled" );
	return( -1 );
    }
    sprintf( buf, "%s/.seq", printspooldir );
    if (( seqfd = open( buf, O_RDWR | O_CREAT, 0661 )) < 0 ) {
	syslog( LOG_ERR, "can't open %s: %m", buf );
	return( -1 );
    }
    if ( flock( seqfd, LOCK_EX ) != 0 ) {
	syslog( LOG_ERR, "can't lock %s: %m", buf );
	return( -1 );
    }
    if (( nr = read( seqfd, buf, sizeof( buf ))) > 0 ) {
	buf[ nr ] = '\0';
	jobseqnum = atoi( buf );
    } else {
	jobseqnum = 0;
    }
    dffile = spoolfilename( "df", jobseqnum, 'B' );
    tffile = spoolfilename( "tf", jobseqnum, 'A' );
    cffile = spoolfilename( "cf", jobseqnum, 'A' );
    lseek( seqfd, 0L , 0 );
    sprintf( buf, "%03d\n", ( jobseqnum + 1 ) % 1000 );
    if ( write( seqfd, buf, 4 ) < 0 ) {
	syslog( LOG_ERR, "write LPD .seq: %m" );
    }
    close( seqfd );
    jobname = literalname = NULL;
    if (( tffd = openspoolfile( tffile )) < 0 ||
      ( dffd = openspoolfile( dffile )) < 0 ) {
	return( -1 );
    }
    return( 0 );
}


finishspoolfiles()
{
    int		rc;
    char	*psetfilelink, *psetfilelinktail, *dffiletail, *s;

    if ( pipedprinter ) {
    /*
     * piped printer: print the files, unlink them, clean up
     */

	close( dffd );
	rc = 0;

	if ( psetfile != NULL ) {
	    rc = doprintcmd( psetfile, 0 );
	    free( psetfile );
	}

	if ( rc == 0 ) {
	    rc = doprintcmd( dffile, 1 );
	}

	unlinktempfiles();
	if ( rc != 0 ) {
	    syslog( LOG_ERR, "problem piping to \"%s\"", printer + 1 );
	}
	return( rc );
    }

/*
 * lpd printer: add file to print (and unlink line) to cf file,
 * rename tf to cf, close files
 */

    s = rindex( dffile, '/' );
    dffiletail = ( s != NULL ? s+1 : dffile );
    if ( psetfile != NULL ) {
	psetfilelink = spoolfilename( "df", jobseqnum, 'A' );
	s = rindex( psetfilelink, '/' );
	psetfilelinktail = ( s != NULL ? s+1 : psetfilelink );
    }

    if ( addcfentry( 'H', host ) < 0 ||
      addcfentry( 'P', operator ) < 0 ||
      ( jobname != NULL && addcfentry( 'J', jobname ) < 0 ) ||
      addcfentry( 'C', host ) < 0 ||
      ( literalname != NULL && addcfentry( 'L', literalname ) < 0 )) {
	return( -1 );
    }
    if ( psetfile != NULL ) {
	struct stat stb;
	char	    statinfo[ 25 ];

	if ( symlink( psetfile, psetfilelink ) < 0 ) {
	    syslog( LOG_ERR, "finishtempfiles symlink: %m" );
	    return( -1 );
	}
    	if ( stat( psetfilelink, &stb ) < 0 ) {
	    syslog( LOG_ERR, "finishtempfiles stat: %m" );
	    return( -1 );
	}
	sprintf( statinfo, "%d %d", stb.st_dev, stb.st_ino );
	if ( addcfentry( 'S', statinfo ) < 0 ||
	  addcfentry( 'l', psetfilelinktail ) < 0 ||
          ( !debugflg && addcfentry( 'U',psetfilelinktail ) < 0 )) {
	    return( -1 );
	}
    } 
    if ( addcfentry( 'l', dffiletail ) < 0 ||
      ( !debugflg && addcfentry( 'U', dffiletail ) < 0 ) ||
      ( jobname != NULL && addcfentry( 'N', jobname ) < 0 )) {
	return( -1 );
    }
    if ( rename( tffile, cffile ) != 0 ) {
	syslog( LOG_ERR, "finishtempfiles rename: %m" );
	return( -1 );
    }

    close( tffd );
    close( dffd );
    free( psetfile );
    free( dffile );
    free( tffile );
    free( cffile );
    return( 0 );
}


addcfentry( tag, s )
    char	tag;
    char	*s;
{
    char	*p;

    sprintf( buf, "%c%s\n", tag, s );
    /*
     * zero the high-bit of any high-bit set characters since under the
     * right conditions lpd will dump core if it encounters one in the cf file
     */
    for ( p = buf; *p != '\0'; ++p ) {
	if ( *p & 0x80 ) {
	    *p = *p & 0x7f;
	}
    }

    if ( write( tffd, buf, strlen( buf )) != strlen( buf )) {
	syslog( LOG_ERR, "can't write control file %s: %m", tffile );
	return( -1 );
    }
    return( 0 );
}


openspoolfile( name )
    char	*name;
{
    int	fd;

    if (( fd = open( name, O_WRONLY | O_CREAT, FILEMODE )) < 0 ) {
	syslog( LOG_ERR, "openspoolfile: open %s: %m", name );
	return( -1 );
    }
    return( fd );
}


unlinktempfiles()
{
    if ( dffile ) {
#ifdef EBUG
	syslog( LOG_INFO, "unlinktempfiles keeping dffile <%s>", dffile );
#else EBUG
	unlink( dffile );
#endif EBUG
	free( dffile );
	dffile = NULL;
    }

    if ( !pipedprinter ) {
	if ( tffile ) {
#ifdef EBUG
	    syslog( LOG_INFO, "unlinktempfiles keeping tffile <%s>", tffile );
#else EBUG
	    unlink( tffile );
#endif EBUG
	    free( tffile );
	    tffile = NULL;
	}

	if ( cffile ) {
#ifdef EBUG
	    syslog( LOG_INFO, "unlinktempfiles keeping cffile <%s>", cffile );
#else EBUG
	    unlink( cffile );
#endif EBUG
	    free( cffile );
	    cffile = NULL;
	}
    }
    return( 0 );
}


char	*
spoolfilename( prefix, seqnum, c )
    char	*prefix;
    int		seqnum;
    char	c;
{
    char	*name;

    if (( name = malloc( 8 + strlen( prefix ) + strlen( printspooldir )
	  + strlen( host ))) == NULL ) {
	syslog( LOG_ERR, "can't get memory, exiting" );
	exit( -1 );
    }
    sprintf( name, "%s/%s%c%03d%s", printspooldir, prefix, c, seqnum, host );
    return( name );
}


char	*
tempfilename()
{
    char	*name;
    extern char	*mktemp();

    if (( name = malloc( 8 + strlen( printspooldir ))) == NULL ) {
	syslog( LOG_ERR, "can't get memory, exiting" );
	exit( -1 );
    }
    sprintf( name, "%s/XXXXXX", printspooldir );
    return( mktemp( name ));
}


readprintcap( printer )
    char	*printer;
{
    int		rc;
    char	*s, *bp = entbuf;

    if (( rc = pgetent( buf, printer )) == 0 ) {
	syslog( LOG_ERR, "unknown printer: %s", printer );
	return( -1 );
    } else if ( rc < 0 ) {
	syslog( LOG_ERR, "unable to open printer capabilities file" );
	return( -1 );
    }

    if (( s = pgetstr( "sd", &bp )) != NULL ) {
	printspooldir = copyarg( s );
    }

    if (( lplockfile = pgetstr( "lo", &bp )) == NULL ) {
	lplockfile = copyarg( LPD_DEF_LOCKFILE );
    }

    return( 0 );
}


doprintcmd( file, closepipe )
    char	*file;
{
    int		f, len, rc = 0;
    static FILE	*pr;


    /*
     * create pipe to printer and open file to be printed
     */
    if ( pr == NULL && ( pr = popen( printer + 1, "w" )) == NULL ) {
	return( -1 );
    }
    if (( f = open( file, O_RDONLY )) < 0 ) {
	pclose( pr );
	return( -1 );
    }

    /*
     * copy file to printer pipe stdin
     */
    while (( len = read( f, buf, sizeof( buf ))) > 0 ) {
	if ( write( fileno( pr ), buf, len ) != len ) {
	    pclose( pr );
	    close( f );
	    return( -1 );
	}
    }

    close( f );
    if ( closepipe && ( rc = pclose( pr )) != 0 ) {
	syslog( LOG_ERR, "pclose() failed: %m" );
    }
    return( rc );
}


startdaemon( printer )
    char	*printer;
/* kick the printer daemon...  (this code is mostly lifted from BSD 4.3 lpr) */
{
    struct sockaddr_un	saun;
    int			s, n;

    if (( s = socket( AF_UNIX, SOCK_STREAM, 0 )) < 0 ) {
	syslog( LOG_ERR, "startdaemon socket: %m" );
	return( -1 );
    }
    saun.sun_family = AF_UNIX;
    strcpy( saun.sun_path, LPDSOCKETNAME );
    if ( connect( s, &saun, strlen( saun.sun_path ) + 2 ) < 0 ) {
	syslog( LOG_ERR, "startdaemon connect: %m" );
	return( -1 );
    }
    sprintf( buf, "\1%s\n", printer );
    n = strlen( buf );
    if ( write( s, buf, n ) != n ) {
	syslog( LOG_ERR, "startdaemon write: %m" );
	return( -1 );
    }
    if ( read( s, buf, 1 ) == 1 ) {
	if ( buf[0] == '\0' ) {		/* everything is OK */
	    close( s );
	    return( 0 );
	}
	putchar( buf[0] );
    }
    return( -1 );
}
