/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/signal.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <pwd.h>
#include <netatalk/at.h>
#include <atalk/atp.h>
#include "pap.h"
#include "lp.h"

extern char	*malloc();

extern int	errno;
extern int	readrequests;

int		debugflg = 0;
int		pid;
static char	statbuf[ PAP_MAXDATA ];
char		*buf;
static char	rbuf[ PAP_MAXSTREAM ];

main( argc, argv )
    int		argc;
    char	**argv;
{
    extern int		optind;
    extern char		*optarg;

    int			c, fd;
    int			errflg = 0;
    char		*printer = NULL;
    char		*statusfile = NULL;
    char		*usage = "usage: %s [-d] [-p printer] [-s statusfile] [file] ...\n";

    pid = getpid();

    /* process arguments */
    while (( c = getopt( argc, argv, "dp:s:" )) != EOF ) {
	switch( c ) {
	case 'd':
	    if ( debugflg ) {
		++errflg;
	    } else {
		++debugflg;
	    }
	    break;
	case 'p':
	    if ( printer != NULL ) {
		++errflg;
	    } else {
		printer = optarg;
	    }
	    break;
	case 's':
	    if ( statusfile != NULL ) {
		++errflg;
	    } else {
		statusfile = optarg;
	    }
	    break;
	default:
	    ++errflg;
	}
    }

    if ( printer == NULL && (( printer = pap_printername()) == NULL )) {
	pap_fperror( stderr, PAP_PRINTERNAMEFILE );
	exit( 2 );
    }

    if ( errflg ) {
	fprintf( stderr, usage, argv[ 0 ] );
	exit( 2 );
    }

    if ( optind == argc ) {
	if ( sendfile( 0, printer, statusfile, argv[ 0 ] ) < 0 ) {
	    exit( 1 );
	}
    } else for ( ; optind < argc; ++optind ) {
	if (( fd = open( argv[ optind ], O_RDONLY )) < 0 ) {
	    pap_fperror( stderr, argv[ optind ] );
	    exit( 2 );
	}
	if ( sendfile( fd, printer, statusfile, argv[ 0 ] ) < 0 ) {
	    fprintf( stderr, "%s[%d]: send file failed\n", argv[ 0 ], pid );
	    exit( 1 );
	}
	close( fd );
    }

    fprintf( stderr, "%s[%d]: successful session\n", argv[ 0 ], pid );
    exit( 0 );
}


sendfile( fd, printer, statusfile, prog )
    int		fd;
    char	*printer;
    char	*statusfile;
    char	*prog;
{
    fd_set		fds;
    PAP			ph;
    u_short		res;
    int			rc, rlen, bufsiz, n, eof, sent_eof;
    struct sockaddr_at	sat;
    struct passwd	*pwent;
    char		*tty, *p;	
    struct timeval	tv, ntv;
    extern char		*ttyname(), *rindex();

    if ( pap_nametoaddr( printer, &sat ) < 0 ) {
	pap_fperror( stderr, printer );
	return( -1 );
    }

    if ( pap_status( &sat, statbuf, sizeof( statbuf )) < 0 ) {
	pap_fperror( stderr, printer );
	return -1;
    }
    fprintf( stderr, "%s[%d]: connecting to %s: %s\n", prog, pid, printer,
		statbuf );
    if ( statusfile != NULL ) {
	update_status_file( statbuf, statusfile );
    }

    tv.tv_sec = ntv.tv_sec = 0;

    do {
	if ( tv.tv_sec != 0 ) {
	    gettimeofday( &ntv, (struct timezone *)NULL );
	}
	res = !PAP_RESULT_READY;
	if (( ph = pap_open( &sat, PAP_MAXQUANTUM, ntv.tv_sec - tv.tv_sec,
			&res, statbuf, sizeof( statbuf ))) == NULL &&
		    pap_status( &sat, statbuf, sizeof( statbuf )) < 0 ) {
	    pap_fperror( stderr, printer );
	    return( -1 );
	}
	if ( res != PAP_RESULT_READY ) {
	    if ( tv.tv_sec == 0 ) {
		gettimeofday( &tv, (struct timezone *)NULL );
	    }
	    if ( statusfile != NULL ) {
		update_status_file( statbuf, statusfile );
	    }
	    if ( ph != NULL ) {
		atp_close( ph->paph_atph );
		free( ph );
	    }
	    sleep( 2 );
	}
    } while ( res != PAP_RESULT_READY );

    fprintf( stderr, "%s[%d]: starting session\n", prog, pid );
    bufsiz = ph->paph_sendquantum * PAP_MAXDATA;
    if (( buf = malloc( bufsiz )) == NULL ) {
	pap_fperror( stderr, "malloc" );
	return( -1 );
    }

    if ( pap_initiate_read( ph ) < 0 ) {
	pap_fperror( stderr, "pap_initiate_read" );
	return( -1 );
    }

    sent_eof = eof = rlen = 0;

    if ( isatty( fd )) { /* set jobname status and enter executive mode */
	if (( pwent = getpwuid( getuid())) == NULL ) {
	    pap_fperror( "getpwuid" );
	    return( -1 );
	}
	sprintf( buf, "statusdict /jobname (%s@", pwent->pw_name );
	gethostname( buf + strlen( buf ), bufsiz - strlen( buf ));
	tty = ttyname( fd );
	if (( p = rindex( tty, '/' )) != NULL ) {
	    tty = ++p;
	}
	sprintf( buf + strlen( buf ), " (%s)) put\nexecutive\n", tty );
	rlen = strlen( buf );
    }

    while ( 1 ) {
	FD_ZERO( &fds );
	if ( !eof && rlen == 0 ) {	/* only read if buffer is empty */
	    FD_SET( fd, &fds );
	}
	FD_SET( pap_ddpsocket( ph ), &fds );

	if ( select( FD_SETSIZE, &fds, (fd_set *)NULL, (fd_set *)NULL,
		(struct timeval *)NULL ) < 0 ) {
	    pap_fperror( stderr, "select" );
	}

	if ( rlen == 0 && FD_ISSET( fd, &fds )) {
#ifdef EBUG
	    if ( debugflg ) {
		fprintf( stderr, "%s[%d] sendfile: reading file\n", prog, pid );
	    }
#endif EBUG
	    if (( rlen = read( fd, buf, bufsiz )) < 0 ) {
		pap_fperror( stderr, "read" );
	    }
#ifdef EBUG
	    if ( debugflg ) {
		fprintf( stderr, "%s[%d] sendfile: read rlen = %d\n", prog,
			pid, rlen );
	    }
#endif EBUG
	    if ( rlen == 0 ) {
		/*
		 * send an end of file
		 */
#ifdef EBUG
		if ( debugflg ) {
		    fprintf( stderr, "%s[%d] sendfile: at EOF\n", prog, pid );
		}
#endif EBUG
		if ( isatty( fd )) {
		    strcpy( buf, "quit\n" );
		    rlen = strlen( buf );
		} else {
		    rlen = 0;
		}
		eof = 1;
	    }
	}

	if ( FD_ISSET( pap_ddpsocket( ph ), &fds )) {
#ifdef EBUG
	    if ( debugflg ) {
		fprintf( stderr, "%s[%d]: calling pap_select\n", prog, pid );
	    }
#endif EBUG
	    switch( pap_select( ph )) {
	    case PAP_SELECT_REQUEST:
#ifdef EBUG
		if ( debugflg ) {
		    fprintf( stderr, "%s[%d] sendfile: got a request\n",
			    prog, pid );
		}
#endif
		if (( rc = pap_get_request( ph )) < 0 ) {
		    pap_fperror( stderr, "pap_get_request" );
		    return( -1 );
		}

		if ( rc == 1 ) {
		    fprintf( stderr, "%s[%d] connection unexpectedly closed\n",
			prog, pid );
		    return( -1 );
		}

		if ( rc == PAP_TICKLE ) {
#ifdef EBUG
		    if ( debugflg ) {
			fprintf( stderr,
			    "%s[%d] sendfile: got a tickle, sending one\n",
			    prog, pid );
		    }
#endif
		    if ( pap_send_tickle( ph, &ph->paph_saddr ) < 0 ) {
			pap_fperror( stderr, "pap_send_tickle" );
			return( -1 );
		    }
		}

		break;

	    case PAP_SELECT_READ:
#ifdef EBUG
		if ( debugflg ) {
		    fprintf( stderr, "%s[%d] sendfile: reading socket\n",
			    prog, pid );
		}
#endif
		if (( n = pap_read_data( ph, rbuf, &rc )) < 0 ) {
		    pap_fperror( stderr, "pap_read_data" );
		    return( -1 );
		}
		print_pap_data( rbuf, n );
		if ( eof && rc ) {	/* we agree that this is end of file */
		    pap_close( ph );
		    goto out;
		}
		if (( rc = pap_initiate_read( ph )) < 0 ) {
		    pap_fperror( stderr, "pap_initiate_read" );
		}
		break;

	    case 0:	/* no new packets */
#ifdef EBUG
		if ( debugflg ) {
		    fprintf( stderr, "%s[%d] sendfile: pap_select zero\n",
			    prog, pid );
		}
#endif

	    default:
		pap_fperror( stderr, "pap_select" );
		return( -1 );
	    }
	}
	
	/*
	 * if there is something to send and we have a request, send it now
	 */
	if (( rlen > 0 || ( eof && !sent_eof )) && readrequests > 0 ) {
	    if ( pap_do_write( ph, buf, rlen, eof ) < 0 ) {
		pap_fperror( stderr, "pap_do_write" );
		return( -1 );
	    }
	    rlen = 0;
	    if ( eof ) {
		sent_eof = 1;
	    }

	} else if ( statusfile != NULL ) {
	    if ( pap_status( &sat, statbuf, sizeof( statbuf )) < 0 ) {
		pap_fperror( stderr, printer );
	    } else {
		update_status_file( statbuf, statusfile );
	    }
	}
    }

out:
    pap_close( ph );
    return ( 0 );
}


print_pap_data( buf, len )
    char	*buf;
    int		len;
{
    int		i;
    char	*p;

    for ( p = buf, i =  len; i > 0; --i ) {
	if ( *p == '\r' ) {
	    *p = '\n';
	}
    }
#ifdef EBUG
    if ( debugflg ) {
	fprintf( stderr, "print_pap_data[%d] %d bytes\n", pid, len );
    }
#endif
    fwrite( buf, len, 1, stdout );
}


update_status_file( s, statusfile )
    char	*s;
    char	*statusfile;
{
    int			fd;
    struct iovec	iov[ 2 ];

    if (( fd = open( statusfile, O_WRONLY | O_TRUNC )) < 0 ) {
	pap_fperror( stderr, statusfile );
	return;
    }

    iov[ 0 ].iov_base = s;
    iov[ 0 ].iov_len = strlen( s );
    iov[ 1 ].iov_base = "\n";
    iov[ 1 ].iov_len = 1;
    writev( fd, iov, 2 );
    close( fd );
}
