/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#ifdef EBUG
#include <stdio.h>
#endif EBUG
#include <sys/types.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <syslog.h>
#include <netinet/in.h>
#undef s_net
#include <netatalk/at.h>
#include <atalk/nbp.h>
#include <atalk/atp.h>
#include "pap.h"
#include "paperrno.h"

extern int	errno;
extern int	debugflg;

static char		dbuf[ 8 ][ ATP_MAXDATA ];
static struct iovec	iov[ 8 ] = {
    { dbuf[ 0 ], ATP_MAXDATA },
    { dbuf[ 1 ], ATP_MAXDATA },
    { dbuf[ 2 ], ATP_MAXDATA },
    { dbuf[ 3 ], ATP_MAXDATA },
    { dbuf[ 4 ], ATP_MAXDATA },
    { dbuf[ 5 ], ATP_MAXDATA },
    { dbuf[ 6 ], ATP_MAXDATA },
    { dbuf[ 7 ], ATP_MAXDATA },
};

int		readrequests = 0;	/* outstanding read request */
u_short		readreqport;
int		writepending = 0;	/* need to write */
char		*writedata;
int		writedlen;
int		writeeof;

pap_initiate_read( paph )
    PAP		paph;
{
    struct atp_block	atpb;
    char		sbuf[ 4 ];
    u_short		seqnum;

    seqnum = htons( ++paph->paph_sendseqnum );
    sbuf[ 0 ] = paph->paph_connid;
    sbuf[ 1 ] = PAP_SENDDATA;
    bcopy( &seqnum, &sbuf[ 2 ], sizeof( short ));
    atpb.atp_saddr = &paph->paph_saddr;
    atpb.atp_sreqdata = sbuf;
    atpb.atp_sreqdlen = 4;
    atpb.atp_sreqto = PAP_DATA_RETRY_TIME;
    atpb.atp_sreqtries = PAP_DATA_RETRY_CNT;
#ifdef EBUG
    if ( debugflg ) {
	timefprint( stderr );
	fprintf( stderr, "sending read request seq num %hu\n", ntohs( seqnum ));
	bfprint( stderr, sbuf, 4 );
	fflush( stderr );
    }
#endif EBUG
    if ( atp_sreq( paph->paph_atph, &atpb, paph->paph_recvquantum,
	    ATP_XO ) < 0 ) {
	return -1;
    }
    return( 0 );
}


pap_read_data( paph, buf, eof )
    PAP		paph;
    char	*buf;
    int		*eof;
{
    struct atp_block	atpb;
    char		*bp;
    int			i;
    int			tries = 0;

    while ( 1 ) {
	for ( i = 0; i < paph->paph_recvquantum; ++i ) {
	    iov[ i ].iov_len = ATP_MAXDATA;
	}
	atpb.atp_saddr = &paph->paph_saddr;
	atpb.atp_rresiovcnt = paph->paph_recvquantum;
	atpb.atp_rresiov = iov;
	if ( atp_rresp( paph->paph_atph, &atpb ) < 0 ) {
	    return( -1 );
	}

	/* sanity check */
#ifndef LIBERAL
	if ( (u_char)iov[ 0 ].iov_base[ 0 ] == paph->paph_connid &&
#else LIBERAL
	if (
#endif LIBERAL
	  iov[ 0 ].iov_base[ 1 ] == PAP_DATA ) {
	    *eof = iov[ 0 ].iov_base[ 2 ];
	    for ( bp = buf, i = 0; i < atpb.atp_rresiovcnt; ++i ) {
		bcopy( iov[ i ].iov_base + 4, bp, iov[ i ].iov_len - 4 );
		bp += ( iov[ i ].iov_len - 4 );
	    }
#ifdef EBUG
	    if ( debugflg ) {
		timefprint( stderr );
		fprintf( stderr, "pap_read %d bytes\n", bp - buf );
		bfprint( stderr, buf, bp - buf );
		fflush( stderr );
	    }
#endif EBUG
	    return( bp - buf );
	}

	syslog( LOG_ERR, "pap_read ignoring connid %X, op=%X",
	    (u_char)iov[ 0 ].iov_base[ 0 ], iov[ 0 ].iov_base[ 1 ] );
	if ( pap_initiate_read( paph ) < 0 ) {
	    return( -1 );
	}
    }

    /* NOT REACHED */
}


pap_read( paph, buf, eof )
    PAP		paph;
    char	*buf;
    int		*eof;
{
	if ( pap_initiate_read( paph ) < 0 ) {
	    return( -1 );
	}

	return( pap_read_data( paph, buf, eof ));
}


pap_select( paph )
    PAP			paph;
{
    struct sockaddr_at	saddr;

    saddr = paph->paph_saddr;
    saddr.sat_port = ATADDR_ANYPORT;
    return( atp_rsel( paph->paph_atph, &saddr, ATP_TRESP | ATP_TREQ ));
}


pap_write( paph, data, dlen, eof )
    PAP		paph;
    char	*data;
    int		dlen;
    int		eof;
{
    if ( dlen > paph->paph_sendquantum * PAP_MAXDATA ) {
	errno = EINVAL;
	return -1;
    }
    if ( readrequests <= 0 ) {
	while ( writepending ) {
	    /* wait for a request to allow us to write waiting data */
	    if ( pap_get_request( paph ) != 0 ) {
		return -1;
	    }
	}
	/* save the write request and just say it went okay */
	writedata = data;
	writedlen = dlen;
	writeeof = eof;
	writepending = 1;
#ifdef EBUG
	if ( debugflg ) {
	    timefprint( stderr );
	    fprintf( stderr, "pap_write -- delaying write\n" );
	    fflush( stderr );
	}
#endif EBUG
	return 0;
    } else {
	/* we have previously received a request... send response */
	return ( pap_do_write( paph, data, dlen, eof ));
    }
}


pap_do_write( paph, data, dlen, eof )
    PAP			paph;
    char		*data;
    int			dlen;
    int			eof;
{
    struct atp_block	atpb;
    struct sockaddr_at	saddr;
    int			i;
    char		*bh;

    --readrequests;
    writepending = 0;
    saddr = paph->paph_saddr;
    saddr.sat_port = readreqport;
    atpb.atp_saddr = &saddr;
    atpb.atp_sresiov = iov;
#ifdef EBUG
    if ( debugflg ) {
	fprintf( stderr, "pap_write %d bytes\n", dlen );
	bfprint( stderr, data, dlen );
	fflush( stderr );
    }
#endif EBUG
    if ( dlen == 0 ) {
	iov[ 0 ].iov_len = 4;
	dbuf[ 0 ][ 0 ] = paph->paph_connid;
	dbuf[ 0 ][ 1 ] = PAP_DATA;
	dbuf[ 0 ][ 2 ] = ( eof != 0 ? 1 : 0 );
	dbuf[ 0 ][ 3 ] = 0;
	i = 1;
    } else for ( i = 0, bh = data; dlen > 0; ++i ) {
	iov[ i ].iov_len = ( dlen > PAP_MAXDATA ? PAP_MAXDATA : dlen );
	bcopy( bh, &dbuf[ i ][ 4 ], iov[ i ].iov_len );
	dbuf[ i ][ 0 ] = paph->paph_connid;
	dbuf[ i ][ 1 ] = PAP_DATA;
	dbuf[ i ][ 2 ] = ( eof != 0 ? 1 : 0 );
	dbuf[ i ][ 3 ] = 0;
	bh += iov[ i ].iov_len;
	dlen -= iov[ i ].iov_len;
	iov[ i ].iov_len += 4;
    }
    atpb.atp_sresiovcnt = i;
#ifdef EBUG
    if ( debugflg ) {
	for ( i = 0; i < atpb.atp_sresiovcnt; ++i ) {
	    timefprint( stderr );
	    fprintf( stderr, "packet %d: ", i );
	    bfprint( stderr, dbuf[ i ], iov[ i ].iov_len );
	}
	fflush( stderr );
    }
#endif EBUG
    if ( atp_sresp( paph->paph_atph, &atpb ) < 0 ) {
	return -1;
    }
    return 0;
}


pap_send_tickle( paph, to )
    PAP			paph;
    struct sockaddr_at	*to;
{
    struct atp_block	atpb;

    dbuf[ 0 ][ 0 ] = paph->paph_connid;
    dbuf[ 0 ][ 1 ] = PAP_TICKLE;
    dbuf[ 0 ][ 2 ] = dbuf[ 0 ][ 3 ] = 0;

    atpb.atp_saddr = to;
    atpb.atp_sreqdata = dbuf[ 0 ];
    atpb.atp_sreqdlen = 4;
    atpb.atp_sreqto = 0;	/* we don't expect a reply */
    atpb.atp_sreqtries = 1;
#ifdef EBUG
    if ( debugflg ) {
	timefprint( stderr );
	fprintf( stderr, "pap_send_tickle\n" );
	fflush( stderr );
    }
#endif EBUG
    if ( atp_sreq( paph->paph_atph, &atpb, 0, 0 ) < 0 ) {
	return -1;
    }
    return 0;
}


pap_close_req( paph, msg )
    PAP			paph;
    char		*msg;
{
    char		sbuf[ 4 ];
    struct atp_block	atpb;
    struct sockaddr_at	saddr;

    if ( msg == NULL ) {
	msg = "Remote Spooler Error";
    }

    if ( pap_write( paph, msg, strlen( msg ), 1 ) < 0 ) {
	return -1;
    }

    sbuf[ 0 ] = paph->paph_connid;
    sbuf[ 1 ] = PAP_CLOSECONN;
    sbuf[ 2 ] = sbuf[ 3 ] = 0;
    saddr = paph->paph_saddr;
    atpb.atp_saddr = &saddr;
    atpb.atp_sreqdata = sbuf;
    atpb.atp_sreqdlen = 4;
    atpb.atp_sreqto = PAP_OPEN_RETRY_TIME;
    atpb.atp_sreqtries = PAP_OPEN_RETRY_CNT;

#ifdef EBUG
    if ( debugflg ) {
	timefprint( stderr );
	fprintf( stderr, "sending closeconn request\n" );
	bfprint( stderr, sbuf, 4 );
	fflush( stderr );
    }
#endif EBUG
    if ( atp_sreq( paph->paph_atph, &atpb, 1, ATP_XO ) < 0 ) {
	return -1;
    }

    iov[ 0 ].iov_len = ATP_MAXDATA;
    atpb.atp_rresiovcnt = 1;
    atpb.atp_rresiov = iov;
    if ( atp_rresp( paph->paph_atph, &atpb ) < 0 ||
	    (u_char)iov[ 0 ].iov_base[ 0 ] != paph->paph_connid ||
	    iov[ 0 ].iov_base[ 1 ] != PAP_CLOSECONNREPLY ) {
	return -1;
    }

    return 0;
}

pap_close_repl( paph, port )
    PAP			paph;
    u_char		port;
{
    struct atp_block	atpb;
    struct sockaddr_at	saddr;

    saddr = paph->paph_saddr;
    saddr.sat_port = port;
    atpb.atp_saddr = &saddr;
    atpb.atp_sresiov = iov;
    atpb.atp_sresiovcnt = 1;
    iov[ 0 ].iov_len = 4;
    dbuf[ 0 ][ 0 ] = paph->paph_connid;
    dbuf[ 0 ][ 1 ] = PAP_CLOSECONNREPLY;
    dbuf[ 0 ][ 2 ] = dbuf[ 0 ][ 3 ] = 0;
#ifdef EBUG
    if ( debugflg ) {
	timefprint( stderr );
	fprintf( stderr, "pap_close_repl: packet 0: " );
	bfprint( stderr, dbuf[ 0 ], iov[ 0 ].iov_len );
	fflush( stderr );
    }
#endif EBUG
    if ( atp_sresp( paph->paph_atph, &atpb ) < 0 ) {
	return -1;
    }
    return 0;
}


/*
 * return: -1 if error
 *         0 if read request received
 *         PAP_TICKLE if tickle received
 *         1 if close connection received
 */
pap_get_request( paph )
    PAP		paph;
{
    struct sockaddr_at	saddr;
    struct atp_block	atpb;
    u_short		seqnum;

    saddr = paph->paph_saddr;
    saddr.sat_port = ATADDR_ANYPORT;
    atpb.atp_saddr = &saddr;
    atpb.atp_rreqdata = dbuf[ 0 ];
    atpb.atp_rreqdlen = ATP_MAXDATA;
    if ( atp_rreq( paph->paph_atph, &atpb ) < 0 ) {
	return -1;
    }
#ifdef EBUG
    if ( debugflg ) {
	timefprint( stderr );
	fprintf( stderr, "got request\n" );
	bfprint( stderr, dbuf[ 0 ], atpb.atp_rreqdlen );	
	fflush( stderr );
    }
#endif EBUG
    switch( dbuf[ 0 ][ 1 ] ) {
    case PAP_SENDDATA:
	bcopy( &dbuf[ 0 ][ 2 ], &seqnum, 2 );
	seqnum = ntohs( seqnum );
	if ( seqnum && seqnum < paph->paph_recvseqnum ) {
	    pap_errno = PAP_EBADSEQNUM;
	    return -1;
	}
	if ( seqnum ) paph->paph_recvseqnum = seqnum;
#ifdef EBUG
	if ( debugflg ) {
	    fprintf( stderr, "send data request seq num %hu\n", seqnum );
	    fflush( stderr );
	}
#endif EBUG
	++readrequests;
	readreqport = atpb.atp_saddr->sat_port;
	if ( writepending ) {
	    pap_do_write( paph, writedata, writedlen, writeeof );
	}
	return 0;
    case PAP_CLOSECONN:
#ifdef EBUG
	if ( debugflg ) {
	    fprintf( stderr, "close connection request\n" );
	    fflush( stderr );
	}
#endif EBUG
	if ( pap_close_repl( paph, atpb.atp_saddr->sat_port ) < 0 ) {
	    return -1;
	}
	return 1;
    default:
    case PAP_TICKLE:
	return PAP_TICKLE;
    }
}
