/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <pwd.h>
#include <stdio.h>
#include <ctype.h>
#include <netinet/in.h>
#undef s_net
#include <netatalk/at.h>
#include <atalk/nbp.h>
#include <atalk/atp.h>
#include "pap.h"
#include "paperrno.h"

extern char	*malloc();
extern int	errno;
extern int	debugflg;

u_char		nextconnid = 0;

pap_nametoaddr( name, sat )
    char		*name;
    struct sockaddr_at	*sat;
{
    struct nbpnve	nn;
    char		*obj, *type, *zone;

    obj = LWOBJ;
    type = LWTYPE;
    zone = "*";
    if ( nbp_name( name, &obj, &type, &zone ) < 0 ) {
	pap_errno = PAP_EBADNAME;
	return -1;
    }
    if ( nbp_lookup( obj, type, zone, &nn, 1 ) != 1 ) {
	pap_errno = PAP_ENAMELOOKUP;
	return -1;
    }
    bcopy( (char *)&nn.nn_sat, (char *)sat, sizeof( struct sockaddr_at ));
    return( 0 );
}


pap_status( sat, status, statlen )
    struct sockaddr_at	*sat;
    char		*status;
    int			statlen;
{
    ATP			atph;
    struct atp_block	atpb;
    char		sbuf[ 4 ];
    char		rbuf[ ATP_MAXDATA ];
    struct iovec	iov;

    /* check parameters */
    if ( statlen < 2 ) {
	errno = EINVAL;
	return -1;
    }

    if (( atph = atp_open( 0 )) == NULL ) {
	return -1;
    }

    /* send request via atp */
    sbuf[ 0 ] = sbuf[ 2 ] = sbuf[ 3 ] = 0;
    sbuf[ 1 ] = PAP_SENDSTATUS;
    atpb.atp_saddr = sat;
    atpb.atp_sreqdata = sbuf;
    atpb.atp_sreqdlen = 4;
    atpb.atp_sreqto = PAP_OPEN_RETRY_TIME;
    atpb.atp_sreqtries = PAP_OPEN_RETRY_CNT;
    if ( atp_sreq( atph, &atpb, 1, ATP_XO ) < 0 ) {
	return -1;
    }

    iov.iov_base = rbuf;
    iov.iov_len = ATP_MAXDATA;
    atpb.atp_rresiov = &iov;
    atpb.atp_rresiovcnt = 1;
    if ( atp_rresp( atph, &atpb ) < 0 ) {
	return( -1 );
    }
    atp_close( atph );

#ifdef EBUG
    if ( debugflg ) {
	fprintf( stderr, "pap_status recv %d bytes:\n", iov.iov_len );
	bfprint( stderr, rbuf, iov.iov_len );
    }
#endif EBUG

    /* sanity check */
    if ( iov.iov_len < 8 || rbuf[ 1 ] != PAP_STATUS ) {
	pap_errno = PAP_EBADREPLY;
	return -1;
    }
    iov.iov_len -= 9;
    status[ 0 ] = '\0';
    if ( iov.iov_len > statlen ) {
	bcopy( &rbuf[ 9 ], status, statlen - 1 );
	status[ statlen-1 ] = '\0';
    } else if ( iov.iov_len > 0 ) {
	bcopy( &rbuf[ 9 ], status, iov.iov_len );
	status[ iov.iov_len ] = '\0';
    }
    return iov.iov_len;
}


char *
pap_printername()
{
    FILE		*f;
    struct passwd	*pwent;
    static char		s[ MAXPATHLEN ];
    char		*q, *pname;

    if (( f = fopen( PAP_PRINTERNAMEFILE, "r" )) == NULL ) {
	if (( pwent = getpwuid( geteuid())) == NULL ) {
	    return( NULL );
	}
	strcpy( s, pwent->pw_dir );
	strcat( s, "/" );
	strcat( s, PAP_PRINTERNAMEFILE );
	if (( f = fopen( s, "r" )) == NULL ) {
	    return( NULL );
	}
    }

    pname = NULL;
    while ( fgets( s, sizeof( s ), f ) != NULL ) {
	s[ strlen( s ) - 1 ] = '\0';	/* remove trailing newline */
	if ( *s == '#' ) {		/* skip comment lines */
	    continue;
	}
	    
	/*
	 * skip leading white space
	 */
	for ( q = s; *q != '\0'; ++q ) {
	    if ( !isspace( *q )) {
		break;
	    }
	}
	if ( *q == '\0' ) {
	    continue;
	}
	pname = q;

	/*
	 * remove trailing white space
	 */
	for ( q = pname + strlen( pname ) - 1; isspace( *q ) && q > pname; --q )
	    ;
	*(++q) = '\0';
	break;
    }
    fclose( f );

    return( pname );
}


PAP
pap_open( sat, quantum, waittime, result, status, statlen )
    struct sockaddr_at	*sat;
    int			quantum;
    u_short		waittime;
    u_short		*result;
    char		*status;
    int			statlen;
{
    PAP			paph;
    ATP			atph;
    struct atp_block	atpb;
    char		sbuf[ 8 ];
    char		rbuf[ ATP_MAXDATA ];
    struct iovec	iov;

    /* check parameters */
    if ( quantum < 1 || quantum > 8 || statlen < 0 ) {
	errno = EINVAL;
	return NULL;
    }

    if (( atph = atp_open( 0 )) == NULL ) {
	return NULL;
    }

    if ( nextconnid == 0 ) {	/* pick a value that is unique to be unique */
	nextconnid = getpid() & 0x000000ff;
    }

    /* send request via atp */
    sbuf[ 0 ] = nextconnid;			/* connection id */
    sbuf[ 1 ] = PAP_OPENCONN;			/* function code */
    sbuf[ 2 ] = sbuf[ 3 ] = 0;			/* zero */
    sbuf[ 4 ] = atp_sockaddr( atph )->sat_port;	/* responding port */
    sbuf[ 5 ] = quantum;			/* flow quantum */
    waittime = htons( waittime );
    bcopy( (char *) &waittime, &sbuf[ 6 ], 2 );	/* wait time */
    atpb.atp_saddr = sat;
    atpb.atp_sreqdata = sbuf;
    atpb.atp_sreqdlen = 8;
    atpb.atp_sreqto = PAP_OPEN_RETRY_TIME;
    atpb.atp_sreqtries = PAP_OPEN_RETRY_CNT;
    if ( atp_sreq( atph, &atpb, 1, ATP_XO ) < 0 ) {
	return NULL;
    }

    iov.iov_base = rbuf;
    iov.iov_len = ATP_MAXDATA;
    atpb.atp_rresiov = &iov;
    atpb.atp_rresiovcnt = 1;
    if ( atp_rresp( atph, &atpb ) < 0 ) {
	return NULL;
    }

#ifdef EBUG
    if ( debugflg ) {
	fprintf( stderr, "pap_open recv %d bytes:\n", iov.iov_len );
	bfprint( stderr, rbuf, iov.iov_len );
	fprintf( stderr, "connid=%d  replycode=%d\n", (u_char)rbuf[ 0 ],
	    (u_char)rbuf[ 1 ] );
    }
#endif EBUG

    /* sanity check */
    if ( iov.iov_len < 8 || (u_char)rbuf[ 0 ] != nextconnid ||
      rbuf[ 1 ] != PAP_OPENCONNREPLY ) {
	atp_close( atph );
	pap_errno = PAP_EBADREPLY;
	return NULL;
    }
    /*  fill the pap handle and the return args */
    if (( paph = (PAP) malloc( sizeof( struct pap_handle ))) == NULL ) {
	atp_close( atph );
	return NULL;
    }
    paph->paph_atph = atph;
    paph->paph_connid = nextconnid;
    paph->paph_sendquantum = rbuf[ 5 ];		/* server's quantum */
    paph->paph_recvquantum = quantum;		/* client (my) quantum */
    bcopy( (char *) atpb.atp_saddr, (char *) &paph->paph_saddr,
      sizeof( struct sockaddr_at ));
    paph->paph_saddr.sat_port = rbuf[ 4 ];	/* server resp. socket */
    bcopy( (char *) &rbuf[ 6 ], (char *) result, 2 );
    *result = ntohs( *result );			/* result code */
    iov.iov_len -= 9;

#ifdef EBUG
    if ( debugflg ) {
	fprintf( stderr, "pap_open result code is %hd\n", *result );
    }
#endif EBUG

    if ( statlen > 0 ) {			/* status string */
	status[ 0 ] = '\0';
    	if ( iov.iov_len > statlen ) {
	    bcopy( &rbuf[ 9 ], status, statlen - 1 );
	    status[ statlen-1 ] = '\0';
	} else if ( iov.iov_len > 0 ) {
	    bcopy( &rbuf[ 9 ], status, iov.iov_len );
	    status[ iov.iov_len ] = '\0';
	}
    }
    paph->paph_sendseqnum = paph->paph_recvseqnum = 0;
    return paph;
}


pap_close( paph )
    PAP		paph;
{
    struct atp_block	atpb;
    char		sbuf[ 4 ];
    char		rbuf[ 4 ];
    struct iovec	iov;
    struct sockaddr_at	saddr;

    /* send request via atp */
    sbuf[ 0 ] = paph->paph_connid;
    sbuf[ 1 ] = PAP_CLOSECONN;
    sbuf[ 2 ] = sbuf[ 3 ] = 0;
    saddr = paph->paph_saddr;
    atpb.atp_saddr = &saddr;
    atpb.atp_sreqdata = sbuf;
    atpb.atp_sreqdlen = 4;
    atpb.atp_sreqto = PAP_OPEN_RETRY_TIME;
    atpb.atp_sreqtries = PAP_OPEN_RETRY_CNT;
    if ( atp_sreq( paph->paph_atph, &atpb, 1, ATP_XO ) < 0 ) {
	return -1;
    }

    iov.iov_base = rbuf;
    iov.iov_len = 4;
    atpb.atp_rresiov = &iov;
    atpb.atp_rresiovcnt = 1;
    if ( atp_rresp( paph->paph_atph, &atpb ) < 0 ) {
	return -1;
    }

    /* sanity check */
    if ( (u_char)rbuf[ 0 ] != paph->paph_connid ||
	    rbuf[ 1 ] != PAP_CLOSECONNREPLY ) {
	pap_errno = PAP_EBADREPLY;
	return -1;
    }
    if ( atp_close( paph->paph_atph ) < 0 ) {
	return -1;
    }
    free( paph );
    return 0;
}
