/*
 * Copyright (c) 1990,1993 Regents of The University of Michigan.
 * All Rights Reserved. See COPYRIGHT.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <netatalk/at.h>
#include <netatalk/endian.h>
#include <stdio.h>
#include <strings.h>
#include <ctype.h>

#include "interface.h"
#include "rtmp.h"

int	seed(), phase(), net(), addr(), zone();

struct param {
    char	*p_name;
    int		(*p_func)();
} params[] = {
    { "seed",	seed },
    { "phase",	phase },
    { "net",	net },
    { "addr",	addr },
    { "zone",	zone },
};

/*
 * Read our config file. If it's not there, return -1. If it is there,
 * but has syntax errors, exit. Format of the file is as follows:
 *
 *	interface [ -seed ] [ -phase number ] [ -net net-range ]
 *	[ -addr net.node ] [ -zone zonename ]...
 * e.g.
 *	le0 -phase 1 -net 7938 -zone Argus
 * or
 *	le0 -phase 2 -net 8043-8044 -zone Argus -zone "Research Systems"
 *	le0 -phase 1 -net 7938 -zone Argus
 *
 * Pretty much everything is optional. Anything that is unspecified is
 * searched for on the network.  If -seed is not specified, the
 * configuration is assumed to be soft, i.e. it can be overridden by
 * another router. If -seed is specified, atalkd will exit if another
 * router disagrees.  If the phase is unspecified, it defaults to phase
 * 2 (the default can be overridden on the command line).  -addr can
 * replace -net, if the network in question isn't a range.  The default
 * zone for an interface is the first zone encountered for that
 * interface.
 */
readconf( cf )
    char		*cf;
{
    struct ifreq	ifr;
    struct interface	*iface, *niface;
    char		line[ 1024 ], *argv[ 128 ], *p;
    int			argc, i, j, s, cc;
    FILE		*conf;

    if ( cf == NULL ) {
	p = _PATH_ATALKDCONF;
    } else {
	p = cf;
    }
    if (( conf = fopen( p, "r" )) == NULL ) {
	if ( cf ) {
	    perror( cf );
	    exit( 1 );
	} else {
	    return( -1 );
	}
    }

    if (( s = socket( AF_APPLETALK, SOCK_DGRAM, 0 )) < 0 ) {
	perror( "socket" );
	exit( 1 );
    }

    while ( fgets( line, sizeof( line ), conf ) != NULL ) {
	if ( *line == '#' ) {
	    continue;
	}
	argc = 0;
	p = line;

	/*
	 * This parser should be made more powerful -- it should
	 * handle various escapes, e.g. \" and \031.
	 */
	while ( *p != '\0' ) {
	    while ( *p != '\0' && *p != '"' && isspace( *p )) {
		p++;
	    }
	    if ( *p == '\0' ) {
		argv[ argc ] = 0;
		break;
	    }
	    if ( *p == '"' ) {
		argv[ argc ] = ++p;
		while ( *p != '\0' && *p != '"' ) {
		    p++;
		}
	    } else {
		argv[ argc ] = p;
		while ( *p != '\0' && !isspace( *p )) {
		    p++;
		}
	    }
	    *p++ = '\0';
	    argc++;
	}
	if ( argv[ 0 ] == '\0' || *argv[ 0 ] == '#' ) {
	    continue;
	}

	/*
	 * Check that av[ 0 ] is a valid interface.
	 */
	strcpy( ifr.ifr_name, argv[ 0 ] );
	if ( ioctl( s, SIOCGIFFLAGS, &ifr ) < 0 ) {
	    perror( argv[ 0 ] );
	    exit( 1 );
	}
	if ( ifr.ifr_flags & ( IFF_LOOPBACK | IFF_POINTOPOINT )) {
	    fprintf( stderr, "%s: can't configure.\n", ifr.ifr_name );
	    exit( 1 );
	}

	if (( niface = (struct interface *)malloc( sizeof( struct interface )))
		== NULL ) {
	    perror( "malloc" );
	    exit( 1 );
	}
	strcpy( niface->i_name, argv[ 0 ] );
	niface->i_flags = 0;
	niface->i_rt = NULL;
	niface->i_time = 0;
	niface->i_addr.sat_family = AF_APPLETALK;
	niface->i_addr.sat_addr.s_net = htons( ATADDR_ANYNET );
	niface->i_addr.sat_addr.s_node = ATADDR_ANYNODE;
	niface->i_caddr.sat_family = AF_APPLETALK;
	niface->i_caddr.sat_addr.s_net = htons( ATADDR_ANYNET );
	niface->i_caddr.sat_addr.s_node = ATADDR_ANYNODE;

	for ( i = 1; i < argc; i += cc ) {
	    if ( argv[ i ][ 0 ] == '-' ) {
		argv[ i ]++;
	    }
	    for ( j = 0; j < sizeof( params ) / sizeof( params[ 0 ] ); j++ ) {
		if ( strcmp( argv[ i ], params[ j ].p_name ) == 0 ) {
		    if ( params[ j ].p_func != NULL ) {
			cc = (*params[ j ].p_func)( niface, &argv[ i + 1 ] );
			break;
		    }
		}
	    }
	    if ( j >= sizeof( params ) / sizeof( params[ 0 ] )) {
		fprintf( stderr, "%s: attribute not found.\n", argv[ i ] );
		exit( 1 );
	    }
	}

	for ( iface = interfaces; iface; iface = iface->i_next ) {
	    if ( strcmp( niface->i_name, iface->i_name ) == 0 &&
		    ((( niface->i_flags & iface->i_flags &
		    ( IFACE_PHASE1|IFACE_PHASE2 )) != 0 ) ||
		    niface->i_flags == 0 || iface->i_flags == 0 )) {
		break;
	    }
	}
	if ( iface ) {	/* Already have this interface and phase */
	    fprintf( stderr, "Specified %s twice.\n", niface->i_name );
	    exit( 1 );
	}

	if ( interfaces == NULL ) {
	    interfaces = niface;
	} else {
	    for ( iface = interfaces; iface->i_next; iface = iface->i_next )
		;
	    iface->i_next = niface;
	}
	niface->i_next = NULL;
    }

    close( s );
    fclose( conf );
    if ( interfaces == NULL ) {
	return( -1 );
    } else {
	return( 0 );
    }
}

/*ARGSUSED*/
seed( iface, av )
    struct interface	*iface;
    char		**av;
{
    iface->i_flags |= IFACE_SEED;
    return( 1 );
}

phase( iface, av )
    struct interface	*iface;
    char		**av;
{
    int			n;
    char		*pnum;

    if (( pnum = av[ 0 ] ) == NULL ) {
	fprintf( stderr, "No phase.\n" );
	exit( 1 );
    }

    switch ( n = atoi( pnum )) {
    case 1 :
	iface->i_flags |= IFACE_PHASE1;
	break;

    case 2 :
	iface->i_flags |= IFACE_PHASE2;
	break;

    default :
	fprintf( stderr, "No phase %d.\n", n );
	exit( 1 );
    }
    return( 2 );
}

net( iface, av )
    struct interface	*iface;
    char		**av;
{
    char		*nrange;
    char		*stop;
    int			net;

    if (( nrange = av[ 0 ] ) == NULL ) {
	fprintf( stderr, "No network.\n" );
	exit( 1 );
    }

    if (( stop = index( nrange, '-' )) != 0 ) {
	stop++;
    }
    net = atoi( nrange );
    if ( net < 0 || net >= 0xffff ) {
	fprintf( stderr, "Bad network: %d\n" );
	exit( 1 );
    }

    if ( iface->i_rt == NULL ) {
	if (( iface->i_rt =
		(struct rtmptab *)malloc( sizeof( struct rtmptab ))) == NULL ) {
	    perror( "malloc" );
	    exit( 1 );
	}
	iface->i_rt->rt_next = iface->i_rt->rt_prev = NULL;
	iface->i_rt->rt_inext = iface->i_rt->rt_iprev = NULL;
	iface->i_rt->rt_hops = 0;
	iface->i_rt->rt_state = 0;
	iface->i_rt->rt_flags = 0;
	iface->i_rt->rt_nzq = 0;
	iface->i_rt->rt_gate = NULL;
	iface->i_rt->rt_zt = NULL;
    }

    if ( iface->i_flags & IFACE_PHASE1 ) {
	if ( stop != 0 ) {
	    fprintf( stderr, "Phase 1 doesn't use an address range.\n" );
	    exit( 1 );
	}
	iface->i_rt->rt_firstnet = iface->i_rt->rt_lastnet = htons( net );
    } else if ( iface->i_flags & IFACE_PHASE2 ) {
	iface->i_rt->rt_firstnet = htons( net );
	if ( stop != 0 ) {
	    net = atoi( stop );
	    if ( net < 0 || net >= 0xffff ) {
		fprintf( stderr, "Bad network: %d\n" );
		exit( 1 );
	    }
	}
	iface->i_rt->rt_lastnet = htons( net );
	if ( iface->i_rt->rt_firstnet != iface->i_rt->rt_lastnet ) {
	    iface->i_rt->rt_flags |= RTMPTAB_EXTENDED;
	}
    } else {
	fprintf( stderr, "Must specify phase before networks.\n" );
	exit( 1 );
    }
    return( 2 );
}

addr( iface, av )
    struct interface	*iface;
    char		**av;
{
    if ( av[ 0 ] == NULL ) {
	fprintf( stderr, "No address.\n" );
	exit( 1 );
    }
    if ( atalk_aton( av[ 0 ], &iface->i_caddr.sat_addr ) == 0 ) {
	fprintf( stderr, "Bad address, %s\n", av[ 0 ] );
	exit( 1 );
    }

    /*
     * Set network range to be the same as the address.
     */
    if ( iface->i_rt == NULL ) {
	if (( iface->i_rt =
		(struct rtmptab *)malloc( sizeof( struct rtmptab ))) == NULL ) {
	    perror( "malloc" );
	    exit( 1 );
	}
	iface->i_rt->rt_next = iface->i_rt->rt_prev = NULL;
	iface->i_rt->rt_inext = iface->i_rt->rt_iprev = NULL;
	iface->i_rt->rt_hops = 0;
	iface->i_rt->rt_state = 0;
	iface->i_rt->rt_flags = 0;
	iface->i_rt->rt_nzq = 0;
	iface->i_rt->rt_gate = NULL;
	iface->i_rt->rt_zt = NULL;
    }
    iface->i_rt->rt_firstnet = iface->i_rt->rt_lastnet =
	    iface->i_caddr.sat_addr.s_net;
    return( 2 );
}

zone( iface, av )
    struct interface	*iface;
    char		**av;
{
    char		*zname;

    if (( zname = av[ 0 ] ) == NULL ) {
	fprintf( stderr, "No zone.\n" );
	exit( 1 );
    }

    if ( iface->i_rt == NULL ) {
	fprintf( stderr, "Must specify networks before zones.\n" );
	exit( 1 );
    }
    addzone( iface->i_rt, strlen( zname ), zname );
    return( 2 );
}

/*
 * Get the configuration from the kernel. Only called if there's no
 * configuration.
 */
getifconf()
{
    struct ifconf	ifc;
    struct ifreq	ifrs[ 10 ], *ifr;
    struct interface	*iface, *niface;
    int			s;

    if (( s = socket( AF_APPLETALK, SOCK_DGRAM, 0 )) < 0 ) {
	perror( "socket" );
	exit( 1 );
    }

    ifc.ifc_len = sizeof( ifrs );
    ifc.ifc_buf = (caddr_t)ifrs;
    if ( ioctl( s, SIOCGIFCONF, &ifc ) < 0 ) {
	perror( "gifconf" );
	exit( 1 );
    }

    for ( ifr = ifc.ifc_req; ifc.ifc_len >= sizeof( struct ifreq );
	    ifc.ifc_len -= sizeof( struct ifreq ), ifr++ ) {
	if ( ioctl( s, SIOCGIFFLAGS, ifr ) < 0 ) {
	    perror( ifr->ifr_name );
	    exit( 1 );
	}
	if ( ifr->ifr_flags & ( IFF_LOOPBACK | IFF_POINTOPOINT )) {
	    continue;
	}

	for ( iface = interfaces; iface; iface = iface->i_next ) {
	    if ( strcmp( iface->i_name, ifr->ifr_name ) == 0 ) {
		break;
	    }
	}
	if ( iface ) {	/* Already have this interface name */
	    continue;
	}

	if (( niface = (struct interface *)malloc( sizeof( struct interface )))
		== NULL ) {
	    perror( "malloc" );
	    exit( 1 );
	}
	strcpy( niface->i_name, ifr->ifr_name );
	niface->i_flags = 0;
	niface->i_rt = NULL;
	niface->i_time = 0;
	/*
	 * Could try to get the address from the kernel...
	 */
	niface->i_addr.sat_family = AF_APPLETALK;
	niface->i_addr.sat_addr.s_net = htons( ATADDR_ANYNET );
	niface->i_addr.sat_addr.s_node = ATADDR_ANYNODE;
	niface->i_caddr.sat_family = AF_APPLETALK;
	niface->i_caddr.sat_addr.s_net = htons( ATADDR_ANYNET );
	niface->i_caddr.sat_addr.s_node = ATADDR_ANYNODE;

	if ( interfaces == NULL ) {
	    interfaces = niface;
	} else {
	    for ( iface = interfaces; iface->i_next; iface = iface->i_next )
		;
	    iface->i_next = niface;
	}
	niface->i_next = NULL;
    }
    if ( ifc.ifc_len != 0 ) {
	fprintf( stderr, "Funky gifconf return.\n" );
	exit( 1 );
    }

    (void)close( s );
    return( 0 );
}
