/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#include <sys/time.h>
#include <sys/syslog.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <netatalk/endian.h>
#include <atalk/adouble.h>
#include <atalk/afp.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include <pwd.h>

#include "directory.h"
#include "file.h"
#include "volume.h"
#include "globals.h"
#include "conf.h"

struct vol	*volumes = NULL;
int		lastvid = 0;

afp_getsrvrparms( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct timeval	tv;
    struct vol		*volume;
    struct passwd	*pwent;
    char		*data;
    int			vcnt, len;

    if ( username == NULL ) {
	readvolfile( defaultvol, NULL );
    } else if (( pwent = getpwnam( username )) != NULL ) {
	/*
	 * Read user's AppleVolumes or .AppleVolumes file
	 * If neither are readable, read the default volumes file
	 */
#ifdef DOWNCASE
	if ( readvolfile( pwent->pw_dir, "applevolumes" ) < 0 &&
	    readvolfile( pwent->pw_dir, ".applevolumes" ) < 0 &&
	    defaultvol != NULL ) {
		readvolfile( defaultvol, NULL );
	}
#else !DOWNCASE
	if ( readvolfile( pwent->pw_dir, "AppleVolumes" ) < 0 &&
	    readvolfile( pwent->pw_dir, ".AppleVolumes" ) < 0 &&
	    defaultvol != NULL ) {
		readvolfile( defaultvol, NULL );
	}
#endif DOWNCASE
    }

    data = rbuf + 5;
    for ( vcnt = 0, volume = volumes;
	    volume;
	    volume = volume->v_next, vcnt++ ) {
	*data++ = 0;
	len = strlen( volume->v_name );
	*data++ = len;
	bcopy( volume->v_name, data, len );
	data += len;
    }

    *rbuflen = data - rbuf;
    data = rbuf;
    if ( gettimeofday( &tv, 0 ) < 0 ) {
	syslog( LOG_ERR, "afp_getsrvrparms: gettimeofday: %m" );
	exit( 1 );
    }
    tv.tv_sec = htonl( tv.tv_sec );
    bcopy( &tv.tv_sec, data, sizeof( tv.tv_sec ));
    data += sizeof( tv.tv_sec );
    *data = vcnt;
    return( AFP_OK );

}

creatvol( path, name )
    char	*path, *name;
{
    struct vol	*volume;
    struct stat	st;
    int		vlen;

    if ( *name == '\0' ) {
	if (( name = rindex( path, '/' )) == NULL ) {
	    return;	/* Obviously not a fully qualified path */
	}
	name++;
    }

    if ( stat( path, &st ) < 0 ) {
	return;		/* Dir doesn't exist */
    }
    if (( st.st_mode & S_IFDIR ) == 0 ) {
	return;		/* Not a dir */
    }
    for ( volume = volumes; volume; volume = volume->v_next ) {
	if ( strcasecmp( volume->v_name, name ) == 0 ) {
	    return;	/* Won't be able to access it, anyway... */
	}
    }

    vlen = strlen( name );
    if ( vlen > 27 ) {
	return;
    }

    if (( volume =
	    (struct vol *)malloc( sizeof( struct vol ))) == NULL ) {
	syslog( LOG_ERR, "creatvol: malloc: %m" );
	exit( 1 );
    }
    if (( volume->v_name =
	    (char *)malloc( vlen + 1 )) == NULL ) {
	syslog( LOG_ERR, "creatvol: malloc: %m" );
	exit( 1 );
    }
    if (( volume->v_path =
	    (char *)malloc( strlen( path ) + 1 )) == NULL ) {
	syslog( LOG_ERR, "creatvol: malloc: %m" );
	exit( 1 );
    }

    strcpy( volume->v_name, name );
    strcpy( volume->v_path, path );

    volume->v_dir = NULL;
    volume->v_did = NULL;
    volume->v_flags = 0;
    volume->v_vid = lastvid++;
    volume->v_lastdid = 3;

    volume->v_next = volumes;
    volumes = volume;
}

char *myfgets( buf, size, fp )
    char	*buf;
    int		size;
    FILE	*fp;
{
    char	*p;
    int		c;

    p = buf;
    while ((( c = getc( fp )) != EOF ) && ( size > 0 )) {
	if ( c == '\n' || c == '\r' ) {
	    *p++ = '\n';
	    break;
	} else {
	    *p++ = c;
	}
	size--;
    }

    if ( p == buf ) {
	return( NULL );
    } else {
	*p = '\0';
	return( buf );
    }
}

/*
 * Read a volume configuration file and add the volumes contained within to
 * the global volume list.  If p2 is non-NULL, the file that is opened is
 * p1/p2
 *
 * Lines that begin with a # are ignored.
 * Volume lines are of the form:  <unix path> <volume name> [<flags>,...]
 */
readvolfile( p1, p2 )
    char	*p1, *p2;
{
    FILE		*fp;
    char		path[ MAXPATHLEN ], volname[ 28 ], buf[ BUFSIZ ];
    int			quoted = 0;

    strcpy( path, p1 );
    if ( p2 != NULL ) {
	strcat( path, "/" );
	strcat( path, p2 );
    }

    if (( fp = fopen( path, "r" )) == NULL ) {
	return( -1 );
    }

    while ( myfgets( buf, sizeof( buf ), fp ) != NULL ) {
	initline( buf );
	parseline( path, username );
	if ( *path == '\0' ) {
	    continue;
	}
	parseline( volname, NULL );

	creatvol( path, volname );
    }
    return( 0 );
}

afp_openvol( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct stat	st;
    char	volname[ MAXNAMLEN ];
    struct vol	*volume;
    struct dir	*dir;
    int		len, ret, buflen;
    u_short	bitmap;

    ibuf += 2;
    bcopy( ibuf, &bitmap, sizeof( u_short ));
    bitmap = ntohs( bitmap );
    ibuf += sizeof( u_short );
    if (( bitmap & (1<<VOLPBIT_VID)) == 0 ) {
	*rbuflen = 0;
	return( AFPERR_BITMAP );
    }

    len = *ibuf++;
    bcopy( ibuf, volname, len );
    volname[ len ] = '\0';

    for ( volume = volumes; volume; volume = volume->v_next ) {
	if ( strcasecmp( volname, volume->v_name ) == 0 ) {
	    break;
	}
    }
    if ( volume == NULL ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    if (( volume->v_flags & AFPVOL_OPEN  ) == 0 ) {
	if (( dir = (struct dir *)malloc( sizeof( struct dir ))) == NULL ) {
	    syslog( LOG_ERR, "afp_openvol: malloc: %m" );
	    exit( 1 );
	}
	dir->d_left = NULL;
	dir->d_right = NULL;
	dir->d_parent = NULL;
	dir->d_child = NULL;
	dir->d_next = NULL;
	dir->d_balance = 0;
	dir->d_did = htonl( 2 );
	dir->d_flags = 0;
	if (( dir->d_name = (char *)malloc( strlen( volume->v_name ) + 1 ))
		== NULL ) {
	    syslog( LOG_ERR, "afp_openvol: malloc: %m" );
	    exit( 1 );
	}
	strcpy( dir->d_name, volume->v_name );
	volume->v_dir = dir;
	volume->v_did = dir;
	volume->v_flags |= AFPVOL_OPEN;
    }

    curdir = volume->v_dir;
    if ( chdir( volume->v_path ) < 0 ) {
	syslog( LOG_DEBUG, "afp_openvol: chdir %s: %m", volume->v_path );
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    if ( stat( ".", &st ) < 0 ) {
	syslog( LOG_DEBUG, "afp_openvol: stat %s: %m", volume->v_path );
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    buflen = *rbuflen - sizeof( u_short );
    if (( ret = getvolparams( bitmap, volume, &st,
	    rbuf + sizeof( u_short ), &buflen )) != AFP_OK ) {
	*rbuflen = 0;
	return( ret );
    }
    *rbuflen = buflen + sizeof( u_short );
    bitmap = htons( bitmap );
    bcopy( &bitmap, rbuf, sizeof( u_short ));

    return( AFP_OK );
}

afp_closevol( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct vol	*vol, *ovol;
    u_short	vid;

    *rbuflen = 0;
    ibuf += 2;
    bcopy( ibuf, &vid, sizeof( u_short ));
    if (( vol = getvolbyvid( vid )) == NULL ) {
	return( AFPERR_PARAM );
    }

    vol->v_flags &= ~AFPVOL_OPEN;
    for ( ovol = volumes; ovol; ovol = ovol->v_next ) {
	if ( ovol->v_flags & AFPVOL_OPEN ) {
	    break;
	}
    }
    if ( ovol != NULL ) {
	curdir = ovol->v_dir;
	if ( chdir( ovol->v_path ) < 0 ) {
	    syslog( LOG_DEBUG, "afp_closevol: chdir %s: %m", ovol->v_path );
	    return( AFPERR_PARAM );
	}
    }

    freedir( vol->v_did );
    vol->v_did = NULL;
    vol->v_dir = NULL;
    return( AFP_OK );
}

freedir( dir )
    struct dir	*dir;
{
    if ( dir->d_left != NULL ) {
	freedir( dir->d_left );
    }
    if ( dir->d_right != NULL ) {
	freedir( dir->d_right );
    }
    free( dir->d_name );
    free( dir );
}

afp_getvolparams( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct stat	st;
    struct vol	*vol;
    int		buflen, ret;
    u_short	vid, bitmap;

    ibuf += 2;
    bcopy( ibuf, &vid, sizeof( u_short ));
    ibuf += sizeof( u_short );
    bcopy( ibuf, &bitmap, sizeof( u_short ));
    bitmap = ntohs( bitmap );

    if (( vol = getvolbyvid( vid )) == NULL ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    if ( stat( vol->v_path, &st ) < 0 ) {
	syslog( LOG_DEBUG, "afp_getvolparams: stat %s: %m", vol->v_path );
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    buflen = *rbuflen - sizeof( u_short );
    if (( ret = getvolparams( bitmap, vol, &st,
	    rbuf + sizeof( u_short ), &buflen )) != AFP_OK ) {
	*rbuflen = 0;
	return( ret );
    }
    *rbuflen = buflen + sizeof( u_short );
    bitmap = htons( bitmap );
    bcopy( &bitmap, rbuf, sizeof( u_short ));
    return( AFP_OK );
}

getvolparams( bitmap, vol, st, buf, buflen )
    u_short	bitmap;
    struct vol	*vol;
    struct stat	*st;
    char	*buf;
    int		*buflen;
{
    struct adouble	ad;
    int			bit = 0, aint, isad = 1, bfree, btotal;
    u_short		ashort;
    char		*data, *nameoff = 0;

    if ( ad_open( vol->v_path, ADFLAGS_HF|ADFLAGS_DIR,
	    O_RDONLY, 0, &ad ) < 0 ) {
	isad = 0;
    }

    if (( bitmap & ( (1<<VOLPBIT_BFREE)|(1<<VOLPBIT_BTOTAL) )) != 0 ) {
	if ( getvolspace( vol->v_path, &bfree, &btotal ) < 0 ) {
	    syslog( LOG_DEBUG, "getvolparams: getvolspace %s: %m",
		    vol->v_name );
	    if ( isad ) {
		ad_close( &ad, ADFLAGS_HF );
	    }
	    return( AFPERR_PARAM );
	}
    }

    data = buf;
    while ( bitmap != 0 ) {
	while (( bitmap & 1 ) == 0 ) {
	    bitmap = bitmap>>1;
	    bit++;
	}

	switch ( bit ) {
	case VOLPBIT_ATTR :
	    ashort = 0;
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case VOLPBIT_SIG :
	    ashort = htons( AFPVOLSIG_FIX );
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case VOLPBIT_CDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_CREATE, &aint,
			sizeof( int ));
		if ( aint == 0 ) {
		    aint = htonl( st->st_ctime );
		}
	    } else {
		aint = htonl( st->st_ctime );
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case VOLPBIT_MDATE :
	    aint = htonl( st->st_ctime );
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case VOLPBIT_BDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ad, ADEID_FILEI ) + FILEIOFF_BACKUP, &aint,
			sizeof( int ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case VOLPBIT_VID :
	    bcopy( &vol->v_vid, data, sizeof( vol->v_vid ));
	    data += sizeof( vol->v_vid );
	    break;

	case VOLPBIT_BFREE :
	    bcopy( &bfree, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case VOLPBIT_BTOTAL :
	    bcopy( &btotal, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case VOLPBIT_NAME :
	    nameoff = data;
	    data += sizeof( u_short );
	    break;

	default :
	    syslog( LOG_DEBUG, "getvolparams: bad bit = %d", bit );
	    if ( isad ) {
		ad_close( &ad, ADFLAGS_HF );
	    }
	    return( AFPERR_BITMAP );
	}
	bitmap = bitmap>>1;
	bit++;
    }
    if ( nameoff != 0 ) {
	ashort = htons( data - buf );
	bcopy( &ashort, nameoff, sizeof( u_short ));
	aint = strlen( vol->v_name );
	*data++ = aint;
	bcopy( vol->v_name, data, aint );
	data += aint;
    }
    if ( isad ) {
	ad_close( &ad, ADFLAGS_HF );
    }
    *buflen = data - buf;
    return( AFP_OK );
}

struct vol *
getvolbyvid( vid )
    short	vid;
{
    struct vol	*vol;

    for ( vol = volumes; vol; vol = vol->v_next ) {
	if ( vid == vol->v_vid ) {
	    break;
	}
    }
    if ( vol == NULL || ( vol->v_flags & AFPVOL_OPEN ) == 0 ) {
	return( NULL );
    }

    return( vol );
}
