/*
 * Copyright (c) 1990,1991 Regents of The University of Michigan.
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appears in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of The University
 * of Michigan not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. This software is supplied as is without expressed or
 * implied warranties of any kind.
 *
 *	Research Systems Unix Group
 *	The University of Michigan
 *	c/o Mike Clark
 *	535 W. William Street
 *	Ann Arbor, Michigan
 *	+1-313-763-0525
 *	netatalk@itd.umich.edu
 */

#include <sys/syslog.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <netatalk/endian.h>
#include <netatalk/at.h>
#include <atalk/afp.h>
#include <atalk/atp.h>
#include <atalk/asp.h>
#include <atalk/adouble.h>
#include <stdio.h>

#include "fork.h"
#include "file.h"
#include "globals.h"
#include "directory.h"
#include "volume.h"
#include "conf.h"

extern int		errno;

static struct ofork	*oforks[ ( NOFILE - 10 ) / 2 ];

pforkdesc( f )
    FILE	*f;
{
    u_short	ofrefnum;

    for ( ofrefnum = 0; ofrefnum < sizeof( oforks ) / sizeof( oforks[ 0 ] );
	    ofrefnum++ ) {
	if ( oforks[ ofrefnum ] != NULL ) {
	    fprintf( f, "%hu <%s>\n", ofrefnum, oforks[ ofrefnum ]->of_name );
	}
    }
}

afp_openfork( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct vol		*vol;
    struct dir		*dir;
    int			did, pathtype, buflen, oflags, ret, adflags, plen;
    u_short		vid, bitmap, access, ofrefnum;
    char		fork, *path;

    ibuf++;
    fork = *ibuf++;
    bcopy( ibuf, &vid, sizeof( u_short ));
    ibuf += sizeof( u_short );

    if (( vol = getvolbyvid( vid )) == NULL ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &did, sizeof( int ));
    ibuf += sizeof( int );

    if (( dir = dirsearch( vol, did )) == NULL ) {
	*rbuflen = 0;
	return( AFPERR_NOOBJ );
    }

    bcopy( ibuf, &bitmap, sizeof( u_short ));
    bitmap = ntohs( bitmap );
    ibuf += sizeof( u_short );
    bcopy( ibuf, &access, sizeof( u_short ));
    access = ntohs( access );
    ibuf += sizeof( u_short );

    if (( pathtype = *ibuf++ ) != 2 ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }
    plen = *ibuf++;
    if (( path = cname( vol, dir, &ibuf, plen )) == NULL ) {
	*rbuflen = 0;
	return( AFPERR_NOOBJ );
    }

    for ( ofrefnum = 0; ofrefnum < sizeof( oforks ) / sizeof( oforks[ 0 ] );
	    ofrefnum++ ) {
	if ( oforks[ ofrefnum ] == NULL ) {
	    break;
	}
    }
    if ( ofrefnum == sizeof( oforks ) / sizeof( oforks[ 0 ] )) {
	*rbuflen = 0;
	return( AFPERR_NFILE );
    }
    if (( oforks[ ofrefnum ] = (struct ofork *)malloc( sizeof( struct ofork )))
	    == NULL ) {
	syslog( LOG_ERR, "afp_openfork: malloc: %m" );
	exit( 1 );
    }
    oforks[ ofrefnum ]->of_dir = curdir;
    if (( oforks[ ofrefnum ]->of_name = (char *)malloc( strlen( path ) + 1 ))
	    == NULL ) {
	syslog( LOG_ERR, "afp_openfork: malloc: %m" );
	exit( 1 );
    }
    strcpy( oforks[ ofrefnum ]->of_name, path );

    if ( access & OPENACC_WR ) {
	oflags = O_RDWR;
    } else {
	oflags = O_RDONLY;
    }
    if ( fork == OPENFORK_DATA ) {
	adflags = ADFLAGS_DF|ADFLAGS_HF;
    } else {
	adflags = ADFLAGS_HF;
    }

    if ( ad_open( mtoupath( path ), adflags, oflags, 0,
		&oforks[ ofrefnum ]->of_ad ) < 0 ) {
	if ( errno == ENOENT && adflags != ADFLAGS_HF ) {
	    ad_close( &oforks[ ofrefnum ]->of_ad, adflags );
	    if ( ad_open( mtoupath( path ), ADFLAGS_DF, oflags, 0,
		    &oforks[ ofrefnum ]->of_ad ) < 0 ) {
		ad_close( &oforks[ ofrefnum ]->of_ad, ADFLAGS_DF );
		free( oforks[ ofrefnum ]->of_name );
		free( oforks[ ofrefnum ] );
		oforks[ ofrefnum ] = NULL;
		*rbuflen = 0;
		return( AFPERR_NOOBJ );
	    }
	} else {
	    free( oforks[ ofrefnum ]->of_name );
	    free( oforks[ ofrefnum ] );
	    oforks[ ofrefnum ] = NULL;
	    *rbuflen = 0;
	    switch ( errno ) {
	    case EMFILE :
	    case ENFILE :
		return( AFPERR_NFILE );
	    case EISDIR :
		return( AFPERR_BADTYPE );
	    case ENOENT :
		return( AFPERR_NOOBJ );
	    case EACCES :
		return( AFPERR_ACCESS );
	    default :
		return( AFPERR_PARAM );
	    }
	}
    }

    if ( ad_getoflags( &oforks[ ofrefnum ]->of_ad, ADFLAGS_HF ) & O_CREAT ) {
	ad_flush( &oforks[ ofrefnum ]->of_ad, adflags );
    }

    buflen = *rbuflen - 2 * sizeof( u_short );
    if (( ret = getforkparams( oforks[ ofrefnum ], bitmap,
	    rbuf + 2 * sizeof( u_short ), &buflen )) != AFP_OK ) {
	*rbuflen = 0;
	ad_close( &oforks[ ofrefnum ]->of_ad, adflags );
	free( oforks[ ofrefnum ]->of_name );
	free( oforks[ ofrefnum ] );
	oforks[ ofrefnum ] = NULL;
	return( ret );
    }

    *rbuflen = buflen + 2 * sizeof( u_short );
    bitmap = htons( bitmap );
    bcopy( &bitmap, rbuf, sizeof( u_short ));
    rbuf += sizeof( u_short );
    bcopy( &ofrefnum, rbuf, sizeof( u_short ));
    return( AFP_OK );
}

afp_setforkparams( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    int		size;
    u_short	ofrefnum;

    ibuf += 2;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));
    ibuf += sizeof( u_short );
    ibuf += sizeof( u_short );
    bcopy( ibuf, &size, sizeof( int ));
    size = ntohl( size );

    *rbuflen = 0;
    if ( oforks[ ofrefnum ] == NULL ) {
	return( AFPERR_PARAM );
    }

    if ( ad_dfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	if ( ad_dtruncate( &oforks[ ofrefnum ]->of_ad, size ) < 0 ) {
	    syslog( LOG_ERR, "afp_setforkparams: ad_dtruncate: %m" );
	    return( AFPERR_PARAM );
	}
    } else {
	ad_refresh( &oforks[ ofrefnum ]->of_ad, ADFLAGS_HF );
	if ( ad_rtruncate( &oforks[ ofrefnum ]->of_ad, size ) < 0 ) {
	    syslog( LOG_ERR, "afp_setforkparams: ad_rtruncate: %m" );
	    return( AFPERR_PARAM );
	}
	if ( ad_flush( &oforks[ ofrefnum ]->of_ad, ADFLAGS_HF ) < 0 ) {
	    syslog( LOG_ERR, "afp_setforkparams: ad_flush: %m" );
	    return( AFPERR_PARAM );
	}
    }

    return( AFP_OK );
}

afp_bytelock( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    int		off;
    u_short	ofrefnum;
    
    ibuf += 2;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));
    ibuf += sizeof( u_short );
    bcopy( ibuf, &off, sizeof( int ));

    bcopy( &off, rbuf, sizeof( int ));
    *rbuflen = sizeof( int );
    return( AFP_OK );
}

afp_read( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    long		offset;
    int			reqcount, cc, rc, eid, eof = 0;
    u_short		ofrefnum;
    u_char		nlmask, nlchar;
    register char	*p, *q;

    ibuf += 2;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));
    ibuf += sizeof( u_short );

    if ( oforks[ ofrefnum ] == NULL ) {
	*rbuflen = 0;
	syslog( LOG_DEBUG, "afp_read: %d is not an open refnum", ofrefnum );
	return( AFPERR_PARAM );
    }

    bcopy( ibuf, &offset, sizeof( long ));
    offset = ntohl( offset );
    ibuf += sizeof( long );
    bcopy( ibuf, &reqcount, sizeof( long ));
    reqcount = ntohl( reqcount );
    ibuf += sizeof( long );

    nlmask = *ibuf++;
    nlchar = *ibuf++;

    if ( ad_dfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	eid = ADEID_DFORK;
    } else {
	eid = ADEID_RFORK;
    }

    if ( reqcount < 0 ) {
	*rbuflen = 0;
	return( AFPERR_EOF );
    }
    rc = MIN( reqcount, *rbuflen );
    cc = ad_read( &oforks[ ofrefnum ]->of_ad, eid, offset, rbuf, rc );
    if ( cc < 0 ) {
	*rbuflen = 0;
	syslog( LOG_DEBUG, "afp_read: read: %m" );
	return( AFPERR_PARAM );
    }
    if ( cc < rc ) {
	eof = 1;
    }

    /*
     * Do Newline check.
     */
    if ( nlmask != 0 ) {
	for ( p = rbuf, q = p + cc; p < q; ) {
	    if (( *p++ & nlmask ) == nlchar ) {
		break;
	    }
	}
	if ( p != q ) {
	    cc = p - rbuf;
	    eof = 0;
	}
    }

#ifdef CRLF
    /*
     * If this file is of type TEXT, then swap \012 to \015.
     */
    if ( eid == ADEID_DFORK &&
	    ( ad_hfileno( &oforks[ ofrefnum ]->of_ad ) == -1 ||
		bcmp( "TEXT", ad_entry( &oforks[ ofrefnum ]->of_ad,
		ADEID_FINDERI ), 4 ) == 0 )) {
	for ( p = rbuf, q = p + cc; p < q; p++ ) {
	    if ( *p == '\012' ) {
		*p = '\015';
	    }
	}
    }
#endif CRLF

    *rbuflen = cc;
    if ( eof ) {
	return( AFPERR_EOF );
    }
    return( AFP_OK );
}

afp_flushfork( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    *rbuflen = 0;
    return( AFP_OK );
}

afp_closefork( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    long	intime, filetime;
    int		adflags, aint;
    u_short	ofrefnum;

    *rbuflen = 0;
    ibuf += 2;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));

    if ( oforks[ ofrefnum ] == NULL ) {
	return( AFPERR_PARAM );
    }

    adflags = 0;
    if ( ad_dfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	adflags |= ADFLAGS_DF;
    }
    if ( ad_hfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	adflags |= ADFLAGS_HF;

	aint = ad_getentrylen( &oforks[ ofrefnum ]->of_ad, ADEID_RFORK );
	bcopy( ad_entry( &oforks[ ofrefnum ]->of_ad, ADEID_FILEI ) +
		FILEIOFF_MODIFY, &intime, sizeof( intime ));
	ad_refresh( &oforks[ ofrefnum ]->of_ad, adflags );
	bcopy( ad_entry( &oforks[ ofrefnum ]->of_ad, ADEID_FILEI ) +
		FILEIOFF_MODIFY, &filetime, sizeof( filetime ));
	if ( aint != ad_getentrylen( &oforks[ ofrefnum ]->of_ad, ADEID_RFORK )
		|| intime != filetime ) {
	    ad_setentrylen( &oforks[ ofrefnum ]->of_ad, ADEID_RFORK, aint );
	    bcopy( &intime, ad_entry( &oforks[ ofrefnum ]->of_ad, ADEID_FILEI )
		    + FILEIOFF_MODIFY, sizeof( filetime ));
	    ad_flush( &oforks[ ofrefnum ]->of_ad, adflags );
	}
    }

    if ( ad_close( &oforks[ ofrefnum ]->of_ad, adflags ) < 0 ) {
	syslog( LOG_DEBUG, "afp_closefork: ad_close: %m" );
	return( AFPERR_PARAM );
    }

    free( oforks[ ofrefnum ]->of_name );
    free( oforks[ ofrefnum ] );
    oforks[ ofrefnum ] = NULL;

    return( AFP_OK );
}

afp_write( asp, ibuf, ibuflen, rbuf, rbuflen )
    ASP		asp;
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    struct timeval	tv;
    register char	*p, *q;
    int			endflag, offset, reqcount, cc, eid;
    u_short		ofrefnum;

    ibuf += 1;
    endflag = *ibuf++;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));
    ibuf += sizeof( u_short );
    bcopy( ibuf, &offset, sizeof( int ));
    offset = ntohl( offset );
    ibuf += sizeof( int );
    bcopy( ibuf, &reqcount, sizeof( int ));
    reqcount = ntohl( reqcount );
    ibuf += sizeof( int );

    if ( oforks[ ofrefnum ] == NULL ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    if ( asp_wrtcont( asp, rbuf, rbuflen ) < 0 ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    if ( ad_dfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	eid = ADEID_DFORK;
    } else {
	eid = ADEID_RFORK;
    }

#ifdef CRLF
    /*
     * If this file is of type TEXT, swap \015 to \012.
     */
    if ( eid == ADEID_DFORK &&
	    ( ad_hfileno( &oforks[ ofrefnum ]->of_ad ) == -1 ||
		bcmp( "TEXT", ad_entry( &oforks[ ofrefnum ]->of_ad,
		ADEID_FINDERI ), 4 ) == 0 )) {
	for ( p = rbuf, q = p + *rbuflen; p < q; p++ ) {
	    if ( *p == '\015' ) {
		*p = '\012';
	    }
	}
    }
#endif CRLF

    if (( cc = ad_write( &oforks[ ofrefnum ]->of_ad,
	    eid, offset, rbuf, *rbuflen )) < 0 ) {
	switch ( errno ) {
	case EDQUOT :
	case EFBIG :
	case ENOSPC :
	    *rbuflen = 0;
	    return( AFPERR_DFULL );
	default :
	    syslog( LOG_DEBUG, "afp_write: ad_write: %m" );
	    *rbuflen = 0;
	    return( AFPERR_PARAM );
	}
    }

    if ( ad_hfileno( &oforks[ ofrefnum ]->of_ad ) != -1 ) {
	if ( gettimeofday( &tv, 0 ) < 0 ) {
	    syslog( LOG_ERR, "afp_write: gettimeofday: %m" );
	    exit( 1 );
	}
	tv.tv_sec = htonl( tv.tv_sec );
	bcopy( &tv.tv_sec, ad_entry( &oforks[ ofrefnum ]->of_ad, ADEID_FILEI ) +
		FILEIOFF_MODIFY, sizeof( tv.tv_sec ));
    }

    offset += cc;
    offset = htonl( offset );
    bcopy( &offset, rbuf, sizeof( int ));
    *rbuflen = 4;
    return( AFP_OK );
}

afp_getforkparams( ibuf, ibuflen, rbuf, rbuflen )
    char	*ibuf, *rbuf;
    int		ibuflen, *rbuflen;
{
    int		buflen, ret;
    u_short	ofrefnum, bitmap;

    ibuf += 2;
    bcopy( ibuf, &ofrefnum, sizeof( u_short ));
    ibuf += sizeof( u_short );
    bcopy( ibuf, &bitmap, sizeof( u_short ));
    bitmap = ntohs( bitmap );
    ibuf += sizeof( u_short );

    if ( oforks[ ofrefnum ] == NULL ) {
	*rbuflen = 0;
	return( AFPERR_PARAM );
    }

    buflen = *rbuflen - sizeof( u_short );
    if (( ret = getforkparams( oforks[ ofrefnum ], bitmap,
	    rbuf + sizeof( u_short ), &buflen )) != AFP_OK ) {
	*rbuflen = 0;
	return( ret );
    }

    *rbuflen = buflen + sizeof( u_short );
    bitmap = htons( bitmap );
    bcopy( &bitmap, rbuf, sizeof( u_short ));
    return( AFP_OK );
}

getforkparams( ofork, bitmap, buf, buflen )
    struct ofork	*ofork;
    u_short		bitmap;
    char		*buf;
    int			*buflen;
{
    struct stat		st;
    char		*data, *nameoff = 0;
    int			bit = 0, isad = 1, aint;
    u_short		ashort;

    if ( ad_hfileno( &ofork->of_ad ) == -1 ) {
	isad = 0;
    }

    if ( isad ) {
	aint = ad_getentrylen( &ofork->of_ad, ADEID_RFORK );
	if ( ad_refresh( &ofork->of_ad ) < 0 ) {
	    return( AFPERR_PARAM );
	}
	ad_setentrylen( &ofork->of_ad, ADEID_RFORK, aint );
    }

    if (( bitmap & ( 1<<FILPBIT_DFLEN )) &&
	    ad_dfileno( &ofork->of_ad ) == -1 ) {
	return( AFPERR_BITMAP );
    }
    if ( bitmap & ( 1<<FILPBIT_DFLEN | 1<<FILPBIT_FNUM )) {
	if ( ad_dfileno( &ofork->of_ad ) == -1 ) {
	    if ( fstat( ad_hfileno( &ofork->of_ad ), &st ) < 0 ) {
		return( AFPERR_BITMAP );
	    }
	} else {
	    if ( fstat( ad_dfileno( &ofork->of_ad ), &st ) < 0 ) {
		return( AFPERR_BITMAP );
	    }
	}
    }

    data = buf;
    while ( bitmap != 0 ) {
	while (( bitmap & 1 ) == 0 ) {
	    bitmap = bitmap>>1;
	    bit++;
	}

	switch ( bit ) {
	case FILPBIT_ATTR :
	    if ( isad ) {
		bcopy( ad_entry( &ofork->of_ad, ADEID_FILEI ) + FILEIOFF_ATTR,
			&ashort, sizeof( u_short ));
	    } else {
		ashort = 0;
	    }
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case FILPBIT_PDID :
	    bcopy( &ofork->of_dir->d_did, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_CDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ofork->of_ad, ADEID_FILEI )
			+ FILEIOFF_CREATE, &aint, sizeof( int ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_MDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ofork->of_ad, ADEID_FILEI )
			+ FILEIOFF_MODIFY, &aint, sizeof( int ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_BDATE :
	    if ( isad ) {
		bcopy( ad_entry( &ofork->of_ad, ADEID_FILEI )
			+ FILEIOFF_BACKUP, &aint, sizeof( int ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_FINFO :
	    if ( isad ) {
		bcopy( ad_entry( &ofork->of_ad, ADEID_FINDERI ), data, 32 );
	    } else {
		bcopy( ufinderi, data, 32 );
	    }
	    data += 32;
	    break;

	case FILPBIT_LNAME :
	    nameoff = data;
	    data += sizeof( u_short );
	    break;

	case FILPBIT_SNAME :
	    ashort = 0;
	    bcopy( &ashort, data, sizeof( u_short ));
	    data += sizeof( u_short );
	    break;

	case FILPBIT_FNUM :
	    bcopy( &st.st_ino, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_DFLEN :
	    aint = htonl( st.st_size );
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	case FILPBIT_RFLEN :
	    if ( isad ) {
		aint = htonl( ad_getentrylen( &ofork->of_ad, ADEID_RFORK ));
	    } else {
		aint = 0;
	    }
	    bcopy( &aint, data, sizeof( int ));
	    data += sizeof( int );
	    break;

	default :
	    return( AFPERR_BITMAP );
	}
	bitmap = bitmap>>1;
	bit++;
    }

    if ( nameoff != 0 ) {
	ashort = htons( data - buf );
	bcopy( &ashort, nameoff, sizeof( u_short ));
	aint = strlen( ofork->of_name );
	aint = ( aint > 31 ) ? 31 : aint;
	*data++ = aint;
	bcopy( ofork->of_name, data, aint );
	data += aint;
    }

    *buflen = data - buf;
    return( AFP_OK );
}
