#include <sys/types.h>
#include <sys/uio.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/param.h>
#if BSD >= 199006
# include <machine/endian.h>
#else
# include <netinet/in.h>
#endif
#include <strings.h>
#include <syslog.h>
#include <ctype.h>
#include <stdio.h>
#include <atalk/adouble.h>
#include "megatron.h"

#define	DEBUG		0

/*	String used to indicate standard input instead of a disk
	file.  Should be a string not normally used for a file
 */
#ifndef	STDIN
#	define	STDIN	"-"
#endif

/*	Yes and no
 */
#define NOWAY		0
#define SURETHANG	1

/*	AppleSingle Entry IDs.
 */
#define ENTRYID_DATA		1
#define ENTRYID_RESOURCE	2
#define ENTRYID_NAME		3
#define ENTRYID_COMMENT		4
#define ENTRYID_ICN		5
#define ENTRYID_CICN		6
#define ENTRYID_FILEINFO	7
#define ENTRYID_DATES		8
#define ENTRYID_FINDERINFO	9
#define ENTRYID_MACFINFO	10
#define ENTRYID_PRODOSFINFO	11
#define ENTRYID_MSDOSFINFO	12
#define ENTRYID_AFPSHORTNAME	13
#define ENTRYID_AFPFINFO	14
#define ENTRYID_AFPDIRID	15
#define ENTRYID_MAX		16

#define ENTRY_INFO_SIZE	12

#define HEADERSIZ	26

/*	This structure holds an entry description, consisting of three
	four byte entities.  The first is the Entry ID, the second is
	the File Offset and the third is the Length.
 */

struct entry_info {
    long		offset;
    long		length;
};

/*	Both input and output routines use this struct and the
	following globals; therefore this module can only be used
	for one of the two functions at a time.
 */
struct single_file_data {
    int			filed;
    char		path[ MAXPATHLEN ];
    struct entry_info	entry[ ENTRYID_MAX ];
} 		single;

extern char	*forkname[];
u_char		header_buf[ HEADERSIZ ];

/* 
 * single_open must be called first.  pass it a filename that is supposed
 * to contain a AppleSingle file.  an single struct will be allocated and
 * somewhat initialized; single_filed is set.
 */

single_open( singlefile, flags, fh )
    char		*singlefile;
    int			flags;
    struct FHeader	*fh;
{
    int			maxlen;
    int			rc;

#if DEBUG
    fprintf( stderr, "entering single_open\n" );
#endif

    if ( flags == O_RDONLY ) {
	if ( strcmp( singlefile, STDIN ) == 0 ) {
	    single.filed = fileno( stdin );
	} else if (( single.filed = open( singlefile, flags )) < 0 ) {
	    perror( singlefile );
	    return( -1 );
	}
#if DEBUG
	fprintf( stderr, "opened %s for read\n", singlefile );
#endif
	if ((( rc = single_header_test()) > 0 ) && 
		( single_header_read( fh, rc ) == 0 )) {
	    return( 0 );
	}
	single_close( TRASH );
	fprintf( stderr, "%s\n", singlefile );
	return( -1 );
    }
}

/* 
 * single_close must be called before a second file can be opened using
 * single_open.  Upon successful completion, a value of 0 is returned.  
 * Otherwise, a value of -1 is returned.
 */

single_close( keepflag )
    int			keepflag;
{
#if DEBUG
    fprintf( stderr, "entering single_close\n" );
#endif
    if ( keepflag == KEEP ) {
	return( close( single.filed ));
    } else if ( keepflag == TRASH ) {
	if (( strcmp( single.path, STDIN ) != 0 ) && 
		( unlink( single.path ) < 0 )) {
	    perror ( single.path );
	}
	return( 0 );
    } else return( -1 );
}

/* 
 * single_header_read is called by single_open, and before any information
 * can read from the fh substruct.  it must be called before any of the
 * bytes of the other two forks can be read, as well.
 */

single_header_read( fh, version )
    struct FHeader	*fh;
    int			version;
{
    u_char		entry_buf[ 16 ];
    u_long		entry_id;
    u_long		time_seconds;
    u_short		mask = 0xfcee;
    u_short		num_entries;
    int			n;
    int			readlen;
    int			date_entry;
    off_t		pos;
    time_t		unix_time;

/*
 * Go through and initialize the array of entry_info structs.  Read in the
 * number of entries, and then read in the info for each entry and save it
 * in the array.
 */

    for ( n = 0 ; n < ENTRYID_MAX; n++ ) {
	single.entry[ n ].offset = 0;
	single.entry[ n ].length = 0;
    }
    bcopy( (char *)&header_buf[ 24 ], (char *)&num_entries,
	    sizeof( num_entries ));
    num_entries = ntohs( num_entries );
    for ( ; num_entries > 0 ; num_entries-- ) {
	if ( read( single.filed, (char *)entry_buf, ENTRY_INFO_SIZE )
		!= ENTRY_INFO_SIZE ) {
	    perror( "Premature end of file :" );
	    return( -1 );
	}
	bcopy( (char *)entry_buf, (char *)&entry_id, sizeof( entry_id ));
	entry_id = ntohl( entry_id );
	bcopy( (char *)&entry_buf[ 4 ],
		(char *)&single.entry[ entry_id ].offset,
		sizeof( single.entry[ entry_id ].offset ));
	single.entry[ entry_id ].offset =
		ntohl( single.entry[ entry_id ].offset );
	bcopy( (char *)&entry_buf[ 8 ],
		(char *)&single.entry[ entry_id ].length,
		sizeof( single.entry[ entry_id ].length ));
	single.entry[ entry_id ].length =
		ntohl( single.entry[ entry_id ].length );
    }

/*
 * Now that the entries have been identified, check to make sure
 * it is a Macintosh file if dealing with version two format file.
 */

    if ( version == 1 ) {
	date_entry = ENTRYID_FILEINFO;
    } else if ( version == 2 ) {
	if ( single.entry[ ENTRYID_MACFINFO ].offset == 0 ) {
	    fprintf( stderr, "%s is not a Macintosh AppleSingle.\n",
		    single.path );
	    return( -1 );
	}
	date_entry = ENTRYID_DATES;
    }

/*
 * Go through and copy all the information you can get from 
 * the informational entries into the fh struct.  The ENTRYID_DATA
 * must be the last one done, because it leaves the file pointer in
 * the right place for the first read of the data fork.
 */

    if ( single.entry[ ENTRYID_NAME ].offset == 0 ) {
	fprintf( stderr, "%s has no name for the mac file.\n", single.path );
	return( -1 );
    } else {
	pos = lseek( single.filed, single.entry[ ENTRYID_NAME ].offset,
		L_SET );
	readlen = single.entry[ ENTRYID_NAME ].length > NAMESIZ ? NAMESIZ
		: single.entry[ ENTRYID_NAME ].length;
	if ( read( single.filed, (char *)fh->name, readlen ) != readlen ) {
	    perror( "Premature end of file :" );
	    return( -1 );
	}
    }
    if ( single.entry[ ENTRYID_FINDERINFO ].offset < 16 ) {
	fprintf( stderr, "%s has bogus FinderInfo.\n", single.path );
	return( -1 );
    } else {
	pos = lseek( single.filed,
		single.entry[ ENTRYID_FINDERINFO ].offset, L_SET );
	if ( read( single.filed, (char *)entry_buf, sizeof( entry_buf )) !=
		sizeof( entry_buf )) {
	    perror( "Premature end of file :" );
	    return( -1 );
	}
	bcopy( (char *)&entry_buf[ FINDERIOFF_TYPE ],
		(char *)&fh->finder_info.fdType,
		sizeof( fh->finder_info.fdType ));
	bcopy( (char *)&entry_buf[ FINDERIOFF_CREATOR ],
		(char *)&fh->finder_info.fdCreator,
		sizeof( fh->finder_info.fdCreator ));
	bcopy( (char *)&entry_buf[ FINDERIOFF_FLAGS ],
		(char *)&fh->finder_info.fdFlags,
		sizeof( fh->finder_info.fdFlags ));
	fh->finder_info.fdFlags = fh->finder_info.fdFlags & mask;
	bcopy( (char *)&entry_buf[ FINDERIOFF_LOC ],
		(char *)&fh->finder_info.fdLocation,
		sizeof( fh->finder_info.fdLocation ));
	bcopy( (char *)&entry_buf[ FINDERIOFF_FLDR ],
		(char *)&fh->finder_info.fdFldr,
		sizeof( fh->finder_info.fdFldr ));
    }
    if ( single.entry[ ENTRYID_COMMENT ].offset == 0 ) {
	fh->comment[0] = '\0';
    } else {
	pos = lseek( single.filed, single.entry[ ENTRYID_COMMENT ].offset,
		L_SET );
	readlen = single.entry[ ENTRYID_COMMENT ].length > COMMENTSIZ 
		? COMMENTSIZ : single.entry[ ENTRYID_COMMENT ].length;
	if ( read( single.filed, (char *)fh->comment, readlen ) != readlen ) {
	    perror( "Premature end of file :" );
	    return( -1 );
	}
    }
    if ( single.entry[ date_entry ].offset == 0 ) {
	time_seconds = htonl( time( &unix_time ) + TIME_DIFF );
	bcopy( (char *)&time_seconds, (char *)&fh->create_date,
		sizeof( fh->create_date ));
	bcopy( (char *)&time_seconds, (char *)&fh->mod_date,
		sizeof( fh->mod_date ));
	fh->backup_date = htonl( TIME_ZERO );
    } else if ( single.entry[ date_entry ].offset != 16 ) {
	fprintf( stderr, "%s has bogus FileInfo.\n", single.path );
	return( -1 );
    } else {
	pos = lseek( single.filed,
		single.entry[ date_entry ].offset, L_SET );
	if ( read( single.filed, (char *)entry_buf, sizeof( entry_buf )) !=
		sizeof( entry_buf )) {
	    perror( "Premature end of file :" );
	    return( -1 );
	}
	bcopy( (char *)&entry_buf[ 0 ], (char *)&fh->create_date,
		sizeof( fh->create_date ));
	bcopy( (char *)&entry_buf[ 4 ], (char *)&fh->mod_date,
		sizeof( fh->mod_date ));
	fh->backup_date = htonl( TIME_ZERO );
    }
    if ( single.entry[ ENTRYID_RESOURCE ].offset == 0 ) {
	fh->forklen[ RESOURCE ] = 0;
    } else {
	fh->forklen[ RESOURCE ] =
		htonl( single.entry[ ENTRYID_RESOURCE ].length );
    }
    if ( single.entry[ ENTRYID_DATA ].offset == 0 ) {
	fh->forklen[ DATA ] = 0;
    } else {
	fh->forklen[ DATA ] = htonl( single.entry[ ENTRYID_DATA ].length );
	pos = lseek( single.filed, single.entry[ ENTRYID_DATA ].offset, L_SET );
    }

    return( 0 );
}

/*
 * single_header_test is called from single_open.  It checks certain
 * values of the file and determines if the file is an AppleSingle version
 * one file something else, and returns a one, or negative one to indicate
 * file type.
 *
 * The Magic Number of the file, the first four bytes, must be hex
 * 0x00051600.  Bytes 4 through 7 are the version number and must be hex
 * 0x00010000.  Bytes 8 through 23 identify the home file system, and we
 * are only interested in files from Macs.  Therefore these bytes must
 * contain hex 0x4d6163696e746f736820202020202020 which is ASCII
 * "Macintosh       " (that is seven blanks of padding).
 */
#define ASMAGIC		0x00051600L
#define VERSIONONE	0x00010000L
#define VERSIONTWO	0x00020000L
#define MACINTOSH	"Macintosh       "
#define SIXTEENZEROS	"0000000000000000"

single_header_test()
{
    int			cc;
    u_long		templong;
    u_short		header_crc;
    u_char		namelen;

#if DEBUG
    fprintf( stderr, "entering single_header_test\n" );
#endif

    cc = read( single.filed, (char *)header_buf, sizeof( header_buf ));
    if ( cc < sizeof( header_buf )) {
	perror( "Premature end of file :" );
	return( -1 );
    }

#if DEBUG
    fprintf( stderr, "was able to read HEADBUFSIZ bytes\n" );
#endif

    bcopy( (char *)&header_buf[ 0 ], (char *)&templong, sizeof( templong ));
    if ( ntohl( templong ) != ASMAGIC ) {
	return( -1 );
    }

#if DEBUG
    fprintf( stderr, "magic number is for AppleSingle\n" );
#endif

    bcopy( (char *)&header_buf[ 4 ], (char *)&templong, sizeof( templong ));
    templong = ntohl( templong );
    if ( templong == VERSIONONE ) {
	cc = 1;
	if ( bcmp( MACINTOSH, &header_buf[ 8 ], sizeof( MACINTOSH ) - 1 ) 
		!= 0 ) {
	    return( -1 );
	}
#if DEBUG
	fprintf( stderr, "version number is %d\n", cc );
	fprintf( stderr, "home file system is Macintosh\n" );
#endif
    } else if ( templong == VERSIONTWO ) {
	cc = 2;
	if ( bcmp( SIXTEENZEROS, &header_buf[ 8 ], sizeof( SIXTEENZEROS ) - 1 )
		!= 0 ) {
	    return( -1 );
	}
#if DEBUG
	fprintf( stderr, "version number is %d\n", cc );
	fprintf( stderr, "filler is properly zeros\n" );
#endif
    } else {
	return( -1 );
    }

    return( cc );
}

/*
 * single_read is called until it returns zero for each fork.  When
 * it returns zero for the first fork, it seeks to the proper place
 * to read in the next, if there is one.  single_read must be called
 * enough times to return zero for each fork and no more.
 *
 */

single_read( fork, buffer, length )
    int			fork;
    char		*buffer;
    int			length;
{
    u_long		entry_id;
    char		*buf_ptr;
    int			readlen;
    int			cc = 1;
    off_t		pos;

#if DEBUG >= 3
    fprintf( stderr, "single_read: fork is %s\n", forkname[ fork ] );
    fprintf( stderr, "single_read: remaining length is %d\n", bin.forklen[fork] );
#endif

    switch ( fork ) {
	case DATA :
	    entry_id = ENTRYID_DATA;
	    break;
	case RESOURCE :
	    entry_id = ENTRYID_RESOURCE;
	    break;
	default :
	    return( -1 );
	    break;
    }

    if ( single.entry[ entry_id ].length < 0 ) {
	fprintf( stderr, "This should never happen, dude!\n" );
	return( single.entry[ entry_id ].length );
    }

    if ( single.entry[ entry_id ].length == 0 ) {
	if ( fork == DATA ) {
	    pos = lseek( single.filed,
		single.entry[ ENTRYID_RESOURCE ].offset, L_SET );
#if DEBUG
	    fprintf( stderr, "current position is %ld\n", pos );
#endif
	}
	return( 0 );
    }

    if ( single.entry[ entry_id ].length < length ) {
	readlen = single.entry[ entry_id ].length;
    } else {
	readlen = length;
    }
#if DEBUG >= 3
    fprintf( stderr, "single_read: readlen is %d\n", readlen );
    fprintf( stderr, "single_read: cc is %d\n", cc );
#endif

    buf_ptr = buffer;
    while (( readlen > 0 ) && ( cc > 0 )) {
	if (( cc = read( single.filed, buf_ptr, readlen )) > 0 ) {
#if DEBUG >= 3
	    fprintf( stderr, "single_read: cc is %d\n", cc );
#endif
	    readlen -= cc;
	    buf_ptr += cc;
	}
    }
    if ( cc >= 0 ) {
	cc = buf_ptr - buffer;
	single.entry[ entry_id ].length -= cc;
    }

#if DEBUG >= 3
    fprintf( stderr, "single_read: chars read is %d\n", cc );
#endif
    return( cc );
}
