#ifndef USE_COLOR
#   ifdef M_SYSV			/* SCO family */
#	   define USE_COLOR
#   endif
#   ifdef ISC				/* not yet */
#   endif
#   ifdef sun				/* not yet */
#   endif
#   ifdef ISCSVR4			/* last I heard from lothar, this worked */
#	   define USE_COLOR
#   endif
#   ifdef ESIXSVR4			/* I heard this does not work */
#   endif
#endif /* USE_COLOR */

/*+-------------------------------------------------------------------------
	ecutty.c - local tty (console) functions
	wht@n4hgf.Mt-Park.GA.US

  Defined functions:
	B_to_timeout_msec(c_cflag,st_rdev)
	_setcolor(clrs)
	color_name_to_num(cname)
	get_initial_colors()
	get_ttymode()
	get_ttyname()
	kbd_test()
	read_colors_file()
	restore_initial_colors()
	ring_bell()
	setcolor(new_colors)
	setcolor_internal(ntokens,tokens)
	termio_to_kbd_chars()
	ttyflush(flush_type)
	ttygetc(xkey_ok)
	ttygets(str,maxsize,flags,delim,pstrpos)
	ttygets(str,maxsize,flags,delim,pstrpos)
	ttygets_esd(tesd,flags,append_flag)
	ttyinit(param)
	ttymode(arg)
	ttyrdchk()

    In SCO versions, ECU keeps the the state of the normal and
    reverse video foreground and background colors in a 32-bit value:

     00000000001111111111222222222233
     01234567890123456789012345678901
     0000|--|0000|--|0000|--|0000|--|
          fg      bk      fg      bk
           reverse      normal

    The color values are per the SCO extended color definitons:

    black    0     gray         8
    blue     1     lt_blue      9
    green    2     lt_green    10
    cyan     3     lt_cyan     11
    red      4     lt_red      12
    magenta  5     lt_magenta  13
    brown    6     yellow      14
    white    7     hi_white    15

    With Lothar's ISC SVR4, the format is

     00000000001111111111222222222233
     01234567890123456789012345678901
     00000000000000000011????0100????
       0   0   0   0   3   f   4  o b
    where f is the foreground color
    and   b is the background color

    (I dont have one, so I'm guessing these colors are chosen from
    the ISO colors):

     BLACK       0
     RED         1
     GREEN       2
     YELLOW      3
     BLUE        4
     MAGENTA     5
     CYAN        6
     WHITE       7

--------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:10-18-1992-14:26-wht@n4hgf-add console xon/xoff control */
/*:10-08-1992-01:12-wht@n4hgf-no more obsolete Metro Link PTS */
/*:10-08-1992-01:06-wht@n4hgf-SVR4 color work + !use_color normal fix */
/*:09-15-1992-18:52-wht@n4hgf-left some debug code in patch01 dammit */
/*:09-13-1992-12:52-wht@n4hgf-add tty_is_scoterm */
/*:09-10-1992-13:59-wht@n4hgf-ECU release 3.20 */
/*:08-30-1992-07:42-wht@n4hgf-implement USE_COLOR+turn off ESIX color for now */
/*:08-22-1992-15:38-wht@n4hgf-ECU release 3.20 BETA */
/*:08-16-1992-03:43-wht@n4hgf-add -F funckeytype */
/*:06-20-1992-21:13-wht@n4hgf-eculibdir was overwritten if no home dir colors */
/*:06-16-1992-11:20-wht@n4hgf-ECUFUNCKEY */
/*:05-05-1992-17:42-wht@n4hgf-repair underscore always on on sun */
/*:04-24-1992-16:55-wht@n4hgf-dont flunk on >43 lines but use only 43 */
/*:04-20-1992-20:31-wht@n4hgf-ttymode now no-op until ttyinit called */
/*:04-20-1992-19:42-wht@n4hgf-kbdtest code in ttygetc messed up-str too short */
/*:04-19-1992-03:21-jhpb@sarto.budd-lake.nj.us-3.18.37 has ESIX SVR4 */
/*:04-19-1992-02:00-wht@n4hgf-if TERM=ansi with WINDOWID, assume scoterm */
/*:08-31-1991-13:29-wht@n4hgf2-look for colors in ECULIBDIR too */
/*:08-30-1991-04:12-wht@n4hgf2-restore colors wrong to do now if not SCO */
/*:08-30-1991-02:49-aega84!lh-use at_ansi.h/kd.h/CONS_GET under ISC SVR4 */
/*:08-25-1991-14:39-wht@n4hgf-SVR4 port thanks to aega84!lh */
/*:08-17-1991-18:29-wht@n4hgf-add kbdtest command */
/*:07-25-1991-12:57-wht@n4hgf-ECU release 3.10 */
/*:07-14-1991-18:18-wht@n4hgf-new ttygets functions */
/*:07-10-1991-16:19-wht@n4hgf-improve multi-char func key read timeout */
/*:03-20-1991-03:07-root@n4hgf-pts driver returns -1 on rdchk success! */
/*:03-19-1991-21:24-root@n4hgf-METROLINK_X11R4_PTS mods */
/*:01-29-1991-14:03-wht@n4hgf-more time for ESC vs fkey discrimination */
/*:01-29-1991-13:44-wht@n4hgf-load colors_normal w/ioctl GIO_ATTR if M_UNIX */
/*:12-01-1990-14:33-wht@n4hgf-more non-ansi - fkey mapping with nonansi.c */
/*:11-28-1990-15:56-wht@n4hgf-add non-ansi terminal support */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#include "ecu.h"
#include "esd.h"
#include "ecufkey.h"
#include "ecukey.h"
#include "ecuxkey.h"
#include "ecuerror.h"
#include "termecu.h"

/*
 * This is how we determine whether we are on a color terminal or not.
 * We handle SCO XENIX, SCO UNIX, ISC 2.2 and ISC SVR4.
 * If you have trouble with this add some kind of thing instead of MYSYS
 * and mail me the full details.  Search for MYSYS throughout here.
 */
#if defined(M_SYSV) && !defined(LINUX)
#   include <sys/machdep.h>
/*
 * Thanks for the G2, er GIO_ATTR, to staceyc@sco.COM (Stacey Campbell)
 * GIO_ATTR was not defined in header files as of this writing
 */
#if !defined(GIO_ATTR)
#define GIO_ATTR  ('a' << 8) | 0	/* Ioctl call for current attribute */
#endif
#else
#if defined(ISC) || defined(ISCSVR4) || defined(ESIXSVR4) /*&&!defined(MYSYS)*/
#  include <sys/at_ansi.h>
#  include <sys/kd.h>
#endif /* ISC || ISCSVR4 || ESIXSVR4 */
#endif /* M_SYSV */

#if defined(LINUX)
#  include <sys/kd.h>
#endif

#define DEFINE_TTY_DATA
#include "ecutty.h"

extern uint tcap_LINES;
extern uint tcap_COLS;
extern int LINES;
extern int COLS;
extern char screen_dump_file_name[];
extern char *dash_f_funckeytype;

char *kde_text();

uint LINESxCOLS;
int current_ttymode = 0;
int ttymode_termecu_on_sigint = 0;
int tty_is_pty;
int tty_is_multiscreen;
int tty_is_scoterm;
int tty_not_char_special;
int tty_use_kbd_sw_flow_control = 1;
int use_colors = 0;		/* set by ttyinit, but default no */
char *ttype;			/* getenv("TERM") */

static int kbd_test_active = 0;
static int ttyinit_has_been_called = 0;
static int got_original_colors = 0;

struct termio tty_termio_at_entry;
struct termio tty_termio_current;
struct stat tty_stat;
struct stat dn;
struct stat tty01;
struct stat ttyp0;
struct stat console;

uchar kbdeof;			/* current input EOF */
uchar kbdeol2;			/* current secondary input EOL */
uchar kbdeol;			/* current input EOL */
uchar kbderase;			/* current input ERASE */
uchar kbdintr;			/* current input INTR */
uchar kbdkill;			/* current input KILL */
uchar kbdquit;			/* current input QUIT */
int echo_erase_char;	/* save users ECHOE bit */
int echo_kill_char;		/* save users ECHOK bit */
char kbd_is_7bit;		/* keyboard has parity */
long tty_escape_timeout = 40L;		/* timeout on waiting for char after ESC */

uchar *dole_out_tgc_accum = (uchar *)0;
int dole_out_tgc_accum_count = 0;

/*
 * color definitions per format described at top of source
 * we handle SCO XENIX and UNIX and ISC SVR4 but not "MYSYS"
 * (search for MYSYS near top of source for clue what MYSYS means)
 *
 * As of this writing, I don't know about colors on ISC 2.2.
 * but they might be the same as for SVR4.
 *
 */
#if defined(M_SYSV) && !defined(LINUX) /* SCO */
ulong colors_initial = 0x04070A00L;	/* default initial colors */
ulong colors_current = 0x04070A00L; /* colors set during execution */
ulong colors_normal  = 0x04070A00L;  /* default lt_green/black red/white */
ulong colors_success = 0x07000A00L;	/* lt_green/black red/white */
ulong colors_alert = 0x0E000E00L;	/* yellow */
ulong colors_error = 0x04000400L;	/* red */
ulong colors_notify = 0x08000800L;	/* gray */
#define COLORS_DEFINED
#endif /* M_SYSV */

#if defined(ISCSVR4)
ulong colors_initial = 0x00003740L;
ulong colors_current = 0x00003740L;
ulong colors_normal  = 0x00003740L;  /* white */
ulong colors_success = 0x00003240L;  /* green */
ulong colors_alert   = 0x00003340L;  /* yellow */
ulong colors_error   = 0x00003140L;  /* red  */
ulong colors_notify  = 0x00003640L;  /* cyan */
#define COLORS_DEFINED
#endif /* ISCSVR4 */

#if defined(ESIXSVR4) || defined(LINUX)
ulong colors_initial = 0x00003740L;
ulong colors_current = 0x00003740L;
ulong colors_normal  = 0x00003740L;  /* ? */
ulong colors_success = 0x00003240L;  /* ? */
ulong colors_alert   = 0x00003340L;  /* ? */
ulong colors_error   = 0x00003140L;  /* ? */
ulong colors_notify  = 0x00003640L;  /* ? */
#define COLORS_DEFINED

static struct color_remapping {
  char *name;
  char ecu_val;
  char svr4_val;
} color_remapping[] =  {
	{ "black",		0,		0x00 },
	{ "blue",		1,		0x04 },
	{ "brown",		6,		0x03 },
	{ "cyan",		3,		0x06 },
	{ "gray",		8,		0x10 },
	{ "green",		2,		0x02 },
	{ "hi_white",	15,		0x17 },
	{ "lt_blue",	9,		0x14 },
	{ "lt_cyan",	11,		0x16 },
	{ "lt_green",	10,		0x12 },
	{ "lt_magenta",	13,		0x15 },
	{ "lt_red",		12,		0x11 },
	{ "magenta",	5,		0x05 },
	{ "red",		4,		0x01 },
	{ "white",		7,		0x07 },
	{ "yellow",		14,		0x13 },
	0, 0, 0,
};
#endif /* ESIXSVR4 */

#if !defined(COLORS_DEFINED)
ulong colors_current = 1; /* dummy */
ulong colors_normal = 2; /* dummy */
ulong colors_initial = 3; /* dummy */
ulong colors_success = 4; /* dummy */
ulong colors_alert = 5; /* dummy */
ulong colors_error = 6; /* dummy */
ulong colors_notify = 7; /* dummy */
#define COLORS_DEFINED
#endif /* !COLORS_DEFINED */


/*+-------------------------------------------------------------------------
	B_to_timeout_msec(c_cflag,st_rdev) - CBAUD code to ESC timeout msec
--------------------------------------------------------------------------*/
/*ARGSUSED*/
ulong
B_to_timeout_msec(c_cflag,st_rdev)
ushort c_cflag;
ushort st_rdev;
{
	long ms = 300L;


	/* make network/xterm/pty sweat, but don't make as many mistakes */
	if(tty_is_pty)
		return(ms);

	/* if multiscreen, 3 ticks is pu-lenty */
	if(tty_is_multiscreen)
		return((long)(1000/hz * 3));

	/* baud rate fiddling */
	switch(c_cflag & CBAUD)
	{
		/*       char times * time/char */
		case B110:	ms = 10 * 100;
		case B300:	ms = 10 * 33;
		case B600:	ms = 10 * 16;
		case B1200:	ms = 10 * 8;
		case B2400:	ms = 10 * 4;
		default:		/* many character times for packetized ... */
			ms = 400L;	/* ... modems used for console */
	}
	return(ms);

}	/* end of B_to_timeout_msec */

/*+-------------------------------------------------------------------------
	color_name_to_num(cname)
--------------------------------------------------------------------------*/
int
color_name_to_num(cname)
char *cname;
{
	register COLOR *color = colors;
	register itmp;

	while(color->name)
	{
		if((itmp = strcmp(color->name,cname)) > 0)
			return(-1);
		if(!itmp)
			return(color->num);
		color++;
	}
	return(-1);

}	/* end of color_name_to_num */

/*+-------------------------------------------------------------------------
	_setcolor(clrs)
--------------------------------------------------------------------------*/
void
_setcolor(clrs)
ulong clrs;
{
#if defined(SVR4) || defined(LINUX)
#if defined(ESIXSVR4) || defined(LINUX)
	struct color_remapping *cr;
	char fgcolor;
	char bgcolor;
#else
	char fg[4], bg[4];
#endif
#endif

#if defined(SVR4) || defined(LINUX)
#if defined(ESIXSVR4) || defined(LINUX)
	/* set foreground color */
	fgcolor = (clrs >> 8) & 0xff;
	for (cr=color_remapping; cr->name; ++cr) {
	  if (fgcolor == cr->ecu_val) {
		fgcolor = cr->svr4_val;
		break;
	  }
	}
	if (!cr->name) {
	  fgcolor = 0x07; /* white */
	}
	if (fgcolor & 0x10) {
	  fgcolor &= ~0x10;
	  ff(se, "\033[1;3%d;m", fgcolor);
	} else {
	  ff(se, "\033[0;3%d;m", fgcolor);
	}

	/* set background color */
	bgcolor = clrs & 0xff;
	for (cr=color_remapping; cr->name; ++cr) {
	  if (bgcolor == cr->ecu_val) {
		bgcolor = cr->svr4_val;
		break;
	  }
	}
	if (!cr->name) {
	  bgcolor = 0x00; /* black */
	}
	if (bgcolor & 0x10) {
	  bgcolor &= ~0x10;
	  ff(se, "\033[5;3%d;m", bgcolor);
	} else {
	  ff(se, "\033[0;3%d;m", bgcolor);
	}
#else	/* ISC SVR4 */
	/* normal */
	sprintf(fg,"%d", (clrs >> 8) & 0xFF);
	sprintf(bg,"%d", clrs & 0xFF);
	ff(se,"\033[%sm", fg);
	ff(se,"\033[%sm", bg);
#endif	/* ESIXSVR4 */
#else	/* not any SVR4 */
#ifdef M_SYSV /* SCO */
	/* normal */
	ff(se,"\033[=%ldF\033[=%ldG",(clrs >> 8) & 0xFF,clrs & 0xFF);

	/* reverse */
	ff(se,"\033[=%ldH\033[=%ldI",(clrs >> 24) & 0xFF,(clrs >> 16) & 0xFF);
#endif /* M_SYSV */
#endif /* SVR4 */
/*#endif */
	colors_current = clrs;
}	/* end of _setcolor */

/*+-------------------------------------------------------------------------
	setcolor(new_colors)

requires termcap init to have been done
--------------------------------------------------------------------------*/
void
setcolor(new_colors)
ulong new_colors;
{
	if(tty_not_char_special)
		return;

/*	if(!use_colors)
	{
		if(new_colors == colors_notify)
			tcap_underscore_on();
		else if(new_colors == colors_alert)
			tcap_bold_on();
		else if(new_colors == colors_error)
			tcap_stand_out();
		else
		{
			tcap_underscore_off();
			tcap_bold_off();
			tcap_stand_end();
		}
		return;
	} */
	_setcolor(new_colors);
	tcap_stand_end();
}	/* end of setcolor */

/*+-------------------------------------------------------------------------
	setcolor_internal(ntokens,tokens)

returns 0 on success, else token number in error + 1
--------------------------------------------------------------------------*/
int
setcolor_internal(ntokens,tokens)
int ntokens;
char **tokens;
{
	ulong fgnd;
	ulong bgnd;

	if(tty_not_char_special || !use_colors)
		return(0);

	if(ntokens < 2)
		return(1);
	else if(ntokens == 2)
		tokens[2] = "black";

	if((fgnd = (ulong)color_name_to_num(tokens[1])) > 15)
		return(2);
	if((bgnd = (ulong)color_name_to_num(tokens[2])) > 15) 
		return(3);

	if(!strcmp(tokens[0],"normal"))
	{
		colors_normal &= 0xFFFF0000L;
		colors_normal |= (fgnd << 8) | bgnd;
		setcolor(colors_normal);
	}
	else if(!strcmp(tokens[0],"reverse"))
	{
		colors_normal &= 0x0000FFFFL;
		colors_normal |= (fgnd << 24) | (bgnd << 16);
		setcolor(colors_normal);
	}
	else if(!strcmp(tokens[0],"notify"))
		colors_notify = (fgnd << 24) | (bgnd << 16) | (fgnd << 8) | bgnd;
	else if(!strcmp(tokens[0],"success"))
		colors_success = (fgnd << 24) | (bgnd << 16) | (fgnd << 8) | bgnd;
	else if(!strcmp(tokens[0],"alert"))
		colors_alert = (fgnd << 24) | (bgnd << 16) | (fgnd << 8) | bgnd;
	else if(!strcmp(tokens[0],"error"))
		colors_error = (fgnd << 24) | (bgnd << 16) | (fgnd << 8) | bgnd;
	else
		return(1);

	return(0);

}	/* end of setcolor_internal */

/*+-------------------------------------------------------------------------
	restore_initial_colors() - make screen safe

On SCO, restore color choices at execution time if we successfully
got them from the driver; in other situations, use tcap to reset.
--------------------------------------------------------------------------*/
void
restore_initial_colors()
{
#if defined(M_SYSV) || defined(SVR4)
	if(use_colors && got_original_colors)
		setcolor(colors_initial);
	else
#endif
	{
		tcap_blink_off();
		tcap_bold_off();
		tcap_underscore_off();
	}
}	/* end of restore_initial_colors */

/*+-------------------------------------------------------------------------
	get_initial_colors() - read colors at time of execution from driver

     00000000001111111111222222222233
     01234567890123456789012345678901
     0000|--|0000|--|0000|--|0000|--|
          fg      bk      fg      bk
           reverse      normal

--------------------------------------------------------------------------*/
#if defined(M_SYSV)
void
get_initial_colors()
{
	uint cur_attr;
	ulong fgnd;
	ulong bgnd;

	colors_initial = colors_normal;	 /* scoterm can use color but ... */
	if(ioctl(TTYIN, GIO_ATTR, 0) == -1) /* ... GIO_ATTR won't work */
		return;
	colors_normal = 0L;

/*
 * first, reverse, so we can end up with normal colors selected
 */
	write(1,"\033[7m",4);		/* select reverse */
	cur_attr = (uint)ioctl(TTYIN, GIO_ATTR, 0);
	fgnd = (ulong)cur_attr & 0x0F;
	bgnd = (ulong) (cur_attr >> 4) & 0x0F;
	colors_normal |= (fgnd << 24) | (bgnd << 16);

/*
 * now, normal
 */
	write(1,"\033[m",3);		/* select normal */
	cur_attr = (uint)ioctl(TTYIN, GIO_ATTR, 0);
	fgnd = (ulong)cur_attr & 0x0F;
	bgnd = (ulong) (cur_attr >> 4) & 0x0F;
	colors_normal |= (fgnd << 8) | bgnd;

	colors_initial = colors_normal;		/* save for restore_initial_colors */
	got_original_colors = 1;

}	/* end of get_initial_colors */
#endif

/*+-------------------------------------------------------------------------
	read_colors_file() - read color definition if present
--------------------------------------------------------------------------*/
void
read_colors_file()
{
	FILE *fp;
	char s128[128];
#define MAX_COLOR_TOKENS 6
	char *tokens[MAX_COLOR_TOKENS];
	int ntokens;
	char *cptr;
	int itmp;

	if(tty_not_char_special)
		return;

#if defined(M_SYSV)
	get_initial_colors();
#endif

	get_home_dir(s128);
	strcat(s128,"/.ecu/colors");
	if(access(s128,4))
		return;

	if(!(fp = fopen(s128,"r")))
	{
		strcpy(s128,eculibdir);
		strcat(s128,"/colors");
		if(!(fp = fopen(s128,"r")))
			return;
	}

	while(fgets(s128,sizeof(s128),fp))
	{
		if(s128[0] == '#')			/* comment? */
			continue;
		if(itmp = strlen(s128))		/* itmp = len; if > 0 ... */
		{
			itmp--;
			s128[itmp] = 0;			/* ... strip trailing NL */
		}
		cptr = s128;				/* first call to str_token, -> buff */
		while((*cptr == 0x20) || (*cptr == TAB))
			cptr++;				/* strip leading spaces */
		if(*cptr == 0)				/* if line all blank, skip it */
			continue;

		build_str_array(s128,tokens,MAX_COLOR_TOKENS,&ntokens);
		if(ntokens < 2)
			continue;

		setcolor_internal(ntokens,tokens);

	}			/* while records left to ready */

#if defined(M_SYSV)
	if(ioctl(TTYIN, GIO_ATTR, 0) == -1)
		colors_initial = colors_normal; /* hack for scoterm */
#endif

	fclose(fp);
}	/* end of read_colors_file */

/*+-------------------------------------------------------------------------
	ring_bell()
--------------------------------------------------------------------------*/
void
ring_bell()
{
	char b = BEL;

	if(tty_not_char_special)
		return;
	write(TTYOUT,&b,1);

}	/* end of ring_bell */

/*+-------------------------------------------------------------------------
	termio_to_kbd_chars()
--------------------------------------------------------------------------*/
void
termio_to_kbd_chars()
{
	kbdintr =  (tty_termio_at_entry.c_cc[VINTR])
		? (tty_termio_at_entry.c_cc[VINTR]  & 0x7F) : '\377';
	kbdquit =  (tty_termio_at_entry.c_cc[VQUIT])
		? (tty_termio_at_entry.c_cc[VQUIT]  & 0x7F) : '\377';
	kbderase = (tty_termio_at_entry.c_cc[VERASE])
		? (tty_termio_at_entry.c_cc[VERASE] & 0x7F) : '\377';
	kbdkill =  (tty_termio_at_entry.c_cc[VKILL])
		? (tty_termio_at_entry.c_cc[VKILL]  & 0x7F) : '\377';
	kbdeof =   (tty_termio_at_entry.c_cc[VEOF])
		? (tty_termio_at_entry.c_cc[VEOF]   & 0x7F) : '\04';
	kbdeol2 =  (tty_termio_at_entry.c_cc[VEOL])
		? (tty_termio_at_entry.c_cc[VEOL]   & 0x7F) : '\377';
	kbdeol =   (tty_termio_at_entry.c_iflag & ICRNL)
		? '\r' : '\n';

	kbd_is_7bit = ((tty_termio_at_entry.c_cflag & PARENB) != 0);
	echo_erase_char = tty_termio_at_entry.c_lflag & ECHOE;
	echo_kill_char = tty_termio_at_entry.c_lflag & ECHOK;

}	/* end of termio_to_kbd_chars */

/*+-------------------------------------------------------------------------
	ttyinit(param)
--------------------------------------------------------------------------*/
void
ttyinit(param)
uchar param;
{
	int fddevtty;
	int itmp;
	char *ftype;
#if defined(CONS_GET) && defined(MONO) && defined(USE_COLOR)
	int monitor_type;
	int cons_get_err;
#endif

	ttype = getenv("TERM");	/* must do this first */

	/*
	 * get control tty control chars in case stdin not tty
	 */
	if((fddevtty = open("/dev/tty",O_RDONLY,0)) >= 0)
	{
		ioctl(fddevtty,TCGETA,(char *)&tty_termio_at_entry);
		close(fddevtty);
		termio_to_kbd_chars();
	}

	sigint = 0;			/* see xmtr signal handlers */

	memset((char *)&tty_stat,0xFF,sizeof(struct stat));
	memset((char *)&ttyp0,0xFF,sizeof(struct stat));
	memset((char *)&console,0xFF,sizeof(struct stat));
	stat("/dev/console",&console);
	stat("/dev/null",&dn);
	stat("/dev/tty01",&tty01);
	stat("/dev/ttyp0",&ttyp0);

	/*
	 * if stdin not open or is /dev/null or is non-character-device
	 */

	itmp = fstat(TTYIN,&tty_stat);
	if(itmp || ((tty_stat.st_mode & S_IFMT) != S_IFCHR) ||
		((dn.st_ino == tty_stat.st_ino) && (dn.st_rdev == tty_stat.st_rdev)))
	{
		tcap_LINES = LINES = 25;	/* fake necessary termcap/curses vars */
		tcap_COLS = COLS = 80;
		LINESxCOLS = tcap_LINES * tcap_COLS;
		shm->scr_lines = tcap_LINES;
		shm->scr_cols = tcap_COLS;
		shm->scr_size = LINESxCOLS;

		tty_not_char_special = 1;
		tty_is_multiscreen = 0;
		return;
	}

	/*
	 * if pty
	 */
	if((tty_stat.st_rdev & 0xFF00) == (ttyp0.st_rdev & 0xFF00))
		tty_is_pty = 1;

	/*
	 * use color if we are on a display that supports it and we know how :-|
	 */
	use_colors = 0;

#if defined(CONS_GET) && defined(MONO) && defined(USE_COLOR)
	if( ((cons_get_err = ioctl(TTYIN,CONS_GET,&monitor_type)) >= 0) &&
		(use_colors = (monitor_type != MONO)))
	{
		read_colors_file();
		setcolor(colors_normal);
	}

#endif /* CONS_GET && MONO && USE_COLOR */

	/*
	 * remember whether or not we are on a multiscreen
	 */
#if defined(M_SYSV) /* for multiscreen and scoterm */
#ifdef LINUX
	tty_is_multiscreen = 1;
	use_colors = 1;
	read_colors_file();
	setcolor(colors_normal);
#else
	tty_is_multiscreen = !(cons_get_err < 0);

	/*
	 * a fuzzy heuristic for scoterm:
	 * 1. presence of WINDOWID and
	 * 2. first four characters of $TERM == "ansi"
	 */
	if(getenv("WINDOWID") && ttype && !strncmp(ttype,"ansi",4))
	{
		use_colors = 1;
		tty_is_scoterm = 1;
		read_colors_file();
		setcolor(colors_normal);
	}
#endif
#endif /* M_SYSV multiscreen and scoterm */

	/*
	 * save initial tty state
	 */
	ioctl(TTYIN,TCGETA,(char *)&tty_termio_at_entry);
	tty_escape_timeout =
		B_to_timeout_msec(tty_termio_at_entry.c_cflag,tty_stat.st_rdev);

	termio_to_kbd_chars();

	tty_termio_current = tty_termio_at_entry;
	current_ttymode = 0;

	get_home_dir(screen_dump_file_name);
	strcat(screen_dump_file_name,"/.ecu/screen.dump");

	ftype = 0;
	if(dash_f_funckeytype)
		ftype =	dash_f_funckeytype;
	else
		ftype = getenv("ECUFUNCKEY");
	if(ttype || ftype)
		funckeymap_read((ftype) ? ftype : ttype);

	/* initialize termcap */
	tcap_init();			/* read termcap strings */

	/* yetch - magic number gretching for lines and columns */
	if(tcap_LINES < 20)
	{
		ff(se,"\7screen height must be >= 20 lines (found %dx%d).\r\n",
			tcap_COLS,tcap_LINES);
		termecu(TERMECU_GEOMETRY);
	}
	if(tcap_LINES > SCREEN_LINES_MAX)
	{
		ff(se,"\7screen height limited to %d lines (found %dx%d).\r\n",
			SCREEN_LINES_MAX,tcap_COLS,tcap_LINES);
		termecu(TERMECU_GEOMETRY);
	}
	if(tcap_COLS != SCREEN_COLS_MAX)
	{
		ff(se,"\7terminal width must be %d columns (found %dx%d).\r\n",
			SCREEN_COLS_MAX,tcap_COLS,tcap_LINES);
		termecu(TERMECU_GEOMETRY);
	}
	LINESxCOLS = tcap_LINES * tcap_COLS;
	shm->scr_lines = tcap_LINES;
	shm->scr_cols = tcap_COLS;
	shm->scr_size = LINESxCOLS;
	ttyinit_has_been_called = 1;

}	/* end of ttyinit */

/*+-----------------------------------------------------------------------
	ttymode(arg) -- control user console (kbd/screen)

  Where arg ==
	0 restore attributes saved at start of execution
	1 raw mode (send xon/xoff, but do not respond to it, no ISIG/SIGINT)
	2 raw mode (same as 1 but allow keyboard interrupts)
	3 same as 2 but terminate program on SIGINT
    4 terminate ecu on sigint

------------------------------------------------------------------------*/
void
ttymode(arg)
int arg;
{

	/*
	 * ignore if no keyboard involved
	 */
	if(tty_not_char_special)
		return;

	/*
	 * usage()->termecu()->ttymode() is possible before ttyinit()
	 */
	if(!ttyinit_has_been_called)
		return;

	switch(arg)
	{
	case 0:
		ioctl(TTYIN,TCSETAW,(char *)&tty_termio_at_entry);
		tty_termio_current = tty_termio_at_entry;
		current_ttymode = 0;
		break;

	case 4:
		ttymode_termecu_on_sigint = 1;
		/* fall thru */
	case 1:
	case 2:
	case 3:
		tty_termio_current = tty_termio_at_entry;

		tty_termio_current.c_cflag &= ~(PARENB | PARODD);
		tty_termio_current.c_cflag |= CS8;

		/* don't want to honor tty xon/xoff, but pass to other end */
		tty_termio_current.c_iflag &=
			~(INLCR | ICRNL | IGNCR | IUCLC | ISTRIP);
		if(tty_use_kbd_sw_flow_control)
			tty_termio_current.c_iflag |= IXON | IXOFF;
		else
			tty_termio_current.c_iflag &= ~(IXON | IXOFF);

		tty_termio_current.c_oflag |= OPOST;
		tty_termio_current.c_oflag &= ~(OLCUC | ONLCR | OCRNL | ONOCR | ONLRET);

		tty_termio_current.c_lflag &= ~(ICANON | ISIG | ECHO);
		if(arg > 1)
			tty_termio_current.c_lflag |= ISIG;

		tty_termio_current.c_cc[VMIN] = 1;
		tty_termio_current.c_cc[VTIME] = 0;

		ioctl(TTYIN,TCSETAW,(char *)&tty_termio_current);
		current_ttymode = arg;
		break;

	default:
		ff(se,"\r\nttymode: invalid argument %d\r\n",arg);
		break;
	}
}	/* end of ttymode */

/*+-------------------------------------------------------------------------
	int	get_ttymode()
--------------------------------------------------------------------------*/
int
get_ttymode()
{
	return(current_ttymode);
}	/* end of get_ttymode */

/*+-----------------------------------------------------------------------
	ttyflush(flush_type) -- flush tty driver input &/or output buffers

0 == input buffer
1 == output buffer
2 == both buffers
------------------------------------------------------------------------*/
void
ttyflush(flush_type)
int flush_type;
{
	if(tty_not_char_special)
		return;

	ioctl(TTYIN,TCXONC,(char *)0); /* stop tty output */

#if !defined(M_I286)
	ioctl(TTYIN,TCFLSH,(char *)flush_type);
#else
	/* avoid 286 compiler warning of cast int to far ptr */
	switch(flush_type)
	{
		case 0:
			ioctl(TTYIN,TCFLSH,(char *)0); break;
		case 1:
			ioctl(TTYIN,TCFLSH,(char *)1); break;
		case 2:
			ioctl(TTYIN,TCFLSH,(char *)2); break;
	}
#endif

	ioctl(TTYIN,TCXONC,(char *)1);	/* restart tty output */

	if(flush_type != 1)
	{
		dole_out_tgc_accum = (uchar *)0;
		dole_out_tgc_accum_count = 0;
	}

}	/* end of ttyflush */

/*+-------------------------------------------------------------------------
	ttyrdchk() - see if key pressed and not read
--------------------------------------------------------------------------*/
int
ttyrdchk()
{
	return(rdchk(TTYIN) || dole_out_tgc_accum_count);
}	/* end of ttyrdchk */

/*+-------------------------------------------------------------------------
	ttygetc(xkey_ok) -- get a key from the keyboard
if UNIX or XENIX, map extended keys to sign-bit-set special value
if xkey_ok is 0, disallow extended keys
--------------------------------------------------------------------------*/
uint
ttygetc(xkey_ok)
int xkey_ok;
{
	uchar ctmp;
	extern int errno;
	register uint itmp = 0;
	long timeout_remaining;
	static uchar tgc_accum[16];
	uchar funckeymap();

	if(tty_not_char_special)	/* this really is unexplored territory */
	{
		ctmp = 255;
		read(TTYIN,(char *)&ctmp,1);
		return((uint)ctmp);
	}

	if(dole_out_tgc_accum_count)
	{
		ctmp = *dole_out_tgc_accum++;
		dole_out_tgc_accum_count--;
		if(kbd_is_7bit)
			ctmp &= 0x7F;
		return((uint)ctmp);
	}

GET_KEY:
	errno = 0;
	if(read(TTYIN,(char *)&ctmp,1) < 0)
	{
		if(errno == EINTR)
			goto GET_KEY;
		perror_errmsg("keyboard");
		termecu(TERMECU_TTYIN_READ_ERROR);
	}

	if(	((ctmp >= 0x01) && (ctmp <= 0x1F) ||
		 (ctmp >= 0x81) && (ctmp <= 0x9F)) &&
		(ctmp != kbderase) && (ctmp != kbdkill) &&
		(ctmp != kbdeol) && (ctmp != kbdeol2) &&
		(ctmp != kbdintr) && (ctmp != kbdeof) )
	{
		tgc_accum[0] = ctmp;
		tgc_accum[itmp = 1] = 0;
		timeout_remaining = tty_escape_timeout;
#if defined(NAP_DEBUG)
		ff(se,"timeout_remaining = %ld hzmsec=%ld\r\n",
			timeout_remaining,hzmsec);
#endif
		while(((ctmp = funckeymap(tgc_accum,itmp)) >= XF_no_way) &&
			(timeout_remaining > 0))
		{
			timeout_remaining -= Nap(hzmsec);
#if defined(NAP_DEBUG)
			ff(se,"timeout_remaining = %ld\r\n",timeout_remaining);
#endif
			if(!rdchk(TTYIN))
				continue;
			read(TTYIN,(char *)&ctmp,1);
			if(itmp == (sizeof(tgc_accum) - 1))	/* do not allow overflow */
			{
				ctmp = XF_no_way;
				break;
			}
			timeout_remaining = tty_escape_timeout;
			tgc_accum[itmp++] = ctmp;
		}
		tgc_accum[itmp] = 0;
		if((ctmp == XF_not_yet) && (itmp == 1))
		{
			if(kbd_is_7bit)
				tgc_accum[0] &= 0x7F;
			return((uint)tgc_accum[0]);
		}
		else if(ctmp < XF_no_way)	/* if we got a map */
		{
			if(kbd_test_active)
			{
				char title[128];
				sprintf(title,"--> func key '%s' (%d key codes received)",
					kde_text(ctmp),itmp);
				hex_dump(tgc_accum,-itmp,title,1);
			}
			if(!xkey_ok)
			{
				ring_bell();
				goto GET_KEY;
			}
			switch(ctmp)
			{
				case IKDE_CU5:
					screen_dump(screen_dump_file_name);
					goto GET_KEY;
				default:
					return((uint)ikde_to_xf(ctmp));
			}
			/*NOTREACHED*/
		}
		/* not func key -- must be typamatic control key */
		if(kbd_test_active)
		{
			char title[128];
			if(itmp > 1)
			{
				sprintf(title,
					"--> no func key recognized (%d key codes received)",itmp);
				hex_dump(tgc_accum,-itmp,title,1);
			}
		}
		dole_out_tgc_accum_count = itmp - 1;
		dole_out_tgc_accum = tgc_accum + 1;
		if(kbd_is_7bit)
			tgc_accum[0] &= 0x7F;
		return((uint)tgc_accum[0]);
	}

	/*
	 * simple key, not special
	 */
	if(kbd_is_7bit)
		ctmp &= 0x7F;
#ifdef LINUX
	/* make backspace key really backspace */
	if (ctmp == kbderase)
		ctmp = 0x08;
#endif
	return((uint)ctmp);
}	/* end if ttygetc */

/*+-----------------------------------------------------------------------
	ttygets(str,maxsize,flags,delim,pstrpos)

flags & TG_CRLF   - echo cr/lf terminator
flags & TG_XDELIM - extended delimiter set
                    (Home, End, PgUp, PgDn, CurUp, CurDn)
flags & TG_EDIT   - redisplay/edit current string
flags & TG_IPOS   - if edit, use initial string pos
------------------------------------------------------------------------*/
void
ttygets(str,maxsize,flags,delim,pstrpos)
register char *str;
int maxsize;
int flags;
uchar *delim;
int *pstrpos;
{
	register inch;
	char ch;
	register strcount = 0;
	register strpos = 0;
	int insert_mode = 0;
	char *bs_str = "\010 \010";
	extern int rcvrdisp_actual2_xmtr_buffer;

	rcvrdisp_actual2_xmtr_buffer = 1; /* let tcap_ rtns buffer */

	--maxsize;		/* decrement for safety */

	if(flags & TG_EDIT)
	{
		strpos = strcount = strlen(str);
		write(TTYOUT,str,strcount);
		if(pstrpos && (*pstrpos > 0) && (*pstrpos <= strcount))
			strpos = *pstrpos;
		tcap_curleft(strcount - strpos);
	}

	while(1)
	{
		rcvrdisp_actual2_xmtr_buffer = 0; /* disallow tcap_ rtns buffer */
		rcvrdisp_actual2_xmtr_buffer = 1; /* let tcap_ rtns buffer */
		inch = ttygetc(1);
		*delim = (uchar)inch;	/* last char will always be the delimiter */
		if((inch == kbdintr) || (inch == ESC))
		{
			tcap_curright(strcount - strpos);
			while(strcount)
			{
				write(TTYOUT,bs_str,strlen(bs_str));
				strcount--;
			}
			str[strcount] = 0;
			*delim = ESC;
			goto RETURN;
		}
		else if(inch == kbdkill)
		{
			tcap_curright(strcount - strpos);
			while(strcount)
			{
				write(TTYOUT,bs_str,strlen(bs_str));
				strcount--;
			}
			strpos = 0;
			*str = 0;
			continue;
		}
		else if(inch == kbderase || inch == 0x08)
		{
			if(strcount)
			{
				if(strcount == strpos)
				{
					write(TTYOUT,bs_str,strlen(bs_str));
					strcount--,strpos--;
				}
				else
				{
					if(!strpos)
						continue;
					mem_cpy(str + strpos - 1,str + strpos,strcount - strpos);
					write(TTYOUT,"\010",1);
					str[--strcount] = 0;
					strpos--;
					write(TTYOUT,str + strpos,strlen(str + strpos));
					write(TTYOUT," ",1);
					tcap_curleft(strcount - strpos + 1);
				}
			}
			str[strcount] = 0;
			continue;
		}
		else if(inch == XFins)
		{
			insert_mode = !insert_mode;
			continue;
		}
		else if(inch == XFcurlf)
		{
			if(strpos)
			{
				strpos--;
				tcap_curleft(1);
			}
			continue;
		}
		else if(inch == XFcurrt)
		{
			if(strpos < strcount)
			{
				strpos++;
				tcap_curright(1);
			}
			continue;
		}

		if(flags & TG_XDELIM)	/* extended delimiter */
		{
			switch(inch)
			{
				case XFhome:
				case XFend:
				case XFpgup:
				case XFpgdn:
				case XFcurup:
				case XFcurdn:
#ifdef notdef
					tcap_curright(strcount - strpos);
					while(strcount)
					{
						write(TTYOUT,bs_str,strlen(bs_str));
						strcount--;
					}
#endif
					str[strcount] = 0;
					goto RETURN;
			}
		}

		switch(inch)
		{
			case CRET:
				*delim = NL;
			case NL:
				str[strcount] = 0;
				tcap_curright(strcount - strpos);
				if((flags & TG_CRLF))
					ff(se,"\r\n");
				goto RETURN;

			case CTL_L:
			case CTL_R:
				tcap_curright(strcount - strpos);
				ff(se,"%s (insert mode %s)\r\n",make_char_graphic(inch,0),
					(insert_mode) ? "ON" : "OFF");
				tcap_eeol();
				write(TTYOUT,str,strcount);
				tcap_curleft(strcount - strpos);
				break;

			default:
				if((inch < SPACE) || (inch >= 0x7F))
				{
					ring_bell();
					break;
				}
				if(strpos == strcount)
				{
					if(strcount == maxsize)
					{
						ring_bell();
						continue;
					}
					str[strcount++] = inch;
					strpos++;
					ch = (char)inch;
					write(TTYOUT,&ch,1);
				}
				else
				{
					if(insert_mode)
					{
						if(strcount == maxsize)
						{
							ring_bell();
							continue;
						}
						mem_cpy(str+strpos+1,str+strpos,strcount-strpos);
						str[strpos] = inch;
						strcount++;
						str[strcount] = 0;
						write(TTYOUT,str + strpos,strcount - strpos);
						strpos++;
						tcap_curleft(strcount - strpos);
					}
					else
					{
						str[strpos++] = inch;
						ch = (char)inch;
						write(TTYOUT,&ch,1);
					}
				}
				str[strcount] = 0;
				continue;
		}
	}

RETURN:
	rcvrdisp_actual2_xmtr_buffer = 0; /* disallow tcap_ rtns buffer */
	if(pstrpos)
		*pstrpos = strpos;

}	/* end of ttygets() */

/*+-------------------------------------------------------------------------
	ttygets_esd(tesd,flags,append_flag)
--------------------------------------------------------------------------*/
ttygets_esd(tesd,flags,append_flag)
ESD *tesd;
int flags;
int append_flag;
{
	char *pb = tesd->pb;
	int maxcb = tesd->maxcb;
	uchar delim;

	if(append_flag)
	{
		pb += tesd->cb;
		maxcb -= tesd->cb;
	}
	else
	{
		pb = tesd->pb;
		maxcb = tesd->maxcb;
		tesd->cb = 0;
	}

	ttygets(pb,maxcb,flags,&delim,(int *)0);

	if(delim == ESC)
	{
		if(!append_flag)
			esdzero(tesd);
		return(eProcAttn_ESCAPE);
	}

	tesd->cb = strlen(tesd->pb);
	plogs(pb);
	if(flags & 1)
		plogc(NL);
	return(0);

}	/* end of ttygets_esd */

/*+-------------------------------------------------------------------------
	kbd_test() - test keyboard handler
--------------------------------------------------------------------------*/
void
kbd_test()
{
	uint ctmp = 0;

	pputs("Press keys to test (ESCape to exit)\n");
	kbd_test_active = 1;
	while(ctmp != ESC)
	{
		ctmp = ttygetc(1);
		
		if((ctmp < 0x80) && dole_out_tgc_accum_count)
		{
			pprintf("    got %d key sequence %s ",
				dole_out_tgc_accum_count + 1,hex_to_ascii_name(ctmp));
			while(dole_out_tgc_accum_count)
			{
				pprintf("%s ",hex_to_ascii_name(*dole_out_tgc_accum++));
				dole_out_tgc_accum_count--;
			}
			pputs("\n");
			ctmp = 0;
			continue;
		}
		pputs("    got ");
		if(ctmp >= 0x80)
			pprintf("fkey '%s'\n",xf_text(ctmp));
		else
			pprintf("key  '%s'\n",hex_to_ascii_name(ctmp));
	}
	kbd_test_active = 0;
	ttyflush(0);
	pputs("keyboard test complete\n\n");
	dole_out_tgc_accum = (uchar *)0;
	dole_out_tgc_accum_count = 0;
	
}	/* end of kbd_test */

/*+-------------------------------------------------------------------------
	char *get_ttyname() - return pointer to static string

This routine is largely a crock and is likely to explode at any rev or twist
--------------------------------------------------------------------------*/
char *
get_ttyname()
{
#ifndef OLD_WAY
	char *ttyname();
	return(ttyname(TTYIN));
#else
	static char ttname[64];
	register unsigned int rdev;
	register char *cptr;

	if(tty_not_char_special)
		return("stdin");
	else if(!tty_is_multiscreen)
		return("non-multiscreen");

	rdev = (unsigned)tty_stat.st_rdev;
	if(rdev == 0x0301)
		strcpy(ttname,"/dev/console");
#if defined(M_UNIX)
	else if(rdev == 0x0000)
		strcpy(ttname,"/dev/syscon");
#endif
	else
	{
		strcpy(ttname,"/dev/tty");
		cptr = ttname + 8;

		if(rdev < 0x000C)
		{
			*cptr++ = '0' + ((rdev + 1) / 10);
			*cptr++ = '0' + ((rdev + 1) % 10);
		}
		else if(!(rdev & ~0x58F))
		{
			*cptr++ = (rdev & 0x0008) ? '2' : '1';
			*cptr++ = ((rdev & 0x0080) ? 'A' : 'a') + (rdev & 0x0007);
		}
		else
		{
			*cptr++ = '?';
			*cptr++ = '?';
		}
		*cptr = 0;
	}

	return(ttname);
#endif
}	/* end of get_ttyname */
/*+-------------------------------------------------------------------------
     set_console_xon_xoff_by_arg(arg)
--------------------------------------------------------------------------*/
int
set_console_xon_xoff_by_arg(arg)
char *arg;
{
     int new_xon_xoff = 0;
      if(ulcmpb(arg,"on") < 0)
     {
             new_xon_xoff = IXON | IXOFF;
             tty_use_kbd_sw_flow_control = 1;
     }
     else if(ulcmpb(arg,"off") < 0)
     {
             new_xon_xoff = 0;
             tty_use_kbd_sw_flow_control = 0;
     }
     else
             return(-1);
     tty_termio_current.c_iflag &= ~(IXON|IXOFF);
     tty_termio_current.c_iflag |= new_xon_xoff;
     ioctl(TTYIN,TCSETA,(char *)&tty_termio_current);
     return(0);
}    /* end of set_console_xon_xoff_by_arg */
/*+-------------------------------------------------------------------------
     console_xon_status()
--------------------------------------------------------------------------*/
char *
console_xon_status()
{
     if(tty_use_kbd_sw_flow_control)
             return("HONOR ^S/^Q locally");
     else
             return("PASS ^S/^Q to remote");
}    /* end of console_xon_status */

/* end of ecutty.c */
/* vi: set tabstop=4 shiftwidth=4: */
