/*
 * tool.c -- user interface for an LSM database
 *
 * Lars Wirzenius
 * "@(#)lsmtool:tool.c,v 1.2 1994/08/09 18:15:38 liw Exp"
 */

 
#include <assert.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <curses.h>
#include <publib.h>
#include "lsm.h"


/* The database and the current location in it */
static struct lsm_database db;
static size_t current = -1;
static int modified = 0;


/* Current search string and filenames */
static char pattern[10240] = "";
static char db_filename[10240] = "LSM.current";
static char del_filename[10240] = "LSM.deleted";
static char read_filename[10240] = "LSM.new";
static char write_filename[10240] = "LSM.save";


/* A key binding */
struct bond {
	int key;
	int (*func)(void);
};


/* Auxiliary functions */
static void display(void);
static void do_endwin(void);
static void read_string(const char *, char *);
static void save_database(void);
static void sort_database(void);

/* Command functions.  These all return -1 for quit, >=0 for ok */
static int search_forward(void);
static int search_next(void);
static int goto_beginning(void);
static int goto_end(void);
static int goto_next(void);
static int goto_prev(void);
static int write_to_file(void);
static int read_from_file(void);
static int delete(void);
static int quit(void);


int main(int argc, char **argv) {
	struct bond bindings[] = {
		{ 'q', quit },
		{ '/', search_forward },
		{ 'n', search_next },
		{ 'g', goto_beginning },
		{ 'G', goto_end },
		{ 'j', goto_next },
		{ ' ', goto_next },
		{ 'k', goto_prev },
		{ 127, goto_prev },
		{ 'w', write_to_file },
		{ 'r', read_from_file },
		{ 'd', delete },
	};
	const size_t nbindings = sizeof bindings / sizeof(*bindings);
	int i, key;
	size_t on_display;
	FILE *f;

	__set_liberror(__exit_on_error | __complain_on_error);
	set_progname(argv[0], "lsmtool");

	initscr();
	cbreak();
	noecho();
	atexit(do_endwin);

	if (argc > 1)
		strmaxcpy(db_filename, argv[1], sizeof db_filename);

	f = fopen(db_filename, "r");
	if (f != NULL) {
		if (lsm_read_database(f, &db) == -1)
			errormsg(1, 0, "error in database");
		xfclose(f);
	}

	current = 0;
	on_display = !current;
	sort_database();
	for (;;) {
		if (current != on_display) {
			display();
			on_display = current;
		}
		refresh();
		key = getch();
		if (key == '\f')
			display();
		else for (i = 0; i < nbindings; ++i)
			if (key == bindings[i].key && bindings[i].func() == -1)
				exit(EXIT_SUCCESS);
	}

	return 0;
}


/* Shutdown curses properly */
static void do_endwin(void) {
	clear();
	refresh();
	endwin();
}


/* Sort the database.  Try to keep track of the current entry.  */
static int field = -1;
static int entry_cmp(const void *e1, const void *e2) {
	const struct lsm_entry *ee1 = e1;
	const struct lsm_entry *ee2 = e2;
	assert(field >= 0);
	return strcmp(ee1->fields[field], ee2->fields[field]);
}
static void sort_database(void) {
	int i, index;
	char helper[] = "xyzzy";

	index = lsm_field_to_index("sort helper");
	db.entries[current].fields[index] = helper;
	field = lsm_field_to_index("Title");
	qsort(db.entries, db.nentries, sizeof(*db.entries), entry_cmp);
	for (i = 0; i < db.nentries; ++i) {
		if (db.entries[i].fields[index] != NULL) {
			db.entries[i].fields[index] = NULL;
			current = i;
			break;
		}
	}
}


/* Display the current item */
static void display(void) {
	struct lsm_entry *e;
	int i, n, w, title;
	const char *p;
	const int width = COLS - 1;

	clear();
	if (current < db.nentries) {
		title = lsm_field_to_index("Title");
		e = &db.entries[current];
		for (i = n = 0; i < MAX_FIELDS && n < LINES-1; ++i) {
			if (e->fields[i] != NULL) {
				move(n, 0);
				if (i == title)
					standout();
				p = lsm_index_to_field(i),
				assert(p != NULL);
				w = strlen(p) > 15 ? 1 : 15-strlen(p);
				printw("%s:%*c", p, w, ' ');
				for (p = e->fields[i]; *p != '\0';) {
					char *q = strchr(p, '\n');
					if (q == NULL || q-p+1 > width)
						w = width;
					else
						w = q-p+1;
					if (w > strlen(p))
						w = strlen(p);
					printw("%.*s", w, p);
					p += w;
					if (p[-1] != '\n') {
						printw("\n");
						if (*p == '\n')
							++p;
						else if (*p != '\0')
							printw("\t\t");
					}
					++n;
				}
				standend();
			}
		}
	}
	move(LINES-1, 0);
	standout();
	printw("(%d/%d)", current+1, db.nentries);
	standend();
	refresh();
}


/* Read a string from the user */
static void read_string(const char *prompt, char *string) {
	int c, dirty;
	size_t pos;

	move(LINES-1, 0);
	standout();
	printw("%s", prompt);
	standend();

	dirty = 1;
	pos = strlen(string);
	for (;;) {
		if (dirty) {
			move(LINES-1, strlen(prompt));
			clrtoeol();
			printw("%s", string);
			move(LINES-1, strlen(prompt) + pos);
			refresh();
			dirty = 0;
		}

		c = getch();
		switch (c) {
		default:
			if (isprint((unsigned char) c)) {
				strcins(string+pos, c);
				++pos;
				dirty = 1;
			}
			break;
		case '\n':
		case '\r':
			return;
		case '\b':
		case 127:
			if (pos > 0) {
				--pos;
		case 4:
				strdel(string+pos, 1);
				dirty = 1;
			}
			break;
		case 2:
			if (pos > 0) {
				--pos;
				dirty = 1;
			}
			break;
		case 6:
			if (string[pos] != '\0') {
				++pos;
				dirty = 1;
			}
			break;
		case 11:
			string[pos] = '\0';
			dirty = 1;
			break;
		case 1:
			pos = 0;
			dirty = 1;
			break;
		case 5:
			pos = strlen(string);
			dirty = 1;
			break;
		case 21:
			pos = 0;
			string[0] = '\0';
			dirty = 1;
			break;
		}
	}
}


/* Search forward for the next entry matching a pattern */
static int search_forward(void) {
	read_string("Search for: ", pattern);
	return search_next();
}



/* Search forward using the previous search string */
static int search_next(void) {
	int title;
	size_t i;

	move(LINES-1, 0);
	standout();
	printw("Searching...");
	standend();
	clrtoeol();
	refresh();

	i = current;
	title = lsm_field_to_index("Title");
	do {
		assert(i < db.nentries);
		i = (i+1) % db.nentries;
		if (strstr(db.entries[i].fields[title], pattern) != NULL) {
			if (current == i)
				display();
			else
				current = i;
			return 0;
		}
	} while (i != current);
	move(LINES-1,0);
	standout();
	printw("Couldn't find `%s'", pattern);
	standend();
	refresh();
	return 0;
}



/* Move between entries */
static int goto_beginning(void) {
	current = 0;
	return 0;
}
static int goto_end(void) {
	current = db.nentries ? db.nentries - 1 : 0;
	return 0;
}
static int goto_next(void) {
	if (current+1 < db.nentries)
		++current;
	return 0;
}
static int goto_prev(void) {
	if (current > 0)
		--current;
	return 0;
}


/* Write current entry to file */
static int write_to_file(void) {
	FILE *f;

	read_string("Write to: ", write_filename);
	f = fopen(write_filename, "a");
	if (f == NULL) {
		move(LINES-1, 0);
		standout();
		printw("Couldn't open `%s' for writing", write_filename);
		refresh();
	} else {
		if (lsm_write_one_entry(f, &db.entries[current]) == -1) {
			move(LINES-1, 0);
			standout();
			printw("Couldn't write to `%s'", write_filename);
			refresh();
		}
		xfclose(f);
		display();
	}
	return 0;
}



/* Read entries from a file */
static int read_from_file(void) {
	FILE *f;

	read_string("Read from: ", read_filename);
	f = fopen(read_filename, "r");
	if (f == NULL) {
		move(LINES-1, 0);
		standout();
		printw("Couldn't open `%s' for reading", read_filename);
		refresh();
	} else {
		current = db.nentries ? db.nentries-1 : 0;
		if (lsm_read_database(f, &db) == -1) {
			move(LINES-1, 0);
			standout();
			printw("Couldn't read from `%s'", read_filename);
			refresh();
		}
		xfclose(f);
		if (current+1 < db.nentries) {
			++current;
			modified = 1;
		} else {
			move(LINES-1, 0);
			standout();
			printw("No entries in `%s'", read_filename);
			refresh();
		}
	}
	return 0;
}



/* Delete current entry */
static int delete(void) {
	FILE *f;

	if (current >= db.nentries)
		return 0;

	f = xfopen(del_filename, "a");
	(void) lsm_write_one_entry(f, &db.entries[current]);
	xfclose(f);

	memdel(db.entries + current,
	       (db.nentries - current) * sizeof(*db.entries),
	       sizeof(*db.entries));
	--db.nentries;
	if (current >= db.nentries)
		current = 0;
	modified = 1;
	display();
	return 0;
}


/* Quit the program. */
static int quit(void) {
	char answer[10240] = "";

	if (modified) {
		read_string("Quit?  "
	            	"y/<cr>=save and quit, "
                    	"n=don't quit, "
	            	"a=quit, but don't save  ", answer);
	} else {
		read_string("Quit? "
			    "y/<cr>/a=quit (no need to save), "
			    "n=don't quit", answer);
	}
	switch (answer[0]) {
	case '\0':
	case 'y':
	case 'Y':
		if (modified)
			save_database();
		/* FALLTHROUGH */
	case 'a':
	case 'A':
		return -1;
	default:
		display();
		return 0;
	}
}



/* Save the database to its file */
static void save_database(void) {
	FILE *f;

	f = xfopen(db_filename, "w");
	if (lsm_write_database(f, &db) == -1) {
		move(LINES-1, 0);
		standout();
		printw("Error saving database to `%s'", db_filename);
		standend();
		refresh();
	}
	xfclose(f);
}
