// Contents ---------------------------------------------------------------
//
//   fingerd.c  -- a Windows Socket Finger Daemon
//
//   Version 1.00
//
//   Copyright (C) Frederick W. Bent 1994
//   All rights reserved.
//
//
// Description
//
//      FINGERD is both the user interface and the network interface.
//	FINGERD is a server, its 'users' are FINGER clients, so the user
//	interface requirements of the daemon are rather simple.  FINGERD
//      simply updates its display with outbound buffer traffic.  We also
//      display winsock errors in the server display.
//
//      FINGERD uses WSAAsyncSelect() to receive the SOCKET_MESSAGE window
//      messages that notify FINGERD of pending client requests, causing
//	FINGERD to respond.  When an FD_ACCEPT is received, the socket is
//      added to the linked list of clients.
//
//      FINGERD speaks the finger protocol, and will reply to finger
//	(e.g., the MS-Windows 3.x winsock finger client).
//
// Ends -------------------------------------------------------------------

// History ----------------------------------------------------------------
//
// 6/28/94  1.0  Fred Bent	First release
//
// Ends -------------------------------------------------------------------

// Legal Stuff ------------------------------------------------------------
//
// Permission to use, modify, and distribute this software and its
// documentation for any purpose and without fee is hereby granted,
// provided that the above copyright notice appears in all copies and
// that both that copyright notice and this permission notice appear in
// supporting documentation.  The author makes no claims as to the
// suitability of this software and documentation for any purpose.
//
// Legal Stuff Ends -------------------------------------------------------

// Interface Dependencies -------------------------------------------------

#define STRICT	1

#include <windows.h>
#include <windowsx.h>
#include <ctl3d.h>
#include <winsock.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <dir.h>
#include "fingerd.h"
#include "file.h"
#include "strlib.h"

#pragma	warn -par

// End Interface Dependencies ---------------------------------------------

// External Declarations --------------------------------------------------

extern  SOCKET  sListen;          // awaits connection requests
extern	HWND	hwndNetwork;      // our invisible network "window"
extern  char	szNetworkClass[]; // class name of network window

// End External Declarations ----------------------------------------------


// Server Implementation --------------------------------------------------

    HWND	hwndMain;      // handle of main window
//    HWND	hwndEdit;
    HWND	hScroll;
    HINSTANCE	hInst;         // our main instance

    char szAppName[] = APP_NAME;	// the name of the application
    char szAppClass[] = APP_CLASS;	// the class name of the main window
    char szMainBuffer[SENDBUFLEN];	// transfer buffer holds in/outbound text
    char szDispBuffer[SENDBUFLEN];	//
    char szLocalFile[MAX_PATH];	    	// filename
    char szFileDir[MAX_PATH];		// file path
    char szHelpFileName[MAX_PATH];    /* Help file name*/

    int  iDebugLevel = DEBUG_OFF;	// level of logging to perform
    BOOL bOnlySnark = FALSE;		// only allow snark file


#ifdef OLD_DSP_LIST
    LINEITEM *pLineItems = NULL;           // ptr to display list LINEITEMS
    LINEITEM *pTopLine;                 // pts to topmost displayable LINEITEM
#else
    HLOCAL	pLineItems = NULL;
    HLOCAL	pTopLine;
#endif

    int      nLineItems = 0;            // number of items in display list
    int      nTopLine = 0;              // line number of topmost displayed line
    int      nClientLines;              // # of text lines in view
	
    int	CharY;

    LPCLIENT fingerClientsHead = NULL;	// linked list root

DECODEWORD frameMsgs[] =               // windows messages & handlers
{
   {WM_CREATE,	   DoCreate},
   {WM_ACTIVATE,   DoActivate},
   {WM_COMMAND,    DoCommand},
   {WM_PAINT,      DoPaint},
   {WM_CLOSE,      DoClose},
   {WM_DESTROY,    DoDestroy},
   {WM_SETFOCUS,	DoSetFocus},
   {WM_INITMENUPOPUP, DoInitMenuPopup},
   {WM_SIZE,	   DoSize},
   {WM_KEYDOWN,	   DoVScroll},
   {WM_VSCROLL,	   DoVScroll}
};

DECODEWORD menuItems[] =            // menu items & associated handlers
{
   {IDM_ABOUT,     DoMenuAbout},
   {IDM_OPTIONS,   DoMenuOptions},
   {IDM_HELP,	   DoMenuHelp},
   {IDM_EXIT,	   DoMenuExit},
   {IDE_EDIT,	   DoEdit},
   {IDM_COPY,	   DoEditCopy},
   {IDM_CLEAR,	DoEditClear},
   {IDM_SELALL,	DoEditSelall},
   {0,		   NULL}
};

	VOID RelScroll(HWND hWnd, int nlines);
	VOID PosView(int nlines);
	VOID Repaint(VOID);
	VOID SetScroll(VOID);



// Main Function

	int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
		     LPSTR lpCmdLine, int nCmdShow)

// Summary ----------------------------------------------------------------
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		WSADATA	WSAData;	// windows sockets info return
		HACCEL	hAccel;
		MSG 	msg;		// holds current message
		int 	err;
		WORD	wVersionRequested;


		hInst = hInstance;           // save this instance handle

		// We can only have one server at a time :-)

		if (hPrevInstance != NULL) return(NULL);

#ifdef USE_CTL3D
		Ctl3dRegister(hInstance);
		Ctl3dAutoSubclass(hInstance);
#endif

		if (!(InitApp(hInstance) && InitInstance(hInstance, nCmdShow)))
		      return(FALSE);                         // can't start, so bail

		// setup global variables

		InitFinger(hInstance, INI_FILE);

		// connect to WinSock.DLL

		wVersionRequested = WSVERSION_REQ;
		err = WSAStartup(wVersionRequested, (LPWSADATA) &WSAData);

		if (err != 0)
		{
			MessageBeep(MB_ICONSTOP);
			MessageBox(hwndMain, WSErrorString(err), szAppName, MB_ICONSTOP | MB_OK);
			DestroyWindow(hwndMain);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif

			return(NULL);
		}

		if ( ( LOBYTE(WSAData.wVersion) < WSVERSION_MAJOR ) ||
		     ( LOBYTE(WSAData.wVersion) == WSVERSION_MAJOR &&
		       HIBYTE(WSAData.wVersion) < WSVERSION_MINOR) )
		{
                	MessageBeep(MB_ICONSTOP);
			MessageBox(hwndMain, WSErrorString(IDS_BAD_VERSION), szAppName, MB_ICONSTOP | MB_OK);
			DestroyWindow(hwndMain);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			WSACleanup();
			return(NULL);
		}

		// OK, now start the server

		if ( netInit() ) {
			MessageBox(hwndMain, WSErrorString(WSAGetLastError()), szAppName, MB_ICONSTOP | MB_OK);
			DestroyWindow(hwndMain);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			WSACleanup();
			return(NULL);
		}

		hAccel = LoadAccelerators(hInstance, szAppClass);
		if ( hAccel == NULL )
		{
                	MessageBeep(MB_ICONSTOP);
			MessageBox(hwndMain, WSErrorString(IDS_INITIALIZE), szAppName, MB_ICONSTOP | MB_OK );
			DestroyWindow(hwndMain);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			WSACleanup();
			return(NULL);
		}

		// loop until WM_QUIT

		while ( GetMessage(&msg, NULL, 0, 0) )
		{
		    if (!TranslateAccelerator(hwndMain, hAccel, &msg))
                    {
		      TranslateMessage(&msg);
		      DispatchMessage(&msg);
		    }
		}

		WSACleanup();
#ifdef USE_CTL3D
		Ctl3dUnregister(hInstance);
#endif
		return(msg.wParam);
	}


// Function

	BOOL InitApp( HINSTANCE hInstance )

// Summary ----------------------------------------------------------------
//
//      Initialization for all instances of the application.  This
//	registers the main window class.
//
//
// Parameters
//
//      hInstance
//
// Return
//
//	BOOL	Returns FALSE if error
//
// Ends -------------------------------------------------------------------

	{
		WNDCLASS    wndclass;

		/*
		 * Has it's own DC because of the font selection
		 */
		wndclass.style         = CS_HREDRAW | CS_VREDRAW;	// CS_OWNDC
		wndclass.lpfnWndProc   = FrameWndProc;
		wndclass.cbClsExtra    = 0;
		wndclass.cbWndExtra    = 0;
		wndclass.hInstance     = hInstance;
		wndclass.hIcon         = LoadIcon(hInstance, "Icon");
		wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW);
#ifndef USE_CTL3D
		wndclass.hbrBackground = CreateSolidBrush( GetSysColor(COLOR_WINDOW) );
#else
		wndclass.hbrBackground = CreateSolidBrush( GetSysColor(COLOR_BTNFACE) );
#endif
		wndclass.lpszMenuName  = "Menu";
		wndclass.lpszClassName = szAppClass;

		return(RegisterClass(&wndclass));
	}


// Function

	BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)

// Summary ----------------------------------------------------------------
//
//      Initializes this specific instance of the application, and creates
//	the main windows.  The handle to the main window is stored in the
//	global variable hwndMain.  The class name is gotten from the
//	global variable szAppName.
//
//
// Parameters
//
//      hInstance
//
//      nCmdShow
//
// Returns
//
//	BOOL	Returns FALSE if there was an error during initialization.
//
// Ends -------------------------------------------------------------------

	{
		hwndMain = CreateWindow(  szAppClass, szAppName,
			   WS_OVERLAPPEDWINDOW | WS_VSCROLL,
			   CW_USEDEFAULT, CW_USEDEFAULT,
			   CW_USEDEFAULT, CW_USEDEFAULT,
			   NULL, NULL, hInstance, NULL);
		if (hwndMain == NULL) return(FALSE);

		SizeWindow(hwndMain);                     // set window & char sizes
		ShowWindow(hwndMain, nCmdShow);
		UpdateWindow(hwndMain);

		return(TRUE);                             // connections
	}


// Function

	BOOL InitFinger(HINSTANCE hInst, PSTR pFilename)

// Summary ----------------------------------------------------------------
//
//	This function is called to initialize some global variables
//	using an .INI file.
//
// Parameters
//
//	pFilename	ASCIIZ string of the .INI file to look for.
//
// Ends -------------------------------------------------------------------


	{
	    LPSTR	lpStr;

	    lstrcpy((LPSTR)szNetworkClass, (LPCSTR) CLASS_NETWORK);

	    GetModuleFileName( hInst, szFileDir, sizeof(szFileDir) );
	    lpStr = lstrrchr( szFileDir, '\\' );
	    *lpStr = '\0';

	    lstrcpy(szLocalFile, szFileDir);
	    lstrcat(szLocalFile, DEFAULTFILE );


	    MakeHelpPathName(hInst, (LPSTR)szHelpFileName);

	    // Get the default file name

	    GetPrivateProfileString( SECTION_MAIN
				   , ENTRY_DEFAULTFILE
				   , (LPCSTR) DEFAULTFILE
				   , (LPSTR) szLocalFile
				   , sizeof(szLocalFile)
				   , (LPCSTR) pFilename);

//	    getcwd( szFileDir, sizeof(szFileDir) );

	    GetPrivateProfileString( SECTION_MAIN
				   , ENTRY_DIR
				   , (LPCSTR) szFileDir
				   , (LPSTR) szFileDir
				   , sizeof(szFileDir)
				   , (LPCSTR) pFilename);

            // Get the debugging level stuff

	    iDebugLevel = GetPrivateProfileInt( SECTION_MAIN
				, ENTRY_DEBUG
				, DEBUG_OFF
				, (LPCSTR) pFilename);

	    bOnlySnark = GetPrivateProfileInt( SECTION_MAIN
				, ENTRY_ONLYSNARK
				, FALSE
				, (LPCSTR) pFilename);

	    fingerClientsHead = NULL;
//	    iDispBufLen = 0;
//	    szDispBuffer[0] = '\0';
	    szMainBuffer[0] = '\0';

//	    bHaveInfo = FALSE;
//	    nCurrent = 0;

	    OpenDisplayList();		// start a new display list
	    pLineItems = NULL;           // ptr to display list LINEITEMS
	    nLineItems = 0;            // number of items in display list

	    return(FALSE);
	}


// Function

	void SizeWindow(HWND hWnd)

// Summary ----------------------------------------------------------------
//
//	Sets the window's character and external dimesions.
//
// Parameters
//
//	hWnd	The handle of the window.
//
// Ends -------------------------------------------------------------------

	{
	   HDC hdc;
	   TEXTMETRIC tm;
	   RECT rect;
	   int cyChar, cxChar;

	   hdc = GetDC(hWnd);
	   SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
	   GetTextMetrics(hdc, &tm);
	   cyChar = tm.tmHeight; // + tm.tmExternalLeading;
	   CharY = cyChar;
	   cxChar = tm.tmAveCharWidth;
	   ReleaseDC(hWnd, hdc);

	   // set initial window width & height in chars
	   GetWindowRect(hWnd, &rect);
	   MoveWindow( hWnd
	   	     , rect.left, rect.top
		     , 80 * cxChar + GetSystemMetrics(SM_CXVSCROLL)
		     , 24 * cyChar + GetSystemMetrics(SM_CYCAPTION)
				  + GetSystemMetrics(SM_CYMENU)
		     , FALSE);
	}


// Function

	void UpdateDisplay(LPSTR szBuffer, int iLen)

// Summary ----------------------------------------------------------------
//
//      Prepares to display the string in szBuffer, so we store the size,
//	and force a repaint by invalidating the main window.
//
// Parameters
//
//      szBuffer	The string to be displayed in the server window.
//
//	len		The length of the string to be displayed.
//
// Return
//
//	BOOL	Returns FALSE if error
//
// Ends -------------------------------------------------------------------

	{
		int 	i;

		if ( iLen == 0 ) return;

		lstrcpyn((LPSTR)szDispBuffer, szBuffer, (iLen+1));

		PushChars(szDispBuffer, iLen);		  // add the buffer
		GetDisplayList(&pLineItems, &nLineItems); // update the pointers

		i = FALSE;
		while ( nLineItems > MAX_DISP_LINES )
		{
                	i = TRUE;
                	RemoveFirstDisplayLine();
			GetDisplayList(&pLineItems, &nLineItems); // get new one
			nTopLine--;
			if (nTopLine < 0) nTopLine = 0;
		}

		if ( i )
		{
			PosView(0);
			SetScroll();
                }
	}



// WindowProc Function

	LRESULT CALLBACK FrameWndProc(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//	The callback function of the network window.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Returns
//
//	BOOL	Returns FALSE when it has processed the message, otherwise
//		returns the result of DefWindowProc().
//
// Ends -------------------------------------------------------------------

	{
		int i;

		for (i = 0; i < dim(frameMsgs); i++)
		{
			if (wMsg == frameMsgs[i].Code)
				return(*frameMsgs[i].Fxn)(hWnd, wMsg, wParam, lParam);
		}

		/* If not handled, then use default... */

 		return(DefWindowProc(hWnd, wMsg, wParam, lParam));
	}



// Window Function

	LRESULT DoCommand(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Demultiplexes WM_COMMAND messages resulting from menu selections,
//	and routes to corresponding menu item handler.  Sends back any
//	unrecognized messages to windows.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		int i;

		for (i = 0; i < dim(menuItems); i++)
		{
			if (wParam == menuItems[i].Code)
				return(*menuItems[i].Fxn)(hWnd, wMsg, wParam, lParam);
		}

		return(DefWindowProc(hWnd, wMsg, wParam, lParam));
	}

// Window Function


	LRESULT DoCreate(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to WM_CLOSE message by refusing pending client connects
// 	and terminating connections in progress, then kills window.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Returns
//
//	LRESULT	Returns zero to create the window.
//
// Ends -------------------------------------------------------------------

	{
		HTASK	hTask;

		hTask = GetCurrentTask();
		fingerLog(DEBUG_OFF | LOG_TIME, IDS_DAEMON_START, hTask);
//		nCurrent = 0;
//		SetScrollRange(hScroll, SB_VERT, 0, 0, FALSE);
//		SetScrollPos(hScroll, SB_VERT, 0, FALSE);
//		hwndEdit = CreateWindow("edit", NULL
//			, WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL
//			  | ES_LEFT | ES_MULTILINE | ES_AUTOHSCROLL | ES_AUTOVSCROLL
//			, 0, 0
//			, 0, 0
//			, hWnd
//			, (HMENU) IDE_EDIT	/* handle of menu or child control id */
//			, hInst
//			, NULL );
//		SendMessage(hwndEdit, EM_LIMITTEXT, 32000, 0L);

                return 0L;
        }


// Window Function

	LRESULT DoClose(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to WM_CLOSE message by refusing pending client connects
// 	and terminating connections in progress, then kills window.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		LPCLIENT	lpClient;
		char	szBuffer[MAXLINE];

                int	result;

		lpClient = fingerClientsHead->lpNextClient;

		if ( lpClient != NULL )
		{
			LoadString(hInst, IDS_EXIT_ACTIVE, (LPSTR)szBuffer, MAXLINE);

			MessageBeep(MB_ICONEXCLAMATION);
			result = MessageBox(hWnd, (LPSTR) szBuffer, szAppName,
				MB_ICONEXCLAMATION | MB_OKCANCEL | MB_DEFBUTTON2 );
		}
		else
                {
			LoadString(hInst, IDS_EXITING, (LPSTR)szBuffer, MAXLINE);
			 result = MessageBox(hWnd, szBuffer, szAppName,
				MB_ICONQUESTION | MB_YESNO);
		}

                if ((result == IDYES) || (result == IDOK))
		{
			netClose();
			DestroyWindow(hWnd);
		}
		return(FALSE);
	}


// Window Function

	LRESULT DoDestroy(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_DESTROY message by posting a WM_QUIT message to
//	the task's win queue, which causes the main translate & dispatch
//	loop to exit, and the app to terminate.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		CloseDisplayList();			  // stop old display
		FreeDisplayList();
		DestroyStrings();
		PostQuitMessage(0);
		return(FALSE);
	}



// Window Function

	LRESULT DoSetFocus(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_DESTROY message by posting a WM_QUIT message to
//	the task's win queue, which causes the main translate & dispatch
//	loop to exit, and the app to terminate.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
//		SetFocus(hwndEdit);
		return(FALSE);
	}


// Window Function

	LRESULT DoSize(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_DESTROY message by posting a WM_QUIT message to
//	the task's win queue, which causes the main translate & dispatch
//	loop to exit, and the app to terminate.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		int	cyClient;

		cyClient = HIWORD(lParam);

		nClientLines = max(1, cyClient / CharY);

		PosView(0);
		SetScroll();

		return(FALSE);
	}


// Window Function

	LRESULT DoPaint(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		HDC hdc;
		PAINTSTRUCT ps;
		int 	i;
#ifdef OLD_DSP_LIST
		LINEITEM	*pline;           // pts to topmost displayable LINEITEM
#else
		HLOCAL		pline;
#endif


		hdc = BeginPaint(hWnd, &ps);

		SelectObject(hdc, GetStockObject(SYSTEM_FIXED_FONT));
#ifndef USE_CTL3D
		SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
		SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
#else
		SetTextColor(hdc, GetSysColor(COLOR_BTNTEXT));
		SetBkColor(hdc, GetSysColor(COLOR_BTNFACE));
#endif

		pline = pTopLine;

		for (i = 0; i <= nClientLines; i++)
		{
			if (pline != NULL)
			{
#ifdef	OLD_DSP_LIST
				TextOut(hdc, 0, i * CharY, (LPSTR)pline->sztext, pline->len);
				pline = pline->next;
#else
				PSTR	sztext;
				LINEITEM *p;
                                HLOCAL	hNext;

				p = LocalLock(pline);
				hNext = p->next;
				if ( p->hText != NULL )
                                {
					sztext = LocalLock(p->hText);

					TextOut(hdc, 0, i * CharY, (LPSTR)sztext, p->len);

					LocalUnlock(p->hText);
				}
				LocalUnlock(pline);
				pline = hNext;
#endif
			} else
				break;
		}

		EndPaint(hWnd, &ps);

		return(FALSE);
	}


// Window Function

	LRESULT DoEdit(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
//		if (LOWORD(lParam))
//		{
//			switch (HIWORD(lParam))
//			{
//			case EN_UPDATE :
//				return(FALSE);
//			case EN_ERRSPACE:
//				MessageBeep(MB_ICONSTOP);
//				MessageBox(hWnd, "Edit control out of space."
//					, szAppName, MB_OK | MB_ICONSTOP );
//				/* Clear entire contents of edit control */
//				SendMessage(hwndEdit, EM_SETSEL, 0, MAKELONG(0,-1));
//				SendMessage(hwndEdit, WM_CLEAR, 0, (LPARAM) 0L);
//				return(FALSE);
//			}
//		}
		return(TRUE);
	}




// Window Function

	LONG DoActivate(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoActivate -- grabs the keyboard focus whenever our deminimized window
// is activated.  This is so we can respond to VK_HOME, VK_END, etc.
// virtual keys for scrolling. HIWORD(lParam) is TRUE for minimized, while
// LOWORD(wParam) is FALSE for activation message.
// 
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		if (!HIWORD(lParam) && LOWORD(wParam))
			SetFocus(hwndMain);

		return FALSE;
	}


// Window Function

	LRESULT DoInitMenuPopup(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		LONG	lSelect;
                WORD	wEnabled;

		/*
		 * Is this the Edit popup?
		 */
		if ( lParam == 1 )
		{
//			lSelect = SendMessage(hwndEdit, EM_GETSEL, 0, 0L);

//			if ( HIWORD(lSelect) == LOWORD(lSelect))
//				wEnabled = MF_GRAYED;
//			else
//				wEnabled = MF_ENABLED;

			GetDisplayList(&pLineItems, &nLineItems); // list and get new one

			if ( nLineItems == 0 )
				wEnabled = MF_GRAYED;
			else
				wEnabled = MF_ENABLED;

			EnableMenuItem((HMENU) wParam, IDM_COPY, wEnabled);
			EnableMenuItem((HMENU) wParam, IDM_CLEAR, wEnabled );
		}
		return(FALSE);
	}


// Window Function

	LRESULT DoEditCopy(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		HGLOBAL hGlobal;

		hGlobal = CopyDisplayList();
		if (hGlobal == NULL)
		{
			MessageBox(hWnd, "Could not allocate global memory buffer!", szAppName, MB_ICONEXCLAMATION | MB_OK );
		} else {
			OpenClipboard(hWnd);
			EmptyClipboard();
			SetClipboardData(CF_TEXT, hGlobal);
			CloseClipboard();
                }

		return(FALSE);
	}



// Window Function

	LRESULT DoEditClear(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
		CloseDisplayList();	// stop old display
		FreeDisplayList();	// dispose old display

		pLineItems = NULL;
		nLineItems = 0;

		OpenDisplayList();	// start new display

		PosView(0);             // position view to top
		SetScroll();            // rescale (or delete)
		Repaint();              // scrollbar & force a repaint

//		SendMessage(hwndEdit, WM_CLEAR, 0, 0L);
		return(FALSE);
	}


	void	FingerFinished(void)
	{
		GetDisplayList(&pLineItems, &nLineItems); // list and get new one

		PosView(0);                               // position view to top
		SetScroll();                              // rescale (or delete)
		Repaint();                                // scrollbar & force a repaint

	}



// Window Function

	LRESULT DoEditSelall(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Responds to a WM_PAINT message by painting the client window with
//	the contents of the transfer buffer.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{
//		SendMessage(hwndEdit, EM_SETSEL, 0, MAKELONG(0, 32767));
		return(FALSE);
	}



// Callback function

	BOOL CALLBACK PaintStrCallback(LPSTR lpString, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//	Called by the string library to add each string to the list box
//
// Parameters
//
//	lpString	A far pointer to the string.
//
//	lParam		Argument to the function.
//
// Returns
//
//	BOOL		True if the string was added to the list box
//
// Ends -------------------------------------------------------------------


	{
		HWND	hDlg;
		UINT	nButton;
		LONG	lResult;

		hDlg = (HWND) HIWORD(lParam);
		nButton = LOWORD(lParam);
		lResult = SendDlgItemMessage(hDlg, nButton, LB_ADDSTRING, 0, (LPARAM) lpString);

		return TRUE;
	}


// Function

	void DisplayWSError(PSTR szString, ...)

// Summary ----------------------------------------------------------------
//
//	Display a WinSock error message in the server window.
//
// Parameters
//
//      szString	An ASCIIZ string
//
//	...		Possible arguments of the string
//
// Return
//
//	BOOL	Returns FALSE if error
//
// Ends -------------------------------------------------------------------

	{
	   va_list	vaArgs;
	   char		szBuf[MAXLINE];
	   char		szTemp[MAXLINE];
	   int err = WSAGetLastError();

	   szBuf[0] = '\0';
	   va_start(vaArgs, szString);
	   if ( szString != NULL )
	   {
		   wvsprintf((LPSTR)szBuf, szString, vaArgs);
		   szMainBuffer[0] = '\0';

		   if ( szString[0] == '\0' )
		   {
			strncpy(szBuf, WSErrorString(err), sizeof(szBuf));
		   }
	   } else { strncpy(szBuf, WSErrorString(err), sizeof(szBuf)); }
	   wsprintf((LPSTR)szTemp, "Windows Sockets error %d: %s", err, (LPCSTR) szBuf);
	   UpdateDisplay((LPSTR)szTemp, strlen(szTemp));
		
	   va_end(vaArgs);
	}


// Function

	PSTR WSErrorString(UINT ResourceID)

// Summary ----------------------------------------------------------------
//
//	Translates WinSock error number into an appropriate string.
//
//
// Parameters
//
//      ResourceID
//
//	The resource number of the string to be printed.
//
// Return
//
//	PSTR	A pointer to the string that is to be printed.
//
// Ends -------------------------------------------------------------------

	{
		int i;
		static char szBuffer[MAXLINE];

		if ( !LoadString(hInst, ResourceID, (LPSTR)szBuffer, MAXLINE) )
			wsprintf( (LPSTR) szBuffer, "Windows Sockets error %d", ResourceID );

		return(szBuffer);
	}


// Debugging function

	void DebugOut(LPSTR lps, ...)

// Summary ----------------------------------------------------------------
//
//	Sends a string to the AUX device (or debugger).
//
//
// Parameters
//
//      lps
//
//	The far pointer to the format string
//
// Return
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
		char	szBuf[MAXLINE];
		va_list vaArgs;
		LPSTR	lpArgs;

		va_start(vaArgs, lps);
		lpArgs = va_arg(vaArgs, LPSTR);
		if (lpArgs != NULL)
		{
			wvsprintf((LPSTR)szBuf, lps, lpArgs);
			OutputDebugString(szBuf);
		}
		va_end(vaArgs);
	}


// Function

	void	fingerLog(int iLevel, int ResourceID, ...)

// Summary ----------------------------------------------------------------
//
//	This function places a string into the string log.
//
// Parameters
//
//	iLevel		The level of the logging
//
//	ResourceID	The ID of the string template to use
//
// Return
//
//	nothing
//
// Ends -------------------------------------------------------------------

	{
		char	szString[MAXLINE];
		char	szLine[MAXLINE];
		char	szTime[30];
		LPSTR	lpArg1, lpArg2;
		va_list	vaArgs;
		DWORD	dwCount;

//		if ( iLevel < iDebugLevel ) return;

		va_start( vaArgs, ResourceID );

		if ( !LoadString(hInst, ResourceID, (LPSTR) szLine, MAXLINE) ) return;

		wvsprintf((LPSTR)szString, szLine, vaArgs);

		if ( iLevel & LOG_TIME )
		{
			netGetTimeAndDate(szTime, sizeof(szTime));
			wsprintf((LPSTR)szLine, "%s> %s", (LPSTR)szTime, (LPSTR)szString);
		} else {
			lstrcpy((LPSTR)szLine,(LPSTR)szString);
		}

		if ( !AddString(szLine) )
		{
			DeleteFirstString();
			AddString(szLine);
		}

		return;
	}


// Function

	BOOL	VerifyDir(PSTR szFilePath)

// Summary ----------------------------------------------------------------
//
//	This function verifies that the specified path does exist.
//
// Parameters
//
//	szFilePath
//
//	The path to be verified.
//
// Return
//
//	BOOL
//
// Ends -------------------------------------------------------------------

	{
		char	szBuffer[MAX_PATH] = "\0";
                int	stat;
		int	drive = -1;
		HLOCAL	hOem;
		PSTR	szOemPath;


                /* Dynamic Allocation */

		hOem = LocalAlloc(LHND, MAX_PATH);
		if ( hOem == NULL ) return(FALSE);

		szOemPath = LocalLock(hOem);
		if ( szOemPath == NULL ) { LocalFree(hOem); return(FALSE); }


		if ( szFilePath[1] == ':' )
                {
			drive = szFilePath[0] - 'A';
                        drive++;
		} else {
			drive = getdisk();
		}
                szBuffer[0] = (char) drive + 'A';
		szBuffer[1] = ':';
		szBuffer[2] = '\\';

		/*
		 * Drive 0 = default, A = 1, B = 2, C = 3, etc.
		 */
		getcurdir(drive, &szBuffer[3]);

		/* MS-DOS uses the OEM character set, so convert */

		AnsiToOem(szFilePath, szOemPath);

		stat = chdir( szOemPath );
		chdir( szBuffer );

		LocalUnlock(hOem);
		LocalFree(hOem);

		if ( stat == -1 ) return TRUE;
		else return FALSE;

	}


// Function

	void MakeHelpPathName(HINSTANCE hInst, LPSTR szFileName)

// Summary ----------------------------------------------------------------
//
//	This function will attempt to generate the full path name of
//	the help file from the path of the executable.  It assumes that
//	the .HLP file is in the same directory as the executable.
//
// Parameters
//
//
//	hInst
//
//	Identifies the module or the instance of the program.
//
//	szFileName
//
//	The buffer to accept the name of the help file.
//
//
// Return
//
//	nothing
//
// Ends -------------------------------------------------------------------

	{
		LPSTR   lpcFileName;
		int     nFileNameLen;
		const char	szHelpFile[] = HELP_FILE;

		nFileNameLen = GetModuleFileName(hInst, szFileName, MAX_PATH);
		lpcFileName = szFileName + nFileNameLen;

		/* Start looking in reverse for a backslash or full colon */
		while (lpcFileName > szFileName)
		{
			if ((*lpcFileName == '\\') || (*lpcFileName == ':'))
			{
				*(++lpcFileName) = '\0';
		break;
			}
			nFileNameLen--;
			lpcFileName--;
		}

		/* Is there enough room to append the help file name? */
		if ( ( nFileNameLen + lstrlen((LPSTR)szHelpFile) ) < MAX_PATH)
		{
			lstrcat( szFileName, (LPSTR) szHelpFile );
		}
		else
		{
			lstrcat( szFileName, "?" );
		}
	
		return;
	}


// Function

	VOID PosView(int nlines)

// Summary ----------------------------------------------------------------
//
// PosView -- repositions the view relative to the top of the display list.
// The view is a logical "window" onto the display list.  The frame window's
// client area is painted with the view's contents.
//
//
// Paramters
//
//	nlines	The position in the display
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
#ifdef OLD_DSP_LIST
   LINEITEM *pline;
#else
   HLOCAL	pline;
   HLOCAL	hNext;
   LINEITEM *p;
#endif
   int i;

   pline = pLineItems;              // root of LINEITEM list

   for (i = 0; i < nlines; i++)
   {
      if (pline == NULL)
	 break;
      else {
#ifdef OLD_DSP_LIST
	 pline = pline->next;
#else
	 p = LocalLock(pline);
	 hNext = p->next;
	 LocalUnlock(pline);
	 pline = hNext;
#endif
      }

   }

   pTopLine = pline;                // ptr to LINEITEM in topmost view line
   nTopLine =+ nlines;              // offset of topmost view line
}


// Function

	VOID Repaint(VOID)

// Summary ----------------------------------------------------------------
//
// Repaint -- force refresh of client window.
// 
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
		InvalidateRect(hwndMain, NULL, TRUE);
	}


// Function

	VOID SetScroll(VOID)

// Summary ----------------------------------------------------------------
//
// SetScroll -- sets the vertical scroll range to the length of the display
// list.  The Scrollbar disappears when the list fits within the view.
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
   if (nLineItems > nClientLines)
      SetScrollRange(hwndMain, SB_VERT, 0, nLineItems - nClientLines, FALSE);
   else
      SetScrollRange(hwndMain, SB_VERT, 0, 0, FALSE);

   SetScrollPos(hwndMain, SB_VERT, nTopLine, TRUE);
}


// number of lines below the bottom of the view.
#define NLINESBELOW (nLineItems - nTopLine - nClientLines)
#define THUMBPOS  LOWORD(lParam)    // Win 16

// Windows Function

	LRESULT DoVScroll(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoVScroll -- process WM_VSCROLL & WM_KEYDOWN for main window.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

{
   switch (LOWORD(wParam))
   {
      case VK_HOME:
      case SB_TOP:
         if (nTopLine > 0) RelScroll(hWnd, -nTopLine);
         break;

      case VK_END:
      case SB_BOTTOM:
         if (NLINESBELOW) RelScroll(hWnd, NLINESBELOW);
	 break;

      case VK_PRIOR:
      case SB_PAGEUP:
	 if (nTopLine > 0) RelScroll(hWnd, max(-nClientLines, -nTopLine));
         break;

      case VK_NEXT:
      case SB_PAGEDOWN:
         if (NLINESBELOW) RelScroll(hWnd, min(nClientLines, NLINESBELOW));
         break;

      case VK_UP:
      case SB_LINEUP:
         if (nTopLine > 0) RelScroll(hWnd, -1);
         break;

      case VK_DOWN:
      case SB_LINEDOWN:
         if (NLINESBELOW) RelScroll(hWnd, 1);
         break;

      case SB_THUMBTRACK:
         RelScroll(hWnd, THUMBPOS - nTopLine);
	 break;
   }

   SetScrollPos(hwndMain, SB_VERT, nTopLine, TRUE);
   return(FALSE);
}


// Function

	VOID RelScroll(HWND hWnd, int nlines)

// Summary ----------------------------------------------------------------
//
// RelScroll -- scroll up/down nlines from present position
//
//
// Parameters
//
//	hWnd	Handle of the window
//
//	nlines	The number of lines to scroll
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
	PosView(nTopLine + nlines);
	ScrollWindow(hWnd, 0, (-nlines * CharY), NULL, NULL);
	UpdateWindow(hWnd);
}
