/******************************************************************************

    TCPINFO - display configuration info to the screen

    Copyright (C) 1991, University of Waterloo
    portions Copyright (C) 1990, National Center for Supercomputer Applications

    This program is free software; you can redistribute it and/or modify
    it, but you may not sell it.

    This program is distributed in the hope that it will be useful,
    but without any warranty; without even the implied warranty of
    merchantability or fitness for a particular purpose.

        Erick Engelke                   or via E-Mail
        Faculty of Engineering
        University of Waterloo          Erick@development.watstar.uwaterloo.ca
        200 University Ave.,
        Waterloo, Ont., Canada
        N2L 3G1

******************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <tcp.h>
#include "com.h"
#include "litecomm.h"

extern longword set_timeout();

struct com_info coms[5];
int currentport = 0;

char *testmsg = "This is a test of the serial port";

static void (*other)();
static void mine(char *name, char *value)
{
    int temp;
    struct com_info *c;

    c = &coms[ currentport ];

    if ( !strcmp( name, "COM.PORT" )) {
	temp = atoi( value );
	if (coms[temp].active)
	    printf("Warning: com port %u was defined more than once\n", temp );
	currentport = temp;
	coms[temp].active = 1;
    } else if ( !strcmp(name,"COM.BAUD" )) {
	if (temp = atoi( value ))
	    c->baud = temp;
	else
	    printf("Warning: com port %u given baud %u\n", currentport, temp );
    } else if ( !strcmp(name,"COM.PARITY")) {
	if (!stricmp(value,"EVEN"))
	    c->parity = EPARITY;
	else if (!stricmp(value,"ODD"))
	    c->parity = OPARITY;
	else if (!stricmp(value,"NONE"))
	    c->parity = NPARITY;
	else if (!stricmp(value,"MARK"))
	    c->parity = MPARITY;
	else if (!stricmp(value,"SPACE"))
	    c->parity = SPARITY;
	else
	    printf("Warning: incorrect parity statement '%s'\n",value);
    } else if (!strcmp(name,"COM.BITS")) {
	c->bits = atoi(value) - 5;
    } else if (!strcmp(name,"COM.STOP")) {
	c->stops = (atoi(value) - 1) * 4;
    } else if (!strcmp(name,"COM.HANDSHAKE")) {
	if (!stricmp(value,"XON")) c->handshake = FALSE;
	if (!stricmp(value,"MODEM")) c->handshake = TRUE;
    } else if (!strcmp(name,"COM.TCPPORT")) {
	c->tcpport = atoi( value );
    } else if (!strcmp(name,"COM.PASSWORD")) {
	strncpy(c->password, value, PASSLENGTH );
	c->password[ PASSLENGTH-1]=0;
    } else if (!strcmp(name,"COM.MODE" )) {
	if (!stricmp(value,"RS232")) c->mode = RS232;
	if (!stricmp(value,"TEST")) c->mode = TEST;
	if (!stricmp(value,"HAYESMODEM")) c->mode = HAYESMODEM;
    } else if (!strcmp(name,"COM.DELAY" )) {
	c->delay = atoi( value );
	c->tickdelay = (c->delay * 9) / 5;
	if (c->delay) c->tickdelay ++;
    } else
	if (other)
	    (*other)( name, value );
}


char *parities[] = { "NONE", "ODD", "ERR", "EVEN", "ERR","MARK", "ERR","SPACE" };
char *modes[] = {"RS232","TEST","HAYESMODEM" };

show_port( int port )
{
    struct com_info *c;

    c = &coms[port];

    printf("    MODE     : %s\n", modes[ c->mode ]);
    printf("    PASSWORD : %s\n", (*c->password) ? "ON" : "OFF");
    printf("    TCP PORT : %u\n", c->tcpport );
    printf("    DELAY    : %0u.%0u seconds\n", c->delay / 10, c->delay % 10 );
printf("delay = %u\n",c->tickdelay);

    printf("    HANDSHAKE: %s\n", (c->handshake) ? "DTR/RTS" : "XON/XOFF");
    printf("    BAUD     : %u\n", c->baud);
    printf("    PARITY   : %s\n", parities[ c->parity / 8 ]);
    printf("    BITS     : %u\n", c->bits + 5);
    printf("    STOP     : %u\n", (c->stops / 4) + 1);
}

interactive( int port, struct com_info *c )
{
    char ch;

    puts("( ^Z to leave interactive mode )");

    do {
	if (kbhit()) {
	    ch = getch();
	    if (ch == 26) return;
	    lc_put(port,ch );
	}
	if (lc_icnt(port)) {
	    ch = lc_getw(port);
	    if (ch == -1)
		puts("ERROR received from port");
	    else putch( ch );
	}
    } while(1);
}

init_coms()
{
    struct com_info *c;
    int status, i;


    for ( i = 1; i < 5; ++i ) {
	c = &coms[i];
	if ( c->active ) {
	    c->status = INIT;
	    status = comm_opn( i , c->baud, c->parity, c->bits,
		c->stops, COMSIZIN, COMSIZOUT, FALSE );
	    if ( status == -1 ) {
		printf("Error: initializing COM %u\n", i );
		show_port( i );

		c->active = 0;
	    } else {
		lc_setmdm( i , DTR | RTS );
		lc_xoff( i, (c->handshake == HS_XOFF)? TRUE : FALSE );
		printf("COM %u : initialized\n", i);
		show_port( i );
	    }

	    if (c->mode == TEST ) {
		lc_puts(i,testmsg, strlen( testmsg ));
		puts("Printed message... now interactive ...");
		interactive(i, &coms[i] );
		puts("Continuing");
	    }
	}
    }
}


poll_read( int port, struct com_info *c)
{
    int count;
    char buffer[ 1024 ];

    if ( !chk_timeout( c->lastxmit ))
	return;

    if (count = lc_icnt( port )) {
	/* read the port and write the data to the socket */
	if (count > sizeof( buffer ))
	    count = sizeof( buffer );
	if (count > sock_tbleft( & c->sock ))
	    count = sock_tbleft( &c->sock);

	/* if we can get anything... */
	if ( count ) {
	    c->lastxmit = set_timeout( 0 ) + (longword)(c->tickdelay);
	    lc_gets( port, buffer, count );
	    sock_fastwrite( &c->sock, buffer, count );
printf("writing %u bytes on port %u\n" , count, port );
    }
    }
}

poll_write( int port, struct com_info *c )
{
    int count, temp;
    char buffer[ 1024 ];

    if (count = sock_rbused( &c->sock )) {
        temp = COMSIZOUT - lc_ocnt( port );
        if (count > temp )
            count = temp;

        if ( count ) {
            sock_fastread( &c->sock, buffer, count );
            lc_puts( port, buffer, count );
        }
    }
}

chk_status( int port, struct com_info *c )
{
    if ( !tcp_tick( &c->sock )) {
	/* generate brk */
	sock_close( &c->sock );
	c->status = CLOSING;
    }

    if ( lc_gotbrk( port ) ) {
	sock_close( &c->sock );
	c->status = CLOSING;
    }
    return( c->status == CLOSING );
}


loop( int port, struct com_info *c )
{
    if (!c->active) return;

    switch ( c->status ) {
	case INIT :
		tcp_listen( &c->sock, c->tcpport, 0, 0, NULL, 0);
		printf( "COM %u : listening on tcp port %u\n",
		    port, c->tcpport );
		c->status = WAITING;
		break;
	case WAITING :
		tcp_tick( NULL );
		if (sock_established( &c->sock )) {
		    printf( "COM %u : incomming session\n", port);
		    c->status = INITCOM;
		}
		break;
	case INITCOM :
		switch ( c->mode ) {
		    case RS232 :
				lc_puts( port, "HELLO\n\r", 7 );
				c->status = ACTIVE;
				break;
		    case HAYESMODEM :
				/* must do dialing */
				c->status = ACTIVE;
				break;
		}
		break;
	case ACTIVE :
		if (!chk_status( port, c )) {
		    poll_read( port, c );
		    poll_write( port, c );
		}
		break;
	case CLOSING :
		/* clean up modem stuff */

		c->status = CLOSED;
		break;
	case CLOSED :
		if (!tcp_tick( &c->sock )) {
		    c->status = INIT;
		}
		break;
    }
}


main( int argc, char **argv)
{
    int i;

    dbuginit();
    printf("Reading Waterloo TCP configuration file.\n");

    memset( coms, 0, sizeof( coms ));

    other = usr_init;
    usr_init = mine;

    sock_init();

    init_coms();

    do {
	for ( i = 1; i < 5 ; ++i )
	    loop( i, &coms[i] );
	kbhit();
    } while ( 1 );

    exit( 0 );
}
