unit Jobbr;
interface
{$I DEF.INC}
uses
  WinProcs,
  WinTypes,
  {$IFNDEF WIN32}
  WinSock,
  {$ELSE}
  WSOCK32,
  {$ENDIF}
  Messages,
  WinFun,
  StrHlp,
  RTimer,
  Classes,
  SysUtils,
  Forms;

const
  MsgBase = WM_USER + 66;


type
  TErrorAction = (eaRelease, eaIgnore, eaRead);
  TPollType = (ptAsync, ptSync);
  TJobEvent = procedure (Job: Integer; JobEvent: Word) of object;
  TCacheRec =
    record
      LastHandle: Integer;
      LastPointer: Pointer;
    end;

  TJobArray = class
    constructor Create;
    destructor Destroy; override;
    function GetPointer(Key: Integer): Pointer;
    function GetPointerRaw(Index: Integer): Pointer;
    procedure Add(Value: Pointer);
    procedure Remove(Value: Pointer);
    function Count: Integer;
    public
      Cache: TCacheRec;
      Jobs: TList;
      property Pointers[Handle: Integer]: Pointer read GetPointer; default;
      property RawPointers[Index: Integer]: Pointer read GetPointerRaw;
  end;

  PJob = ^TJob;
  TJob = record
    ISocket: TSocket;
    JobEvent: TJobEvent;
    AsyncOpts: Integer;
    Timer: Boolean;
    TimeOut: Integer;
    ID: Integer;
    Tag: Integer;
    Handle: Integer;
  end;

  ETCPIPException = class(Exception);

  TTCPIPJob = class
    constructor Create;
    destructor Destroy; override;
    function CreateJob(FJob: TJobEvent; SType, SProt, SAsync: Integer): Integer;
    function CreateLocalJob(FJob: TJobEvent; SType, SProt, SAsync: Integer): Integer;
    function CreateJobIndirect(FJob: TJobEvent; SType, SProt, SAsync: Integer;
             Sckt: TSocket): Integer;
    function GetJobSocket(Job:Integer): TSocket;
    function ConnectJob(Job:Integer; const Host: String; Port: Word): Boolean;
    function ConnectJobImplicit(Job: Integer; Host: TIn_addr; Port: Word): Boolean;
    function ResolveAddress(const hname: String; var rsa: TIn_Addr): Boolean;
    function DeleteJob(Job: Integer): Integer;
    procedure SelectEvent(Job: Integer; Event: Word);
    function GetJobAsyncEvent(Job: integer): Integer;
    function AddrToString(const s: TIn_Addr): String;
    procedure SocketWindowProc(var Msg: TMessage);
    procedure ProcessTimer(idTimer: Word);
    function BindJob(Job: Integer; const Address: String; Port: Word): Boolean;
    function SetTimeOut(Job: Integer; TimeOut: Integer): Boolean;
    function GetJobTag(Job: Integer): Integer;
    procedure SetJobTag(Job: Integer; Tag: Word);
    function StringAddress(const tin: TIn_Addr): String;
    procedure RestoreEvents;
    function RestoreEvent(Job: Integer): Boolean;
    function GetError(Error :Word): String;
    function GetJobPort(Job: Integer): Integer;
    procedure SetError(error: word);
    function Execute: Boolean;
    function SendErrorStatus(Error: Word): TErrorAction;
    function ReadErrorStatus(Error: Word): TErrorAction;
    procedure CheckJobs(Sender: TObject; var Done: Boolean);

    private
      IPCache: TStringList;
      SockWin: Word;
      JobArray: TJobArray;
      NumHandle: Integer;
      PollType: TPollType;
      function IsValidJob(Job: Integer): Boolean;
  end;

var
  ply: LongInt;
  TCPExitProc: Pointer;
  InetJob: TTCPIPJob;
  wsInitialized: Boolean;
  wsInfo: TWSAData;

implementation

constructor TJobArray.Create;
begin
  inherited Create;
  Jobs:=TList.Create;
end;

destructor TJobArray.Destroy;
begin
  Jobs.Free;
  inherited Destroy;
end;

function TJobArray.GetPointer(Key: Integer): Pointer;
var
  n: Integer;
begin
  result:=nil;
  try
    if (Cache.LastHandle=Key) and (Cache.LastHandle<>0) then
    begin
      result:=Cache.LastPointer;
      Exit;
    end;
  except
  end;

  for n:=0 to Jobs.Count-1 do
  if PJob(Jobs[n])^.Handle=Key then
  begin
    result:=Jobs[n];
    break;
  end;
end;

procedure TJobArray.Add(Value: Pointer);
begin
  Jobs.Add(Value);
end;

procedure TJobArray.Remove(Value: Pointer);
begin
  Jobs.Remove(Value);
end;

function TJobArray.Count: Integer;
begin
  result:=Jobs.Count;
end;

function TJobArray.GetPointerRaw(Index: Integer): Pointer;
begin
  try
    result:=Jobs[Index];
  except
    result:=nil;
  end;
end;

function TTCPIPJob.SendErrorStatus(Error: Word): TErrorAction;
begin
  result:=eaRelease;
  case Error of
    WSAEWOULDBLOCK:
      result:=eaRead;
    WSAEINPROGRESS, WSAEINTR, WSAEFAULT, WSAENETDOWN:
      result:=eaIgnore;
  end;
end;

function TTCPIPJob.ReadErrorStatus(Error: Word): TErrorAction;
begin
  result:=eaRelease;
  case Error of
    WSAEWOULDBLOCK:
      result:=eaRead;
    WSAEINPROGRESS, WSAEINTR, WSAEFAULT, WSAENETDOWN:
      result:=eaIgnore;
  end;
end;

function TTCPIPJob.Execute: Boolean;
var
  ReqdVersion: Word;
begin
  result:=wsInitialized;
  if result then exit;

  asm
    mov ReqdVersion,0101h
  end;

  if WSAStartUp(ReqdVersion, @wsInfo)<>0 then Exit;

  if (LOBYTE(wsInfo.wVersion) <> 1) or
     (HIBYTE(wsInfo.wVersion) <> 1) then Exit;

  result:= True;
  wsInitialized:= Result;
end;

procedure TTCPIPJob.SetError(error: word);
begin
  WSASetLastError(error);
end;

function TTCPIPJob.GetError(Error: Word): String;
  begin
   Case WSAGetLastError of
     WSAEINTR           : result:='Interrupted system call';{WSAEINTR}
     WSAEBADF           : result:='Bad file number'; {WSAEBADF}
     WSAEACCES          : result:='Permission denied'; {WSAEINTR}
     WSAEFAULT          : result:='Bad address';{WSAEFAULT}
     WSAEINVAL          : result:='Invalid argument';{WSAEINVAL}
     WSAEMFILE          : result:='Too many open files';{WSAEMFILE}
     WSAEWOULDBLOCK     : result:='Operation would block';{WSAEWOULDBLOCK}
     WSAEINPROGRESS     : result:='Operation now in progress';{WSAEINPROGRESS}
     WSAEALREADY        : result:='Operation already in progress';{WSAEALREADY}
     WSAENOTSOCK        : result:='Socket operation on nonsocket';{WSAENOTSOCK}
     WSAEDESTADDRREQ    : result:='Destination address required';{WSAEDESTADDRREQ}
     WSAEMSGSIZE        : result:='Message too long';{WSAEMSGSIZE}
     WSAEPROTOTYPE      : result:='Protocol wrong type for socket';{WSAEPROTOTYPE}
     WSAENOPROTOOPT     : result:='Protocol not available';{WSAENOPROTOOPT}
     WSAEPROTONOSUPPORT : result:='Protocol not supported';{WSAEPROTONOSUPPORT}
     WSAESOCKTNOSUPPORT : result:='Socket not supported';{WSAESOCKTNOSUPPORT}
     WSAEOPNOTSUPP      : result:='Operation not supported on socket';{WSAEOPNOTSUPP}
     WSAEPFNOSUPPORT    : result:='Protocol family not supported';{WSAEPFNOSUPPORT}
     WSAEAFNOSUPPORT    : result:='Address family not supported';{WSAEAFNOSUPPORT}
     WSAEADDRINUSE      : result:='Address already in use';{WSAEADDRINUSE}
     WSAEADDRNOTAVAIL   : result:='Can''t assign requested address';{WSAEADDRNOTAVAIL}
     WSAENETDOWN        : result:='Network is down';{WSAENETDOWN}
     WSAENETUNREACH     : result:='Network is unreachable';{WSAENETUNREACH}
     WSAENETRESET       : result:='Network dropped connection on reset';{WSAENETRESET}
     WSAECONNABORTED    : result:='Software caused connection abort';{WSAECONNABORTED}
     WSAECONNRESET      : result:='Connection reset by peer';{WSAECONNRESET}
     WSAENOBUFS         : result:='No buffer space available';{WSAENOBUFS}
     WSAEISCONN         : result:='Socket is already connected';{WSAEISCONN}
     WSAENOTCONN        : result:='Socket is not connected';{WSAENOTCONN}
     WSAESHUTDOWN       : result:='Can''t send after socket shutdown';{WSAESHUTDOWN}
     WSAETOOMANYREFS    : result:='Too many references:can''t splice';{WSAETOOMANYREFS}
     WSAETIMEDOUT       : result:='Connection timed out';{WSAETIMEDOUT}
     WSAECONNREFUSED    : result:='Connection refused';{WSAECONNREFUSED}
     WSAELOOP           : result:='Too many levels of symbolic links';{WSAELOOP}
     WSAENAMETOOLONG    : result:='File name is too long';{WSAENAMETOOLONG}
     WSAEHOSTDOWN       : result:='Host is down';{WSAEHOSTDOWN}
     WSAEHOSTUNREACH    : result:='No route to host';{WSAEHOSTUNREACH}
     WSAENOTEMPTY       : result:='Directory is not empty';{WSAENOTEMPT}
     WSAEPROCLIM        : result:='Too many processes';{WSAEPROCLIM}
     WSAEUSERS          : result:='Too many users';{WSAEUSERS}
     WSAEDQUOT          : result:='Disk quota exceeded';{WSAEDQUOT}
     WSAESTALE          : result:='Stale NFS file handle';{WSAESTALE}
     WSAEREMOTE         : result:='Too many levels of remote in path';{WSAEREMOTE}
     WSASYSNOTREADY     : result:='Network subsystem is unusable';{WSASYSNOTREADY}
     WSAVERNOTSUPPORTED : result:='Winsock DLL cannot support this appliaction';{WSAVERNOTSUPPORTED}
     WSANOTINITIALISED  : result:='Winsock not initialized';{WSANOTINITIALISED}
     WSAHOST_NOT_FOUND  : result:='Host not found';{WSAHOST NOT FOUND}
     WSATRY_AGAIN       : result:='Non authoritative - host not found';{WSATRY_AGAIN}
     WSANO_RECOVERY     : result:='Non recoverable error';
     WSANO_DATA         : result:='Valid name, no data record of requested type';
     INADDR_NONE        : result:='Invalid IP address given';
     else result:='Unknown error';
   end;
end;

function TTCPIPJob.IsValidJob(Job: Integer): Boolean;
begin
  result:=JobArray<>nil;
  if not result then Exit;
  result:=(JobArray[Job]<>nil);
end;

function TTCPIPJob.GetJobAsyncEvent(Job: Integer): Integer;
begin
  result:=-1;
  if IsValidJob(Job) then
    result:=PJob(JobArray[Job])^.AsyncOpts;
end;

function TTCPIPJob.GetJobTag(Job: Integer): Integer;
begin
  result:=-1;
  if IsValidJob(Job) then
    result:=PJob(JobArray[Job])^.Tag;
end;

procedure TTCPIPJob.SetJobTag(Job: Integer; Tag: Word);
begin
  if IsValidJob(Job) then
    PJob(JobArray[Job])^.Tag:=Tag;
end;

function TTCPIPJob.BindJob(Job: Integer; const Address: String; Port: Word): Boolean;
var
  Socket: TSocket;
  ba: TSockAddr_in;
  ta: String;
begin
  result:=false;
  Socket:=GetJobSocket(Job);
  if Socket=INVALID_SOCKET then Exit;
  ta:=Address;

  with ba do
  begin
    if Address='0' then sin_addr.s_addr:=INADDR_ANY else
    begin
      ta:=ta+#0;
      sin_addr.s_addr:=inet_addr(@ta[BegChar]);
    end;

    sin_family:=AF_INET;
    sin_port:=htons(Port);
  end;
  result:=(bind(Socket, @ba, SizeOf(ba))=0);
end;

procedure TTCPIPJob.SelectEvent(Job: Integer; Event: Word);
var
  Socket:TSocket;
begin
  Socket:=GetJobSocket(Job);
  if (Socket<=0) then exit;
  WSAAsyncSelect(Socket, SockWin, MsgBase+Job, Event);
end;

function TTCPIPJob.AddrToString(const s:tIn_Addr):string;
begin
  result:=StrPas(Inet_NToA(s));
end;

function TTCPIPJob.ConnectJobImplicit(Job: Integer; Host: TIn_Addr;
                                      Port: Word): Boolean;
var
  sa: TSockAddr_In;
  Socket: TSocket;
begin
  result:=False;
  with sa do
  begin
     sin_family:=AF_INET;
     sin_port:=htons(Port);
     sin_addr:=Host;
  end;
  Socket:=GetJobSocket(Job);
  if Socket=INVALID_SOCKET then Exit;
  Connect(Socket, @sa, SizeOf(TSockAddr_In));
  result:=True;
end;

function TTCPIPJob.ConnectJob(Job: Integer; const Host: String;
                              Port: Word): Boolean;
var
  sa: TSockAddr_In;
  Socket: TSocket;
begin
  result:=False;
  Socket:=GetJobSocket(Job);
  if Socket=INVALID_SOCKET then Exit;
  with sa do
  begin
     sin_family:=AF_INET;
     sin_port:=htons(Port);
     if not ResolveAddress(Host, sin_addr) then Exit;
  end;
  Connect(Socket, @sa, SizeOf(TSockAddr_In));
  result:=True;
end;

function TTCPIPJob.GetJobSocket(Job:integer): TSocket;
begin
  result:=INVALID_SOCKET;
  if not (IsValidJob(Job)) then Exit;
  result:=PJob(JobArray[Job])^.ISocket;
  if result=0 then result:=INVALID_SOCKET;
end;

constructor TTCPIPJob.Create;
begin
  inherited Create;
  try
    IPCache:=TStringList.Create;
    SockWin:=AllocateWindow(SocketWindowProc);
    AllocateTimer($1234, 1000, ProcessTimer);
    JobArray:=TJobArray.Create;
  except
    raise ETCPIPException.Create('Error initialising TCP/IP Job Class');
  end;
end;

destructor TTCPIPJob.Destroy;
var
  n: Integer;
begin
  DeAllocateTimer($1234);
  DeAllocateWindow(SockWin);
  for n:=0 to NumHandle do
    DeleteJob(n);
  JobArray.Free;
  JobArray:=nil;
  IPCache.Free;

  inherited Destroy;
end;

function TTCPIPJob.DeleteJob(Job: Integer): Integer;
begin
  result:=-1;
  if not IsValidJob(Job) then Exit;
  if PJob(JobArray[Job])^.ISocket>0 then
  begin
    while WSAIsBlocking do
      WSACancelBlockingCall;
    ShutDown(PJob(JobArray[Job])^.ISocket, 1);
    CloseSocket(PJob(JobArray[Job])^.ISocket);
  end;
  Dispose(PJob(JobArray[Job]));
  JobArray.Jobs.Remove(JobArray[Job]);
  JobArray.Jobs.Pack;
  JobArray.Cache.LastHandle:=0;
  result:=Job;
end;

procedure TTCPIPJob.CheckJobs(Sender: TObject; var Done: Boolean);
var
  n: Integer;
  res: LongInt;
begin
  for n:=0 to JobArray.Count-1 do
  begin
    ioctlsocket(PJob(JobArray[n])^.ISocket, FIONREAD, @res);
    if (res<>INVALID_SOCKET) and (res<>0) then
    begin
      if Assigned(PJob(JobArray[n])^.JobEvent) then
        PJob(JobArray[n])^.JobEvent(n, FD_READ);
    end;
  end;
end;

function TTCPIPJob.CreateJob(FJob: TJobEvent; SType, SProt,
                             SAsync: Integer): Integer;
var
  p: PJob;
begin
  result:=-1;
  try
    New(p);
  except
    raise ETCPIPException.Create('No memory left!');
    Exit;
  end;

  with p^ do
  begin
    ISocket:=Socket(PF_INET, SType, SProt);
    if ISocket=INVALID_SOCKET then
    begin
      ISocket:=0;
      Exit;
    end;
    JobEvent:=FJob;
    AsyncOpts:=SAsync;
    TimeOut:=-1;
    Tag:=0;
    if PollType=ptAsync then Application.OnIdle:=CheckJobs;
{    if (SAsync and FD_READ)=1 then SAsync:=SAsync and (not FD_READ);}
    WSAAsyncSelect(ISocket, SockWin, MsgBase+NumHandle, SAsync);
    Handle:=NumHandle;
  end;

  try
    JobArray.Add(p);
  except
    Exit;
  end;

  result:=NumHandle;
  Inc(NumHandle);
end;

function TTCPIPJob.CreateLocalJob(FJob: TJobEvent; SType, SProt,
                                  SAsync: Integer): Integer;
var
  p: PJob;
begin
  result:=-1;
  try
    New(p);
  except
    raise ETCPIPException.Create('No memory left!');
    Exit;
  end;

  with p^ do
  begin
    ISocket:=0;
    JobEvent:=FJob;
    AsyncOpts:=SAsync;
    TimeOut:=-1;
    Tag:=0;
    Handle:=NumHandle;
  end;

  try
    JobArray.Add(p);
  except
    Exit;
  end;

  result:=NumHandle;
  Inc(NumHandle);
end;

function TTCPIPJob.SetTimeOut(Job: Integer; TimeOut: Integer): Boolean;
begin
  result:=False;
  if not(IsValidJob(Job)) then Exit;
  PJob(JobArray[Job])^.TimeOut:=TimeOut;
  result:=True;
end;

function TTCPIPJob.CreateJobIndirect(FJob: TJobEvent; SType, SProt,
                                     SAsync: Integer;
                                     Sckt: TSocket): Integer;
var
  p: PJob;

begin
  result:=-1;

  try
    New(p);
  except
    raise ETCPIPException.Create('No memory left!');
    Exit;
  end;

  if PollType=ptAsync then Application.OnIdle:=CheckJobs;

  with p^ do
  begin
    ISocket:=Sckt;
    if ISocket=INVALID_SOCKET then ISocket:=0;
    JobEvent:=FJob;
    AsyncOpts:=SAsync;
    TimeOut:=-1;
    Tag:=0;
{    if (SAsync and FD_READ)=1 then SAsync:=SAsync and (not FD_READ);}
    WSAAsyncSelect(ISocket, SockWin, MsgBase+NumHandle, SAsync);
    Handle:=NumHandle;
  end;

  try
    JobArray.Add(p);
  except
    exit;
  end;

  result:=NumHandle;
  Inc(NumHandle);
end;

procedure TTCPIPJOB.ProcessTimer(idTimer: Word);
var
  Job: Integer;
begin
  for Job:=0 to JobArray.Count-1 do
  if (JobArray.RawPointers[Job]<>nil) and
     (PJob(JobArray.RawPointers[Job])^.ISocket<>0) then
    with PJob(JobArray.RawPointers[Job])^ do
    begin
      if (TimeOut>0) then Dec(TimeOut);
      if TimeOut=0 then
      if System.Assigned(JobEvent) then
        JobEvent(Job, FD_CLOSE);
    end;
end;

procedure TTCPIPJob.SocketWindowProc(var msg: TMessage);
const
  retry: Integer = 255;
var
  Event:Word;
  Job:Integer;
begin
  if JobArray=nil then Exit;
  with Msg do
  begin
    if (Msg<MsgBase) or (Msg>MsgBase+NumHandle-1) then Exit;
    Job:=Msg - MsgBase;
    if not(IsValidJob(Job)) then Exit;
    Event:=LoWord(lParam);
  end;

  if WSAIsBlocking then
  begin
    while retry>0 do
    begin
      if RestoreEvent(Job) then Break;
      Dec(retry);
    end;
    Exit;
  end;

  with PJob(JobArray[Job])^ do
  if Assigned(JobEvent) then
    JobEvent(Job, Event);
end;

function TTCPIPJob.RestoreEvent(Job: Integer): Boolean;
var
  n: Integer;
begin
  result:=False;
  n:=GetJobSocket(Job);
  if n=-1 then Exit;
  with PJob(JobArray[Job])^ do
    n:=WSAAsyncSelect(ISocket, SockWin, MsgBase+Job, AsyncOpts);
  result:=(n=0);
end;

procedure TTCPIPJob.RestoreEvents;
var
  n: Integer;
begin
  for n:=0 to JobArray.Count-1 do
  if IsValidJob(n) then
  with PJob(JobArray[n])^ do
  if ISocket<>0 then
    WSAAsyncSelect(ISocket, SockWin, MsgBase+n, AsyncOpts);
end;

function TTCPIPJob.GetJobPort(Job: Integer): Integer;
var
  Socket: TSocket;
  SockName: TSockAddr;
  SLen:Integer;
begin
  result:=-1;
  Socket:=GetJobSocket(Job);
  if Socket=-1 then Exit;
  SLen:=SizeOf(SockName);
  if GetSockName(Socket, @SockName, @SLen)<>0 then exit;
  result:=PSockAddr_In(@SockName)^.sin_port;
end;

function TTCPIPJob.StringAddress(const tin: TIn_Addr): String;
begin
  result:=StrPas(inet_ntoa(tin));
end;

function TTCPIPJob.ResolveAddress(const hname: String;
                                  var rsa: TIn_Addr): Boolean;
var
  hs: array [BegChar..MaxStrLen+1] of Char;
  ent: PHostEnt;
  ip: PChar;
  srch: Integer;
begin
  result:=false;
  StrPCopy(@hs, hname);
  FillChar(rsa, SizeOf(rsa), 0);
  if (IPCache.Count<>-1) then
    srch:=IPCache.IndexOf(UpCaseStr(hname));

  if srch=-1 then
  begin
    longint(ip):=inet_addr(@hs);
    if longint(ip)<>INADDR_NONE then
    begin
      rsa.S_addr:=longint(ip);
      result:=True;
      Exit;
    end else
    begin
      ent:=GetHostByName(@hs);
      RestoreEvents;
      if ent=nil then exit;
      move(ent^.h_addr_list^, ip, SizeOf(ip));
      move(ip^, rsa.s_addr, SizeOf(rsa.s_addr));
      IPCache.AddObject(UpCaseStr(hname), pointer(rsa.s_addr));
    end;
  end else
    rsa.s_addr:=longint(IPCache.Objects[srch]);
  result:=True;
end;

procedure TCPFree; far;
begin
  InetJob.Free;
  if wsInitialized then WSACleanUp;
  ExitProc:=TCPExitProc;
end;

begin
  InetJob:=TTCPIPJob.Create;
  Jobbr.TCPExitProc:=ExitProc;
  ExitProc:=@TCPFree;
end.
