#include "HttpUpload.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <io.h>
#define SNPRINTF _snprintf
#else
#include <unistd.h>
#define SNPRINTF snprintf
#endif
#include "curl/curl.h"
#include "curl/types.h"
#include "curl/easy.h"
#include "misc.h"

const char HttpUpload::pServerName[] = "upload.securityfocus.com";

const char HttpUpload::pServerLoginFormat[] = "https://%s/Upload/verifpwd.asp?"
  "Username=%s&Password=%s&URL=/Upload/member/_blank.asp";


const char HttpUpload::pServerUploadURL[] = "https://%s/Upload/member/_LogUpload.asp";

int *HttpUpload::m_pNow = NULL;
int *HttpUpload::m_pTotal = NULL;	
int *HttpUpload::m_pState = NULL;


HttpUpload::HttpUpload(const char *infile, const char *username, const char *password, const char *tmp)
{
  m_Status = SUCCESS;
  m_pInfile = strdup(infile);
  m_pHttpUser = strdup(username);
  m_pHttpPass = strdup(password);
  m_pTmpTemplate = strdup(tmp);

  // 
  // This variable is also used to indicate
  // that a proxy should or shouldn't be used
  //
  m_pProxyServer = NULL;
  m_pProxyUser = NULL;

  m_nRefs = 1;
  m_pAlternateHost = NULL;
}

HttpUpload::~HttpUpload()
{
  free(m_pInfile);
  free(m_pHttpUser);
  free(m_pHttpPass);
  free(m_pTmpTemplate);

  if(m_pProxyServer) 
    free(m_pProxyServer);
  if(m_pProxyUser) {
    free(m_pProxyUser);
    free(m_pProxyPassword);
  }
  if(m_pAlternateHost)
    free(m_pAlternateHost);
}
void HttpUpload::setAlternateHost(const char *host)
{
  if(!host)
    return;

  m_pAlternateHost = strdup(host);

}
int HttpUpload::progressCallback(void *p, size_t dltotal, size_t dlnow, size_t ultotal, size_t ulnow)
{
	if (m_pTotal && m_pNow && m_pState)
	{
		if (*m_pState == READING)
			*m_pState = UPLOADING;
		*m_pTotal = ultotal;
		*m_pNow = ulnow;
		if (*m_pState == CANCEL)
		{
			*m_pState = CANCELLED;
			return -1;					// abort transfer
		}
	}
	return 0;							// continue transfer
}

void HttpUpload::SetProgressVars (int* pNow, int* pTotal, int *pState)
{
	m_pNow = pNow;
	m_pTotal = pTotal;
	m_pState = pState;
	*m_pState = NONE;
}

void HttpUpload::setProxyInfo(const char *server, const char *user, const char *password)
{
  m_pProxyServer = strdup(server);

  if(user && password) {
    m_pProxyUser = strdup(user);
    m_pProxyPassword = strdup(password);
  }
  else {
    m_pProxyUser = NULL;
    m_pProxyPassword = NULL;
  }

}

HttpUpload::err_code HttpUpload::errorInfo(char *buffer, int size)
{
  switch(m_Status) {
  case SUCCESS:
    strncpy(buffer, "Operation completed successfully", size);
    break;
  case ERR_SERVAUTH:
    strncpy(buffer, "Unspecified error authenticating to server", size);
    break;
  case ERR_AUTH:
    strncpy(buffer, "Authentication error logging into server", size);
    break;
  case ERR_SSL:
	strncpy(buffer, "SSL Connect error", size);
	break;
  case ERR_UPLOAD:
    strncpy(buffer, "Unspecified error uploading log file to server", size);
    break;
  case ERR_FILE:
    strncpy(buffer, "Error opening input file", size);
    break;
  case ERR_TIMEOUT:
	strncpy(buffer, "Error: Partial upload", size);
	break;
  case ERR_WRITE:
	strncpy(buffer, "Socket write error", size);
	break;
  case ERR_SERVER:
	strncpy(buffer, "Server did not accept log", size);
	break;
  case ERR_SIZE:
        strncpy(buffer, "Log file too large", size);
	break;
  }
  buffer[size - 1] = '\0';
  return m_Status;
}

void HttpUpload::useProxy(CURL *curl)
{
  const  int size = 1024;
  static char urlBuf[size], authBuf[size];

  if(m_pProxyServer) {
    SNPRINTF(urlBuf, size, "http://%s", m_pProxyServer);
    curl_easy_setopt(curl, CURLOPT_PROXY, urlBuf);
    if(m_pProxyUser && m_pProxyPassword) {
      SNPRINTF(authBuf, size, "%s:%s", m_pProxyUser, m_pProxyPassword);
      curl_easy_setopt(curl, CURLOPT_PROXYUSERPWD);
    }
  }
}

bool HttpUpload::doTransfer()
{

  // Ensure m_pInfile exists and that we can read it 
  FILE *f;
  if((f = fopen(m_pInfile, "r")) == NULL) {
    m_Status = ERR_FILE;
	if (m_pState)
		*m_pState = DONE;	
	// verbose file transfer
	debug("ERROR: Could not read input file, aborting.\n");
    return false;
  }
  fclose(f);

  if (m_pState)
	*m_pState = LOGIN;

  if(!doLogin()) 
  {
    if (m_pState)
      *m_pState = DONE;	
	
	// verbose file transfer
	debug( "ERROR: Login failed.\n" );
    
	return false;
  }
  
  if (m_pState)
	*m_pState = READING;
  
  m_Status = doUpload ();
  
  if(m_Status != SUCCESS) 
  {
	if (m_pState)
		*m_pState = DONE;	
    // verbose file transfer
    debug("ERROR: %d error code returned from upload.\n", (int) m_Status);

    return false;
  }

  if (m_pState)
  {
	  if (*m_pState != CANCELLED)
		  *m_pState = DONE;	
  }
  
  debug( "Transfer completed.\n\n");
  
  return true;
}

extern int curlverbose;

bool HttpUpload::doLogin()
{
  const int size = 1024;
  char buffer[size];
  CURL *curl;
  CURLcode res;
  FILE *headerfile;
  bool ret = false;

  SNPRINTF(buffer, size, pServerLoginFormat, m_pAlternateHost?m_pAlternateHost:pServerName,
	   m_pHttpUser, m_pHttpPass );

  debug( "URL: %s\n", buffer );

  curl = curl_easy_init();
  if(curl) 
  {
  	debug("\n**LOGIN**\n");

#ifdef _WIN32
	char tempname[_MAX_PATH+1];
	TCHAR szTempDir[_MAX_PATH + 1];
	GetTempPath (_MAX_PATH, szTempDir);
	if (GetTempFileName (szTempDir, m_pTmpTemplate, 0, tempname) == 0)
	{
		AfxMessageBox("doTransfer: GetTempFileName()");
		return false;
	}
	if ((headerfile = fopen (tempname, "w+")) == NULL)
	{
		AfxMessageBox("fopen()");
		return false;
	}
#else
    char *tempname = strdup(m_pTmpTemplate);
    int fd;
	if((fd = mkstemp(tempname)) == -1) {
      perror("mkstemp()");
      exit(1);
    }
    if((headerfile = fdopen(fd, "r+")) == NULL) {
      perror("fdopen()");
      exit(1);
    }
#endif

    curl_easy_setopt(curl, CURLOPT_URL, buffer);
    curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1);
    //    curl_easy_setopt(curl, CURLOPT_NOBODY, 1);
    curl_easy_setopt(curl, CURLOPT_WRITEHEADER, headerfile);
    curl_easy_setopt(curl, CURLOPT_FILE, headerfile);
    
	// for larger files use SSLv2
    curl_easy_setopt(curl, CURLOPT_SSLVERSION, 2);

    if( curlverbose )
      curl_easy_setopt(curl, CURLOPT_VERBOSE, true);

    if(m_pProxyServer)
      useProxy(curl);
    res = curl_easy_perform(curl);

    // dump the cookie to the debug log
    debug( "<Cookie>\n" );
    char sLineBuffer[size];
    rewind( headerfile );
    while( !feof(headerfile) )
    {
      if (fgets (sLineBuffer, size, headerfile) != sLineBuffer)
        break;	
      debug( "%s", sLineBuffer );
    }
    debug( "</Cookie>\n" );

    curl_easy_cleanup(curl);

    ret = fileContains(headerfile, "Set-Cookie: FormsAuth=");
    if(!ret) 
	{
      if(fileContains(headerfile, "Authentication Failed"))
	  {
	    m_Status = ERR_AUTH;
		debug( "ERROR: Authentication failed.\n");
	  }
      else
	  {
	    m_Status = ERR_SERVAUTH;
		debug( "ERROR: Authentication failed, cookie did not return'Authentication Failed'.\n");
	  }

      unlink(tempname);
    }
    fclose(headerfile);
#ifdef _WIN32
	m_pCookieFilename = strdup (tempname);
#else
    m_pCookieFilename = tempname;
#endif
  }
  return ret;
}

HttpUpload::err_code HttpUpload::doUpload()
{

  const int size = 1024;
  char buffer[size];
  char URLbuffer[size];
  struct HttpPost *httppost = NULL, *last_post = NULL;
  CURL *curl;
  FILE *bodyfile;
  err_code ret = ERR_UPLOAD;

  curl = curl_easy_init();
  if(curl) {

	debug("\n**UPLOAD**\n");

#ifdef _WIN32
	char tempname[_MAX_PATH+1];
	TCHAR szTempDir[_MAX_PATH + 1];
	GetTempPath (_MAX_PATH, szTempDir);
	if (GetTempFileName (szTempDir, m_pTmpTemplate, 0, tempname) == 0)
	{
		AfxMessageBox("doUpload: GetTempFileName()");
		return ERR_UPLOAD;
	}
	if ((bodyfile = fopen (tempname, "w+")) == NULL)
	{
		AfxMessageBox("fopen()");
		return ERR_UPLOAD;
	}
#else
    char *tempname = strdup(m_pTmpTemplate);
    int fd;
	if((fd = mkstemp(tempname)) == -1) {
      perror("mkstemp()");
      exit(1);
    }
    if((bodyfile = fdopen(fd, "r+")) == NULL) {
      perror("fdopen()");
      exit(1);
    }
#endif

    SNPRINTF(URLbuffer, size, pServerUploadURL, m_pAlternateHost?m_pAlternateHost:pServerName);
    debug( "URL: %s\n", URLbuffer );
  
    curl_easy_setopt(curl, CURLOPT_URL, URLbuffer);
    curl_easy_setopt(curl, CURLOPT_COOKIEFILE, m_pCookieFilename);
    curl_easy_setopt(curl, CURLOPT_FILE, bodyfile);
    curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, progressCallback);
    if(m_pProxyServer)
      useProxy(curl);

    // for larger files use SSLv2
    curl_easy_setopt(curl, CURLOPT_SSLVERSION, 2);

    char errBuffer[CURL_ERROR_SIZE];
    curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errBuffer );

    SNPRINTF(buffer, size, "FILE1=@%s;type=text/plain", m_pInfile);

    if(curl_formparse(buffer, &httppost, &last_post)) {
#ifdef _WIN32
		AfxMessageBox ("curl_formparse() failed!");
		return ERR_UPLOAD;
#else
	  fprintf(stderr, "curl_formparse() failed!\n");
      exit(1);
#endif
    }

    curl_easy_setopt(curl, CURLOPT_HTTPPOST, httppost);

    // print out the contents of httppost
    debug( "Http post contents:\n" );
    struct HttpPost *temppost = httppost;
    while( temppost )
    {
      debug( "name     :  %s\n", temppost->name );
      debug( "contents :  %s\n", temppost->contents );
      debug( "type     :  %s\n", temppost->contenttype );
      debug( "flags    :  %ld\n", temppost->flags );
      temppost = temppost->more;
    }

    debug( "\nBegin perform... " );
    CURLcode errcode = curl_easy_perform(curl);
    debug( "finished.\n");
    debug( "Error code:   %d\n", errcode);
    debug( "Error buffer: %s\n", errcode?errBuffer:"(none)" );

    // cph debug
    unsigned long httpcode = 0;
    double connecttime = 0;
    unsigned long headerbytes = 0;
    unsigned long reqbytes = 0;
    double dlbytes = 0;
    double upbytes = 0;
    CURLcode e;
        
    e = curl_easy_getinfo( curl, CURLINFO_HTTP_CODE, &httpcode );
    debug( "%s" "last HTTP code: %ld\n", (e==CURLE_OK)?"":"(INVALID) ", httpcode);

    e = curl_easy_getinfo( curl, CURLINFO_CONNECT_TIME, &connecttime );
    debug( "%s" "connect time: %g\n", (e==CURLE_OK)?"":"(INVALID) ", connecttime);

    e = curl_easy_getinfo( curl, CURLINFO_HEADER_SIZE, &headerbytes);
    debug( "%s" "download header bytes: %ld\n", (e==CURLE_OK)?"":"(INVALID) ", headerbytes);

    e = curl_easy_getinfo( curl, CURLINFO_REQUEST_SIZE, &reqbytes);
    debug( "%s" "request bytes: %ld\n", (e==CURLE_OK)?"":"(INVALID) ", reqbytes);

    e = curl_easy_getinfo( curl, CURLINFO_SIZE_DOWNLOAD, &dlbytes ); 
    debug( "%s" "download bytes: %g\n", (e==CURLE_OK)?"":"(INVALID) ", dlbytes);

    e = curl_easy_getinfo( curl, CURLINFO_SIZE_UPLOAD, &upbytes ); 
    debug( "%s" "upload bytes: %g\n", (e==CURLE_OK)?"":"(INVALID) ", upbytes);

    // free the memory associated with the http post
#ifdef HAVE_CURL_FORMFREE
    curl_formfree(httppost);
#endif

    curl_easy_cleanup(curl);

    if (errcode == CURLE_ABORTED_BY_CALLBACK)
		ret = SUCCESS;
	else if (errcode == CURLE_PARTIAL_FILE)
		ret = ERR_TIMEOUT;
	else if (errcode == CURLE_WRITE_ERROR)
		ret = ERR_WRITE;
	else if (errcode == CURLE_SSL_CONNECT_ERROR)
		ret = ERR_SSL;
        else if (errcode == CURLE_OK) {
	  if(fileContains(bodyfile, "sent and saved"))
	    ret = SUCCESS;
	  else if(fileContains(bodyfile, "error saving file"))
		ret = ERR_SIZE;
	  else if(fileContains(bodyfile, "Size exceeded for this file")) 
	    ret = ERR_SIZE;
	  else
	    ret = ERR_SERVER;
	}
	else
	{
		ret = ERR_UPLOAD;
#ifdef _WIN32
		CString x; x.Format ("Extended Error: %d", errcode); ::MessageBox (NULL, x, NULL, MB_OK);
#endif
	}
    
	fclose(bodyfile);

    unlink(tempname);
    if(m_pCookieFilename)
	  unlink(m_pCookieFilename);
#ifndef _WIN32
    free(tempname);
#endif
  }

  if(m_pCookieFilename) {
    free(m_pCookieFilename);
    m_pCookieFilename = NULL;
  }
  return ret;
}

bool HttpUpload::fileContains(FILE *file, char *str)
{
  const int size = 1024;
  char buffer[size];

  rewind(file);
  while(fgets(buffer, size, file) != NULL) 
    if(strstr(buffer, str))
      return true;

  return false;
  
}

