///////////////////////////////////////////////////////////////////////////
//                                                                       //
// PGPManager.cpp : implementation file                                  //
//                                                                       //
// Copyright (c) 2000 Samopal Corporation.                               //
// All right are all right (tm)                                          //
//                                                                       //
// For news and updates visit http://www.samopal.com/soft/pgpicq/        //
// Email your comments to pgpicq@samopal.com                             //
//                                                                       //
// Free use and distribution of this source code allowed                 //
// under the condition of keeping this header intact.                    //
//                                                                       //
///////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "pgpKeys.h"
#include "pgpManager.h"
#include "pgpKeysDefs.h"
#include "WaitWnd.h"
#include "SearchDlg.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


CPgpManager::CPgpManager()
{
	m_pgpContext = kInvalidPGPContextRef;
	m_defKeySet  = kInvalidPGPKeySetRef;
}


CPgpManager::~CPgpManager()
{
	ReleaseKeyrings();

	PGPsdkUILibCleanup();

	if ( PGPContextRefIsValid(m_pgpContext) )
		PGPFreeContext (m_pgpContext);

	PGPsdkCleanup();
}


BOOL CPgpManager::IsPGPInstalled()
{
	HKEY hKey;

	if ( RegOpenKeyEx (HKEY_LOCAL_MACHINE, _T(kRegPathPGP),
				0, KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS )
	{
		DWORD dwType, dwDataLen;
		char  chPath[MAX_PATH];
		if ( RegQueryValueEx (hKey, _T(kStrInstPath), NULL, 
			&dwType, (unsigned char*)chPath, &dwDataLen) == ERROR_SUCCESS )
			m_strPGPPath = CString(chPath);

		RegCloseKey(hKey);
		return true;
	}
	else
		return false;
}


BOOL CPgpManager::ObtainPassphrase()
{
	/*
	CPassDlg passDlg;
	if ( passDlg.DoModal() == IDOK )
		m_strPassphrase = passDlg.m_strPass;

	if ( m_strPassphrase.IsEmpty() )
		return false;
	else	
		return true;
	*/

	return false;
}


BOOL CPgpManager::OpenKeyrings(CString strPublicKeyring, CString strSecretKeyring)
{
	int			nErr;
	PGPFileSpecRef	pubFileRef;
	PGPFileSpecRef	secFileRef;

	ReleaseKeyrings();

	nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strPublicKeyring, &pubFileRef);
	nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strSecretKeyring, &secFileRef);

	nErr = PGPOpenKeyRingPair (
				m_pgpContext,
				kPGPKeyRingOpenFlags_Mutable | kPGPKeyRingOpenFlags_Create,
				pubFileRef, secFileRef,
				&m_defKeySet
			);

	if ( IsPGPError(nErr) )
	{
		m_defKeySet = kInvalidPGPKeySetRef;
		return false;
	}

	PGPsdkSavePrefs(m_pgpContext);

	return true;
}

BOOL CPgpManager::ReleaseKeyrings()
{
	if ( m_defKeySet == kInvalidPGPKeySetRef)
		return false;

	PGPFreeKeySet (m_defKeySet);
	m_defKeySet = kInvalidPGPKeySetRef;

	return true;
}



BOOL CPgpManager::Initialize(CString strPublicKeyring, CString strSecretKeyring)
{
	int nErr;

	try {

		// Initialize the PGPsdk libs
		nErr = PGPsdkInit();
		if ( IsPGPError(nErr) )
			return false;

		// Create the PGPContext
		nErr = PGPNewContext (kPGPsdkAPIVersion, &m_pgpContext);
		if ( IsPGPError(nErr) )
			return false;

		nErr = PGPsdkUILibInit();

		if ( !OpenKeyrings(strPublicKeyring, strSecretKeyring) )
			return false;
	
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPgpManager::Initialize()");
		e->Delete();
		return false;
	}	

	return true;
}



void CPgpManager::GetAllKeys (KeyObjectListT *vecKeyList, int nCol, BOOL bDesc)
{
	int				nErr;
	PGPKeyListRef	keyList;
	PGPKeyIterRef	keyIter;
	PGPKeyRef		key;
	int				nSortType;

	nSortType = kPGPAnyOrdering;
	switch ( nCol )
	{
	case 0:	
		if ( !bDesc ) nSortType = kPGPReverseUserIDOrdering;
		else nSortType = kPGPUserIDOrdering;
		break;
	case 2:
		if ( !bDesc ) nSortType = kPGPReverseEncryptKeySizeOrdering;
		else nSortType = kPGPEncryptKeySizeOrdering;
		break;
	case 3:
		if ( !bDesc ) nSortType = kPGPReverseCreationOrdering;
		else nSortType = kPGPCreationOrdering;
		break;
	}


	// Create a keylist from keyset
	nErr = PGPOrderKeySet (m_defKeySet, (PGPKeyOrdering_)nSortType, &keyList );
	if ( IsPGPError(nErr) )
		return;

	// Create a key iterator for browsing thru the keys
	nErr = PGPNewKeyIter (keyList, &keyIter);
	if ( IsPGPError(nErr) )
		return;
	
	nErr = PGPKeyIterNext (keyIter, &key);

	while ( key )
	{
		CKeyObject keyObject;
		keyObject.Create(&m_pgpContext, &m_defKeySet, key);
			
		vecKeyList->push_back(keyObject);
		nErr = PGPKeyIterNext ( keyIter, &key);
	}

	nErr = PGPFreeKeyIter (keyIter);
	nErr = PGPFreeKeyList (keyList);
}



void CPgpManager::ExportKeys(KeyPointerListT *pKeyList, CString strFile)
{
	int					nErr;
	PGPFilterRef		keyFilter = kInvalidPGPFilterRef;
	PGPKeySetRef		singleKeySet = kInvalidPGPKeySetRef;
	PGPKeySetRef		multKeySet = kInvalidPGPKeySetRef;
	
	// Look for selected keys and add them to a set
	nErr = PGPNewKeySet (m_pgpContext, &multKeySet);
	KeyPointerListT::iterator first = pKeyList->begin();
	KeyPointerListT::iterator last = pKeyList->end();

	while ( first != last )
	{
		CKeyObject*pKey = (CKeyObject*)(*first++);
		nErr = PGPNewKeyIDFilter (m_pgpContext, pKey->GetKeyID(), &keyFilter);

		nErr = PGPFilterKeySet (m_defKeySet, keyFilter, &singleKeySet);
		nErr = PGPAddKeys (singleKeySet, multKeySet);

		PGPFreeKeySet (singleKeySet);
		PGPFreeFilter (keyFilter);
	}
				
	/*
	PGPUInt32 numKeys;
	nErr = PGPCountKeys (multKeySet, &numKeys);
	CString strMsg;
	strMsg.Format ("Exporting %d key(s) to file\n%s", numKeys, strFile);
	AfxMessageBox (strMsg);
	*/

	if ( strFile.IsEmpty() )
	{
		// to the Clipboard

		PGPSize	nOutBuffSize;
		char	chBuffer[MAX_BUFFER_LENGTH];

		PGPOptionListRef outputBufferOpt = PGPOOutputBuffer (m_pgpContext, 
			chBuffer, MAX_BUFFER_LENGTH, &nOutBuffSize);

		nErr = PGPExportKeySet (
					multKeySet,
					outputBufferOpt,
					PGPOArmorOutput (m_pgpContext, true),
					PGPONullOption (m_pgpContext),
					PGPOLastOption (m_pgpContext)
				);
		
		if ( !nErr )
		{
			if ( !OpenClipboard (AfxGetApp()->GetMainWnd()->m_hWnd) ) 
				return; 
			EmptyClipboard();

			// Allocate a global memory object for the text.
			LPTSTR  lptstrCopy; 
			HGLOBAL hglbCopy;    
 
			hglbCopy = GlobalAlloc(GMEM_DDESHARE, nOutBuffSize);
			if (hglbCopy == NULL) 
			{ 
				CloseClipboard(); 
				return; 
			} 
 
			// Lock the handle and copy the text to the buffer. 
 
			lptstrCopy = (char*)GlobalLock(hglbCopy); 
			memcpy ( lptstrCopy, chBuffer, nOutBuffSize); 
			lptstrCopy[nOutBuffSize-2] = 0;
			GlobalUnlock(hglbCopy); 
 
			// Place the handle on the clipboard. 
			SetClipboardData(CF_TEXT, hglbCopy); 
			CloseClipboard();
		}

	}
	else
	{
		// to the file

		PGPFileSpecRef fileRef;
		nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strFile, &fileRef);

		nErr = PGPExportKeySet (
					multKeySet,
					PGPOOutputFile (m_pgpContext, fileRef),
					PGPOArmorOutput (m_pgpContext, true),
					PGPONullOption (m_pgpContext),
					PGPOLastOption (m_pgpContext)
				);
	}

	PGPFreeKeySet (multKeySet);
}


// Imports public keys from a file or Clipboard
void CPgpManager::ImportKeys(CString strFile)
{
	int					nErr;
	PGPKeySetRef		impKeySet;
	PGPFileSpecRef		fileRef;
	PGPKeyListRef		impKeyList;
	PGPKeyIterRef		keyIter;
	PGPKeyRef			key;
	char				chName[MAX_PATH];
	CString				strMsg;
	PGPSize				usedLength;
	int					nKeys = 0;
	

	if ( strFile.IsEmpty() )
	{
		// from Clipboard
		HGLOBAL	hglb; 
		LPTSTR	lptstr; 
		char	chBuffer[MAX_BUFFER_LENGTH];
		
		if ( !OpenClipboard (AfxGetApp()->GetMainWnd()->m_hWnd) ) 
            return; 
 
        hglb = GetClipboardData(CF_TEXT); 
        if (hglb != NULL) 
        { 
            lptstr = (char*)GlobalLock(hglb); 
            if (lptstr != NULL) 
            { 
				strcpy(chBuffer, lptstr);
                GlobalUnlock(hglb); 
            } 
        } 
        CloseClipboard();

		// create the input buffer and import
		PGPOptionListRef inputBufferOpt = PGPOInputBuffer (m_pgpContext, 
			chBuffer, strlen(chBuffer) );

		nErr = PGPImportKeySet (
				m_pgpContext,
				&impKeySet,
				inputBufferOpt,
				PGPOLastOption(m_pgpContext)
			);
 
	}
	else
	{
		nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strFile, &fileRef);

		nErr = PGPImportKeySet (
				m_pgpContext,
				&impKeySet,
				PGPOInputFile (m_pgpContext, fileRef),
				PGPOLastOption(m_pgpContext)
			);
	}

	nErr = PGPOrderKeySet (impKeySet, kPGPAnyOrdering, &impKeyList );
	nErr = PGPNewKeyIter (impKeyList, &keyIter);

	key = NULL;
	strMsg = kStrKeyTypePromptImport;

	do {
		nErr = PGPKeyIterNext (keyIter, &key);
		
		if ( key != NULL )
		{
			nKeys++;
			nErr = PGPGetPrimaryUserIDNameBuffer (key, MAX_PATH, chName, &usedLength);
			strMsg += _T(chName);
			strMsg += _T("\n");
		}
	} while ( key != NULL );
	PGPFreeKeyList (impKeyList);

	if ( nKeys > 0 )
	{
		strMsg += _T(kStrKeyTypePromptSure);

		if ( MessageBox ( NULL, strMsg, _T(kStrTitleConfirmation), 
				MB_OKCANCEL|MB_APPLMODAL|MB_ICONEXCLAMATION) == IDOK )
		{
			nErr = PGPAddKeys (impKeySet, m_defKeySet);
			if ( nErr = PGPCommitKeyRingChanges (m_defKeySet) )
				::MessageBox ( NULL, _T(kStrErrorImport), _T(kStrTitleError),
					MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		}
	}
	else
	{
		if ( strFile.IsEmpty() )
			::MessageBox ( NULL, _T(kStrKeyTypePromptNoInfoClip), _T(kStrTitleNeutral),
				MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		else
			::MessageBox ( NULL, _T(kStrKeyTypePromptNoInfoFile), _T(kStrTitleNeutral),
				MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
	}

	nErr = PGPFreeKeySet (impKeySet);
}



int CPgpManager::GenerateKeyPair(CString strUserID, CString strPass, long nSize, long nExpdays)
{
	int					nErr;
	PGPOptionListRef	optionList		= kInvalidPGPOptionListRef;
	BOOL				bNewDefaultKey	= true;
	PGPKeyRef			keyDefault;
	PGPKeyRef			newKey;
	PGPSubKeyRef		newSubkey;
	PGPUInt32			neededEntropyBits = 1000;

	
	nErr =  PGPCollectRandomDataDialog (m_pgpContext,
		neededEntropyBits, PGPOLastOption(m_pgpContext) );
	

	PGPGetDefaultPrivateKey (m_defKeySet, &keyDefault);
	if (PGPRefIsValid (keyDefault)) 
		bNewDefaultKey = false;

	nErr = PGPBuildOptionList (
				m_pgpContext, 
				&optionList,
				PGPOKeySetRef (m_pgpContext, m_defKeySet),
				PGPOKeyGenParams (m_pgpContext, kPGPPublicKeyAlgorithm_DSA, 1024),
				PGPOKeyGenFast (m_pgpContext, true),
				PGPOKeyGenName (m_pgpContext, strUserID, strUserID.GetLength()),
				PGPOPassphrase (m_pgpContext, strPass),
				PGPOExpiration (m_pgpContext, nExpdays),
				//PGPOPreferredAlgorithms (ctx, pAlgs, numAlgs),
				//PGPOEventHandler (ctx, sKeyGenCallback, pkgi),
				PGPOLastOption (m_pgpContext)
			);

	nErr = PGPGenerateKey (m_pgpContext, &newKey, optionList, PGPOLastOption (m_pgpContext)); 
	PGPFreeOptionList (optionList);

	if ( nErr )
	{
		::MessageBox ( NULL, _T(kStrErrorGenerateKey), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		return false;
	}
	
	nErr = PGPGenerateSubKey (
				m_pgpContext, &newSubkey,
				PGPOKeyGenMasterKey (m_pgpContext, newKey),
				PGPOKeyGenParams (m_pgpContext, kPGPPublicKeyAlgorithm_ElGamal, nSize),
				PGPOKeyGenFast (m_pgpContext, true),
				PGPOPassphrase (m_pgpContext, strPass),
				PGPOExpiration (m_pgpContext, nExpdays),
				//PGPOEventHandler (ctx, sKeyGenCallback, pkgi),
				PGPOLastOption (m_pgpContext)
			);

	if ( nErr )
	{
		::MessageBox ( NULL, _T(kStrErrorGenerateSubkey), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		return false;
	}
	

	nErr = PGPCommitKeyRingChanges (m_defKeySet);
	if ( nErr )
	{
		::MessageBox ( NULL, _T(kStrErrorStoreNew), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		return false;
	}

	if ( bNewDefaultKey )
	{
		CKeyObject keyObj;
		keyObj.Create(&m_pgpContext, &m_defKeySet, newKey);
		keyObj.SetAsDefault();
	}

	return true;
}



void CPgpManager::SearchKeyServer(CString strServerURL)
{
	int			nErr;
	CSearchDlg	searchDlg;

	searchDlg.m_strServer = strServerURL;
	searchDlg.m_pgpContext = m_pgpContext;
	searchDlg.m_pPGPMan = (void*)this;
	
	if ( searchDlg.DoModal() != IDOK )
		return;

	if ( !searchDlg.m_returnKeySet )
		return;

	/*
	PGPUInt32 numKeys;
	PGPCountKeys (searchDlg.m_returnKeySet, &numKeys);
	CString strMsg;
	strMsg.Format ("Importing %d key(s)", numKeys);
	AfxMessageBox (strMsg);
	*/

	nErr = PGPAddKeys (searchDlg.m_returnKeySet, m_defKeySet);
	nErr = PGPFreeKeySet (searchDlg.m_returnKeySet);

	if ( nErr = PGPCommitKeyRingChanges (m_defKeySet) )
		::MessageBox ( NULL, _T(kStrErrorImport), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
}



void CPgpManager::UploadToKeyServer(CString m_strServer, CString strKeyID)
{
	int					nErr;
	PGPKeyServerRef		keyServer;
	PGPFilterRef		keyFilter;
	PGPKeySetRef		keysToUpload;
	PGPKeySetRef		keysThatFailed;
	PGPtlsSessionRef	tlsSession;


	nErr = PGPNewUserIDStringFilter (m_pgpContext, LPCSTR(strKeyID), 
		kPGPMatchSubString, &keyFilter);
	nErr = PGPFilterKeySet (m_defKeySet, keyFilter, &keysToUpload);


	nErr = PGPKeyServerInit();

	nErr = PGPNewKeyServer (m_pgpContext, kPGPKeyServerClass_PGP,
			&keyServer, PGPONetURL(m_pgpContext, m_strServer),
			PGPOLastOption(m_pgpContext) );

	nErr = PGPGetKeyServerTLSSession (keyServer, &tlsSession);

	nErr = PGPKeyServerOpen (keyServer, tlsSession);
	if ( nErr )
	{
		::MessageBox (NULL, _T(kStrErrorServerConnect), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		return;
	}

	nErr = PGPUploadToKeyServer (keyServer, keysToUpload, &keysThatFailed);
	if ( nErr )
	{
		::MessageBox (NULL, _T(kStrErrorServerUpload), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
		return;
	}


	PGPUInt32	numKeys;
	CString		strMsg;
	PGPCountKeys (keysThatFailed, &numKeys);

	if ( numKeys )
	{
		::MessageBox (NULL, _T(kStrErrorServerRejected), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
	}
	else
	{
		::MessageBox (NULL, _T(kStrKeyTypePromptUploadDone), _T(kStrTitleError), 
			MB_OK|MB_APPLMODAL|MB_ICONEXCLAMATION);
	}
	
	PGPFreeKeySet (keysToUpload);
	PGPFreeKeySet (keysThatFailed);
	PGPFreeTLSSession (tlsSession);
	PGPFreeKeyServer(keyServer);
	PGPFreeFilter(keyFilter);

}
