///////////////////////////////////////////////////////////////////////////
//                                                                       //
// PGPManager.cpp : implementation file                                  //
//                                                                       //
// Copyright (c) 2000 Samopal Corporation.                               //
// All right are all right (tm)                                          //
//                                                                       //
// For news and updates visit http://www.samopal.com/soft/pgpicq/        //
// Email your comments to pgpicq@samopal.com                             //
//                                                                       //
// Free use and distribution of this source code allowed                 //
// under the condition of keeping this header intact.                    //
//                                                                       //
///////////////////////////////////////////////////////////////////////////


#include "stdafx.h"
#include "pgpicq.h"
#include "PGPManager.h"
#include "PassDlg.h"
#include "PGPICQDlg.h"


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


CPGPManager::CPGPManager()
{
	m_pgpContext = kInvalidPGPContextRef;
	m_defKeySet  = kInvalidPGPKeySetRef;
}



CPGPManager::~CPGPManager()
{
	if ( PGPKeySetRefIsValid (m_defKeySet) )
		PGPFreeKeySet (m_defKeySet);

	if ( PGPContextRefIsValid(m_pgpContext) )
		PGPFreeContext (m_pgpContext);

	PGPsdkCleanup();
}


BOOL CPGPManager::OpenKeyrings()
{
	/*
	// Try to open the default keyring
	int nErr = PGPOpenDefaultKeyRings (m_pgpContext, 
		kPGPKeyRingOpenFlags_Mutable, &m_defKeySet );
	*/
	

	int				nErr;
	CString			strPublicKeyring("");
	CString			strSecretKeyring("");
	PGPFileSpecRef	pubFileRef;
	PGPFileSpecRef	secFileRef;

	HKEY			hKey;
	DWORD			dwType, dwDataLen;
	unsigned char  *chStr;


	if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathPgpKeys),
			0, KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS )
	{
		dwDataLen = 512;
		chStr = (unsigned char *) malloc(dwDataLen);

		dwDataLen = 512;
		nErr = RegQueryValueEx (hKey, "PublicPath", NULL, &dwType, chStr, &dwDataLen);
		if ( nErr == ERROR_SUCCESS )
			strPublicKeyring = CString((LPCTSTR )chStr, dwDataLen);

		dwDataLen = 512;
		nErr = RegQueryValueEx (hKey, "PrivatePath", NULL, &dwType, chStr, &dwDataLen);
		if ( nErr == ERROR_SUCCESS )
			strSecretKeyring = CString((LPCTSTR )chStr, dwDataLen);

		RegCloseKey(hKey);
		free(chStr);
	}
	

	nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strPublicKeyring, &pubFileRef);
	nErr = PGPNewFileSpecFromFullPath (m_pgpContext, strSecretKeyring, &secFileRef);

	nErr = PGPOpenKeyRingPair (
				m_pgpContext,
				kPGPKeyRingOpenFlags_Mutable | kPGPKeyRingOpenFlags_Create,
				pubFileRef, secFileRef,
				&m_defKeySet
			);
	

	if ( IsPGPError(nErr) )
		return false;
	else
		return true;
}


BOOL CPGPManager::ReleaseKeyrings()
{
	if ( PGPKeySetRefIsValid (m_defKeySet) )
		PGPFreeKeySet (m_defKeySet);

	return true;
}


BOOL CPGPManager::IsPGPInstalled()
{
	HKEY hKey;

	if ( RegOpenKeyEx (HKEY_LOCAL_MACHINE, _T(kRegPathPGP),
				0, KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS )
	{

		DWORD dwType, dwDataLen;
		char  chPath[MAX_PATH];
		if ( RegQueryValueEx (hKey, _T(kStrInstPath), NULL, 
			&dwType, (unsigned char*)chPath, &dwDataLen) == ERROR_SUCCESS )
			m_strPGPPath = CString(chPath);

		RegCloseKey(hKey);
		return true;
	}
	else
		return false;
}


BOOL CPGPManager::ObtainPassphrase()
{
	CPassDlg passDlg;
	if ( passDlg.DoModal() == IDOK )
		m_strPassphrase = passDlg.m_strPass;

	if ( m_strPassphrase.IsEmpty() )
		return false;
	else
		return true;
}


void CPGPManager::ForgetPassphrase()
{
	m_strPassphrase = "";
	((CPGPICQDlg*)(AfxGetApp()->m_pMainWnd))->KillTimer(kPassTimerId);
}


BOOL CPGPManager::Initialize()
{
	int nErr;

	try {

		// Initialize the PGPsdk libs
		nErr = PGPsdkInit();
		if ( IsPGPError(nErr) )
			return false;

		// Create the PGPContext
		nErr = PGPNewContext (kPGPsdkAPIVersion, &m_pgpContext);
		if ( IsPGPError(nErr) )
			return false;
	
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::Initialize()");
		e->Delete();
		return false;
	}	


	if ( !OpenKeyrings() )
	{
		if ( MessageBox ( NULL, _T(kMsgPgpInitNeeded), _T(kTitleNeutral), 
						  MB_OKCANCEL|MB_APPLMODAL|MB_ICONEXCLAMATION) != IDOK )
			return false;
		else
		{

			char  chInstPath[MAX_PATH];
			HKEY  hKey;

			if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathPgpIcq), 0, 
								KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS )
			{

				DWORD dwType, dwDataLen;
				dwDataLen = MAX_PATH;
				
				if ( RegQueryValueEx (hKey, _T(kStrInstPath), NULL, &dwType, 
					(unsigned char*)chInstPath, &dwDataLen) == ERROR_SUCCESS )

				RegCloseKey(hKey);
			}
			else
			{
				::MessageBox (NULL, _T(kMsgReinstall), _T(kTitleError), MB_APPLMODAL|MB_ICONEXCLAMATION);
				exit(0);
				return false;
			}


			CString m_strPath;
			m_strPath.Format ("%s\\%s", chInstPath, kStrPathPgpKeys);
			if ( int(::ShellExecute ( NULL, "open", m_strPath, NULL, "", SW_SHOWNORMAL)) < 32 )
			{
				::MessageBox (NULL, _T(kMsgPgpKeysFailed), _T(kTitleError), MB_APPLMODAL|MB_ICONEXCLAMATION);
				return false;
			}

			exit(0);
			return false;
		}
	}
	else
		ReleaseKeyrings();

	LoadPairs();

	return true;
}


BOOL CPGPManager::GetUserList(StringListT *vecUserList)
{
	int nErr;
	PGPKeyListRef	keyList;
	PGPKeyIterRef	keyIter;
	PGPKeyRef		key;
	PGPUserIDRef	userID;
	char			chName[MAX_PATH];
	PGPSize			usedLength;

	try 
	{
		// Open the default keyring
		if ( !OpenKeyrings() )
			return false;

		// Create a keylist from keyset
		nErr = PGPOrderKeySet (m_defKeySet, kPGPAnyOrdering, &keyList );
		if ( IsPGPError(nErr) )
			return false;

		// Create a key iterator for browsing thru the keys
		nErr = PGPNewKeyIter (keyList, &keyIter);
		if ( IsPGPError(nErr) )
			return false;
	
		key = NULL;

		do {
			nErr = PGPKeyIterNext ( keyIter, &key);
			if ( IsPGPError(nErr) )	
				break;
	
			nErr = PGPGetPrimaryUserID (key, &userID);
			if ( IsPGPError(nErr) )	
				break;

			nErr = PGPGetPrimaryUserIDNameBuffer (key, MAX_PATH, chName, &usedLength);
			if ( IsPGPError(nErr) )	
				break;

			vecUserList->push_back(chName);

		} while ( key != NULL );

	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::Initialize()");
		e->Delete();
		return false;
	}

	// clean up
	PGPFreeKeyIter (keyIter);
	PGPFreeKeyList (keyList);

	ReleaseKeyrings();

	return true;
}


BOOL CPGPManager::AddPair(CString strUin, PGPKeyIdT keyId)
{
	UinKeyPairStruct *pExistingPair = HasPair(strUin);
	HKEY hKey;

	try
	{
		if ( pExistingPair == NULL ) {
			// add new
			UinKeyPairStruct pair;
			pair.strUin = strUin;
			pair.keyId = keyId;
			m_UinKeyPairs.push_back(pair);
		}
		else// just modify keyId
			pExistingPair->keyId = keyId;
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::AddPair()");
		e->Delete();
		return false;
	}


	try
	{
		if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathAssociations),
					0, KEY_ALL_ACCESS, &hKey) != ERROR_SUCCESS ) 
		{
			if ( RegCreateKey (HKEY_CURRENT_USER, kRegPathAssociations, &hKey) != ERROR_SUCCESS )
			{
				MessageBox(NULL, CString(_T(kMsgSaveAssocFailed)), 
					CString(_T(kTitleError)), MB_APPLMODAL|MB_ICONEXCLAMATION);
				return false;
			}
		}

		RegSetValueEx (hKey, strUin, 0, REG_SZ,
				(unsigned char*)keyId.GetBuffer(keyId.GetLength()),
				keyId.GetLength());
		
		RegCloseKey(hKey);
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::AddPair()");
		e->Delete();
		return false;
	}
	
	return true;
}


BOOL CPGPManager::RemovePair(CString strUin)
{
	BOOL	bFound = false;
	HKEY	hKey;
	CString	strValue;

	try
	{
		vector<UinKeyPairStruct>::iterator first = m_UinKeyPairs.begin();
		vector<UinKeyPairStruct>::iterator last = m_UinKeyPairs.end();

		while ( first != last )	{
			if ( !first->strUin.Compare(strUin) ) {
				strValue = first->strUin;
				bFound = true;
				m_UinKeyPairs.erase(first);
				break;
			}
			first++;
		}
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::RemovePair()");
		e->Delete();
		return false;
	}

	if ( !bFound )
		return false;

	try
	{
		if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathAssociations),
				0, KEY_ALL_ACCESS, &hKey) == ERROR_SUCCESS ) 
		{
			RegDeleteValue (hKey, strValue);
 		}
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::RemovePair()");
		e->Delete();
		return false;
	}
	
	return true;
}


UinKeyPairStruct * CPGPManager::HasPair(CString strUin)
{
	try
	{
		vector<UinKeyPairStruct>::iterator first = m_UinKeyPairs.begin();
		vector<UinKeyPairStruct>::iterator last = m_UinKeyPairs.end();

		while ( first != last )	{
			if ( !first->strUin.Compare(strUin) )
				return first;
			first++;
		}
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::HasPair()");
		e->Delete();
	}

	return NULL;
}



BOOL CPGPManager::LoadPairs()
{
	HKEY	hKey;
	char	chVarName[MAX_PATH];
	DWORD	dwVarNameLen;
	unsigned char chVarValue[MAX_PATH];
	DWORD	dwVarValueLen;

	DWORD	dwIndex = 0;
	int		nRes = ERROR_SUCCESS;

	try 
	{
		if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathAssociations),
				0, KEY_QUERY_VALUE, &hKey) != ERROR_SUCCESS )
			return false;

		while ( nRes != ERROR_NO_MORE_ITEMS ) {

			dwVarNameLen = MAX_PATH;
			dwVarValueLen = MAX_PATH;

			nRes = RegEnumValue (hKey, dwIndex++, chVarName, 
				&dwVarNameLen, 0, NULL, chVarValue, &dwVarValueLen);

			if ( nRes == ERROR_SUCCESS )
			{
				UinKeyPairStruct pair;
				pair.strUin = chVarName;
				pair.keyId = chVarValue;
				m_UinKeyPairs.push_back(pair);
			}
 		}

		RegCloseKey(hKey);
		return true;
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::LoadPairs()");
		e->Delete();
		return false;
	}

}


BOOL CPGPManager::SavePairs()
{
	HKEY	hKey;

	try
	{
		if ( RegOpenKeyEx (HKEY_CURRENT_USER, _T(kRegPathAssociations),
				0, KEY_ALL_ACCESS, &hKey) != ERROR_SUCCESS ) 
		{
			if ( RegCreateKey (HKEY_CURRENT_USER, kRegPathAssociations, &hKey) != ERROR_SUCCESS )
			{
				MessageBox(NULL, CString(_T(kMsgSaveAssocFailed)), 
					CString(_T(kTitleError)), MB_APPLMODAL|MB_ICONEXCLAMATION);
				return false;
			}
		}
	
		vector<UinKeyPairStruct>::iterator first = m_UinKeyPairs.begin();
		vector<UinKeyPairStruct>::iterator last = m_UinKeyPairs.end();

		while ( first != last )	{
			RegSetValueEx (hKey, first->strUin, 0, REG_SZ,
			(unsigned char*)first->keyId.GetBuffer(first->keyId.GetLength()),
			first->keyId.GetLength());
 			first++;
		}

		RegCloseKey(hKey);
		return true;
	}
	catch (CException* e)
	{
		AfxMessageBox("Unexpected exeption in CPGPManager::RemovePair()");
		e->Delete();
		return false;
	}
}



long CPGPManager::EncryptMessage(CString keyId, char* chMessage, char* chEncMessage)
{
	int nErr = 0;
	char const *chUserID = keyId;

	PGPFilterRef		keyFilter = kInvalidPGPFilterRef;
	PGPFilterRef		recipientFilter = kInvalidPGPFilterRef;
	PGPFilterRef		ownerFilter = kInvalidPGPFilterRef;
	PGPKeySetRef		foundKeySet = kInvalidPGPKeySetRef;
	PGPKeyRef			privateKey = kInvalidPGPKeyRef;
	PGPKeyID			ownerKeyID;
	PGPSize				nOutBuffSize;
	PGPOptionListRef	inputBufferOpt;
	PGPOptionListRef	outputBufferOpt;


	// Open the default keyring
	if ( !OpenKeyrings() )
		nErr = -1;

	// Create a filter to look for the recipient's key
	if ( !IsPGPError(nErr) )
		nErr = PGPNewUserIDStringFilter (m_pgpContext, chUserID, 
			kPGPMatchSubString, &recipientFilter/*keyFilter*/);


	// Find owner's key, so the owner can decrypt sent messages
	if ( !IsPGPError(nErr) )
		nErr = PGPGetDefaultPrivateKey (m_defKeySet, &privateKey);

	if ( !IsPGPError(nErr) )
		nErr = PGPGetKeyIDFromKey (privateKey, &ownerKeyID);

	if ( !IsPGPError(nErr) )
		nErr = PGPNewKeyIDFilter (m_pgpContext, &ownerKeyID,
			&ownerFilter);


	// Join filters
	if ( !IsPGPError(nErr) )
		nErr = PGPUnionFilters (recipientFilter, ownerFilter, &keyFilter);


	// Look for the key
	if ( !IsPGPError(nErr) )
		nErr = PGPFilterKeySet (m_defKeySet, keyFilter, &foundKeySet);


	// Create buffer options
	if ( !IsPGPError(nErr) )
	{
		inputBufferOpt = PGPOInputBuffer (m_pgpContext, 
			chMessage, strlen(chMessage) );

		outputBufferOpt = PGPOOutputBuffer (m_pgpContext, 
			chEncMessage, MAX_BUFFER_LENGTH, &nOutBuffSize);
	}

		
	if ( !IsPGPError(nErr) )
		nErr = PGPEncode ( m_pgpContext,
			PGPOEncryptToKeySet (m_pgpContext, foundKeySet),
			inputBufferOpt,
			outputBufferOpt,
			PGPOArmorOutput (m_pgpContext, true),
			PGPOOutputLineEndType (m_pgpContext, kPGPLineEnd_CRLF),
			PGPOLastOption (m_pgpContext) );


	// Clean up
	if ( PGPKeySetRefIsValid (foundKeySet) )
		PGPFreeKeySet (foundKeySet);
	if( PGPFilterRefIsValid (keyFilter) )
		PGPFreeFilter (keyFilter);
	if( PGPFilterRefIsValid (ownerFilter) )
		PGPFreeFilter (ownerFilter);
	
	ReleaseKeyrings();

	//nOutBuffSize = MakeArmoredMessage (chEncMessage, nOutBuffSize);

	if ( !IsPGPError(nErr) )
		return nOutBuffSize;
	else
		return 0;
}


long CPGPManager::MakeArmoredMessage(char *chMessage, long nMessageLength)
{
	long	i;
	long	nArmorLen;
	char	ch;

	for (i=0; i<nMessageLength; i++)
	{
		ch = chMessage[i];
		if ( ch == 0 )
			chMessage[i] = 1;
	}

	nArmorLen = strlen(chMessage);
	return nArmorLen;


	/*
	char	*chArmoredMsg;
	long	i, j;
	long	nHoles;
	long	nArmorLen;
	char	ch;

	nHoles = 0;

	for (i=0; i<nMessageLength; i++)
	{
		ch = chMessage[i];
		if ( ch ==  )
			nHoles++;
	}

	nArmorLen = nMessageLength + nHoles*strlen(kPGPICQArmorBlock) - nHoles;
	chArmoredMsg = (char*) malloc(nArmorLen);

	for (i=0,j=0; i<nMessageLength; i++)
	{
		ch = chMessage[i];

		if ( ch != 0 )
			chArmoredMsg[j++] = ch;
		else
		{
			memcpy (chArmoredMsg+j, kPGPICQArmorBlock, strlen(kPGPICQArmorBlock));
			j += strlen(kPGPICQArmorBlock);
		}
	}

	chArmoredMsg[j] = 0;

	if ( nArmorLen == strlen(chArmoredMsg) )
	{
		strcpy (chMessage, chArmoredMsg);
		return nArmorLen;
	}
	else
		return 0;
	*/
}



long CPGPManager::MakeDearmoredMessage(char *chArmoredMessage)
{
	long	i;
	long	nDearmorLen;
	char	ch;

	nDearmorLen = strlen(chArmoredMessage);

	for (i=0; i<nDearmorLen; i++)
	{
		ch = chArmoredMessage[i];
		if ( ch == 1 )
			chArmoredMessage[i] = 0;
	}

	return nDearmorLen;

	/*
	char	*chDearmoredMsg;
	long	i, j;
	long	nBlocks;
	long	nDearmorLen;
	long	nMessageLen;
	char	chBlock[32];
	char   *chMessage;

	nBlocks = 0;

	chMessage = (char*)malloc(strlen(*chArmoredMessage));
	strcpy (chMessage, *chArmoredMessage);
	nMessageLen = strlen(chMessage);
	
	for (i=0; i < nMessageLen-strlen(kPGPICQArmorBlock); i++)
	{
		memcpy(chBlock, chMessage+i, strlen(kPGPICQArmorBlock));
		chBlock[strlen(kPGPICQArmorBlock)]=0;
		if ( !strcmp(chBlock, kPGPICQArmorBlock) )
			nBlocks++;
	}

	nDearmorLen = nMessageLen - nBlocks*strlen(kPGPICQArmorBlock) + nBlocks;
	chDearmoredMsg = (char*) malloc(nDearmorLen);
	
	for (i=0,j=0; i<nMessageLen; i++)
	{
		memcpy(chBlock, chMessage+j, strlen(kPGPICQArmorBlock));
		chBlock[strlen(kPGPICQArmorBlock)]=0;

		if ( !strcmp(chBlock, kPGPICQArmorBlock) )
		{
			chDearmoredMsg[i] = 0;
			j += strlen(kPGPICQArmorBlock);
		}
		else
			chDearmoredMsg[i] = chMessage[j++];
	}

	chDearmoredMsg[j] = 0;

	*chArmoredMessage = chDearmoredMsg;
	return nDearmorLen;
	*/
}



long CPGPManager::DecryptMessage(CString keyId, char *chMessage, char *chDecMessage)
{
	int	nErr = 0;
	char const *chUserID = keyId;

	PGPFilterRef		keyFilter = kInvalidPGPFilterRef;
	PGPKeySetRef		foundKeySet = kInvalidPGPKeySetRef;
	PGPKeyRef			privateKey = kInvalidPGPKeyRef;
	PGPSize				nOutBuffSize;
	PGPSize				nInBuffSize;
	PGPOptionListRef	inputBufferOpt;
	PGPOptionListRef	outputBufferOpt;


	nInBuffSize = MakeDearmoredMessage (chMessage);
	
	// Open the default keyring
	if ( !OpenKeyrings() )
		nErr = -1;

	// Get and validate the passphrase
	if ( m_strPassphrase == "" )
		if ( !ObtainPassphrase() )
			return kPGPError_MissingPassphrase;

	nErr = PGPGetDefaultPrivateKey (m_defKeySet, &privateKey);

	if ( !IsPGPError(nErr) )
		if ( !PGPPassphraseIsValid ( privateKey,
			PGPOPassphrase (m_pgpContext, m_strPassphrase),
			PGPOLastOption (m_pgpContext)) )
		{
			ForgetPassphrase();
			return kPGPError_BadPassphrase;
		}

	// Since the passphrase is valid, reset the cache timer
	((CPGPICQDlg*)(AfxGetApp()->m_pMainWnd))->ResetCacheTimer();

	// Create a filter to look for the key
	if ( !IsPGPError(nErr) )
		nErr = PGPNewUserIDStringFilter (m_pgpContext, chUserID, kPGPMatchSubString, &keyFilter);
	
	// Look for the key
	if ( !IsPGPError(nErr) )
		nErr = PGPFilterKeySet (m_defKeySet, keyFilter, &foundKeySet);

	// Create buffer options
	if ( !IsPGPError(nErr) )
	{
		inputBufferOpt = PGPOInputBuffer (m_pgpContext, 
			chMessage, nInBuffSize );

		outputBufferOpt = PGPOOutputBuffer (m_pgpContext, 
			chDecMessage, MAX_BUFFER_LENGTH, &nOutBuffSize);
	}

	// make a copy of the original text first
	//strcpy (chDecMessage, chMessage);

	if ( !IsPGPError(nErr) )
		nErr = PGPDecode ( m_pgpContext, 
			inputBufferOpt, 
			outputBufferOpt,
			PGPOKeySetRef (m_pgpContext, m_defKeySet),
			PGPOPassphrase (m_pgpContext, m_strPassphrase),
			PGPOLastOption (m_pgpContext) );


	// Clean up
	if ( PGPKeySetRefIsValid (foundKeySet) )
		PGPFreeKeySet (foundKeySet);
	if( PGPFilterRefIsValid (keyFilter) )
		PGPFreeFilter (keyFilter);

	ReleaseKeyrings();

	if ( !IsPGPError(nErr) )
		return nOutBuffSize;
	else
		return 0;
}



/*
enum PGPLineEndType_
{
	kPGPLineEnd_Default	= 0,
	kPGPLineEnd_LF		= 1,
	kPGPLineEnd_CR		= 2,
	kPGPLineEnd_CRLF	= (kPGPLineEnd_LF | kPGPLineEnd_CR),
	PGP_ENUM_FORCE( PGPLineEndType_ )
};
*/





long CPGPManager::GetPublicKey(CString keyId, char *chKey)
{
	int nErr = 0;

	PGPFilterRef		keyFilter = kInvalidPGPFilterRef;
	PGPKeySetRef		foundKeySet = kInvalidPGPKeySetRef;
	PGPKeyRef			privateKey = kInvalidPGPKeyRef;
	PGPKeyID			ownerKeyID;
	PGPSize				nOutBuffSize;
	PGPOptionListRef	outputBufferOpt;


	// Open the default keyring
	if ( !OpenKeyrings() )
		nErr = -1;


	if ( keyId.IsEmpty() )
	{
		// Add owner to the filter
		if ( !IsPGPError(nErr) )
			nErr = PGPGetDefaultPrivateKey (m_defKeySet, &privateKey);
		if ( !IsPGPError(nErr) )
			nErr = PGPGetKeyIDFromKey (privateKey, &ownerKeyID);
		if ( !IsPGPError(nErr) )
			nErr = PGPNewKeyIDFilter (m_pgpContext, &ownerKeyID, &keyFilter);
	}
	else
	{
		// Create a filter to look for the recipient's key
		if ( !IsPGPError(nErr) )
			nErr = PGPNewUserIDStringFilter (m_pgpContext, keyId, 
					kPGPMatchSubString, &keyFilter);
	}


	// Look for the key(s)
	if ( !IsPGPError(nErr) )
		nErr = PGPFilterKeySet (m_defKeySet, keyFilter, &foundKeySet);

	// Create the buffer
	if ( !IsPGPError(nErr) )
		outputBufferOpt = PGPOOutputBuffer (m_pgpContext, 
			chKey, MAX_BUFFER_LENGTH, &nOutBuffSize);

	// Export public keys(s) to the buffer

	nErr = PGPExportKeySet (
				foundKeySet,
				outputBufferOpt,
				PGPOArmorOutput (m_pgpContext, true),
				PGPONullOption (m_pgpContext),
				PGPOLastOption (m_pgpContext)
			);

	ASSERT ( nOutBuffSize == strlen(chKey) );

	// Clean up
	if ( PGPKeySetRefIsValid (foundKeySet) )
		PGPFreeKeySet (foundKeySet);
	if( PGPFilterRefIsValid (keyFilter) )
		PGPFreeFilter (keyFilter);

	ReleaseKeyrings();

	return nErr;
}



long CPGPManager::ImportPublicKey(char *chKey)
{
	int nErr = 0;

	PGPFilterRef		keyFilter = kInvalidPGPFilterRef;
	PGPKeySetRef		impKeySet = kInvalidPGPKeySetRef;
	PGPKeyRef			privateKey = kInvalidPGPKeyRef;
	PGPKeyListRef		keyList;
	PGPKeyIterRef		keyIter;
	PGPKeyRef			key;
	PGPOptionListRef	inputBufferOpt;
	char				chName[MAX_PATH];
	PGPSize				nUsedLength;

	CString				strMsg (_T(kMsgGotPublicKeys));


	// Create the buffer
	if ( !IsPGPError(nErr) )
		inputBufferOpt = PGPOInputBuffer (m_pgpContext, 
			chKey, strlen(chKey) );

	// Create a key set from the buffer
	nErr = PGPImportKeySet (
				m_pgpContext,
				&impKeySet,
				inputBufferOpt,
				PGPOLastOption(m_pgpContext)
			);

	// Create a keylist from keyset
	nErr = PGPOrderKeySet (impKeySet, kPGPUserIDOrdering, &keyList );
	if ( IsPGPError(nErr) )
		return -1;

	// Browse thru the keys
	nErr = PGPNewKeyIter (keyList, &keyIter);
	nErr = PGPKeyIterNext (keyIter, &key);

	while ( key )
	{
		PGPGetPrimaryUserIDNameBuffer (key, MAX_PATH, chName, &nUsedLength);
		strMsg += _T(chName);
		strMsg += _T("\n");
		nErr = PGPKeyIterNext ( keyIter, &key);
	}

	strMsg += _T(kMsgImportPublicKeys);

	if ( ::MessageBox (NULL, strMsg, _T(kTitleNeutral), 
			MB_APPLMODAL|MB_ICONASTERISK|MB_YESNO) == IDYES )
	{
		if ( OpenKeyrings() )
		{
			nErr = PGPAddKeys (impKeySet, m_defKeySet);

			if ( nErr = PGPCommitKeyRingChanges (m_defKeySet) )
				::MessageBox ( NULL, _T(kMsgErrorImport), _T(kTitleError),
					MB_APPLMODAL|MB_ICONEXCLAMATION|MB_OK);

			ReleaseKeyrings();
		}
		else
			nErr = -1;
	}
	
	
	// Clean up
	nErr = PGPFreeKeyIter (keyIter);
	nErr = PGPFreeKeyList (keyList);
	if ( PGPKeySetRefIsValid (impKeySet) )
		PGPFreeKeySet (impKeySet);

	return nErr;
}
