/************************************************************
MODULO: Openclos.c
	In questo modulo sono definite le funzioni che il driver
	di protocollo registra presso l'NDIS per apertura/chiusura
	e reset di una delle sue istanze.
************************************************************/
#include <basedef.h>
#include <vmm.h>
#include <ndis.h>
#include <vwin32.h>
#include "debug.h"
#include "packet.h"
#pragma VxD_LOCKED_CODE_SEG
#pragma VxD_LOCKED_DATA_SEG

void YieldExecution( void )
{
	VMMCall(Release_Time_Slice);
	VMMCall(Begin_Nest_Exec);
	VMMCall(Resume_Exec);
	VMMCall(End_Nest_Exec);
}

/************************************************************
Funzione registrata presso l'NDIS utile per aggiornare i dati
del VXD quando si aggiunge un nuovo driver MAC.
INPUT:	BindAdapterContext - Handler che verr utilizzato da 
				NDIS per chiamare la NdisCompleteBindAdapter.
		AdapterName - Nome (UNICODE) dell'adattore NIC su cui 
				si effettua il bind. 
		SystemSpecific1	- Punta ad un percorso nel file di 
				registro per la NdisOpenProtocolConfiguration.
		Systemspecific2 - Riservato per il sistema in uso.
OUTPUT:	Status - Variabile in cui verr ritornato il valore
				della operazione. Pu assumere valori:
			NDIS_STATUS_SUCCES 
				questo valore indica che l'op. si  conclusa 
				con successo e dunque  stato realizzato il
				bind con l'adattore di cui  fornito il nome
				in AdapterName.
			NDIS_STATUS_PENDING
				l'NDIS comunica che l'operazione sar completata
				in maniera asincrona chiamando la 
				NdisCompleteBindAdapter.
			NDIS_STATUS_XXX
				riporta un errore nel bind dovuto ad un errore
				a livello inferiore.
************************************************************/
VOID NDIS_API PacketBindAdapter( OUT PNDIS_STATUS Status,
						 IN  NDIS_HANDLE  BindAdapterContext,
						 IN  PNDIS_STRING AdapterName,
						 IN  PVOID        SystemSpecific1,
						 IN  PVOID        SystemSpecific2 )
{
	PDEVICE_EXTENSION	pde;
	POPEN_INSTANCE 	oiNew;
	NDIS_STATUS			nsErrorStatus, nsOpenStatus;
	UINT           	uiMedium;
	NDIS_MEDIUM    	nmMediumArray=NdisMedium802_3;
	UINT           	i;
	PWRAPPER_PROTOCOL_BLOCK				pWPBlock;
	PNDIS_PROTOCOL_CHARACTERISTICS	pNPChar;

	TRACE_ENTER( "BindAdapter" );
	/*Testa della lista degli adattatori aperti*/
	pde = GlobalDeviceExtension;
	/*Alloca un elemento che descrive un adattore*/
	NdisAllocateMemory( (PVOID *)&oiNew, sizeof( OPEN_INSTANCE ), 0, -1 );
	if ( oiNew == NULL ) 
	{
		*Status = NDIS_STATUS_RESOURCES;
		return;
	}
	NdisZeroMemory( (PVOID)oiNew, sizeof( OPEN_INSTANCE ) );
	/*Alloca un pool per gli header dei pacchetti*/
	NdisAllocatePacketPool( &nsErrorStatus,
							&(oiNew->PacketPool),
							TRANSMIT_PACKETS,
							sizeof(PACKET_RESERVED) );
	IF_TRACE_MSG( "PACKET_RESERVED :%lx",sizeof(PACKET_RESERVED));
	if ( nsErrorStatus != NDIS_STATUS_SUCCESS ) 
	{
		IF_TRACE_MSG( "Failed to allocate packet pool AllocStatus=%x", nsErrorStatus );
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		*Status = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "BindAdapter" );
		return;
	}
	/*Alloca un pool per i dati dei pacchetti*/
	NdisAllocateBufferPool( &nsErrorStatus,
							&(oiNew->BufferPool),
							TRANSMIT_PACKETS );
	if ( nsErrorStatus != NDIS_STATUS_SUCCESS )
	{
		IF_TRACE_MSG( "Failed to allocate packet pool AllocStatus=%x", nsErrorStatus );
		NdisFreePacketPool( oiNew->PacketPool );
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		*Status = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "BindAdapter" );
		return;
	}
	/*Inizializza la lista di richiste di reset al 
		dispositivo.*/
	NdisAllocateSpinLock( &(oiNew->ResetSpinLock) );
	InitializeListHead( &(oiNew->ResetIrpList) );
	/*Inizializza la lista di richieste di ricezione 
		pacchetti al dispositivo.*/
	NdisAllocateSpinLock( &(oiNew->RcvQSpinLock) )
	InitializeListHead( &(oiNew->RcvList) );
	/*Inizializza la lista richieste al dispositivo. */
	NdisAllocateSpinLock( &(oiNew->RequestSpinLock) );
	InitializeListHead( &(oiNew->RequestList) );
	/*Trasferisce le richieste interne nella lista delle 
		richieste*/
	for ( i=0;i<MAX_REQUESTS;i++ ) 
	{
		InsertTailList( &(oiNew->RequestList), &(oiNew->Requests[i].Reserved.ListElement) );
	}
	oiNew->Status = NDIS_STATUS_PENDING;
	oiNew->BindAdapterContext = BindAdapterContext;
	/*Apre il driver Mac chiamando l'NDIS*/
	NdisOpenAdapter( &nsOpenStatus,
					 &nsErrorStatus,
					 &oiNew->AdapterHandle,
					 &uiMedium,
					 &nmMediumArray,
					 1,
					 pde->NdisProtocolHandle,
					 oiNew,
					 AdapterName,
					 0,
					 NULL );
	IF_TRACE_MSG( "Open Status                   : %lx", nsOpenStatus );
	IF_TRACE_MSG( "Error Status                  : %lx", nsErrorStatus );
	IF_TRACE_MSG( "Completion Status             : %lx", oiNew->Status );
	if ( nsOpenStatus == NDIS_STATUS_PENDING )
	{
		/*Se apertura asincrona, rilascia la schedulazione
			in attesa che finisca*/
		while ( oiNew->Status == NDIS_STATUS_PENDING )
			YieldExecution();
	}
	else
	{
		/*Se l'apertura si  gi conclusa allora completa*/
		PacketBindAdapterComplete( oiNew, nsOpenStatus, nsErrorStatus );
	}
	pWPBlock = ((PWRAPPER_OPEN_BLOCK)(oiNew->AdapterHandle))->ProtocolHandle;
	pNPChar  = &pWPBlock->ProtocolCharacteristics;
	IF_TRACE_MSG( "Protocol                      : %s",  pNPChar->Name.Buffer );
	IF_TRACE_MSG( "Protocol Handle               : %lx", pde->NdisProtocolHandle );
	IF_TRACE_MSG( "PWRAPPER_OPEN_BLOCK           : %lx", oiNew->AdapterHandle );
	IF_TRACE_MSG( "PWRAPPER_PROTOCOL_BLOCK       : %lx", pWPBlock );
	IF_TRACE_MSG( "NDIS_PROTOCOL_CHARACTERISTICS : %lx", pNPChar );
	IF_TRACE_MSG( "Name                          : %lx", &pNPChar->Name );
	IF_TRACE_MSG( "Adapter Name                  : %s",  AdapterName->Buffer );
	*Status = oiNew->Status;
	/*Nel caso di errore, si libera la memoria*/
	if ( *Status != NDIS_STATUS_SUCCESS ) 
	{
		NdisFreeMemory( oiNew, sizeof( OPEN_INSTANCE ) ,  0 );
		IF_TRACE( "Bind Operation FAILED!" );
	}
	else
	{
		/*Qui si possono inserire eventuali inizializzazioni
			per personalizzare il driver all'apertura.*/
	}
	TRACE_LEAVE( "BindAdapter" );
	return;
}

/************************************************************
Funzione richiamata dall'NDIS quando occorre completare 
asincronamente l'apertura di un nuovo driver a livello MAC.
INPUT:	ProtocolOpenAdapterComplete - Handle ad una struttura
			per-bind del driver, cio una struttura contenete 
			dati specifici all'istanza del bind completato.
OUTPUT:	Status - Ritorna lo stato dell'operazione che a questo
			punto pu solo essere NDIS_STATUS_SUCCES o nel
			caso di problemi nel NIC NDIS_STATUS_ADAPTER_NOT_READY
		OpenErrorStatus - Specifica ulteriormente l'errore
			segnalato da status.
************************************************************/
VOID NDIS_API
PacketBindAdapterComplete(
   IN NDIS_HANDLE  ProtocolBindingContext,
   IN NDIS_STATUS  Status,
   IN NDIS_STATUS  OpenErrorStatus )
{
	POPEN_INSTANCE	Open;
	TRACE_ENTER( "BindAdapterComplete" );
	IF_TRACE_MSG2( "ErrorStatus=%x Status=%x", OpenErrorStatus, Status );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	if ( Status == NDIS_STATUS_SUCCESS ) 
	{
		/*Inserisce il nuovo NIc aperto nella lista di quelli
			gi inizializzati*/
		InsertTailList( &GlobalDeviceExtension->OpenList, &Open->ListElement );
	}
	else
	{
		/*Libera le risorse occupate da driver.*/
		PacketFreeResources( Open );
	}
	Open->Status = Status;
	/*Completa segnalando all'NDIS l'avvenuto completamento 
		dell'operazione*/
	NdisCompleteBindAdapter( Open->BindAdapterContext, Status, OpenErrorStatus );
	TRACE_LEAVE( "BindAdapterComplete" );
	return;
}

/************************************************************
Funzione che inizia l'operazione di unbind di un NIC dal 
driver di protocollo.
INPUT:	ProtocolBindingContext - Handle della struttura per-bind
			da liberare.
		UnbindContext - Handle fornito dall'NDIS che verr poi
			usato dalla NdisCompleteUnbindAdapter per completare
			la fine delle operazioni.
OUTPUT:	Status - Stato della operazione. Vedere la PacketBindAdapter
			per i possibili valori.
************************************************************/
VOID NDIS_API
PacketUnbindAdapter( OUT PNDIS_STATUS	Status,
					 IN  NDIS_HANDLE	ProtocolBindingContext,
					 IN  NDIS_HANDLE	UnbindContext )
{
	POPEN_INSTANCE	Open;
	NDIS_STATUS		nsCloseStatus;
	TRACE_ENTER( "CloseAdapter" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	/*Tiene traccia dell'handle da usare con NdisCompleteUnbindAdapter*/
	Open->BindAdapterContext = UnbindContext;
	/*Ripulisce richieste pendenti*/
	PacketCleanUp( Status, Open );
	Open->Status = NDIS_STATUS_PENDING;
	/*Invoca l'NDIS per chiudere l'adattatore scelto*/
	NdisCloseAdapter( &nsCloseStatus, Open->AdapterHandle );
	if ( nsCloseStatus == NDIS_STATUS_PENDING )
	{
		while ( Open->Status == NDIS_STATUS_PENDING )
			YieldExecution();
	}
	else
	{
		PacketUnbindAdapterComplete( Open, nsCloseStatus );
	}
	*Status = Open->Status;
	if ( *Status == NDIS_STATUS_SUCCESS )
	{
		NdisFreeMemory( Open, sizeof( OPEN_INSTANCE ) ,  0 );
	}
	else
	{
		IF_TRACE( "Unbind Operation FAILED!" );
	}
	TRACE_LEAVE( "CloseAdapter" );
	return;
}

/************************************************************
Funzione che completa l'operazione di unbind di un NIC dal 
driver di protocollo.
INPUT:	ProtocolBindingContext - Handle della struttura per-bind
			da liberare.
OUTPUT:	Status - Stato della operazione. 
************************************************************/
VOID NDIS_API 
PacketUnbindAdapterComplete( IN NDIS_HANDLE ProtocolBindingContext,
							 IN NDIS_STATUS Status	)
{
	POPEN_INSTANCE Open;
	TRACE_ENTER( "UnbindAdapterComplete" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	if ( Status == NDIS_STATUS_SUCCESS )
	{
		PacketFreeResources( Open );
	}
	Open->Status = Status;
	/*Invoca l'NDIS per completare l'unbind*/
	NdisCompleteUnbindAdapter( Open->BindAdapterContext, Status );
	TRACE_LEAVE( "UnbindAdapterComplete" );
	return;
}

/************************************************************
Funzione che libera le risorse occupate da un adattore.
INPUT:	Open - Struttura che descrive l'istanza da liberare.
************************************************************/
VOID PacketFreeResources( POPEN_INSTANCE Open )
{
	RemoveEntryList( &Open->ListElement );
	NdisFreeSpinLock( &Open->RequestSpinLock );
	NdisFreeSpinLock( &Open->RcvQSpinLock );
	NdisFreeSpinLock( &Open->ResetSpinLock );
	NdisFreeBufferPool( Open->BufferPool );
	NdisFreePacketPool( Open->PacketPool );
}

/************************************************************
Funzione che libera le richieste pendenti.
INPUT:	Open - Struttura che descrive l'istanza da azzerare.
OUTPUT:	Status - Stato dell'operazione.
************************************************************/
VOID
PacketCleanUp(	PNDIS_STATUS	Status,
				POPEN_INSTANCE	Open )
{
	PLIST_ENTRY			PacketListEntry;
	PNDIS_PACKET   	pPacket;
	PPACKET_RESERVED  Reserved;
	TRACE_ENTER( "Cleanup" );
	/*Ripuliamo prima tutte le richieste pendenti*/
	NdisAcquireSpinLock( &(Open->RcvQSpinLock) );
	while( (PacketListEntry = PacketRemoveHeadList( &(Open->RcvList) )) != NULL )
	{
		IF_VERY_LOUD( "CleanUp - Completing read" );
		Reserved = CONTAINING_RECORD( PacketListEntry, PACKET_RESERVED, ListElement );
		pPacket  = CONTAINING_RECORD( Reserved, NDIS_PACKET, ProtocolReserved );
		/*Emula la fine di un trasferimento dati al fine di sbloccare processi 
			in attesa della loro fine.*/
		PacketTransferDataComplete( Open, pPacket, NDIS_STATUS_SUCCESS, 0 );
	}
	NdisReleaseSpinLock( &(Open->RcvQSpinLock) );
	PacketReset( Status, Open );
	TRACE_LEAVE( "Cleanup" );
	return;
}

/************************************************************
Funzione che inizia il reset di una istanza del driver.
INPUT:	pOpen - Istanza da ripulire.
OUTPUT:	pStatus - Stato dell'operazione.
************************************************************/
VOID
PacketReset( PNDIS_STATUS	pStatus,
			 POPEN_INSTANCE	pOpen )
{
	PLIST_ENTRY	ResetListEntry;
	TRACE_ENTER( "PacketReset" );
	/*Rileva la lista delle richieste per inserirla in coda a quella delle 
		richieste di reset.*/
	NdisAcquireSpinLock( &pOpen->RequestSpinLock );
	ResetListEntry = PacketRemoveHeadList( &pOpen->RequestList );
	NdisReleaseSpinLock( &pOpen->RequestSpinLock );
	if ( ResetListEntry == NULL ) 
	{
		*pStatus = NDIS_STATUS_RESOURCES;
		TRACE_LEAVE( "PacketReset" );
		return;
	}
	NdisAcquireSpinLock( &pOpen->ResetSpinLock );
	InsertTailList( &pOpen->ResetIrpList, ResetListEntry );
	NdisReleaseSpinLock( &pOpen->ResetSpinLock );
	/*Richiama l'NDIS per effettuare il reset dell'adattore.*/
	NdisReset( pStatus, pOpen->AdapterHandle );
	if ( *pStatus != NDIS_STATUS_PENDING ) 
	{
		/*Reset sincrono dell'adattore*/
		PacketResetComplete( pOpen, *pStatus );
	}
	TRACE_LEAVE( "PacketReset" );
	return;
}
	
/************************************************************
Funzione che completa il reset di una istanza.
INPUT:	ProtocolBindingContext - Istanza da ripulire.
OUTPUT:	Status - Stato dell'operazione.
************************************************************/
VOID NDIS_API
PacketResetComplete( IN NDIS_HANDLE	ProtocolBindingContext,
					 IN NDIS_STATUS	Status   )
{
	POPEN_INSTANCE		Open;
	PLIST_ENTRY			ResetListEntry;
	TRACE_ENTER( "PacketResetComplete" );
	Open = (POPEN_INSTANCE)ProtocolBindingContext;
	/*Ripulisce la coda delle richieste di reset.*/
	NdisAcquireSpinLock( &Open->ResetSpinLock );
	ResetListEntry = PacketRemoveHeadList( &Open->ResetIrpList );
	NdisReleaseSpinLock( &Open->ResetSpinLock );
	if ( ResetListEntry == NULL ) 
	{
		IF_VERY_LOUD( "Reset List Empty Error" );
		TRACE_LEAVE( "PacketResetComplete" );
		return;
	}
	/*Passare le richieste di reset nella lista delle richieste*/
	NdisAcquireSpinLock( &Open->RequestSpinLock );
	InsertTailList( &Open->RequestList, ResetListEntry );
	NdisReleaseSpinLock( &Open->RequestSpinLock );
	TRACE_LEAVE( "PacketResetComplete" );
	return;
}
