/*
 * rpc scanner - Simple Nomad
 *
 * Adapted for VLAD usage from a private NMRC collection.
 *
 * $Id: rpcscan.c,v 1.4 2001/07/17 20:40:41 ccoffin Exp $
 * $Log: rpcscan.c,v $
 * Revision 1.4  2001/07/17 20:40:41  ccoffin
 * added Sans number headers to both bannereater.c and rpcscan.c
 *
 * Revision 1.3  2001/07/17 17:10:20  loveless
 * Removed references to the -q parameter (which doesn't exist anymore).
 *
 * Revision 1.2  2001/06/28 21:19:04  loveless
 * Major changes. This code does 'exploit' mode where it shows details on
 * services it is aware of as exploitable. It's also handy for straight
 * command-line exploration of what rpc services are running on a *nix target.
 *
 */
#include <stdio.h>
#include <unistd.h>
#ifndef __OpenBSD__
#include <getopt.h>
#endif

#include <sys/types.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <rpc/pmap_prot.h>
#include <arpa/inet.h>
#ifdef __OpenBSD__
#include <sys/socket.h>
#include <stdlib.h>
#include <limits.h>
#endif

#define PORTMAP_PORT 111

void caughtsig(int sig)
{
  fprintf(stdout,"Unable to talk to host\n\n");  
  exit(sig);
}

void usage(char *prog)
{
  fprintf(stdout,"USAGE:\n");
  fprintf(stdout,"%s [opts] hostname\n",prog);
  fprintf(stdout,"  options include\n");
  fprintf(stdout,"    -t timeout in seconds (default 10)\n");
  fprintf(stdout,"    -v verbose\n");
  fprintf(stdout,"    -e exploit mode, report known exploitable services\n");
  fprintf(stdout,"    -h this screen\n");
  fprintf(stdout,"\n");
}

void print_info(unsigned long vers, unsigned long prot, unsigned long port, int pprot)
{
  fprintf(stdout," vers: %u  ",vers);
  if (pprot)
    switch(prot)
    {
      case 6: fprintf(stdout,"tcp port: ");break;
      case 17: fprintf(stdout,"udp port: ");break;
      default: fprintf(stdout,"unknown protocol port: ");break;
    }
  else fprintf(stdout,"port: ");
  fprintf(stdout,"%u\n",port);
}
int main(int argc, char **argv)
{
  struct sockaddr_in saddr;
  struct hostent *hp;
  register struct pmaplist *list;
  register struct rpcent *rpc;
  int verbose = 0, exploit = 0, sans = 0;
  int timeout = 10, ret;
  int rpcbind = 0, pcnfs = 0, tooltalk = 0, mountd = 0, cmsd = 0;
  int sadmind = 0, statd = 0;
  char ch, *prog;
  char buf[1024];
  

  prog = argv[0];

  if (argc <= 1)
  {
    usage(prog);
    exit(-1);      
  }

  while ((ch = getopt(argc, argv, "set:hv")) != EOF)
    switch(ch)
    {
      case 's':
        sans = 1;
        break;
      case 'e':
        exploit = 1;
        break;
      case 't':
        timeout = (int) strtol(optarg, NULL, 10);
        break;
      case 'h':
        usage(prog);
        exit(-1);
      case 'v':
        verbose = 1;
        break;
      default:
        usage(prog);
        exit(-1);
    }

  argc -= optind;
  argv += optind;  

  if (sans)
  {
    verbose = 0;
    exploit = 1;
  }

  if (verbose) fprintf(stdout,"RPC Scanner -- v1.0\nSimple Nomad\nhttp://www.nmrc.org\n");

  if (sans) fprintf(stdout,"\n#3 and #6 - Checks for the presence of possible vulnerable RPC services\n\n");
  
  memset(&saddr, 0, sizeof(saddr));
  memset(&hp, 0, sizeof(hp));

  signal(SIGALRM, caughtsig);
  alarm(timeout);

  hp = gethostbyname(*argv);
  if (hp == NULL)
  {
    fprintf(stdout,"ERROR: Unable to resolve %s, unknown host\n", argv[1]);
    exit(-1);
  }

  saddr.sin_addr = *(struct in_addr *)*hp->h_addr_list; 
  saddr.sin_family = PF_INET;
  saddr.sin_port = htons(PORTMAP_PORT);

  if (verbose) fprintf(stdout,"Checking %s (%s)...\n",hp->h_name,inet_ntoa(saddr.sin_addr));

  for (list = pmap_getmaps(&saddr); list; list = list->pml_next)
  {
    rpc = getrpcbynumber((int) list->pml_map.pm_prog);
    if (exploit)
    {
      switch(list->pml_map.pm_prog)
      {
        case 100000:
          if (rpcbind) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    rpcbind        ");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/rpcbind");
            ret=system(buf);
          }
          rpcbind++;
          break;
        case 100001:
          if (statd) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    rstatd         ");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/statd");
            ret=system(buf);
          } 
          statd++;
          break;
        case 100005:
          if (mountd) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    mountd         ");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/mountd");
            ret=system(buf);
          }
          mountd++;
          break;
        case 100068:
          if (cmsd) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    rpc.cmsd       ");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/cmsd");
            ret=system(buf);
          }
          cmsd++;
          break;
        case 100083:
          if (tooltalk) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    rpc.ttdbserverd");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/tooltalk");
            ret=system(buf);
          }
          tooltalk++;
          break;
        case 100232:
          if (sadmind) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    sadmind        ");
          if (verbose) fprintf(stdout," - Solstice");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/sadmind");
            ret=system(buf);
          }
          sadmind++;
          break;
        case 150001:
          if (pcnfs) break;
          if (!sans) fprintf(stdout,"\n%u ",list->pml_map.pm_prog);
          fprintf(stdout,"    pcnfs          ");
          print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,0);
          fprintf(stdout,"Severity: High  Possible remote root compromise\n\n");
          if ((verbose)||(sans))
          {
            sprintf(buf,"cat rpc/pcnfs");
            ret=system(buf);
          }
          pcnfs++;
          break;
        default:
          break;
      }
    }
    else 
    {
      fprintf(stdout,"%u ",list->pml_map.pm_prog);
      switch(list->pml_map.pm_prog)
      {
        case 100000: fprintf(stdout,"    rpcbind        ");break;
        case 100001: fprintf(stdout,"    rstatd         ");break;
        case 100002: fprintf(stdout,"    rusersd        ");break;
        case 100003: fprintf(stdout,"    nfs            ");break;
        case 100004: fprintf(stdout,"    ypserv         ");break;
        case 100005: fprintf(stdout,"    mountd         ");break;
        case 100006: fprintf(stdout,"    remote_dbx     ");break;
        case 100007: fprintf(stdout,"    ypbind         ");break;
        case 100008: fprintf(stdout,"    walld          ");break;
        case 100009: fprintf(stdout,"    yppasswdd      ");break;
        case 100010: fprintf(stdout,"    etherstatd     ");break;
        case 100011: fprintf(stdout,"    rquotad        ");break;
        case 100012: fprintf(stdout,"    sprayd         ");break;
        case 100013: fprintf(stdout,"    3270_mapper    ");break;
        case 100014: fprintf(stdout,"    rje_mapper     ");break;
        case 100015: fprintf(stdout,"    selection_svc  ");break;
        case 100016: fprintf(stdout,"    database_svc   ");break;
        case 100017: fprintf(stdout,"    rexd           ");break;
        case 100018: fprintf(stdout,"    alis           ");break;
        case 100019: fprintf(stdout,"    sched          ");break;
        case 100020: fprintf(stdout,"    llockmgr       ");break;
        case 100021: fprintf(stdout,"    nlockmgr       ");break;
        case 100022: fprintf(stdout,"    x25.inr        ");break;
        case 100023: fprintf(stdout,"    statmon        ");break;
        case 100024: fprintf(stdout,"    status         ");break;
        case 100025: fprintf(stdout,"    select_lib     ");break;
        case 100026: fprintf(stdout,"    bootparam      ");break;
        case 100027: fprintf(stdout,"    mazewars       ");break;
        case 100028: fprintf(stdout,"    ypupdated      ");break;
        case 100029: fprintf(stdout,"    keyserv        ");break;
        case 100030: fprintf(stdout,"    securelogin    ");break;
        case 100031: fprintf(stdout,"    nfs_fwdlnit    ");break;
        case 100032: fprintf(stdout,"    nfs_fwdtrns    ");break;
        case 100033: fprintf(stdout,"    sunlink_mapper ");break;
        case 100034: fprintf(stdout,"    net_monitor    ");break;
        case 100035: fprintf(stdout,"    database       ");break;
        case 100036: fprintf(stdout,"    passwd_auth    ");break;
        case 100037: fprintf(stdout,"    tfsd           ");break;
        case 100038: fprintf(stdout,"    nsed           ");break;
        case 100039: fprintf(stdout,"    nsemntd        ");break;
        case 100043: fprintf(stdout,"    showfhd        ");break;
        case 100055: fprintf(stdout,"    rpc.ioadmd     ");break;
        case 100062: fprintf(stdout,"    NETlicense     ");break;
        case 100065: fprintf(stdout,"    sunisamd       ");break;
        case 100066: fprintf(stdout,"    debug_svc      ");break;
        case 100068: fprintf(stdout,"    rpc.cmsd       ");break;
        case 100069: fprintf(stdout,"    rpc.ypxfrd     ");break;
        case 100071: fprintf(stdout,"    bugtraqd       ");break;
        case 100078: fprintf(stdout,"    kerbd          ");break;
        case 100083: fprintf(stdout,"    rpc.ttdbserverd");break;
        case 100087: fprintf(stdout,"    admind         ");break;
        case 100099: fprintf(stdout,"    autofsd        ");break;
        case 100101: fprintf(stdout,"    na.event       ");if (verbose) fprintf(stdout," - SunNet Manager"); break;
        case 100102: fprintf(stdout,"    na.logger      ");if (verbose) fprintf(stdout," - SunNet Manager"); break;
        case 100104: fprintf(stdout,"    na.sync        ");break;
        case 100105: fprintf(stdout,"    na.diskinfo    ");if (verbose) fprintf(stdout," - SunNet Manager"); break;
        case 100106: fprintf(stdout,"    na.iostat      ");break;
        case 100107: fprintf(stdout,"    na.hostperf    ");break;
        case 100109: fprintf(stdout,"    na.activity    ");if (verbose) fprintf(stdout," - SunNet Manager"); break;
        case 100110: fprintf(stdout,"    db_mgr         ");break;
        case 100111: fprintf(stdout,"    na.lpstat      ");if (verbose) fprintf(stdout," - SunNet Manager"); break;
        case 100112: fprintf(stdout,"    na.hostmem     ");break;
        case 100113: fprintf(stdout,"    na.sample      ");break;
        case 100114: fprintf(stdout,"    na.x25         ");break;
        case 100115: fprintf(stdout,"    na.ping        ");break;
        case 100116: fprintf(stdout,"    na.rpcnfs      ");break;
        case 100117: fprintf(stdout,"    na.hostif      ");break;
        case 100118: fprintf(stdout,"    na.etherif     ");break;
        case 100119: fprintf(stdout,"    na.ippath      ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 100120: fprintf(stdout,"    na.iproutes    ");break;
        case 100121: fprintf(stdout,"    na.layers      ");break;
        case 100122: fprintf(stdout,"    na.snmp        ");break;
        case 100123: fprintf(stdout,"    na.traffic     ");break;
        case 100124: fprintf(stdout,"    na.dni         ");if (verbose) fprintf(stdout," - DNIneT");break;
        case 100131: fprintf(stdout,"    na.layers2     ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 100133: fprintf(stdout,"    nsm_addrand    ");if (verbose) fprintf(stdout," - Solaris's statd");break;
        case 100135: fprintf(stdout,"    na.etherif2    ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 100136: fprintf(stdout,"    na.hostmem2    ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 100137: fprintf(stdout,"    na.iostat2     ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 100138: fprintf(stdout,"    na.snmpv2      ");if (verbose) fprintf(stdout," - SNM Version 2.2.2");break;
        case 100139: fprintf(stdout,"    cc_sender      ");if (verbose) fprintf(stdout," - Cooperative Consoles");break;
        case 100146: fprintf(stdout,"    amiserv        ");break;
        case 100147: fprintf(stdout,"    amiaux         ");break;
        case 100150: fprintf(stdout,"    ocfserv        ");break;
        case 100221: fprintf(stdout,"    kcms_server    ");if (verbose) fprintf(stdout," - SunKCMS Profile Server");break;
        case 100227: fprintf(stdout,"    nfs_acl        ");break;
        case 100229: fprintf(stdout,"    metad          ");if (verbose) fprintf(stdout," - rpc.metad");break;
        case 100230: fprintf(stdout,"    metamhd        ");if (verbose) fprintf(stdout," - rpc.metamhd");break;
        case 100232: fprintf(stdout,"    sadmind        ");if (verbose) fprintf(stdout," - Solstice");break;
        case 100233: fprintf(stdout,"    ufsd           ");break;
        case 100234: fprintf(stdout,"    gss            ");if (verbose) fprintf(stdout," - GSS Daemon");break;
        case 100235: fprintf(stdout,"    cachefsd       ");if (verbose) fprintf(stdout," - CacheFS Daemon");break;
        case 100249: fprintf(stdout,"    snmpXdmid      ");break;
        case 100300: fprintf(stdout,"    rpc.nisd       ");break;
        case 100301: fprintf(stdout,"    nis_cache      ");break;
        case 100302: fprintf(stdout,"    nis_callback   ");break;
        case 100303: fprintf(stdout,"    rpc.nispasswdd ");break;
        case 100304: fprintf(stdout,"    fnsypd         ");if (verbose) fprintf(stdout," - Federated Naming Service (FNS)");break;
        case 105004: fprintf(stdout,"    Protoserver    ");if (verbose) fprintf(stdout," - Advanced Printing Software");break;
        case 120126: fprintf(stdout,"    nf_snmd        ");if (verbose) fprintf(stdout," - SunNet Manager");break;
        case 120127: fprintf(stdout,"    nf_snmd        ");break;
        case 150001: fprintf(stdout,"    pcnfs          ");break;
        case 200000: fprintf(stdout,"    PyramidLock    ");break;
        case 200001: fprintf(stdout,"    PyramidSys5    ");if (verbose) fprintf(stdout," - Sys5");break;
        case 200002: fprintf(stdout,"    CADDS_Image    ");if (verbose) fprintf(stdout," - CV CADDS images.");break;
        case 200005: fprintf(stdout,"    pdbDA          ");break;
        case 200023: fprintf(stdout,"    exportmap      ");break;
        case 200201: fprintf(stdout,"    ecodisc        ");if (verbose) fprintf(stdout," - EcoTools daemons/programs");break;
        case 200202: fprintf(stdout,"    ecolic         ");if (verbose) fprintf(stdout," - EcoTools daemons/programs");break;
        case 200203: fprintf(stdout,"    eamon          ");if (verbose) fprintf(stdout," - EcoTools daemons/programs");break;
        case 200205: fprintf(stdout,"    ecoad          ");if (verbose) fprintf(stdout," - EcoTools daemons/programs");break;
        case 211637: fprintf(stdout,"    rpc.dbserv     ");break;
        case 217854: fprintf(stdout,"    rpc.taped      ");break;
        case 300001: fprintf(stdout,"    ADTFileLock    ");if (verbose) fprintf(stdout," - ADT file locking service.");break;
        case 300006: fprintf(stdout,"    fmclient       ");if (verbose) fprintf(stdout," - FrameMaker Client");break;
        case 300007: fprintf(stdout,"    fmeditor       ");if (verbose) fprintf(stdout," - FrameMaker Editor");break;
        case 300009: fprintf(stdout,"    fmserver       ");if (verbose) fprintf(stdout," - FrameMaker Server");break;
        case 300019: fprintf(stdout,"    amd            ");break;
        case 300029: fprintf(stdout,"    rpc.ldmd       ");if (verbose) fprintf(stdout," - Unidata LDM");break;
        case 300326: fprintf(stdout,"    dr_daemon      ");if (verbose) fprintf(stdout," - Sun Enterprise Server Alternate Pathing");break;
        case 300214: fprintf(stdout,"    fm_flb         ");if (verbose) fprintf(stdout," - FrameMaker");break;
        case 300215: fprintf(stdout,"    fm_fls         ");if (verbose) fprintf(stdout," - FrameMaker licnese server");break;
        case 300375: fprintf(stdout,"    rmd            ");if (verbose) fprintf(stdout," - FA&O Command Post App");break;
        case 300376: fprintf(stdout,"    agcd           ");if (verbose) fprintf(stdout," - FA&O Command Post App");break;
        case 300433: fprintf(stdout,"    bssd           ");break;
        case 300434: fprintf(stdout,"    drdd           ");break;
        case 300473: fprintf(stdout,"    ap_daemon      ");if (verbose) fprintf(stdout," - SUNWapu - Alternate Pathing (AP)");break;
        case 300483: fprintf(stdout,"    cnxmond        ");if (verbose) fprintf(stdout," - cluster node monitor (Digital UNIX)");break;
        case 300484: fprintf(stdout,"    cnxagentd      ");if (verbose) fprintf(stdout," - cluster agent (Digital UNIX)");break;
        case 300527: fprintf(stdout,"    cluinfod       ");if (verbose) fprintf(stdout," - cluster information server (Digital UNIX)");break;
        case 300598: fprintf(stdout,"    dmispd         ");if (verbose) fprintf(stdout," - Sun Solstice Enterprise DMI Service Provider");break;
        case 300632: fprintf(stdout,"    prpasswd       ");break;
        case 390100: fprintf(stdout,"    prestoctl_svc  ");if (verbose) fprintf(stdout," - Prestoserve control daemon");break;
        case 390101: fprintf(stdout,"    rap            ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390102: fprintf(stdout,"    rapserv        ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390103: fprintf(stdout,"    nsrd           ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390104: fprintf(stdout,"    nsrmmd         ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390105: fprintf(stdout,"    nsrindexd      ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390107: fprintf(stdout,"    nsrmmdbd       ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390109: fprintf(stdout,"    nsrstat        ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390110: fprintf(stdout,"    nsrjbd         ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390113: fprintf(stdout,"    nsrexecd       ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390400: fprintf(stdout,"    nsrnotd        ");if (verbose) fprintf(stdout," - Legato Networker");break;
        case 390600: fprintf(stdout,"    arserverd      ");if (verbose) fprintf(stdout," - Remedy AR System daemons");break;
        case 390601: fprintf(stdout,"    ntserverd      ");if (verbose) fprintf(stdout," - Remedy AR System daemons");break;
        case 390602: fprintf(stdout,"    ntclientd      ");if (verbose) fprintf(stdout," - Remedy AR System daemons");break;
        case 390604: fprintf(stdout,"    arservtcd      ");if (verbose) fprintf(stdout," - Remedy AR System daemons");break;
        case 391000: fprintf(stdout,"    sgi_snoopd     ");break;
        case 391001: fprintf(stdout,"    sgi_toolkitbus ");
        case 391002: fprintf(stdout,"    sgi_fam        ");break;
        case 391003: fprintf(stdout,"    sgi_notepad    ");break;
        case 391004: fprintf(stdout,"    sgi_mountd     ");if (verbose) fprintf(stdout," - showmount");break;
        case 391005: fprintf(stdout,"    sgi_smtd       ");break;
        case 391006: fprintf(stdout,"    sgi_pcsd       ");break;
        case 391007: fprintf(stdout,"    sgi_nfs        ");break;
        case 391008: fprintf(stdout,"    sgi_rfind      ");break;
        case 391009: fprintf(stdout,"    sgi_pod        ");break;
        case 391010: fprintf(stdout,"    sgi_iphone     ");break;
        case 391011: fprintf(stdout,"    sgi_videod     ");break;
        case 391012: fprintf(stdout,"    sgi_testcd     ");break;
        case 391013: fprintf(stdout,"    sgi_ha_hbeat   ");break;
        case 391014: fprintf(stdout,"    sgi.ha_nc      ");break;
        case 391015: fprintf(stdout,"    sgi.ha_appmon  ");break;
        case 391016: fprintf(stdout,"    sgi_xfsmd      ");break;
        case 391017: fprintf(stdout,"    sgi_mediad     ");break;
        case 391018: fprintf(stdout,"    sgi.ha_orcl    ");break;
        case 391019: fprintf(stdout,"    sgi.ha_ifmx    ");break;
        case 391020: fprintf(stdout,"    sgi.ha_sybs    ");break;
        case 391021: fprintf(stdout,"    sgi.ha_ifa     ");break;
        case 391022: fprintf(stdout,"    sgi_reserved   ");break;
        case 391023: fprintf(stdout,"    sgi_reserved   ");break;
        case 391024: fprintf(stdout,"    sgi_reserved   ");break;
        case 391025: fprintf(stdout,"    sgi_reserved   ");break;
        case 391026: fprintf(stdout,"    sgi_reserved   ");break;
        case 391027: fprintf(stdout,"    sgi_reserved   ");break;
        case 391028: fprintf(stdout,"    sgi_reserved   ");break;
        case 391029: fprintf(stdout,"    sgi_espd       ");break;
        case 391030: fprintf(stdout,"    sgi_reserved   ");break;
        case 391031: fprintf(stdout,"    sgi_reserved   ");break;
        case 391032: fprintf(stdout,"    sgi_reserved   ");break;
        case 391033: fprintf(stdout,"    sgi_reserved   ");break;
        case 391034: fprintf(stdout,"    sgi_reserved   ");break;
        case 391035: fprintf(stdout,"    sgi_reserved   ");break;
        case 391036: fprintf(stdout,"    sgi_reserved   ");break;
        case 391037: fprintf(stdout,"    sgi_reserved   ");break;
        case 391038: fprintf(stdout,"    sgi_reserved   ");break;
        case 391039: fprintf(stdout,"    sgi_reserved   ");break;
        case 391040: fprintf(stdout,"    sgi_reserved   ");break;
        case 391041: fprintf(stdout,"    sgi_reserved   ");break;
        case 391042: fprintf(stdout,"    sgi_reserved   ");break;
        case 391043: fprintf(stdout,"    sgi_reserved   ");break;
        case 391044: fprintf(stdout,"    sgi_reserved   ");break;
        case 391045: fprintf(stdout,"    sgi_reserved   ");break;
        case 391046: fprintf(stdout,"    sgi_reserved   ");break;
        case 391047: fprintf(stdout,"    sgi_reserved   ");break;
        case 391048: fprintf(stdout,"    sgi_reserved   ");break;
        case 391049: fprintf(stdout,"    sgi_reserved   ");break;
        case 391050: fprintf(stdout,"    sgi_reserved   ");break;
        case 391051: fprintf(stdout,"    sgi_reserved   ");break;
        case 391052: fprintf(stdout,"    sgi_reserved   ");break;
        case 391053: fprintf(stdout,"    sgi_reserved   ");break;
        case 391054: fprintf(stdout,"    sgi_reserved   ");break;
        case 391055: fprintf(stdout,"    sgi_reserved   ");break;
        case 391056: fprintf(stdout,"    sgi_reserved   ");break;
        case 391057: fprintf(stdout,"    sgi_reserved   ");break;
        case 391058: fprintf(stdout,"    sgi_reserved   ");break;
        case 391059: fprintf(stdout,"    sgi_reserved   ");break;
        case 391060: fprintf(stdout,"    sgi_reserved   ");break;
        case 391061: fprintf(stdout,"    sgi_reserved   ");break;
        case 391062: fprintf(stdout,"    sgi_reserved   ");break;
        case 391063: fprintf(stdout,"    sgi_reserved   ");break;
        case 395175: fprintf(stdout,"    asedirector    ");break;
        case 395176: fprintf(stdout,"    aseagent       ");break;
        case 395177: fprintf(stdout,"    asehsm         ");break;
        case 395179: fprintf(stdout,"    aselogger      ");break;
        case 20000000: fprintf(stdout,"  na.retherif    ");break;
        case 20000001: fprintf(stdout,"  na.genagnt     ");break;
        case 20000114: fprintf(stdout,"  rpc.acedbd     ");if (verbose) fprintf(stdout," - ACeDB database server");break;
        case 20000115: fprintf(stdout,"  rpc.aboutdb    ");if (verbose) fprintf(stdout," - ACeDB database server");break;
        case 20000116: fprintf(stdout,"  rpc.aatdb      ");if (verbose) fprintf(stdout," - ACeDB database server");break;
        case 20000777: fprintf(stdout,"  seagent        ");if (verbose) fprintf(stdout," - Memco/Platinum/CA SeOS security product");break;
        case 28000205: fprintf(stdout,"  egts           ");if (verbose) fprintf(stdout," - Eurogam Tape Server, (10205/udp)");break;
        case 28000220: fprintf(stdout,"  ers            ");if (verbose) fprintf(stdout," - Eurogam Register Server, (10220/udp)");break;
        case 28000230: fprintf(stdout,"  sas            ");if (verbose) fprintf(stdout," - Spectrum Access Server, (10230/udp)");break;
        case 536870915: fprintf(stdout," uidd           ");break;
        case 536870916: fprintf(stdout," userd          ");break;
        case 536870917: fprintf(stdout," bondd          ");break;
        case 536870918: fprintf(stdout," staticd        ");break;	
        case 536870919: fprintf(stdout," curved         ");break;
        case 536870920: fprintf(stdout," msglogd        ");break;	
        case 536870921: fprintf(stdout," aliasd         ");break;
        case 536870922: fprintf(stdout," ticketd        ");break;
        case 536870923: fprintf(stdout," glossd         ");break;
        case 536870924: fprintf(stdout," futured        ");break;
        case 536870928: fprintf(stdout," priced         ");break;
        case 536870929: fprintf(stdout," ladderd        ");break;
        case 536870937: fprintf(stdout," optiond        ");if (verbose) fprintf(stdout," - FIRM option server");break;
        case 545580417: fprintf(stdout," bwnfsd         ");break;
        case 555555554: fprintf(stdout," inetray.start  ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555555: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555556: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555557: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555558: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555559: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 555555560: fprintf(stdout," inetray        ");if (verbose) fprintf(stdout," - Inetray / Raytrace");break;
        case 600100069: fprintf(stdout," freebsd-ypxfrd ");break;
        case 788585389: fprintf(stdout," bwnfsd         ");if (verbose) fprintf(stdout," - (PC)NFS server by Beame & Whiteside, Inc.");break;
        case 805306368: fprintf(stdout," dmispd         ");if (verbose) fprintf(stdout," - Sun Solstice Enterprise DMI Service Provider");break;
        case 824395111: fprintf(stdout," cfsd           ");break;
        case 1073741824: fprintf(stdout,"fmproduct      ");if (verbose) fprintf(stdout," - _Frame_RPC, #FrameMaker");break;
        case 1092830567: fprintf(stdout,"cfsd           ");break;
        case 1289637086: fprintf(stdout,"dtcm           ");break;
        case 1342177279: fprintf(stdout,"ttsession      ");break;
        default: fprintf(stdout," *UNKNOWN* ");break;
      }
    print_info(list->pml_map.pm_vers,list->pml_map.pm_prot,list->pml_map.pm_port,1);
    }
  } 
  exit(0);
}
 
