#!/usr/bin/perl -w

use strict;
use IO::Socket;
use Socket;
use Getopt::Long;


# verbosity level
my $verbosity = 0;
my $timeout = 10;
my $version = "1.0";
my $quiet = 0;
my $help = 0;
my $found = 0;
my $sansmode = 0;

sub tcp_port_open {
    my ($host, $port) = @_;
    my $socket;
    my $rc;

    printf "connecting to %s:%s\n", $host, $port unless $verbosity < 1;
    socket (TCP_SOCK, PF_INET, SOCK_STREAM, getprotobyname("tcp"));

    my $addr = inet_aton ($host) or die "Couldn't inet_aton\n";
    my $paddr = sockaddr_in($port, $addr);

    eval {
        local $SIG{ALRM} = sub { die "alarm\n" };
        alarm $timeout;
        if (!connect (TCP_SOCK, $paddr)) {
            printf "can't connect\n" unless $verbosity < 1;
            $rc = 0;
        } else {
            $rc = 1;
        }
        alarm 0;
    };
    if ($@) {
        # timed out
        $rc = 0;
    }

    close (TCP_SOCK);
    return $rc;
}


sub udp_port_open {
    my ($host, $port) = @_;
    my $rc;

    socket (UDP_SOCK, PF_INET, SOCK_DGRAM, getprotobyname("udp"));
    my $ipaddr = inet_aton ($host);
    my $addr = sockaddr_in ($port, $ipaddr);
    send (UDP_SOCK, "hackershield", 0, $addr) or return 0;
    my $resp;
    eval {
        local $SIG{ALRM} = sub { die "alarm\n" };
        alarm $timeout;
        $addr = recv (UDP_SOCK, $resp, 500, 0);
        alarm 0;
    };
    if ($@) {
        $rc = 1;
    } else {
        $rc = 0;
    }
    close (UDP_SOCK);
    return $rc;
}


sub check_host {
    my ($host) = @_;

    if (tcp_port_open ($host, 139)) {
        printf "Found possible SMB server on port 139\n";
        $found = 1;
    }
    if (tcp_port_open ($host, 445)) {
        printf "Found possible SMB server on port 445\n";
        $found = 1;
    }
    if (tcp_port_open ($host, 427)) {
        printf "Found possible AppleShare/IP server on port 427\n";
        $found = 1;
    }
    if (tcp_port_open ($host, 548)) {
        printf "Found possible AppleShare/IP server on port 548\n";
        $found = 1;
    }
    if (udp_port_open ($host, 2049)) {
        printf "Possible NFS on port 2049 (unless udp is filtered/firewalled)\n";
        $found = 1;
    }
}

sub usage {
    printf "Usage: ./netfs.pl [options] hostname\n";
    printf "options are -v           (verbose, use more for more verbosity\n";
    printf "            -t timeout   (timeout (in secs) for connections, defaults to 30)\n";
    printf "            -h           (help)\n";
    printf "            -q           (quiet, suppress version info)\n";
    exit (0);
}



#
# main
#
#parse arguments
GetOptions ("v+" => \$verbosity,
            "t=i" => \$timeout,
            "q" => \$quiet,
            "s" => \$sansmode,
            "h" => \$help
            );
if ($help) { usage (); }

if ($sansmode) { $quiet = 1; $verbosity = 0; }

printf "RAZOR Network filesystem scanner - version: %s\n", $version unless $quiet;
printf "#7 - Global file sharing and inappropriate information sharing\n\n" unless !($sansmode);

my $host = shift    or usage ();
printf "checking %s\n", $host unless $sansmode;

check_host $host;

if ($sansmode)
{
  printf "  Possible vulnerabilities found. See Docs/shares.html for details.\n\n" unless !($found);
  printf "  No problems related to #7\n\n" unless ($found);
}

#
# netfs.pl  - check for network filesystems
#
# $Id: netfs.pl,v 1.1 2000/11/06 15:32:05 loveless Exp $
# $Log: netfs.pl,v $
# Revision 1.1  2000/11/06 15:32:05  loveless
# Moved to sectools section of CVS
#
# Revision 1.7  2000/07/24 17:14:43  loveless
# Updated Docs references to reflect changes to filenames
#
# Revision 1.6  2000/07/20 19:16:43  loveless
# Fixed reference to the Docs dir
#
# Revision 1.5  2000/07/18 20:47:19  loveless
# Added more verbage to NFS (2049) output
#
# Revision 1.4  2000/07/13 21:22:48  loveless
# Fixed typo in output.
#
# Revision 1.3  2000/06/28 21:38:48  loveless
# Changed default timeout to 10 seconds.
#
# Revision 1.2  2000/06/28 21:30:47  loveless
# Added hooks for vlad.pl script.
#
# Revision 1.1  2000/06/28 21:01:13  tsabin
# initial version
#
#
