/*

   Banner Eater 0.1
   ----------------
   Michal Zalewski <lcamtuf@bos.bindview.com>

$Id: bannereater.c,v 1.5 2001/07/17 20:40:41 ccoffin Exp $

$Log: bannereater.c,v $
Revision 1.5  2001/07/17 20:40:41  ccoffin
added Sans number headers to both bannereater.c and rpcscan.c

Revision 1.4  2001/06/29 21:29:19  loveless
Added a new routine for handling output, so everything is in one place. More
cleanup, added support for displaying description files, removed quiet mode
as it was not needed.

Revision 1.3  2001/06/29 19:08:26  loveless
Minor adjustment so we only scan ports when the identified OS matches the
database entry.

Revision 1.2  2001/06/28 22:27:18  loveless
For some reason I thought it would be fun to do rpc stuff from here, and
then I came to my senses. Removed all rpc code.


*/

#include <unistd.h>
#include <stdio.h>
#ifndef __OpenBSD__
#include <getopt.h>
#endif
#ifdef __OpenBSD__
#include <limits.h>
#endif
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <signal.h>

#include <sys/socket.h>
#include <sys/un.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "bannereater.h"

struct be_entry (*db)[];
struct be_port  (*port)[];
int    entries;
int    platform;
int    ports;
int    verbose;
int    sans;
int    timeout;
struct in_addr rhost;

#define fatal(x...) { fprintf(stderr,x); exit(1); }


/************************************************************
 * This function performs stateful configurationn database  *
 * processing. It is messy a little bit, as we have to work *
 * on human-generated textfile. Note that 'db' is dynamic,  *
 * and we have to resize it from time to time.              *
 ************************************************************/

void load_database(const char* fn) {

  FILE* d;
  char buf[MAX_LINE+1];
  int line=0;
  int state=STATE_TI;
  char tmp;
  int tmp2;

  d=fopen(fn,"r");
  if (!d) { perror(fn); exit(1); }
  buf[MAX_LINE]=0;

  while (!feof(d)) {

    fgets(buf,MAX_LINE,d); line++;
    if ((buf[0]=='#') || (!strchr(buf,':'))) continue;
    if (buf[strlen(buf)-1]=='\n') buf[strlen(buf)-1]=0;

    if (*(strchr(buf,':')+1)!=' ') 
      fatal("Expected ' ' after ':' in %s, line %d.\n",fn,line);

    switch (state) {

      case STATE_TI:

        if (strncasecmp(buf,"TI",2)) 
          fatal("Expected TI in %s, line %d.\n",fn,line);

        db=realloc(db,(entries+1)*sizeof(struct be_entry)); 
        (*db)[entries].title=strdup(strchr(buf,':')+2);
        state=STATE_DE;

        break;

      case STATE_DE:

        if (strncasecmp(buf,"DE",2)) 
          fatal("Expected DE in %s, line %d.\n",fn,line);

        (*db)[entries].description=strdup(strchr(buf,':')+2);
        state=STATE_PL;

        break;

      case STATE_PL:

        if (strncasecmp(buf,"PL",2)) 
          fatal("Expected PL in %s, line %d.\n",fn,line);

        tmp=tolower(*(strchr(buf,':')+2));

        if (tmp=='w') (*db)[entries].platform=PLATFORM_MS;
        else if (tmp=='n') (*db)[entries].platform=PLATFORM_NON_MS;
        else if (tmp=='a') (*db)[entries].platform=PLATFORM_ANY;
        else fatal("Unknown platform in %s, line %d.\n",fn,line);

        state=STATE_PO;

        break;

      case STATE_PO:

        if (strncasecmp(buf,"PO",2)) 
          fatal("Expected PO in %s, line %d.\n",fn,line);

        tmp2=(*db)[entries].port=atoi(strchr(buf,':')+2);
        if (tmp2<=0 || tmp2>65535) 
          fatal("Incorrect port number in %s, line %d.\n",fn,line);
        state=STATE_SE;

        break;

      case STATE_SE:

        if (strncasecmp(buf,"SE",2)) 
          fatal("Expected SE in %s, line %d.\n",fn,line);

        tmp=tolower(*(strchr(buf,':')+2));

        if (tmp=='l') (*db)[entries].severity=SEVERITY_LOW;
        else if (tmp=='m') (*db)[entries].severity=SEVERITY_MEDIUM;
        else if (tmp=='h') (*db)[entries].severity=SEVERITY_HIGH;
        else if (tmp=='u') (*db)[entries].severity=SEVERITY_UNKNOWN;
        else fatal("Incorrect severity in %s, line %d.\n",fn,line);

        state=STATE_CO;

        break;

      case STATE_CO:

        if (strncasecmp(buf,"CO",2)) 
          fatal("Expected CO in %s, line %d.\n",fn,line);

        tmp=tolower(*(strchr(buf,':')+2));

        if (tmp=='n') (*db)[entries].type=COMP_NONE;
        else if (tmp=='a') (*db)[entries].type=COMP_ANY;
        else if (tmp=='p') (*db)[entries].type=COMP_PARTIAL;
        else if (tmp=='f') (*db)[entries].type=COMP_FULL;
        else fatal("Incorrect comparsion in %s, line %d.\n",fn,line);

        state=STATE_BA;

        break;

      case STATE_BA:

        if (strncasecmp(buf,"BA",2)) 
          fatal("Expected BA in %s, line %d.\n",fn,line);

        (*db)[entries].lookfor=strdup(strchr(buf,':')+2);
        state=STATE_TI;
        entries++;

        break;

      default: fatal("Parser bug %d (%s:%d).\n",state,fn,line);

    }

  }

  if (state!=STATE_TI) fatal("Unexpected EOF in file %s.\n",fn);

  if (verbose) fprintf(stderr,"Database: %d lines (%d entries) loaded succesfully.\n",line,entries);

  fclose(d);

}

/********************************************************
 * This function performs ICMP-based OS fingerprinting. *
 ********************************************************/

void determine_os(const char* hostname) {
  char buf[1024];
  int ret;
  sprintf(buf,"./pingwin '%s' 2>&1 >/dev/null",hostname);
  ret=system(buf);
  switch (ret / 256) {

    case 0: 
       platform=PLATFORM_NON_MS;
       if (verbose) fprintf(stdout,"Detected OS: Unix, MacOS or other non-Microsoft.\n");
       break;

    case 1: 
       platform=PLATFORM_MS;
       if (verbose) fprintf(stdout,"Detected OS: Microsoft Windows.\n");
       break;

    case 100: 
       platform=PLATFORM_ANY;
       if (verbose) fprintf(stdout,"Unknown error while determining OS type. Assuming any OS.\n");
       if (verbose) fprintf(stdout,"NOTE: Try invoking './pingwin' manually for more info.\n");
       sleep(1);
       break;

    case 127: 
       platform=PLATFORM_ANY;
       if (verbose) fprintf(stdout,"Timeout while trying to determine OS type. Assuming any OS.\n");
       if (verbose) fprintf(stdout,"NOTE: This host might be down.\n");
       sleep(1);
       break;

    default:
      fatal("Cannot invoke './pingwin' utility to determine OS type\n");

  }
}



void sigalrm(int dummy) { /* Do nothing! */ }
struct sigaction siga;


/*******************************************************
 * This function gathers banners on all listed ports.  *
 * connect() and recv() is not performed if we already *
 * have a banner for given port already.               *
 *******************************************************/

void scan_host(void) {
  int i,j;
  char buf[BANNER_MAX+1];
  struct sockaddr_in sin;
  int f,bail_out;

  for (i=0;i<entries;i++) {

    // skip looking for a port on a system unless it matches the OS
    if (!(platform & (*db)[i].platform)) continue;

    // See if we have this port already...

    bail_out=0;

    for (j=0;j<ports;j++)
      if ((*db)[i].port == (*port)[j].port) { bail_out = 1; break; }

    // printf("Trying port %d, bail=%d\n",(*db)[i].port,bail_out);

    if (bail_out) continue;

    port=realloc(port,(ports+1)*sizeof(struct be_port));
    ports++;

    (*port)[ports-1].port=(*db)[i].port;
    (*port)[ports-1].banner="";
    (*port)[ports-1].status=STATUS_CONNFAIL;

    f=socket(PF_INET,SOCK_STREAM,0);
    if (f<0) fatal("socket");
    sin.sin_family = PF_INET;
    sin.sin_port=htons((*db)[i].port);
    memcpy(&sin.sin_addr,&rhost,sizeof(rhost));

    siga.sa_handler=sigalrm;
    sigaction(SIGALRM,&siga,0);
    alarm(timeout);

    if (connect(f,&sin,sizeof(struct sockaddr_in))) {
      alarm(0);
      close(f);
      // printf("  - no connection\n");
      continue;
    }

    (*port)[ports-1].status=STATUS_NOBANNER;
    bzero(buf,sizeof(buf));

    sigaction(SIGALRM,&siga,0);
    alarm(timeout);

    if (read(f,buf,BANNER_MAX)<0) {
      alarm(0);
      close(f);
      // printf("  - no banner\n");
      continue;
    }

    (*port)[ports-1].status=STATUS_BANNER;
    (*port)[ports-1].banner=strdup(buf);

    // printf("  Banner: %s\n",buf);

    close(f);
    alarm(0);

  }

}


char* sev[]= { "unknown", "low", "medium", "HIGH" };
char* comps[]= { "no banner", "any banner", "full banner", "banner substring" };

/********************************************************
 * This function handles the printing of vuln output as *
 * we find each vulnerable service.                     *
 ********************************************************/
void print_vuln_info(struct be_entry print, int port, int type)
{
  char buf[1024];
  int ret;

  fprintf(stdout,"\n%s on Port: %d  Severity: %s ",print.title,port,sev[(int)print.severity]);
  if (verbose) fprintf(stdout,"(%s)",comps[type]);
  fprintf(stdout,"\n");
  if (port != print.port) fprintf(stdout,"  Normally found on port %d\n",print.port);
  if (!verbose) fprintf(stdout,"For more info see %s\n",print.description);
  else
  {
    fprintf(stdout,"Description (from %s):\n\n",print.description);
    sprintf(buf,"cat '%s'",print.description);
    ret=system(buf);
  }
  fprintf(stdout,"\n");
}

int vulns;

/********************************************************
 * This function compares banners with specified rules, *
 * determining if this rule should be triggered or not. *
 * In second run, it performs lookups to find known     *
 * programs on wrong ports.                             *
 ********************************************************/

void match_banners(void) {
  int i,j;
  for (i=0;i<entries;i++) {

    for (j=0;j<ports;j++)  {

      if (!(platform & (*db)[i].platform)) continue;
      if ((*port)[j].status == STATUS_CONNFAIL) continue;

      // printf("Checking banner [%s] for port %d with [%s]\n",
      //       (*port)[j].banner,(*db)[i].port,(*db)[i].lookfor);

      switch ((*db)[i].type) {

        case COMP_NONE:

          // Do not check for it if port is incorrect.
          if ((*db)[i].port != (*port)[j].port) break;

          if (!strlen((*port)[j].banner)) {
            print_vuln_info((*db)[i],(*port)[j].port,COMP_NONE);
            vulns++;
          }

          break;

        case COMP_ANY:
  
          if ((*db)[i].port != (*port)[j].port) break;
  
          if (strlen((*port)[j].banner)) {
            print_vuln_info((*db)[i],(*port)[j].port,COMP_ANY);
            vulns++;
          }

          break;

        case COMP_FULL:

          if (!strncmp((*db)[i].lookfor,(*port)[j].banner,strlen((*db)[i].lookfor))) {
            print_vuln_info((*db)[i],(*port)[j].port,COMP_FULL);
            vulns++;
          }
  
          break;
  
        case COMP_PARTIAL:
  
          if (strstr((*port)[j].banner,(*db)[i].lookfor)) {
            print_vuln_info((*db)[i],(*port)[j].port,COMP_PARTIAL);
            vulns++;
          }
  
          break;

      }
    }
  }
}

void usage(char *prog)
{
  fprintf(stderr,"USAGE: ");  
  fprintf(stderr,"%s [opts] -d dbfile target\n\n",prog);
  fprintf(stderr,"  opts are h ? v q s\n");
  fprintf(stderr,"    -h this help screen\n");
  fprintf(stderr,"    -? this help screen\n");
  fprintf(stderr,"    -v verbose\n");
  fprintf(stderr,"    -s sans top ten mode\n");
  fprintf(stderr,"  -d the banner database file (default is banner.db)\n");
  fprintf(stderr,"\n");
}

/***************
 * Entry point *
 ***************/

int main(int argc, char **argv) 
{
  struct hostent* x;
  char ch;
  char *bannerdb = "banner.db";
  char *prog;
  
  prog = argv[0];
  verbose = 0;
  sans = 0;
  timeout = 10;

  while ((ch = getopt(argc, argv, "vd:h?rst:")) != EOF) 
    switch(ch)
    {
      case 'v':
        verbose = 1;
        break;
      case 'd':
        bannerdb = optarg;
        break;
      case 'h':
      case '?':
        usage(prog);
        exit(0);
      case 's':
        sans = 1;
        break;
      case 't':
        timeout = (int) strtol(optarg, NULL, 10);
        break;
      default:
        fprintf(stderr,"Invalid option...\n");
        usage(prog);
        exit(-1);
    }

  argc -= optind;
  argv += optind;

  /* post arg processing */
  if(!argv[0] || !strlen(argv[0]))
  {
    fprintf(stderr,"You must specify a target...\n");
    usage(prog);
    exit(-1);
  }
  
  x=gethostbyname(argv[0]);
  if (!x || !x->h_addr) fatal("cannot determine host address");
  memcpy(&rhost,x->h_addr,sizeof(rhost));
  /* end post arg processing */

  if (sans) fprintf(stderr,"#5 and #9 - Vulnerable versions of software are detected based on the banner\n");

  if (!sans) fprintf(stderr,"Banner Eater by Michal Zalewski <lcamtuf@bos.bindview.com>\n");

  load_database(bannerdb);

  determine_os(argv[0]);
  scan_host();
  match_banners();
  if (verbose) fprintf(stdout,"\nScanning finished (%d ports, %d vulnerabilities).\n",ports,vulns);
  return 0;
}

