/****************************************************************/
#define VERSIONSTR "Toxyn Ping && Scan 2.0a, by pmsac, 1997\n"
/* gcc -Wall -O6 -o tps tps.c                                   */
/****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <linux/ip.h>
#include <linux/icmp.h>
#include <linux/tcp.h>

#define MINARGS 4
#define MAXARGS 10
#define STARTIPARG 1
#define ENDIPARG 2
#define ENDIPALTFORM "--"
#define OWNIPARG 3
#define FORCESCAN "-scan"
#define MINFORCESCANLEN 2
#define FILENAME "-file"
#define MINFILENAMELEN 2
#define FILENAMESEPARATOR ":"
#define MAXPINGS "-maxpings"
#define MINMAXPINGSLEN 5
#define MAXPINGSSEPARATOR ":"
#define MINMAXPINGS 1
#define MAXMAXPINGS 20
#define PINGDELAY "-pingdelay"
#define MINPINGDELAYLEN 2
#define PINGDELAYSEPARATOR ":"
#define MINPINGDELAY 100
#define MAXPINGDELAY 999
#define MAXTCPS "-maxtcps"
#define MINMAXTCPSLEN 5
#define MAXTCPSSEPARATOR ":"
#define MINMAXTCPS 1
#define MAXMAXTCPS 15
#define TCPDELAY "-tcpdelay"
#define MINTCPDELAYLEN 2
#define TCPDELAYSEPARATOR ":"
#define MINTCPDELAY 100
#define MAXTCPDELAY 999
#define SOURCEPORT "-port"
#define MINSOURCEPORTLEN 2
#define SOURCEPORTSEPARATOR ":"
#define MINSOURCEPORT 0
#define MAXSOURCEPORT 65535

#define CONFIGFILE "./tps.conf"
#define MAXPACKET 10240
#define MAXSCAN 100
#define MAXMESSAGE 61
#define MAXCOMMAND 101
#define MAXBUFFER (2*(MAXMESSAGE+MAXCOMMAND))
#define MINSERVICE 0
#define MAXSERVICE 65535
#define PINGSERVICE MINSERVICE

#define DEFAULTMAXPINGS 8
#define DEFAULTPINGDELAY 500
#define EXTRAPINGWAITS 1
#define EXTRAPINGWAITU 0
#define DEFAULTMAXTCPS 10
#define DEFAULTTCPDELAY 250
#define EXTRATCPWAITS 1
#define EXTRATCPWAITU 0
#define DEFAULTSOURCEPORT 65534

#define OK 0
#define ERROR -1
#define RAWSOCKETERROR -2
#define RAWPINGERROR -3
#define FATALERROR -4
#define NOFILEERROR -5
#define SENDERROR -6
#define RECEIVEERROR -7
#define REMOTEDOWN -8
#define HITALL -9
#define ALLOVER -10

/* This code is all about _not_ regarding little/big endian. Works on 
 * 
 * * intel Pentiums... I suppose that's little endian */
typedef struct {
    int sourceIP, destinationIP;
    char zero, protocol;
    char lenHi, lenLo;
} tcpPseudoHeaderType;

typedef enum {
    INITIAL = 0, COPYING, ESCAPE
} parsingStateType;

typedef enum {
    ZIP = 0, ACCEPTED, REJECTED
} tcpStateType;

typedef struct {
    tcpStateType state;
    int sentCounter;
} machineStatusEntryType;

typedef struct {
    int finishedEntries;
    machineStatusEntryType x[MAXSCAN];
} machineStatusType;

typedef struct {
    int service;
    char message[MAXMESSAGE];
    char command[MAXCOMMAND];
} scanTableEntryType;

typedef struct {
    int numEntries;
    int pingHitWork;
    char pingMessage[MAXMESSAGE];
    char pingCommand[MAXCOMMAND];
    scanTableEntryType x[MAXSCAN];
} scanTableType;

/* This function looks for a "" bounded message at source ignoring *
 * leading spaces and tabs and copies it to destination (ignoring the 
 * 
 * * "" delimiters also). It returns the position of the last
 * character  * checked in the endOfParse parameter. The destination
 * space is * restricted by the maxChars parameter to avoid overruns */
int parse(char *source, char *destination,
	  int *endOfParse, int maxChars)
{
    parsingStateType parseState = INITIAL;
    int sourcePosition = 0;
    int destinationPosition = 0;

    /* Search for initial `"' */
    while (parseState == INITIAL) {
	if (source[sourcePosition] == '\0') {
	    destination[0] = '\0';
	    *endOfParse = sourcePosition;
	    return (OK);
	}
	else if (source[sourcePosition] == '"') {
	    sourcePosition++;
	    parseState = COPYING;
	}
	else if ((source[sourcePosition] == ' ') ||
		 (source[sourcePosition] == '\t'))
	    sourcePosition++;
	else {
	    *endOfParse = sourcePosition;
	    return (ERROR);
	}
    }
    for (;;) {
	/* Got last `"' */
	if ((parseState == COPYING) &&
	    (source[sourcePosition] == '"')) {
	    destination[destinationPosition] = '\0';
	    *endOfParse = sourcePosition;
	    return (OK);
	}
	else
	    /* Normal copy */
	    if ((parseState == COPYING) &&
		(source[sourcePosition] != '\\'))
	    if (destinationPosition < maxChars)
		destination[destinationPosition++] =
		    source[sourcePosition++];
	    else {
		*endOfParse = sourcePosition;
		return (ERROR);
	    }
	else
	    /* Entering escape mode */
	    if ((parseState == COPYING) &&
		(source[sourcePosition] == '\\')) {
	    sourcePosition++;
	    parseState = ESCAPE;
	}
	else
	    /* Resume copying state */
	    if ((parseState == ESCAPE) &&
		((source[sourcePosition] == '"') ||
		 (source[sourcePosition] == '\\'))) {
	    /* Enough state to copy escaped char ? */
	    if (destinationPosition < maxChars) {
		destination[destinationPosition++] = source[sourcePosition++];
		parseState = COPYING;
	    }
	    else {
		*endOfParse = sourcePosition;
		return (ERROR);
	    }
	}
	/* None of the above */
	else {
	    *endOfParse = sourcePosition;
	    return (ERROR);
	}
    }
    return (OK);
}

int processBuffer(char *theBuffer, scanTableType * scanTable)
{
    int theService;
    char dummy;
    int i;
    int endOfMessageParse, endOfCommandParse;

    /* Check if we get called with a null buffer or commentary */
    if ((theBuffer[0] == '\0') || (theBuffer[0] == '#') ||
	(theBuffer[0] == '\n'))
	return (OK);

    /* Remove unecessary \n, 0x0a0x0d chars from end of buffer */
    while ((theBuffer[strlen(theBuffer) - 1] == 0x0a) ||
	   (theBuffer[strlen(theBuffer) - 1] == 0x0d))
	theBuffer[strlen(theBuffer) - 1] = 0;

    /* Check if we have space to handle this service */
    if (scanTable->numEntries >= MAXSCAN) {
	fprintf(stderr, "Service is one too many (%i max.):\n", MAXSCAN);
	fprintf(stderr, "%s\n", theBuffer);
	return (ERROR);
    }
    /* Check if the service number is well formed (can have '+' and * 
     * '-' as prefix) */
    if (sscanf(strtok(theBuffer, " \t\n"), "%i%c", &theService, &dummy)
	!= 1) {
	fprintf(stderr, "Invalid service:\n");
	fprintf(stderr, "%s\n", theBuffer);
	return (ERROR);
    }
    /* Check if service is in range */
    if ((theService < MINSERVICE) || (theService > MAXSERVICE)) {
	fprintf(stderr, "Service %i is out of range (%i..%i)\n",
		theService, MINSERVICE, MAXSERVICE);
	return (ERROR);
    }
    /* Check for duplicate lines for the same service. If not *
     * duplicate start filling the corresponding structure */
    if (theService == 0) {
	if (scanTable->pingHitWork) {
	    fprintf(stderr, "Duplicate 'on ping' line:\n");
	    fprintf(stderr, "%s\n", theBuffer);
	    return (ERROR);
	}
	else
	    scanTable->pingHitWork = 1;
    }
    else {
	for (i = 0; i < scanTable->numEntries; i++)
	    if (scanTable->x[i].service == theService)
		break;
	if (i == scanTable->numEntries) {
	    scanTable->x[scanTable->numEntries].service = theService;
	    /* All the other fields were zeroed at readConfigFile() */
	    scanTable->numEntries++;
	}
	else {
	    fprintf(stderr, "Duplicate 'on service' line:\n");
	    fprintf(stderr, "%s\n", theBuffer);
	    return (ERROR);
	}
    }

    /* The usage of &theBuffer[strlen(theBuffer)+N] bellow is caused
     * * by using strtok(theBuffer,) above. Now let's see if we got a
     * * message associated with the service */
    if (theService == 0) {
	if (parse(&theBuffer[strlen(theBuffer) + 1],
		  scanTable->pingMessage,
		  &endOfMessageParse, MAXMESSAGE - 1) != OK) {
	    fprintf(stderr, "Error parsing message at position %i:\n",
		    endOfMessageParse);
	    fprintf(stderr, "%s\n", &theBuffer[strlen(theBuffer) + 1]);
	    return (ERROR);
	}
	else
	    /* Check for a command associated with the service */
	    if (parse(&theBuffer[strlen(theBuffer) + 1 + endOfMessageParse + 1],
		      scanTable->pingCommand,
		      &endOfCommandParse, MAXCOMMAND - 1) != OK) {
	    fprintf(stderr, "Error parsing command at position %i:\n",
		    endOfCommandParse);
	    fprintf(stderr, "%s\n", &theBuffer[strlen(theBuffer) + 1 +
					    endOfMessageParse + 1]);
	    return (ERROR);
	}
    }
    else if (parse(&theBuffer[strlen(theBuffer) + 1],
		   scanTable->x[scanTable->numEntries - 1].message,
		   &endOfMessageParse, MAXMESSAGE - 1) != OK) {
	fprintf(stderr, "Error parsing message at position %i:\n",
		endOfMessageParse);
	fprintf(stderr, "%s\n", &theBuffer[strlen(theBuffer) + 1]);
	return (ERROR);
    }
    else
	/* Check for a command associated with the service */
	if (parse(&theBuffer[strlen(theBuffer) + 1 + endOfMessageParse + 1],
		  scanTable->x[scanTable->numEntries - 1].command,
		  &endOfCommandParse, MAXCOMMAND - 1) != OK) {
	fprintf(stderr, "Error parsing command at position %i:\n",
		endOfCommandParse);
	fprintf(stderr, "%s\n",
	 &theBuffer[strlen(theBuffer) + 1 + endOfMessageParse + 1]);
	return (ERROR);
    }
/************* Some debug we used to put this to work *************
 printf("SERVICE:%i\n"
        "MESSAGE:%s\n"
        "COMMAND:%s\n",
        theService,
        scanTable->x[scanTable->numEntries-1].message,
        scanTable->x[scanTable->numEntries-1].command);
******************************************************************/

    return (OK);
}

int readConfigFile(char *fileName, scanTableType * scanTable)
{
    FILE *configFile;
    char theBuffer[MAXBUFFER];

    scanTable->numEntries = 0;
    scanTable->pingHitWork = 0;
    bzero(scanTable->pingMessage, MAXMESSAGE * sizeof (char));
    bzero(scanTable->pingCommand, MAXCOMMAND * sizeof (char));

    bzero(scanTable->x, MAXSCAN * sizeof (scanTableEntryType));
    if ((configFile = fopen(fileName, "r")) == NULL)
	return NOFILEERROR;
    while (!feof(configFile)) {
	bzero(theBuffer, MAXBUFFER);
	if (fgets(theBuffer, MAXBUFFER - 1, configFile) != NULL)
	    if (processBuffer(theBuffer, scanTable) != OK) {
		fclose(configFile);
		return (FATALERROR);
	    }
    }
    if (fclose(configFile) != 0)
	return (FATALERROR);
    return OK;
}

unsigned short checkSum(u_short * addr, int len)
{
    register int nleft = len;
    register u_short *w = addr;
    register int sum = 0;
    u_short answer = 0;

/* 
 * Our algorithm is simple, using a 32 bit accumulator (sum), we add
 * sequential 16 bit words to it, and at the end, fold back all the
 * carry bits from the top 16 bits into the lower 16 bits.
 */
    while (nleft > 1) {
	sum += *w++;
	nleft -= 2;
    }

    /* mop up an odd byte, if necessary */
    if (nleft == 1) {
	*(u_char *) (&answer) = *(u_char *) w;
	sum += answer;
    }
    /* add back carry outs from top 16 bits to low 16 bits */
    sum = (sum >> 16) + (sum & 0xffff);		/* add hi 16 to low * 
						 * 16 */
    sum += (sum >> 16);		/* add carry */
    answer = ~sum;		/* truncate to 16 bits */
    return (answer);
}

int sendPing(int theSocket, struct sockaddr_in *whoToScanSockAddr,
	     int packetIdentifier)
{
    struct icmphdr icmpPacket;
    int sendToReturnCode;

    icmpPacket.type = ICMP_ECHO;
    icmpPacket.code = 0;
    icmpPacket.checksum = 0;
    icmpPacket.un.echo.id = packetIdentifier;
    icmpPacket.un.echo.sequence = packetIdentifier;
    icmpPacket.checksum = checkSum((u_short *) & icmpPacket,
				   sizeof (struct icmphdr));

    sendToReturnCode = sendto(theSocket,
			      &icmpPacket,
			      sizeof (struct icmphdr),
			      0,
			       (struct sockaddr *)whoToScanSockAddr,
			      sizeof (struct sockaddr_in)
    );

    if ((sendToReturnCode < 0) ||
	(sendToReturnCode !=
	 sizeof (struct icmphdr)))
	 return (ERROR);

    return (OK);
}

int receiveICMP(int theSocket, struct sockaddr_in *whoToScanSockAddr,
		int packetIdentifier)
{
    int bytesRead;
    struct iphdr *ipHeader;
    struct protoent *theProtocol;
    struct icmphdr *icmpHeader;
    char thePacket[MAXPACKET];

/* Used in debuging */
/* struct in_addr inAddr; */

    /* See if it's from machine we are pinging, don't ignore *
     * packetIdentifier to avoid spoofs. Relaxed way of doing would * 
     * use any packet with source address == address we are pinging */

    /* Read the packet */
    if ((bytesRead = read(theSocket, thePacket, MAXPACKET - 1))
	<= sizeof (struct iphdr))
	/* Never mind the error, we'll jump right back on next *
	 * select() */
	 return (ERROR);

    /* Start with IPheader */
    ipHeader = (struct iphdr *)thePacket;

    /* Check if we got all packet, avoiding surprises later */
    if (bytesRead < ntohs(ipHeader->tot_len))
	return (ERROR);

    /* Just for safe, check out the protocol, _should_ be ICMP */
    if ((theProtocol = getprotobyname("icmp")) == NULL)
	return (ERROR);

    /* No net/host byte differences here */
    if (ipHeader->protocol != theProtocol->p_proto)
	return (ERROR);

    /* Now check ICMPheader */
    icmpHeader = (struct icmphdr *)&thePacket[ipHeader->ihl << 2];

/************* Some debug we used to put this to work *************
inAddr.s_addr = ipHeader->saddr;
printf("From(victim):%s", inet_ntoa(inAddr));
inAddr.s_addr = ipHeader->daddr;
printf(" To(us):%s\n", inet_ntoa(inAddr));
printf("Protocol:%u TTL:%u TOS:%u HeaderLen:%u Total:%u\n",
       ipHeader->protocol, ipHeader->ttl, ipHeader->tos,
       ipHeader->ihl, ntohs(ipHeader->tot_len));
printf("Type:%u Code:%u Id:%u Seq:%u\n",
       icmpHeader->type, icmpHeader->code, icmpHeader->un.echo.id,
       icmpHeader->un.echo.sequence);
printf("Expected identifier:%u\n", packetIdentifier);
******************************************************************/

    if ((ipHeader->saddr == whoToScanSockAddr->sin_addr.s_addr) &&
	(icmpHeader->type == ICMP_ECHOREPLY) &&
	(icmpHeader->code == 0) &&
	(icmpHeader->un.echo.id == packetIdentifier) &&
	(icmpHeader->un.echo.sequence == packetIdentifier))
	return (OK);
    return (ERROR);
}

int pingMachine(struct sockaddr_in *whoToScanSockAddr, int maxPings,
		int pingDelay)
{
    int packetIdentifier = getpid() & 0xffff;
    struct protoent *theProtocol;
    int theSocket;
    fd_set theMainSet, theSet;
    static struct timeval timeOut;
    int pingsSent = 0;

    if ((theProtocol = getprotobyname("icmp")) == NULL)
	return (RAWSOCKETERROR);
    if ((theSocket = socket(AF_INET, SOCK_RAW, theProtocol->p_proto)) < 0)
	return (RAWSOCKETERROR);
    FD_ZERO(&theMainSet);
    FD_SET(theSocket, &theMainSet);
    /* PREPARING LINUX SPECIFIC CODE */
    timeOut.tv_sec = 0;
    timeOut.tv_usec = pingDelay;
    /* Send first ping */
    if (sendPing(theSocket, whoToScanSockAddr, packetIdentifier) == ERROR) {
	if (close(theSocket) != 0)
	    return (FATALERROR);
	return (RAWPINGERROR);
    }
    pingsSent++;
    fprintf(stderr, ".");
    fflush(stderr);
    for (;;) {
	memcpy(&theSet, &theMainSet, sizeof (theMainSet));
	select(theSocket + 1, &theSet, NULL, NULL, &timeOut);
	if (FD_ISSET(theSocket, &theSet)) {
	    /* 1 (or more) ICMP hit our machine, check it out */
	    if (receiveICMP(theSocket, whoToScanSockAddr, packetIdentifier)
		== OK) {
		if (close(theSocket) != 0)
		    return (FATALERROR);
		return (OK);
	    }
	}
	else
	    /* THIS CODE IS LINUX SPECIFIC, SEE MAN SELECT FOR *
	     * DETAILS */
	if ((timeOut.tv_sec == 0) && (timeOut.tv_usec == 0)) {
	    /* Timed out */
	    if (pingsSent == maxPings) {
		/* Force one extra 'wait state' without sending pings 
		 * 
		 */
		pingsSent++;
		timeOut.tv_sec = EXTRAPINGWAITS;
		timeOut.tv_usec = EXTRAPINGWAITU;
	    }
	    else if (pingsSent > maxPings) {
		/* No valid reply, remote machine is assumed down */
		if (close(theSocket) != 0)
		    return (FATALERROR);
		return (REMOTEDOWN);
	    }
	    else {
		/* Send one more ping and prepare for waiting again */
		if (sendPing(theSocket,
			     whoToScanSockAddr,
			     packetIdentifier) == ERROR) {
		    if (close(theSocket) != 0)
			return (FATALERROR);
		    return (RAWPINGERROR);
		}
		pingsSent++;
		fprintf(stderr, ".");
		fflush(stderr);
		timeOut.tv_sec = 0;
		timeOut.tv_usec = pingDelay;
	    }
	}
    }
    if (close(theSocket) != 0)
	return (FATALERROR);
    return (OK);
}

int sendTCPPacket(int theSocket, int ownIp, int theProtocol,
		  struct sockaddr_in *whoToScanSockAddr,
		  int ourPort, int theService, int packetIdentifier)
{
    char tcpPacketBuffer[MAXBUFFER];
    struct tcphdr *tcpPacket;
    tcpPseudoHeaderType *tcpPseudoHeader;
    int sendToReturnCode;

#define DATASIZE 4
#define MSSOPTION 2
#define MSSLEN 4
#define MSSVALUE 536

    bzero(tcpPacketBuffer, MAXBUFFER);
    tcpPacket = (struct tcphdr *)tcpPacketBuffer;
    tcpPseudoHeader =
	(tcpPseudoHeaderType *) & tcpPacketBuffer[sizeof (struct tcphdr) + 4];

/****** Is this somewhat messed up ??? ******/
    tcpPacket->source = htons(theService & 0xffff);
    tcpPacket->dest = htons(ourPort & 0xffff);
/********************************************/
    tcpPacket->seq = packetIdentifier;
    tcpPacket->ack_seq = 0;
    tcpPacket->doff = ((sizeof (struct tcphdr) + DATASIZE) >> 2);

    tcpPacket->syn = 1;
    tcpPacket->ack = 0;
    tcpPacket->rst = 0;
    tcpPacket->fin = 0;
    tcpPacket->urg = 0;
    tcpPacket->psh = 0;
    tcpPacket->res1 = 0;
    tcpPacket->res2 = 0;
    tcpPacket->window = htons(2048);
    tcpPacket->check = 0;
    tcpPacket->urg_ptr = 0;

    /* This code is all about _not_ regarding little/big endian. *
     * Works on intel Pentiums... I suppose that's little endian */
    tcpPacketBuffer[sizeof (struct tcphdr)] = MSSOPTION;
    tcpPacketBuffer[sizeof (struct tcphdr) + 1] = MSSLEN;
    tcpPacketBuffer[sizeof (struct tcphdr) + 2] = (MSSVALUE / 0x100);
    tcpPacketBuffer[sizeof (struct tcphdr) + 3] = (MSSVALUE % 0x100);

    tcpPseudoHeader->sourceIP = ownIp;
    tcpPseudoHeader->destinationIP = whoToScanSockAddr->sin_addr.s_addr;
    tcpPseudoHeader->zero = 0;
    tcpPseudoHeader->protocol = theProtocol & 0xff;
    tcpPseudoHeader->lenHi = ((sizeof (struct tcphdr) + DATASIZE) / 0x100);
    tcpPseudoHeader->lenLo = ((sizeof (struct tcphdr) + DATASIZE) % 0x100);

    /* End of lame code (I hope) */

    tcpPacket->check = checkSum((u_short *) tcpPacket,
				sizeof (struct tcphdr) + DATASIZE +
				sizeof (tcpPseudoHeaderType));

    sendToReturnCode = sendto(theSocket,
			      tcpPacket,
			      sizeof (struct tcphdr) + DATASIZE,
			      0,
			       (struct sockaddr *)whoToScanSockAddr,
			      sizeof (struct sockaddr_in)
    );

    if ((sendToReturnCode < 0) ||
	(sendToReturnCode != (sizeof (struct tcphdr) + DATASIZE)))
	 return (ERROR);

    return (OK);
}

int sendTCP(int theSocket, int ownIp, int theProtocol,
	    struct sockaddr_in *whoToScanSockAddr,
	    int ourPort, int packetIdentifier, int maxTCPs,
	    scanTableType * scanTable,
	    machineStatusType * machineStatus)
{
    static lastEntry = -1;

    if (machineStatus->finishedEntries == scanTable->numEntries) {
	lastEntry = -1;
	return (ALLOVER);
    }
    if (lastEntry == -1)
	lastEntry = 0;
    else
	for (;;) {
	    lastEntry++;
	    lastEntry %= scanTable->numEntries;
	    if (machineStatus->x[lastEntry].sentCounter < maxTCPs)
		break;
	}

    machineStatus->x[lastEntry].sentCounter++;
    if (machineStatus->x[lastEntry].sentCounter == maxTCPs)
	machineStatus->finishedEntries++;
    return (sendTCPPacket(theSocket, ownIp, theProtocol, whoToScanSockAddr,
			  scanTable->x[lastEntry].service, ourPort,
			  packetIdentifier));
}

void scanTableHitMark(int service, int syn, int rst, int maxTCPs,
		      scanTableType * scanTable,
		      machineStatusType * machineStatus)
{
    int i;

    if ((syn && rst) || (!syn && !rst))
	return;
    for (i = 0; i < scanTable->numEntries; i++)
	if ((scanTable->x[i].service == service) &&
	    (machineStatus->x[i].state == ZIP)) {
	    if (syn)
		machineStatus->x[i].state = ACCEPTED;
	    else
		machineStatus->x[i].state = REJECTED;
	    if (machineStatus->x[i].sentCounter != maxTCPs) {
		machineStatus->x[i].sentCounter = maxTCPs;
		machineStatus->finishedEntries++;
	    }
	    break;
	}
}

int receiveTCP(int theSocket, struct sockaddr_in *whoToScanSockAddr,
	       int packetIdentifier, int maxTCPs, int ourPort,
	       scanTableType * scanTable,
	       machineStatusType * machineStatus)
{
    int bytesRead;
    struct iphdr *ipHeader;
    struct protoent *theProtocol;
    struct tcphdr *tcpHeader;
    char thePacket[MAXPACKET];

/* Used in debuging */
/* struct in_addr inAddr; */

    /* See if it's from machine we are scanning, don't ignore *
     * packetIdentifier to avoid spoofs. Relaxed way of doing would * 
     * use any packet with source address == address we are scanning */

    /* Read the packet */
    if ((bytesRead = read(theSocket, thePacket, MAXPACKET - 1))
	< (sizeof (struct iphdr) + sizeof (struct tcphdr)))
	 return (ERROR);

    /* Start with IPheader */
    ipHeader = (struct iphdr *)thePacket;

    /* Check if we got all packet, avoiding surprises later */
    if (bytesRead < ntohs(ipHeader->tot_len))
	return (ERROR);

    /* Just for safe, check out the protocol, _should_ be TCP */
    if ((theProtocol = getprotobyname("tcp")) == NULL)
	return (ERROR);

    /* No net/host byte differences here */
    if (ipHeader->protocol != theProtocol->p_proto)
	return (ERROR);

    /* Now check TCPheader */
    tcpHeader = (struct tcphdr *)&thePacket[ipHeader->ihl << 2];

    if ((ipHeader->saddr == whoToScanSockAddr->sin_addr.s_addr) &&
	(tcpHeader->ack == 1) &&
	(tcpHeader->dest == ntohs(ourPort & 0xffff))) {
	scanTableHitMark(ntohs(tcpHeader->source), tcpHeader->syn,
			 tcpHeader->rst,
			 maxTCPs, scanTable, machineStatus);
	return (OK);
    }
    return (OK);
}

int scanMachine(struct sockaddr_in *whoToScanSockAddr, int ownIp,
		scanTableType * scanTable,
		machineStatusType * machineStatus,
		int maxTCPs, int tcpDelay, int ourPort)
{
    int packetIdentifier = getpid();
    struct protoent *theProtocol;
    int theSocket;
    fd_set theMainSet, theSet;
    static struct timeval timeOut;
    int extraWait = 0;

    bzero(machineStatus, sizeof (machineStatusType));
    if ((theProtocol = getprotobyname("tcp")) == NULL)
	return (RAWSOCKETERROR);
    if ((theSocket = socket(AF_INET, SOCK_RAW, theProtocol->p_proto)) < 0)
	return (RAWSOCKETERROR);
    FD_ZERO(&theMainSet);
    FD_SET(theSocket, &theMainSet);
    /* PREPARING LINUX SPECIFIC CODE */
    timeOut.tv_sec = 0;
    timeOut.tv_usec = tcpDelay;
    /* We should start sending our TCP packets here... let's just * * 
     * timeout */
    for (;;) {
	memcpy(&theSet, &theMainSet, sizeof (theMainSet));
	select(theSocket + 1, &theSet, NULL, NULL, &timeOut);
	if (FD_ISSET(theSocket, &theSet)) {
	    /* 1 (or more) TCP hit our machine, check it out */
	    /* The OK means it's all over, can get out */
	    switch (receiveTCP(theSocket, whoToScanSockAddr, packetIdentifier, maxTCPs,
			       ourPort, scanTable, machineStatus)) {
		case HITALL:
		    if (close(theSocket) != 0)
			return (FATALERROR);
		    return (OK);
		    break;
		case OK:
		    break;
		case ERROR:
		    if (close(theSocket) != 0)
			return (FATALERROR);
		    return (RECEIVEERROR);
		    break;
		default:
		    fprintf(stderr, "WTF ?\n");
		    fflush(stderr);
		    if (close(theSocket) != 0)
			return (FATALERROR);
		    return (FATALERROR);
		    break;
	    }
	}
	/* THIS CODE IS LINUX SPECIFIC, SEE MAN SELECT FOR DETAILS */
	if ((timeOut.tv_sec == 0) && (timeOut.tv_usec == 0)) {
	    switch (extraWait) {
		case 1:
		    /* Let's wait an extra 'wait state' */
		    timeOut.tv_sec = 1;
		    timeOut.tv_usec = 0;
		    extraWait = 2;
		    break;
		case 2:
		    if (close(theSocket) != 0)
			return (FATALERROR);
		    return (OK);
		    break;
		default:
		    /* Let's send another TCP round */
		    switch (sendTCP(theSocket, ownIp, theProtocol->p_proto,
		       whoToScanSockAddr, ourPort, packetIdentifier,
			       maxTCPs, scanTable, machineStatus)) {
			case ALLOVER:
			    extraWait = 1;
			    timeOut.tv_sec = 0;
			    timeOut.tv_usec = tcpDelay;
			    break;
			case OK:
			    timeOut.tv_sec = 0;
			    timeOut.tv_usec = tcpDelay;
			    fprintf(stderr, ".");
			    fflush(stderr);
			    break;
			case ERROR:
			    if (close(theSocket) != 0)
				return (FATALERROR);
			    return (SENDERROR);
			    break;
			default:
			    fprintf(stderr, "WTF ??\n");
			    fflush(stderr);
			    if (close(theSocket) != 0)
				return (FATALERROR);
			    return (FATALERROR);
			    break;
		    }
		    break;
	    }
	}
    }
    if (close(theSocket) != 0)
	return (FATALERROR);
    return (OK);
}

void printScanInfo(scanTableType * scanTable, machineStatusType * machineStatus)
{
    int i;

    for (i = 0; i < scanTable->numEntries; i++)
	if (machineStatus->x[i].state == ACCEPTED)
	    fprintf(stderr, "%s\n", scanTable->x[i].message);
}

void executeCommands(struct in_addr *whoToScanInAddr,
		     scanTableType * scanTable,
		     machineStatusType * machineStatus)
{
    int i;
    char commandBuffer[MAXBUFFER];

    for (i = 0; i < scanTable->numEntries; i++)
	if ((machineStatus->x[i].state == ACCEPTED) &&
	    (scanTable->x[i].command != NULL)) {
	    bzero(commandBuffer, MAXBUFFER);
	    sprintf(commandBuffer, "MACHINE=%s;SERVICE=%u;%s",
	       inet_ntoa(*whoToScanInAddr), scanTable->x[i].service,
		    scanTable->x[i].command);
	    system(commandBuffer);
	}
}

int addressOk(u_long hostIp)
{
    int residual = hostIp & 0x000000ff;

    if ((residual == 0x00) || (residual == 0xff))
	return 0;
    return 1;
}

int resolveDNS(char *hostName, int *IP)
{
    struct sockaddr_in addr;
    struct hostent *host;

    (void)bzero((char *)&addr, sizeof (struct sockaddr_in));

    /* first, check if the address is an ip address */
    addr.sin_addr.s_addr = inet_addr(hostName);
    if ((int)addr.sin_addr.s_addr == -1) {
	/* it wasn't.. so we try it as a long host name */
	host = gethostbyname(hostName);
	if (host) {
	    /* wow.  It's a host name.. set the fields */
	    /* ?? addr.sin_family = host->h_addrtype; */
	    bcopy(host->h_addr, (char *)&addr.sin_addr,
		  host->h_length);
	    *IP = addr.sin_addr.s_addr;
	}
	else
	    /* oops.. can't find it.. */
	    return (ERROR);
    }
    /* all done. */
    return (OK);

}

void main(int argc, char *argv[])
{
    int forceScanLen, forceScan = 0, forceScanUsed = 0;
    int fileNameLen;
    char *fileName = NULL;
    int fileNameUsed = 0;
    int maxPingsLen, maxPings = 0, maxPingsUsed = 0;
    int pingDelayLen, pingDelay = 0, pingDelayUsed = 0;
    int maxTCPsLen, maxTCPs = 0, maxTCPsUsed = 0;
    int tcpDelayLen, tcpDelay = 0, tcpDelayUsed = 0;
    int sourcePortLen, sourcePort = 0, sourcePortUsed = 0;
    int i;
    char dummy = 0;
    u_long hostIp;
    int startIp, endIp, ownIp;
    struct in_addr whoToScanInAddr;
    struct sockaddr_in whoToScanSockAddr;
    int machineOk;
    scanTableType scanTable;
    machineStatusType machineStatus;
    char commandBuffer[MAXBUFFER];

    /* Rough check on args */
    if ((argc < MINARGS) || (argc > MAXARGS)) {
	fprintf(stderr,
		VERSIONSTR
		"Usage: %s <startIP> <endIP> <yourIP> [options]\n"
	     "If <endIP> is `%s' then only <startIP> will be used\n"
	  "<yourIP> is needed for stealthier scanning than normal\n"
		"Options:\t%s\t\tforce scan, no ping\n"
		"\t\t%s%sFILE\tconfiguration file (default %s)\n"
	    "\t\t%s%sN\tmax ping packets per machine (default %i)\n"
		"\t\t%s%sN\tping delay in useconds (default %i)\n"
	     "\t\t%s%sN\tmax tcp packets per service (default %i)\n"
		"\t\t%s%sN\ttcp delay in useconds (default %i)\n"
	   "\t\t%s%sN\t\tsource port for tcp packets (default %i)\n"
		"\n",
		argv[0], ENDIPALTFORM, FORCESCAN,
		FILENAME, FILENAMESEPARATOR, CONFIGFILE,
		MAXPINGS, MAXPINGSSEPARATOR, DEFAULTMAXPINGS,
		PINGDELAY, PINGDELAYSEPARATOR, DEFAULTPINGDELAY,
		MAXTCPS, MAXTCPSSEPARATOR, DEFAULTMAXTCPS,
		TCPDELAY, TCPDELAYSEPARATOR, DEFAULTTCPDELAY,
		SOURCEPORT, SOURCEPORTSEPARATOR, DEFAULTSOURCEPORT);
	exit(-1);
    }
    /* Check start, end and own IPs */
    if ((startIp = inet_addr(argv[STARTIPARG])) == -1)
	if (strncmp(argv[ENDIPARG], ENDIPALTFORM, strlen(ENDIPALTFORM))) {
	    fprintf(stderr, "<startIP> should be in xxx.xxx.xxx.xxx form !\n");
	    exit(-1);
	}
    /* It's just one IP, so let's resolve it */
	else if (resolveDNS(argv[STARTIPARG], &startIp) != OK) {
	    fprintf(stderr, "Can't resolve %s !\n", argv[STARTIPARG]);
	    exit(-1);
	}
    if (!strncmp(argv[ENDIPARG], ENDIPALTFORM, strlen(ENDIPALTFORM)))
	endIp = startIp;
    else if ((endIp = inet_addr(argv[ENDIPARG])) == -1) {
	fprintf(stderr, "<endIP> should be in xxx.xxx.xxx.xxx form"
		" or put as `%s' !\n", ENDIPALTFORM);
	exit(-1);
    }
    if ((ownIp = inet_addr(argv[OWNIPARG])) == -1) {
	fprintf(stderr, "<yourIP> should be in xxx.xxx.xxx.xxx form !\n");
	exit(-1);
    }
    /* Check possible options */
    if (argc > MINARGS) {
	for (i = MINARGS; i < argc; i++) {
	    forceScanLen = strlen(argv[i]);
	    if (forceScanLen < MINFORCESCANLEN)
		forceScanLen = MINFORCESCANLEN;
	    if (!strncmp(argv[i], FORCESCAN, forceScanLen)) {
		if (forceScanUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", FORCESCAN);
		    exit(-1);
		}
		else
		    forceScanUsed = 1;
		forceScan = 1;
		continue;
	    }
	    fileNameLen = strlen(argv[i]);
	    if (fileNameLen > strcspn(argv[i], FILENAMESEPARATOR))
		fileNameLen = strcspn(argv[i], FILENAMESEPARATOR);
	    if (fileNameLen < MINFILENAMELEN)
		fileNameLen = MINFILENAMELEN;
	    if (!strncmp(argv[i], FILENAME, fileNameLen)) {
		if (fileNameUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", FILENAME);
		    exit(-1);
		}
		else
		    fileNameUsed = 1;
		if (!strncmp(&argv[i][fileNameLen], FILENAMESEPARATOR,
			     sizeof (char)))
		     fileName = &argv[i][fileNameLen + 1];

		continue;
	    }
	    maxPingsLen = strlen(argv[i]);
	    if (maxPingsLen > strcspn(argv[i], MAXPINGSSEPARATOR))
		maxPingsLen = strcspn(argv[i], MAXPINGSSEPARATOR);
	    if (maxPingsLen < MINMAXPINGSLEN)
		maxPingsLen = MINMAXPINGSLEN;
	    if (!strncmp(argv[i], MAXPINGS, maxPingsLen)) {
		if (maxPingsUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", MAXPINGS);
		    exit(-1);
		}
		else
		    maxPingsUsed = 1;
		if ((!strncmp(&argv[i][maxPingsLen], MAXPINGSSEPARATOR,
			      sizeof (char))) &&
		     (sscanf(&argv[i][maxPingsLen + 1], "%u%c", &maxPings, &dummy) == 1) &&
		     (dummy == 0)) {
		    if ((maxPings < MINMAXPINGS) || (maxPings > MAXMAXPINGS)) {
			fprintf(stderr,
				"Invalid N in %s%sN, N should be in range (%u..%u)\n",
			   MAXPINGS, MAXPINGSSEPARATOR, MINMAXPINGS,
				MAXMAXPINGS);
			exit(-1);
		    }
		}
		else {
		    fprintf(stderr, "%s option badly formed\n", MAXPINGS);
		    exit(-1);
		}
		continue;
	    }
	    pingDelayLen = strlen(argv[i]);
	    if (pingDelayLen > strcspn(argv[i], PINGDELAYSEPARATOR))
		pingDelayLen = strcspn(argv[i], PINGDELAYSEPARATOR);
	    if (pingDelayLen < MINPINGDELAYLEN)
		pingDelayLen = MINPINGDELAYLEN;
	    if (!strncmp(argv[i], PINGDELAY, pingDelayLen)) {
		if (pingDelayUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", PINGDELAY);
		    exit(-1);
		}
		else
		    pingDelayUsed = 1;
		if ((!strncmp(&argv[i][pingDelayLen], PINGDELAYSEPARATOR,
			      sizeof (char)))
		    && (sscanf(&argv[i][pingDelayLen + 1], "%u%c", &pingDelay,
			       &dummy) == 1) && (dummy == 0)) {
		    if ((pingDelay < MINPINGDELAY) || (pingDelay > MAXPINGDELAY)) {
			fprintf(stderr,
				"Invalid N in %s%sN, N should be in range (%u..%u)\n",
			PINGDELAY, PINGDELAYSEPARATOR, MINPINGDELAY,
				MAXPINGDELAY);
			exit(-1);
		    }
		}
		else {
		    fprintf(stderr, "%s option badly formed\n", PINGDELAY);
		    exit(-1);
		}
		continue;
	    }
	    maxTCPsLen = strlen(argv[i]);
	    if (maxTCPsLen > strcspn(argv[i], MAXTCPSSEPARATOR))
		maxTCPsLen = strcspn(argv[i], MAXTCPSSEPARATOR);
	    if (maxTCPsLen < MINMAXTCPSLEN)
		maxTCPsLen = MINMAXTCPSLEN;
	    if (!strncmp(argv[i], MAXTCPS, maxTCPsLen)) {
		if (maxTCPsUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", MAXTCPS);
		    exit(-1);
		}
		else
		    maxTCPsUsed = 1;
		if ((!strncmp(&argv[i][maxTCPsLen], MAXTCPSSEPARATOR,
			      sizeof (char))) &&
		 (sscanf(&argv[i][maxTCPsLen + 1], "%u%c", &maxTCPs,
			 &dummy) == 1) &&
		     (dummy == 0)) {
		    if ((maxTCPs < MINMAXTCPS) || (maxTCPs > MAXMAXTCPS)) {
			fprintf(stderr,
				"Invalid N in %s%sN, N should be in range (%u..%u)\n",
			      MAXTCPS, MAXTCPSSEPARATOR, MINMAXTCPS,
				MAXMAXTCPS);
			exit(-1);
		    }
		}
		else {
		    fprintf(stderr, "%s option badly formed\n", MAXTCPS);
		    exit(-1);
		}
		continue;
	    }
	    tcpDelayLen = strlen(argv[i]);
	    if (tcpDelayLen > strcspn(argv[i], TCPDELAYSEPARATOR))
		tcpDelayLen = strcspn(argv[i], TCPDELAYSEPARATOR);
	    if (tcpDelayLen < MINTCPDELAYLEN)
		tcpDelayLen = MINTCPDELAYLEN;
	    if (!strncmp(argv[i], TCPDELAY, tcpDelayLen)) {
		if (tcpDelayUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", TCPDELAY);
		    exit(-1);
		}
		else
		    tcpDelayUsed = 1;
		if ((!strncmp(&argv[i][tcpDelayLen], TCPDELAYSEPARATOR, sizeof (char))) &&
		     (sscanf(&argv[i][tcpDelayLen + 1], "%u%c", &tcpDelay, &dummy) == 1)
		    && (dummy == 0)) {
		    if ((tcpDelay < MINTCPDELAY) || (tcpDelay > MAXTCPDELAY)) {
			fprintf(stderr,
				"Invalid N in %s%s:N, N should be in range (%u..%u)\n",
			   TCPDELAY, TCPDELAYSEPARATOR, MINTCPDELAY,
				MAXTCPDELAY);
			exit(-1);
		    }
		}
		else {
		    fprintf(stderr, "%s option badly formed\n", TCPDELAY);
		    exit(-1);
		}
		continue;
	    }
	    sourcePortLen = strlen(argv[i]);
	    if (sourcePortLen > strcspn(argv[i], SOURCEPORTSEPARATOR))
		sourcePortLen = strcspn(argv[i], SOURCEPORTSEPARATOR);
	    if (sourcePortLen < MINSOURCEPORTLEN)
		sourcePortLen = MINSOURCEPORTLEN;
	    if (!strncmp(argv[i], SOURCEPORT, sourcePortLen)) {
		if (sourcePortUsed) {
		    fprintf(stderr, "Repeated use of option %s\n", SOURCEPORT);
		    exit(-1);
		}
		else
		    sourcePortUsed = 1;
		if ((!strncmp(&argv[i][sourcePortLen], SOURCEPORTSEPARATOR,
			      sizeof (char))) &&
		     (sscanf(&argv[i][sourcePortLen + 1], "%u%c", &sourcePort,
			     &dummy) == 1)
		    && (dummy == 0)) {
		    if ((sourcePort < MINSOURCEPORT) ||
			(sourcePort > MAXSOURCEPORT)) {
			fprintf(stderr,
				"Invalid N in %s%s:N, N should be in range (%u..%u)\n",
				SOURCEPORT, SOURCEPORTSEPARATOR, MINSOURCEPORT,
				MAXSOURCEPORT);
			exit(-1);
		    }
		}
		else {
		    fprintf(stderr, "%s option badly formed\n", SOURCEPORT);
		    exit(-1);
		}
		continue;
	    }
	    fprintf(stderr, "Invalid option \"%s\"\n", argv[i]);
	    exit(-1);
	}
    }
    if (fileName == NULL)
	fileName = CONFIGFILE;
    if (maxPings == 0)
	maxPings = DEFAULTMAXPINGS;
    if (pingDelay == 0)
	pingDelay = DEFAULTPINGDELAY;
    if (maxTCPs == 0)
	maxTCPs = DEFAULTMAXTCPS;
    if (tcpDelay == 0)
	tcpDelay = DEFAULTTCPDELAY;
    if (sourcePort == 0)
	sourcePort = DEFAULTSOURCEPORT;

/************* Some debug we used to put this to work *****************
 printf("A-Ok, force scan == %u, file == %s, max pings == %u, "
        "ping delay == %u, max tcps == %u, tcp delay == %u\n",
        forceScan, fileName, maxPings, pingDelay, maxTCPs, tcpDelay);
**********************************************************************/
    switch (readConfigFile(fileName, &scanTable)) {
	case OK:
	    break;
	case NOFILEERROR:
	    fprintf(stderr, "Unable to open configuration file (%s)\n",
		    fileName);
	    exit(-1);
	    break;
	case FATALERROR:
	    fprintf(stderr, "Errors in configuration file (%s)\n", fileName);
	    exit(-1);
	    break;
    }
    startIp = ntohl(startIp);
    endIp = ntohl(endIp);
    for (hostIp = startIp; hostIp <= endIp; hostIp++) {
	if (!addressOk(hostIp))
	    continue;
	whoToScanInAddr.s_addr = htonl(hostIp);
	whoToScanSockAddr.sin_family = PF_INET;
	whoToScanSockAddr.sin_port = htons(0);
	whoToScanSockAddr.sin_addr.s_addr = whoToScanInAddr.s_addr;
	machineOk = 0;
	if (!forceScan) {
	    fprintf(stderr, "Pinging %s ", inet_ntoa(whoToScanInAddr));
	    switch (pingMachine(&whoToScanSockAddr, maxPings, pingDelay)) {
		case OK:
		    machineOk = 1;
		    fprintf(stderr, "\n");
		    if (scanTable.pingMessage[0] != '\0')
			fprintf(stderr, "%s\n", scanTable.pingMessage);
		    if (scanTable.pingCommand != NULL) {
			bzero(commandBuffer, MAXBUFFER);
			sprintf(commandBuffer, "MACHINE=%s;SERVICE=0;%s",
				inet_ntoa(whoToScanInAddr),
				scanTable.pingCommand);
			system(commandBuffer);
		    }
		    break;
		case RAWSOCKETERROR:
		    fprintf(stderr, " RSERROR\n");
		    break;
		case RAWPINGERROR:
		    fprintf(stderr, " RPERROR\n");
		    break;
		case REMOTEDOWN:
		    fprintf(stderr, " no answer, assumed down\n");
		    break;
		case FATALERROR:
		    fprintf(stderr, " Fatal error (closing socket ?)\n");
		    exit(-1);
		    break;
		default:
		    fprintf(stderr, " WTF ???\n");
		    exit(-1);
		    break;
	    }
	}
	if (machineOk || forceScan) {
	    fprintf(stderr, "Scanning %s ", inet_ntoa(whoToScanInAddr));
	    switch (scanMachine(&whoToScanSockAddr, ownIp,
				&scanTable, &machineStatus,
				maxTCPs, tcpDelay, sourcePort)) {
		case OK:
		    fprintf(stderr, "\n");
		    printScanInfo(&scanTable, &machineStatus);
		    executeCommands(&whoToScanInAddr, &scanTable, &machineStatus);
		    break;
		case RAWSOCKETERROR:
		    fprintf(stderr, " RSERROR\n");
		    break;
		case SENDERROR:
		    fprintf(stderr, " Error sending packet\n");
		    break;
		case RECEIVEERROR:
		    fprintf(stderr, " Error receiving packet\n");
		    break;
		case FATALERROR:
		    fprintf(stderr, " Fatal error (closing socket ?)\n");
		    exit(-1);
		    break;
		default:
		    fprintf(stderr, " WTF ????\n");
		    exit(-1);
		    break;
	    }
	}
    }
}
