/* Nessuslib -- the Nessus Library
 * Copyright (C) 1998-2000 Renaud Deraison
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Plugutils -- plugin-specific stuff
 */

#define EXPORTING
#include <includes.h>
#include "comm.h"
#include "harglists.h"
#include "diff.h"
#include "rand.h"

/* want version stuff */
#include "libvers.h"


#undef DEBUG_DIFF_SCAN



char *nessuslib_version()
{
  return strdup(VERSION);
}
ExtFunc
void nessus_lib_version(major, minor, rev)
 int * major, *minor, *rev;
{
 *major = NL_MAJOR;
 *minor = NL_MINOR;
 *rev   = NL_REV;
}

#ifdef USE_PTHREADS
ExtFunc 
int nessuslib_pthreads_enabled()
{
 int enabled = 1;
 return(enabled);
}
#endif





/*
 * Escapes \n and \r properly. The resulting string
 * is copied in another buffer.
 */
ExtFunc char * 
addslashes(in)
	char * in;
{
 char * ret;
 char * out = malloc(strlen(in) * 2 + 1);
 bzero(out, strlen(in) * 2 + 1);
 ret = out;
 while(in[0])
 {
  if(in[0] == '\\')
  {
   out[0] = '\\'; out++;
   out[0] = '\\'; out++;
  }

  else if(in[0] == '\n')
  {
   out[0] = '\\'; out++;
   out[0] = 'n'; out++;
  }
  else if(in[0] == '\r')
  {
    out[0] = '\\'; out++;
    out[0] = 'r';  out++;
  }
  else {
	  out[0] = in[0];
	  out++;
  }
  in++;
 }
 return realloc(ret, strlen(ret) + 1);
}

/*
 * Replaces escape codes (\n, \r) by the real value
 * The resulting string is stored in another buffer
 */
ExtFunc char * 
rmslashes(in)
 char * in;
{
 char * out = malloc(strlen(in) + 1);
 char * ret = out;
 bzero(out, strlen(in) + 1);
 while(in[0])
 {
  if(in[0] == '\\')
  {
   switch(in[1])
   {
    case 'r' :
      out[0] = '\r';
      in++;
      break;
    case 'n' :
      out[0] =  '\n';
      in++;
      break;
    case '\\' :
      out[0] = '\\';
      in++;
      break;
    default :
      fprintf(stderr, "Unknown escape sequence '\\%c'\n", in[1]);
   }
  }
  else out[0] = in[0];
  in++;
  out++;
 }
 return realloc(ret, strlen(ret) + 1);
}


ExtFunc
void plug_set_version(desc, version)
 struct arglist * desc;
 const char* version;
{
 arg_add_value(desc, "VERSION", ARG_STRING, sizeof(int), estrdup((char*)version));
}

ExtFunc
void plug_set_id(desc, id)
 struct arglist * desc;
 int id;
{
 arg_add_value(desc, "ID", ARG_INT, sizeof(int), (void*)id);
}

ExtFunc int
plug_get_id(desc)
 struct arglist * desc;
{
 return (int)arg_get_value(desc, "ID");
}


ExtFunc
void plug_set_cve_id(desc, id)
 struct arglist * desc;
 char * id;
{
 arg_add_value(desc, "CVE_ID", ARG_STRING, strlen(id), estrdup(id));
}


ExtFunc char *
plug_get_cve_id(desc)
 struct arglist * desc;
{
 return arg_get_value(desc, "CVE_ID");
}

ExtFunc 
void plug_set_see_also(desc, url)
 struct arglist * desc;
 char * url;
{
 struct arglist * seealso = arg_get_value(desc, "SEE_ALSO");
 if(!seealso)
 {
  seealso = emalloc(sizeof(*seealso));
  arg_add_value(desc, "SEE_ALSO",  ARG_ARGLIST, -1, seealso);
 }
 arg_add_value(seealso, "url", ARG_STRING, strlen(url), estrdup(url));
}


ExtFunc
void plug_set_family(desc, family, language)
 struct arglist * desc; 
 const char * family;
 const char * language;
{
  char * s_language;
  struct arglist * prefs = arg_get_value(desc, "preferences");
  
  s_language = arg_get_value(prefs,"language");
  if(s_language && language)
   {
    if(!strcmp(s_language, language))
    {
    if(family)
    arg_add_value(desc, "FAMILY", ARG_STRING,
    			strlen(family), estrdup(family));
    }
   }
  else if(family)
    {
     if(!arg_get_value(desc, "FAMILY"))
      arg_add_value(desc, "FAMILY", ARG_STRING,
    			strlen(family), estrdup(family));
    }
}

ExtFunc
void plug_require_key(desc, keyname)
 struct arglist * desc;
 const char * keyname;
{
 struct arglist * keys;
 if(keyname)
 {
  keys = arg_get_value(desc, "required_keys");
  if(!keys)
  {
   keys = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "required_keys", ARG_ARGLIST, -1, keys);
  }
  arg_add_value(keys, "key", ARG_STRING, strlen(keyname), estrdup(keyname));
 }
}

ExtFunc
void plug_exclude_key(desc, keyname)
 struct arglist * desc;
 const char * keyname;
{
 struct arglist * keys;
 if(keyname)
 {
  keys = arg_get_value(desc, "excluded_keys");
  if(!keys)
  {
   keys = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "excluded_keys", ARG_ARGLIST, -1, keys);
  }
  arg_add_value(keys, "key", ARG_STRING, strlen(keyname), estrdup(keyname));
 }
}


ExtFunc 
void plug_require_port(desc, portname)
 struct arglist * desc;
 const char * portname;
{
 struct arglist * ports;
 int iport = atoi(portname);
 
 if(portname)
 {
  ports = arg_get_value(desc, "required_ports");
  if(!ports)
  {
   ports = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "required_ports", ARG_ARGLIST, -1, ports);
  }
  

  if(iport)
   arg_add_value(ports, portname, ARG_INT, sizeof(iport), (char*)iport);
  else 
   arg_add_value(ports, portname, ARG_STRING, strlen(portname), estrdup(portname));

 }
}

ExtFunc 
void plug_require_udp_port(desc, portname)
 struct arglist * desc;
 const char * portname;
{
 struct arglist * ports;
 int iport = atoi(portname);
 
 if(portname)
 {
  ports = arg_get_value(desc, "required_udp_ports");
  if(!ports)
  {
   ports = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "required_udp_ports", ARG_ARGLIST, -1, ports);
  }
  

  if(iport)
   arg_add_value(ports, portname, ARG_INT, sizeof(iport), (char*)iport);
  else 
   arg_add_value(ports, portname, ARG_STRING, strlen(portname), estrdup(portname));

 }
}
 

ExtFunc
void plug_set_keyword(desc, keyword)
 struct arglist * desc;
 const char * keyword;
{
 struct arglist * keywords;
 if(keyword)
 {
  keywords = arg_get_value(desc, "KEYWORDS");
  if(!keywords)
  {
   keywords = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "KEYWORDS", ARG_ARGLIST, -1, keywords);
  }
  arg_add_value(keywords, estrdup(keyword), ARG_STRING, 0, estrdup(""));
 }
}


ExtFunc
void plug_set_dep(desc, depname)
 struct arglist * desc;
 const char * depname;
{
 struct arglist * deps;
 if(depname)
 {
  deps = arg_get_value(desc, "DEPENDENCIES");
  if(!deps){
   deps = emalloc(sizeof(struct arglist));
   arg_add_value(desc, "DEPENDENCIES", ARG_ARGLIST, -1, deps);
   }
  arg_add_value(deps, estrdup(depname), ARG_STRING, 0, estrdup(""));
 }
}


ExtFunc
void plug_set_timeout(desc, timeout)
 struct arglist * desc; 
 int timeout;
{
    arg_add_value(desc, "TIMEOUT", ARG_INT,
    			sizeof(int), (void *)timeout);
}


ExtFunc
const char * plug_get_family(desc)
 struct arglist * desc;
{
 return(arg_get_value(desc, "FAMILY"));
}
		

ExtFunc
void plug_set_launch(desc, launch)
 struct arglist * desc;
 int launch;
{
  if(arg_set_value(desc, "ENABLED", sizeof(int), (void *)launch))
  {
   arg_add_value(desc, "ENABLED", ARG_INT, sizeof(int), (void *)launch);
  }
}


ExtFunc
int plug_get_launch(desc)
 struct arglist * desc;
{
 	return((int)arg_get_value(desc, "ENABLED"));
}	
	
	
ExtFunc
void plug_set_name(desc, name, language)
 struct arglist * desc; 
 const char * name; 
 const char * language;
{
 char * s_language;
 struct arglist * prefs = arg_get_value(desc, "preferences");
  
  s_language = arg_get_value(prefs,"language");
  if(s_language && language)
   {
    if(!strcmp(s_language, language))
    {
    if(name)
    arg_add_value(desc, "NAME", ARG_STRING,
    			strlen(name), estrdup(name));
    }
   }
  else if(name)
  {
    if(!arg_get_value(desc, "NAME"))
    	arg_add_value(desc, "NAME", ARG_STRING,
    			strlen(name), estrdup(name));	
  }
}


ExtFunc
void plug_set_summary(desc, summary,language)
 struct arglist * desc;
 const char * summary;
 const char * language;
{
 char * s_language;
 struct arglist * prefs = arg_get_value(desc, "preferences");
  
  s_language = arg_get_value(prefs,"language");
  if(s_language && language)
   {
    if(!strcmp(s_language, language))
    {
    if(summary)
    arg_add_value(desc, "SUMMARY", ARG_STRING,
    			strlen(summary), estrdup(summary));
    }
   }
  else if(summary)
  {
    if(!arg_get_value(desc, "SUMMARY"))
    	arg_add_value(desc, "SUMMARY", ARG_STRING,
    			strlen(summary), estrdup(summary));	
  }
}


ExtFunc
void plug_set_description(desc, description,language)
 struct arglist * desc;
 const char * description;
 const char * language;
{
  char * s_language;
 struct arglist * prefs = arg_get_value(desc, "preferences");
  
  s_language = arg_get_value(prefs,"language");
  if(s_language && language)
   {
    if(!strcmp(s_language, language))
    {
    if(description)
    arg_add_value(desc, "DESCRIPTION", ARG_STRING,
    			strlen(description), estrdup(description));
    }
   }
  else if(description)
  {
    if(!arg_get_value(desc, "DESCRIPTION"))
    	arg_add_value(desc, "DESCRIPTION", ARG_STRING,
    			strlen(description), estrdup(description));	
  }
}


ExtFunc
void plug_set_copyright(desc, copyright,language)
 struct arglist * desc;
 const char * copyright;
 const char * language;
{
 char * s_language;
 struct arglist * prefs = arg_get_value(desc, "preferences");
  
  s_language = arg_get_value(prefs,"language");
  if(s_language && language)
   {
    if(!strcmp(s_language, language))
    {
    if(copyright)
    arg_add_value(desc, "COPYRIGHT", ARG_STRING,
    			strlen(copyright), estrdup(copyright));
    }
   }
  else if(copyright)
  {
    if(!arg_get_value(desc, "COPYRIGHT"))
    	arg_add_value(desc, "COPYRIGHT", ARG_STRING,
    			strlen(copyright), estrdup(copyright));	
  }
}


ExtFunc
void plug_set_category(desc, category)
 struct arglist * desc;
 int category;
{
       arg_add_value(desc, "CATEGORY", ARG_INT, sizeof(int), (void *)category);
}


ExtFunc
void plug_set_attacklevel(desc, attacklevel)
 struct arglist * desc;
 int attacklevel;
{
	arg_add_value(desc, "ATTACKLEVEL", ARG_INT, sizeof(int), (void *)attacklevel);
}


ExtFunc
void plug_add_host(desc, hostname)
 struct arglist * desc;
 struct arglist * hostname;
{
	struct arglist * h;
	
	h = arg_get_value(desc, "HOSTNAME");
	if(!h)arg_add_value(desc, "HOSTNAME", ARG_ARGLIST, sizeof(hostname), hostname);
	else arg_set_value(desc, "HOSTNAME", sizeof(hostname), hostname);
}


ExtFunc
void host_add_port_proto(args, portnum, state, proto)
 struct arglist * args;
 int portnum;
 int state;
 char * proto;
{
 char * port_s = emalloc(50+strlen(proto));
 sprintf(port_s, "Ports/%s/%d", proto, portnum);
 if(!plug_get_key(args, port_s))
 {
  plug_set_key(args, port_s, ARG_INT, (void*)1);
  sprintf(port_s, "/tmp/Ports/%s/%d", proto, portnum);
  plug_set_key(args, port_s, ARG_INT, (void*)1);
 }
 free(port_s);
}


ExtFunc
void host_add_port(hostdata, portnum, state)
 struct arglist * hostdata;
 int portnum;
 int state;
{
 host_add_port_proto(hostdata, portnum, state, "tcp");
}

ExtFunc
void host_add_port_udp(hostdata, portnum, state)
 struct arglist * hostdata;
 int portnum;
 int state;
{
 host_add_port_proto(hostdata, portnum, state, "udp");
}
  

int port_in_ports(port, ports, s, e)
	u_short port, * ports;
	int s, e;
{
 int mid = (s+e)/2;
 if(s==e)return(port == ports[e]);
 if(port > ports[mid])return(port_in_ports(port, ports, mid+1, e));
 else return(port_in_ports(port, ports, s, mid));
}
 	


static int
unscanned_ports_as_closed(plugdata)
 struct arglist * plugdata;
{
 static struct arglist * prefs;
 char * unscanned;
 if(!prefs)prefs = arg_get_value(plugdata, "preferences");
 unscanned = arg_get_value(prefs, "unscanned_closed");
 if(unscanned && !strcmp(unscanned, "yes"))
  return 0;
 else
  return 1;
}
           
ExtFunc
int host_get_port_state_proto(plugdata, portnum, proto)
 struct arglist * plugdata;
 int portnum;
 char * proto;
{ 
 char * port_s;
 unsigned short * range;
 int flag;
 int num;

 if(!proto)
  proto = "tcp";
  
 /* Check that we actually scanned the port */
 
 if(!strcmp(proto, "tcp") && !plug_get_key(plugdata, "Host/scanned")){
#ifdef DEBUG
	printf("HOST NOT SCANNED\n");
#endif
	return unscanned_ports_as_closed(plugdata);
	}

 else if(!strcmp(proto, "udp") && !plug_get_key(plugdata, "Host/udp_scanned"))
       {
        return unscanned_ports_as_closed(plugdata);
      }
      
      		
 range = arg_get_value(plugdata, "ports");
 if(!range){
#ifdef DEBUG
	printf("unknown port range\n");
#endif	
 	return(1);
	}
 num = (int)arg_get_value(plugdata, "ports_num");
 if(!port_in_ports(portnum, range, 0, num)){
       return unscanned_ports_as_closed(plugdata);
       }
 

 /* Ok, we scanned it. What is its state ? */
 port_s = emalloc(20+strlen(proto));
 sprintf(port_s, "Ports/%s/%d", proto, portnum);
 flag = (int)plug_get_key(plugdata, port_s);
 free(port_s);
#ifdef DEBUG
 printf("%d -> %d\n", portnum, flag);
#endif
 return(flag ? 1:0);
}


ExtFunc
int host_get_port_state(plugdata, portnum)
 struct arglist * plugdata;
 int portnum;
{
 return(host_get_port_state_proto(plugdata, portnum, "tcp"));
}

ExtFunc
int host_get_port_state_udp(plugdata, portnum)
 struct arglist * plugdata;
 int portnum;
{
 return(host_get_port_state_proto(plugdata, portnum, "udp"));
}


ExtFunc
const char * plug_get_hostname(desc)
 struct arglist * desc;
{
 struct arglist * hinfos = arg_get_value(desc, "HOSTNAME");
 if(hinfos)return((char*)arg_get_value(hinfos, "NAME"));
 else return(NULL);
}

ExtFunc
const char * plug_get_host_fqdn(desc)
 struct arglist * desc;
{
 struct arglist * hinfos = arg_get_value(desc, "HOSTNAME");
 if(hinfos)return((char*)arg_get_value(hinfos, "FQDN"));
 else return(NULL);
}


ExtFunc
struct in_addr * plug_get_host_ip(desc)
 struct arglist * desc;
{
 struct arglist * hinfos = arg_get_value(desc, "HOSTNAME");
 if(hinfos)return((struct in_addr*)arg_get_value(hinfos, "IP"));
 else return(NULL);
}



static void 
mark_successful_plugin(desc)
 struct arglist * desc;
{
 int id = (int)arg_get_value(desc, "ID");
 char data[512];
 

 bzero(data, sizeof(data));
 sprintf(data, "Success/%d", id);
 if(!plug_get_key(desc, data))
  plug_set_key(desc, data, ARG_INT, (void*)1);
 
 
 /*
  * KB entries starting by /tmp/ are not saved
  */
 sprintf(data, "/tmp/Success/%d", id);
 if(!plug_get_key(desc, data))
  plug_set_key(desc, data, ARG_INT, (void*)1);
  
}

static void
mark_post(desc, action, content)
 struct arglist * desc;
 char * action;
 char * content;
{
 char * entry_name = emalloc(strlen(action) + 50);
 int num_post = (int)arg_get_value(desc, "NUM_POST");
 
#ifdef DEBUG_DIFF_SCAN
 printf("===>MARK_POST\n");
#endif
 sprintf(entry_name, "SentData/%d/%s/%d", (int)arg_get_value(desc, "ID"), action, num_post);
 plug_set_key(desc, entry_name, ARG_STRING, content);
 
 efree(&entry_name);
}

static int
post_sent_already(desc, action, content)
 struct arglist * desc;
 char * action;
 char * content;
{
 char * trunc_name = emalloc(strlen(action) + 50);
 int num_post = (int)arg_get_value(desc, "NUM_POST");
 struct arglist * key = (struct arglist *)arg_get_value(desc, "key");
 
 sprintf(trunc_name, "SentData/%d/%s/%d", (int)arg_get_value(desc, "ID"), action,num_post);
 while(key && key->next)
 {
#ifdef DEBUG_DIFF_SCAN
  printf("%s & %s\n", trunc_name, key->name);
#endif  
  if(!strcmp(trunc_name, key->name))
  {
#ifdef DEBUG_DIFF_SCAN
   printf("Compare %s and %s\n", trunc_name, key->name);
#endif   
   if(banner_diff(content, key->value))
   {
#ifdef DEBUG_DIFF_SCAN
    printf("DIFF\n");
#endif   
    efree(&trunc_name);
    return 0;
   }
   else return 1;
  }
  key = key->next;
 }
 efree(&trunc_name);
 return 0;
}
 


 
/* Pluto 24.6.00: reduced to one, and left the orig in place */
void 
proto_post_wrapped(desc, port, proto, action, what)
 struct arglist * desc;
 int port;
 const char * proto;
 const char * action;
 const char * what;
{

 char *t;
 char * buffer;
 int soc;
 struct arglist * globs;
 struct arglist * hostdata;
 char * naction;
 int len;
 ntp_caps* caps = arg_get_value(desc, "NTP_CAPS");
 char * cve;
 int do_send = 1;
 int num_post = (int)arg_get_value(desc, "NUM_POST");
 
 if(!num_post)
 {
  arg_add_value(desc, "NUM_POST", ARG_INT, sizeof(int), (void*)1);
  num_post = 1;
 }
 else
 {
  arg_set_value(desc, "NUM_POST", sizeof(int), (void*)++num_post);
 }
 
 
 
 if(!action)
 {
  action = arg_get_value(desc, "DESCRIPTION");
 }
 
 cve = arg_get_value(desc, "CVE_ID");
 if(!action)return;
 len = strlen(action);
 if(cve)len+=strlen(cve)+20;

 

 if(!caps)return;

 hostdata = arg_get_value(desc, "HOSTNAME");
 naction = emalloc(len+1);
 if(cve)sprintf(naction, "%s\nCVE : %s\n", action, cve);
 else strncpy(naction, action, len);
 
 if(!caps->escape_crlf)
   while((t=strchr(naction, '\n'))||(t=strchr(naction, '\r')))t[0]=';';
 else
  {
   char * old = naction;
   len -= strlen(naction);
   naction = addslashes(naction);
   len += strlen(naction);
   efree(&old);
  }
  
 buffer=malloc(1024+len);
 if(caps->ntp_11) {
   struct servent * service = NULL;
   char idbuffer[32];
   service = getservbyport(htons((unsigned short)port), proto);
#ifndef NESSUSNT
   endservent();
#endif
   if (caps->scan_ids) { 
     if (arg_get_type(desc, "ID") == -1) {
       *idbuffer = '\0';
     } else {
       int id = (int)arg_get_value(desc, "ID");
       sprintf(idbuffer, "<|> %d ", id);
     }
   } else {
     *idbuffer = '\0';
 }
  if(port>0){
     sprintf(buffer,
	     "SERVER <|> %s <|> %s <|> %s (%d/%s) <|> %s %s<|> SERVER\n",
	     what,
  	(char *)arg_get_value(hostdata, "NAME"),
        service ? service->s_name:"unknown",
	     port, proto, naction, idbuffer);
    
   } else
     sprintf(buffer,
	     "SERVER <|> %s <|> %s <|> general/%s <|> %s %s<|> SERVER\n",
	     what,
  	(char *)arg_get_value(hostdata, "NAME"), 
	     proto, naction, idbuffer);
 } else {
   sprintf(buffer, "SERVER <|> %s <|> %s <|> %d:%s <|> SERVER\n", 
	   what,
	   (char *)arg_get_value(hostdata, "NAME"), port, naction);
 }
 
 
 /*
  * XXX only send data to the client if the option
  * DIFF_SCAN is not set (or if the plugin used to
  * not be successful [that is, if there is no Success/<id>
  * key in the KB]).
  */
 if(arg_get_value(desc, "DIFF_SCAN"))
 {
  char str[128];
#ifdef DEBUG_DIFF_SCAN  
  fprintf(stderr, "** DIFF_SCAN enabled\n");
#endif  
  bzero(str, sizeof(str));
  
  if(!post_sent_already(desc, what, action))
 	 {
	  mark_post(desc, what, action);
 	 }
  else do_send = 0;
 }
  else
    mark_post(desc, what, action);

 
 if(do_send)
 {
  soc = (int)arg_get_value(desc, "SOCKET");
  globs = emalloc(sizeof(struct arglist));
  arg_add_value(globs, "global_socket", ARG_INT, sizeof(int), (void *)soc);
  auth_send(globs, buffer);
  arg_free_all(globs);
 }
 
 /*
  * Mark in the KB that the plugin was sucessful
  */
 mark_successful_plugin(desc);
 efree(&buffer);
 efree(&naction);
 return;
}

/* Pluto end */

ExtFunc
void proto_post_hole(desc, port, proto, action)
 struct arglist * desc;
 int port;
 const char * proto;
 const char * action;
{
  proto_post_wrapped(desc, port, proto, action, "HOLE");
  return;
}


ExtFunc
void post_hole(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
  proto_post_hole(desc, port, "tcp", action);
} 


ExtFunc
void post_hole_udp(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
 proto_post_hole(desc, port, "udp", action);
}


ExtFunc
void post_info(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
  proto_post_info(desc, port, "tcp", action);
} 


ExtFunc
void post_info_udp(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
 proto_post_info(desc, port, "udp", action);
}


ExtFunc
void proto_post_info(desc, port, proto, action)
 struct arglist * desc;
 int port;
 const char * proto;
 const char * action;
{
  proto_post_wrapped(desc, port, proto, action, "INFO");
  return;
}
 
ExtFunc
void post_note(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
  proto_post_note(desc, port, "tcp", action);
} 

     
ExtFunc
void post_note_udp(desc, port, action)
 struct arglist * desc;
 int port;
 const char * action;
{
 proto_post_note(desc, port, "udp", action);
}
	   

ExtFunc
void proto_post_note(desc, port, proto, action)
 struct arglist * desc;
 int port;
 const char * proto;
 const char * action;
{
  /*
   * Backward compatibility. We only use the notes if the remote
   * client accepts them
   */
  char * allow_notes = get_preference(desc, "ntp_client_accepts_notes");
  if(allow_notes && !strcmp(allow_notes, "yes")) 
   proto_post_wrapped(desc, port, proto, action, "NOTE");
  else
   proto_post_wrapped(desc, port, proto, action, "INFO");
 return;
} 
 
 
ExtFunc
char * get_preference(desc, name)
 struct arglist *desc;
 const char * name;
{
 struct arglist * prefs;
 prefs = arg_get_value(desc, "preferences");
 if(!prefs)return(NULL);
 return((char *)arg_get_value(prefs, name));
}


ExtFunc
void add_plugin_preference(desc, name, type, defaul)
 struct arglist *desc;
 const char * name;
 const char * type;
 const char * defaul;
{
 struct arglist * prefs = arg_get_value(desc, "preferences");
 char * p_name = arg_get_value(desc, "NAME");
 char * pref;
 char * cname;
 
 cname = estrdup(name);
 while(cname[strlen(cname)-1]==' ')cname[strlen(cname)-1]='\0';
 
 if(!prefs || !p_name)return;
 pref = emalloc(strlen(p_name)+10+strlen(type)+strlen(cname));
 sprintf(pref, "%s[%s]:%s", p_name, type, cname);
 arg_add_value(prefs, pref, ARG_STRING, strlen(defaul), estrdup(defaul));
 efree(&cname);
 efree(&pref);
}


ExtFunc char * 
get_plugin_preference(desc, name)
  struct arglist * desc;
  const char * name;
{
 struct arglist * prefs = arg_get_value(desc, "preferences");
 char * plug_name = arg_get_value(desc, "NAME");
 char * cname = estrdup(name);
 while(cname[strlen(cname)-1]==' ')cname[strlen(cname)-1]='\0';
 
 while(prefs && prefs->next)
 {
  char * a= NULL, *b = NULL;
  int c = 0;
  char * t = estrdup(prefs->name);
  
  a = strchr(t, '[');
  if(a)b=strchr(t, ']');
  if(b)c=(b[1]==':');
  
  if(c)
  {
   b+=2*sizeof(char);
   if(!strcmp(cname, b)){
   	a[0] = 0;
	if(!strcmp(t, plug_name)){
		efree(&t);
		efree(&cname);
		return(prefs->value);
		}
	}
  }
  efree(&t);
  prefs = prefs->next;
 }
 efree(&cname);
 return(NULL);
}

ExtFunc const char * 
get_plugin_preference_fname(desc, filename)
 struct arglist * desc;
 const char * filename;
{
 struct arglist * globals = arg_get_value(desc, "globals");
 harglst * trans;
 if(!globals) 
  return NULL;
  
 trans = arg_get_value(globals, "files_translation");
 if(!trans)
  return NULL;
 
 return harg_get_string(trans, filename);
}


ExtFunc
void plug_set_key(args, name, type, value)
 struct arglist * args;
 char * name;
 int type;
 void * value;
{
 int pip = (int)arg_get_value(args, "pipe");
 char * str = NULL;
#ifdef DEBUG
 printf("set key %s -> %d\n", name, value);
#endif 
 
 if(!name || !value)return;
 switch(type)
 {
  case ARG_STRING :
   value = addslashes(value);
   str = emalloc(strlen(name)+strlen(value)+10);
   sprintf(str, "%d %s=%s;\n", ARG_STRING, name, (char *)value);
   efree(&value);
   break;
  case ARG_INT :
   str = emalloc(strlen(name)+20);
   sprintf(str, "%d %s=%d;\n", ARG_STRING, name, (int)value);
   break;
 }
 if(str)
 {
   int len = strlen(str);
   int sent = 0;
   
   while(sent < len)
   {
     int e;
    e = send(pip, str+sent, len+1, 0);
    if(e < 0){
    	perror("send ");
	return;
	}
    sent+=e;
   }
  }
} 



ExtFunc void
scanner_add_port(args, port, proto)
 struct arglist * args;
 int port;
 char * proto;
{
 ntp_caps* caps = arg_get_value(args, "NTP_CAPS");
 struct arglist * hostdata = arg_get_value(args, "HOSTNAME");
 char * buf;
 struct servent * serv = getservbyport(htons(port), proto);
 char * hn = arg_get_value(hostdata, "NAME");
 int len;
 int soc;
 struct arglist * globs;
 int do_send = 1;
 static int confirm = -1;
 
 if(confirm < 0)
 {
  struct arglist * globals = arg_get_value(args, "globals");
  if(globals)confirm = (int)arg_get_value(globals, "confirm");
 }

#ifndef NESSUSNT
  endservent();
#endif 
 
 
 /*
  * Diff scan stuff : if the port was known to be open,
  * there is no need to report it again.
  */
 if(arg_get_value(args, "DIFF_SCAN"))
 {
   char * port_s = emalloc(50+strlen(proto));
   sprintf(port_s, "Ports/%s/%d", proto, port);
   if(plug_get_key(args, port_s))do_send = 0;
   efree(&port_s);
 }


 host_add_port_proto(args, port, 1, proto);
 
 len = 255 + (hn ? strlen(hn):0) + 
 	    (serv ?strlen(serv->s_name):strlen("unknown"));
 buf = emalloc(len);
 if(caps->ntp_11)
  sprintf(buf, "SERVER <|> PORT <|> %s <|> %s (%d/%s) <|> SERVER\n",
 		hn,serv ? serv->s_name:"unknown",
		port, proto);
 else
  {
   if(!strcmp(proto, "tcp"))
     sprintf(buf, "SERVER <|> PORT <|> %s <|> %d <|> SERVER\n",
  		hn, port);
  }
   
 if(do_send)
 {
 soc = (int)arg_get_value(args, "SOCKET");
 globs = emalloc(sizeof(struct arglist));
 arg_add_value(globs, "global_socket", ARG_INT, sizeof(int), (void *)soc);
 arg_add_value(globs, "confirm", ARG_INT, sizeof(int), (void*)confirm);
 
 auth_send(globs, buf);
 arg_free(globs);
 }
 efree(&buf);
}


/*
 * plug_get_key() may fork(). We use this signal handler to kill
 * its son in case the process which calls this function is killed
 * itself
 */
#ifndef NESSUSNT
static int _plug_get_key_son;

static void 
plug_get_key_sighand_term(int sig)
{
 if(_plug_get_key_son)
 {
  kill(_plug_get_key_son, SIGTERM);
  _plug_get_key_son = 0;
 }
 exit(0);
}

static void
sig_n(n, f)
 int n;
 void * f;
{
 #ifdef HAVE_SIGACTION
  struct sigaction sa;
  sa.sa_handler = f;
  sa.sa_flags = 0;
  sigemptyset(&sa.sa_mask);
  sigaction(n,&sa,(struct sigaction *) 0);
#else
  signal(n, f);
#endif
}

static void
sig_term(f)
 void *f;
{
 sig_n(SIGTERM, f);
}

static void
sig_alarm(f)
 void * f;
{
 sig_n(SIGALRM, f);
}
#endif

void * 
plug_get_key(args, name)
 struct arglist * args;
 char * name;
{
#ifndef NESSUSNT
 struct arglist * key = (struct arglist *)arg_get_value(args, "key");
 int type;
 
 if(!key)return(NULL);
 
 type = arg_get_type(key, name);
 if(type >= 0)
 {
  if(type == ARG_STRING)
     {
      return arg_get_value(key, name);
     }
   else if(type == ARG_INT)
    return arg_get_value(key,name);
  else if(type == ARG_ARGLIST)
  {
   struct arglist * value = arg_get_value(key, name);
   while(value && value->next)
   {
     int pid;  
    if(!(pid = fork()))
     {
     /* so that different plugins do not use the same see */
     nessus_init_random();

     sig_term(exit);
     sig_alarm(exit);
     alarm(120);
     arg_set_value(key, name, -1, value->value);
     arg_set_type(key, name, value->type);
     return value->value;
     }
    else
     {
      if(pid < 0)
      {
       fprintf(stderr, "nessus-libraries:libnessus:plugutils.c:plug_get_key(): fork() failed : %s", strerror(errno));	      
       return NULL;
      }
      else
      {
      _plug_get_key_son = pid;
      sig_term(plug_get_key_sighand_term);
      waitpid(pid,NULL, 0);
      _plug_get_key_son = 0;
      sig_term(exit);
      }
    }
    value = value->next;
   }
   exit(0);
  }
 }
#endif
 return NULL;
}

ExtFunc unsigned int 
plug_get_host_open_port(struct arglist * desc)
{
 struct arglist * h = arg_get_value(desc, "key");
 char * str = "Ports/tcp/";
 char * t;
 int port = 0;
 
 while(h && h->next)
 {
  if((strlen(h->name) > strlen(str)) && 
     !strncmp(h->name, str, strlen(str))){
     	t = h->name + strlen(str);
	port = atoi(t);
	break;
	}
  h = h->next;	
 }
 
 return(port);
}
       
 
/*
 * Those brain damaged functions should probably be in another file
 * They are use to remember who speaks SSL or not
 */
   
ExtFunc
void plug_set_port_transport(args, port, tr)
     struct arglist * args;
     int		port, tr;
{
  static char	s[256];

  sprintf(s, "Transports/TCP/%d", port);
  plug_set_key(args, s, ARG_INT, (void*) tr);
}

ExtFunc
int plug_get_port_transport(args, port)
     struct arglist * args;
     int		port;
{
  static char	s[256];
  char * trp;
  sprintf(s, "Transports/TCP/%d", port);
  
  trp = plug_get_key(args, s);
  if(trp)
   return atoi(trp);
  else
   return NESSUS_ENCAPS_IP;
}

ExtFunc
const char* plug_get_port_transport_name(args, port)
     struct arglist * args;
     int		port;
{
  return get_encaps_name(plug_get_port_transport(args, port));
}

#ifdef HAVE_SSL
static void
plug_set_ssl_item(args, item, itemfname)
 struct arglist * args;
 char * item;
 char * itemfname;
{
 static char s[256];
 struct arglist * key = (struct arglist*)arg_get_value(args, "key");
 sprintf(s, "SSL/%s", item);
 plug_set_key(args, s, ARG_STRING, itemfname);

 /*
  * Ugly hack - the KB is not updated before the next plugin call, 
  * so we manually add this key
  */
 arg_add_value(key, s, ARG_STRING, strlen(itemfname), estrdup(itemfname));
}

ExtFunc void
plug_set_ssl_cert(args, cert)
 struct arglist * args;
 char * cert;
{
 plug_set_ssl_item(args, "cert", cert);
}

ExtFunc void 
plug_set_ssl_key(args, key)
 struct arglist * args;
 char * key;
{
 plug_set_ssl_item(args, "key", key);
}
ExtFunc void
plug_set_ssl_pem_password(args, key)
 struct arglist * args;
 char * key;
{
 plug_set_ssl_item(args, "password", key);
}

ExtFunc void
plug_set_ssl_CA_file(args, key)
 struct arglist * args;
 char * key;
{
 plug_set_ssl_item(args, "CA", key);
}
#else
ExtFunc  void
plug_set_ssl_cert(args, cert)
 struct arglist * args;
 char * cert;
{
 fprintf(stderr, "plug_set_ssl_cert(): not implemented\n");
}

ExtFunc void
plug_set_ssl_key(args, key)
 struct arglist * args;
 char * key;
{
 fprintf(stderr, "plug_set_ssl_key(): not implemented\n");
}
#endif /* HAVE_SSL */


ExtFunc char *
find_in_path(name, safe)
     char	*name;
     int	safe;
{
  char		*buf = getenv("PATH"), *pbuf, *p1, *p2;
  static char	cmd[MAXPATHLEN];
  int		len = strlen(name);
  
  if (len >= MAXPATHLEN)
    return NULL;

  if (buf == NULL)		/* Should we use a standard PATH here? */
    return NULL;

  pbuf = buf;
  while (*pbuf != '\0')
    {
      for (p1 = pbuf, p2 = cmd; *p1 != ':' && *p1 != '\0'; )
	*p2 ++ = *p1 ++;
      *p2 = '\0';
      if (*p1 == ':')
	p1 ++;
      pbuf = p1;
      if (p2 == cmd)		/* :: found in $PATH */
	strcpy(cmd, ".");

      if (cmd[0] != '/' && safe)
	continue;
      if (p2 - cmd + 1 + len >= MAXPATHLEN)
	/* path too long: cannot be reached */
	continue;

      sprintf(p2, "/%s", name);
      if (access(cmd, X_OK) == 0)
	{
	  *p2 = '\0';
#if 0
	  fprintf(stderr, "find_in_path: %s found in %s\n", name, cmd);
#endif
	  return cmd;
	}
#if 0
      else
	fprintf(stderr, "find_in_path: No %s\n", cmd);
#endif
    }
  return NULL;
}

ExtFunc int 
is_shell_command_present(name)
 char * name;
{
  return find_in_path(name, 0) != NULL;
}
