/* Nessuslib -- the Nessus Library
 * Copyright (C) 1999 Renaud Deraison
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <includes.h>
#ifndef _CYGWIN_
#include "../libpcap-nessus/pcap.h"
#else
#include <pcap.h>
#endif

#include <net/if.h>
#ifdef HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif

#define IFNAME_MAX 64
#define IF_MAX 256
#define LOCALNET htonl(0x7F000000)

struct iface {
    char if_name[IFNAME_MAX];
    struct in_addr if_addr;
};

static int getinterfaces(struct iface*, int *);



/*
 * On systems with few /dev/bpf's, we may need to wait a little
 * before we can get one.
 *
 * These function calls are not perfect per se, but really helps
 * on most systems
 */
#ifdef HAVE_DEV_BPFN
ExtFunc int 
wait_for_free_bpf(timeout)
 int timeout;
{
  int fd;
  int n = 0;
  char device[sizeof "/dev/bpf000"];
  struct timeval now, then;
  
  gettimeofday(&then, NULL);
  for(;;)
  {
  /*
   * Go through all the minors and find one that isn't in use.
   */
  do {
 	 (void)sprintf(device, "/dev/bpf%d", n++);
 	 fd = open(device, O_RDONLY);
  } while (fd < 0 && errno == EBUSY);
 if(fd > 0)
 { 
  return fd;
 }
 usleep(10000);
 gettimeofday(&now, NULL);
 if(now.tv_sec - then.tv_sec > timeout)
  return -1;
 }
 return -1;
}

ExtFunc int
release_bpf(bpf)
 int bpf;
{
 return close(bpf);
}
#else
ExtFunc int
wait_for_free_bpf(timeout)
 int timeout;
{
 return 1;
}

ExtFunc int
release_bpf(bpf)
 int bpf;
{
 return 0;
}
#endif /* HAVE_DEV_BPFN */



int is_local_ip(addr)
 struct in_addr addr;
{
 int ifaces = 256;
 struct iface ifs[256];
 int i;
 
 if ( getinterfaces(ifs, &ifaces) < 0) 
 	return -1;
 for(i=0;i<ifaces;i++)
 {
  bpf_u_int32 net, mask;
  char errbuf[PCAP_ERRBUF_SIZE];
  pcap_lookupnet(ifs[i].if_name, &net, &mask, errbuf);
  if((net & mask) == (addr.s_addr & mask))
  	return 1;
 }
 return 0;
}



/* 
 * We send an empty UDP packet to the remote host, and read back
 * its mac address. 
 *
 * (we should first interrogate the kernel's arp cache - we may
 * rely on libdnet in the future to do that)
 *
 * As a bonus, this function works well as a local ping
 *
 */
int 
get_mac_addr(addr, mac)
 struct in_addr addr;
 char ** mac;
{
 int soc = socket(AF_INET, SOCK_DGRAM, 0);
 struct sockaddr_in soca;
 int b;
 pcap_t * pcap;
 struct in_addr me;
 char * iface = routethrough(&addr, &me);
 char errbuf[PCAP_ERRBUF_SIZE];
 char filter[255];
 bpf_u_int32 netmask, network;
 struct bpf_program filter_prog; 
 char * src_host, * dst_host;
 unsigned char * packet;
 struct pcap_pkthdr head;
 
 *mac = NULL;
 if(soc < 0)
  return -1;
  
 b = wait_for_free_bpf(15);
 if(b < 0)
 {
   close(soc);
   return -1;
 }
   
 release_bpf(b);
 pcap = pcap_open_live(iface, 64, 0, 100, errbuf);
 if(!pcap)
 {
  close(soc);
  return -1;
 }
 
 /*
  * We only deal with ethernet
  */
 if(pcap_datalink(pcap) != DLT_EN10MB)
 {
  pcap_close(pcap);
  close(soc);
  return -1;
 }
 
 pcap_lookupnet(iface, &network, &netmask, 0);
 src_host = estrdup(inet_ntoa(me));
 dst_host = estrdup(inet_ntoa(addr));
 sprintf(filter, "ip and (src host %s and dst host %s)",
 	src_host, dst_host);
 efree(&src_host);
 efree(&dst_host);
 
 pcap_compile(pcap, &filter_prog, filter, 0, netmask);
 pcap_setfilter(pcap, &filter_prog);	
 
 soca.sin_addr.s_addr = addr.s_addr;
 soca.sin_port = htons(9); /* or whatever */
 soca.sin_family = AF_INET;
 if(sendto(soc, NULL, 0, 0, (struct sockaddr*)&soca, sizeof(soca)) == 0)
 {
  packet = (unsigned char*)pcap_next(pcap, &head);
  if(packet)
  { 
   if(head.caplen >= get_datalink_size(pcap_datalink(pcap)))
   {
    int i;
    for(i=0;i<6;i++)
    	if(packet[i]!=0xFF)break;
    
    if(i == 6)
    {
     pcap_close(pcap);
     close(soc);
     return 1;
    }
    
    *mac = emalloc(22);
    sprintf(*mac, "%.2x.%.2x.%.2x.%.2x.%.2x.%.2x",
    		(unsigned char)packet[0],
		(unsigned char)packet[1], 
		(unsigned char)packet[2],
		(unsigned char)packet[3],
		(unsigned char)packet[4],
		(unsigned char)packet[5]);
   pcap_close(pcap);
   close(soc);
   return 0;		
   }
  }
  else
  {
   pcap_close(pcap);
   close(soc);
   return 1;
  }
 }
 pcap_close(pcap);
 close(soc);
 return -1;
}

static int
getinterfaces(ifaces, num)
   struct iface *ifaces;
   int *num;
{
  int soc;
  int len;
  struct ifconf ifconf;
  struct ifreq *ifreq;
  char buf[32768];
  char * b = buf;
  int i = 0;
  int tot_len = 0;
  
  
  if(!ifaces || !num)
   return -1;
  
  soc = socket(AF_INET, SOCK_DGRAM, 0);
  if(soc < 0)
  {
   perror("nessus-libraries:pcap.c:getinterfaces:socket() ");
   return -1;
  }
  
  ifconf.ifc_len = sizeof(buf);
  ifconf.ifc_buf = b;
  
  if(ioctl(soc, SIOCGIFCONF, &ifconf)  < 0)
  {
   perror("nessus-libraries:pcap.c:getinterfaces:ioctl() ");
   return -1;
  }
  
  closesocket(soc);
  
  ifreq = (struct ifreq*) buf;
  if(ifconf.ifc_len == 0)
  {
   fprintf(stderr, "nessus-libraries:pcap.c:getinterfaces: nothing found\n");
   return -1;
  }
  
#ifdef HAVE_SOCKADDR_SA_LEN
  len = ifreq->ifr_addr.sa_len;
#else
  len = sizeof(struct sockaddr);
#endif

  for(;tot_len<ifconf.ifc_len;)
  {
   struct sockaddr_in *soca;
   char *t;
   soca = (struct sockaddr_in*)(&(ifreq->ifr_addr));
   memcpy(&(ifaces[i].if_addr), 
   	  &(soca->sin_addr),
	  sizeof(struct in_addr)
	  );
   strncpy(ifaces[i].if_name, ifreq->ifr_name, IFNAME_MAX);
   ifaces[i].if_name[IF_MAX-1] = '\0';
   if((t = strchr(ifaces[i].if_name, ':')) != NULL)
   	t[0] = '\0';
   i++;
   if(i >= *num)
    {
    fprintf(stderr, "nessus-libraries:pcap.c:getinterfaces: too many interfaces\n");
    return -1; /* Yuck */
    }
#ifdef HAVE_SOCKADDR_SA_LEN
    len = ifreq->ifr_addr.sa_len;
#endif
    b = &(b[sizeof(ifreq->ifr_name) + len]);
    tot_len += sizeof(ifreq->ifr_name) + len;
    ifreq = (struct ifreq*)b;
  }
  
  *num = i;
  return 0;
}




/*
 * Convert an IP address to its associated network interface
 */
static int 
ipaddr2devname( dev, addr)
 char * dev;
 struct in_addr * addr;
{
 struct iface ifaces[IF_MAX];
 int num_ifaces = IF_MAX;
 int i;

 if(getinterfaces(ifaces, &num_ifaces) < 0)
 	return -1;
 
 for ( i = 0 ; i < num_ifaces ; i ++ )
 {
  if(addr->s_addr == ifaces[i].if_addr.s_addr)
   {
    strcpy(dev, ifaces[i].if_name);
    return 0;
   }
 }
 return -1;
}



/* Tests whether a packet sent to  IP is LIKELY to route 
 through the kernel localhost interface */
 
int 
islocalhost(addr)
 struct in_addr * addr;
{
 char ifname[IFNAME_MAX];
 
  if(addr == NULL)		/* Error */
   return -1;	
   
 
  return (
     (addr->s_addr == 0) 			    ||
     (addr->s_addr & htonl(0xFF000000) == LOCALNET) ||
     (ipaddr2devname(ifname, addr) ==  0)
     );
}


int
get_datalink_size(datalink)
 int datalink;
{
 switch(datalink) {
  case DLT_NULL : 
  	return 0;
	break;
  case DLT_EN10MB :
  	return 14;
	break;
  case DLT_IEEE802 :
  	return 22;
  case DLT_SLIP :
#if (FREEBSD || OPENBSD || NETBSD || BSDI)
	return 16;
#else
	return 24;
#endif
	break;
  case DLT_PPP :
#if defined(SOLARIS)
	return 8;
#elif (FREEBSD || OPENBSD || NETBSD || BSDI)
 	return 4;
#else
	return 24;
#endif
  case DLT_RAW :
  	return 0;
  default:	
  	return -1;
  }
}




static int
getsourceip(src, dst)
 struct in_addr * src, * dst;
{
 int soc;
 struct sockaddr_in sin;
 struct sockaddr_in myname;
 int len;
 
 soc = socket(AF_INET, SOCK_DGRAM, 0);
 if(soc < 0)
 {
  perror("nessus-libraries:pcap.c:getsourceip():socket() ");
  return -1;
 }
 
 sin.sin_port   = htons(5000); 
 sin.sin_family = AF_INET;
 sin.sin_addr  = *dst;
 
 if (connect(soc, (struct sockaddr *) &sin, sizeof(sin)) == -1)
 {
  perror("nessus-libraries:pcap.c:getsourceip():connect() ");
  closesocket(soc);
  return -1;
 }
 
 bzero(&myname, sizeof(myname));
 len = sizeof(myname);
 if(getsockname(soc, (struct sockaddr*)&myname, &len) == -1)
 {
  perror("nessus-libraries:pcap.c:getsourceip():getsockname() ");
  closesocket(soc);
  return -1;
 }
 close(soc);
 *src = myname.sin_addr;
 return 1;
}


char * 
routethrough(dst, src)
 struct in_addr * dst;
 struct in_addr * src;
{
 static char dev[IFNAME_MAX];
 
 if(getsourceip(src, dst) < 0)
  return NULL;
  
 if(ipaddr2devname(dev, src) < 0)
  return NULL;
  
 return dev;
}



