/* Nessus
 * Copyright (C) 1998 - 2001 Renaud Deraison
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * $Id: rtree.c,v 1.7 2002/05/24 10:37:08 renaud Exp $
 */
 
 

#include <includes.h>
#include "rtree.h"




/*=======================================================================*
 *                                                                       *
 * Tree management                                                       *
 *                                                                       *
 * We'll use some sort of rooted balanced tree to build our list of      *
 * plugins we can launch at the same time                                *
 *                                                                       *
 *=======================================================================*/
 
 
static int rtree_delete(struct rtree*, int);
 
/* 
 * Creates a new node with root <root> and with the potential of
 * holding approx. <num> subnodes
 */
struct rtree * 
rtree_node_new(root, num)
  struct rtree * root;
  int num;
{
  struct rtree * node = emalloc(sizeof(struct rtree));
  node->num_elems = 0;
  node->id = 0;
  node->elems = num ? emalloc(num*(sizeof(struct rtree*))) : NULL;
  if(node->elems)
  node->allocated_elems = num;
  node->root = root;
  node->data = NULL;
  if(root)
    rtree_insert(root, node);
  return node;
}


/*
 * Sets the data associated with a node
 */
int
rtree_node_set_data(node, data)
  struct rtree * node;
  void * data;
{
  if(!node) 
    return -1;
  else 
    node->data = data;
    
  return 0;  
}


/*
 * Gets the data associated with a node
 */
void *
rtree_node_data(node)
 struct rtree * node;
{
  if(!node)
    return RTREE_INVAL_NODE;
  else
    return node->data;
}


/*
 * Returns the number of elements <node> can contain
 */
int
rtree_node_size(node)
  struct rtree * node;
{
  if(!node)
    return -1;
  else 
    return node->allocated_elems;
}


int
rtree_node_num_elems(node)
 struct rtree * node;
{
 if(!node)
  return -1;
 else
  return node->num_elems;
}


/*
 * Resize the number of elements <node> can contain
 */
static int
rtree_node_resize(node, newsize)
  struct rtree * node;
  int newsize;
{
 if(!node)
    return -1;

  if(newsize < node->num_elems)
    node->num_elems = newsize;

  node->allocated_elems = newsize;
  if(!node->elems)
    {
    node->elems = emalloc(newsize*sizeof(struct rtree*));
    }
  else
   {
    node->elems = realloc(node->elems, newsize*sizeof(struct rtree*));
   }
  return 0;
}


/*
 * Get the n'th subnode of a node
 */
struct rtree *
rtree_node_get_nth_node(node, num)
  struct rtree * node;
  int num;
{
  if(num >= node->num_elems)
    return RTREE_INVAL_NODE;
  else
    return node->elems[num];
}
  
  
/*
 * Creates a new node, and associates data to it
 */
struct rtree *
rtree_node_new_with_data(root, num, data)
  struct rtree * root;
  int num;
  void * data;
{
 struct rtree * ret = rtree_node_new(root, num);
 rtree_node_set_data(ret, data);
 return ret;
}

struct rtree *
rtree_node_detach_from_root(node)
 struct rtree * node;
{
 int i;
 if((!node) || (!node->root))
  return node;
 
 for(i=0;i<node->root->num_elems;i++)
 {
  if(node->root->elems[i] == node)break;
 }
 
 if(node->root->elems[i] == node)
 {
  rtree_delete(node->root, i);
 }
 return node;
}


/*
 * Insert a node under <root>
 */ 
int
rtree_insert(root, node)
  struct rtree * root, * node;
{
  if(!root->allocated_elems)
    rtree_node_resize(root, 200);

  if(root->num_elems >= root->allocated_elems)
    rtree_node_resize(root, root->allocated_elems + 200);

  root->elems[root->num_elems] = node;
  if(node)
  {
    if(node->root != root)
      {
      if(node->root)
      {
       rtree_node_detach_from_root(node);
       rtree_free(rtree_root(node->root));
      }
      node->root = root;
      }
  }
  node->id =  root->num_elems;
  return root->num_elems++;
}

/*
 * Suppresses a node from under <root>. Note that
 * nothing is being freed from memory here, so we should
 * use this function with caution !
 */
static int
rtree_delete(root, num)
  struct rtree* root;
  int num;
{
  if(num >= root->num_elems)
   return -1;
  
  if(root->elems[num])
  {
   root->elems[num]->root = NULL;
   root->elems[num] = NULL;
  }
  memmove(&(root->elems[num]), &(root->elems[num+1]), sizeof(struct rtree*)* root->num_elems - num);
  root->num_elems--;
  return 0;
}


/*
 * Frees completely a tree from memory
 */
void
rtree_free(tree)
 struct rtree *tree;
{
  if(tree->elems)
  {
   int i;
   for(i=0;i<tree->num_elems;i++)
   {
    if(tree->elems[i])rtree_free(tree->elems[i]);
   }
  efree(&(tree->elems));
  }
  bzero(tree, sizeof(struct rtree));
  efree(&tree);
}


/* 
 * See function below
 */
static int
_rtree_node_depth(node, depth)
 struct rtree * node;
 int depth;
{
  if(node->root)
   return _rtree_node_depth(node->root, depth + 1);
  else
   return depth;
}

/*
 * Returns the depth (or level) of a node
 */
int
rtree_node_depth(node)
 struct rtree * node;
{
 return _rtree_node_depth(node, 0);
}

/*
 * Move the node to the level of its root
 *
 * That is, we have the tree :
 *    a
 *    |\
 *    b c
 *    | 
 *    d
 *
 * and we call rtree_node_move_up(d), and we get :
 *
 *      a
 *     /|\
 *    b c d
 *
 * If we have :
 *
 *      a
 *     /|
 *    b c
 *
 * And we call rtree_node_move_up(c), we obtain :
 *
 *      .
 *      |\
 *      a c
 *      |
 *      b
 *
 * So it's wise to call rtree_root() after making this call.
 *
 * This call returns the root above the node
 *
 */
struct rtree *
rtree_node_move_up(node)
 struct rtree * node;
{
 struct rtree * root;
 
 if(!node->root)
  return node; /* at top already */
 
 if(node->root->root)
 { 
 root = node->root->root;
 rtree_delete(node->root, node->id);
 rtree_insert(root, node);
 }
 else
 {
  struct rtree * old_root = node->root;
  rtree_delete(node->root, node->id);
  root = rtree_node_new(NULL, 200);
  rtree_insert(root, old_root);
  rtree_insert(root, node);
 }
 return root;
}


/*
 * Same as above, but we only move the node data, not
 * the subnodes. After this call, node->data = NULL
 * This call returns the root above the node
 */
struct rtree *
rtree_node_data_move_up(node)
 struct rtree * node;
{
 if(!node || !node->data)
  return NULL;
 
 if(!node->root)
 {
  /*
   * XXX if I wanted to write perfect code, I'd have to take
   * care of that situation. In practice, this never happens, so
   * why bother ?
   */
  fprintf(stderr, "rtree_node_data_move_up>HU?\n");
  exit(0);
 }
 
 
 if(node->root->root)
  {
   if(!node->root->root->data)
   {
   node->root->root->data = node->data;
   node->data = NULL;
   }
  else
  {
   struct rtree * new_node;
   new_node = rtree_node_new_with_data(node->root->root, 0, node->data);
   node->data = NULL;
  }
  return node->root->root;
 }
 else
 {
  struct rtree * root = rtree_node_new_with_data(NULL, 200, node->data);
  node->data = NULL;
  rtree_insert(root, node->root);
  return root;
 }
 /*NOTREACHED*/
}


/*
 * Returns the top-level root of a tree
 */
struct rtree *
rtree_root(tree)
 struct rtree * tree;
{
 if(tree->root)
  return rtree_root(tree->root);
 else
  return tree;
}


