/* Nessus
 * Copyright (C) 1998 - 2001 Renaud Deraison
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Plugins scheduler / launcher.
 *
 */
 
 
#include <includes.h>
#include "pluginload.h"
#include "piic.h"
#include "plugs_req.h"
#include "rtree.h"
#include "utils.h"
#include "preferences.h"
#include "log.h"

struct running {
	nthread_t pid;
	int soc[2];
	struct arglist * globals;
	struct arglist * kb;
	char           * name;
	struct arglist * plugin;
	struct timeval start;
	int timeout;
	};
	
	
/*
 * This is the 'hard' limit of the max. number of concurrent
 * plugins
 */	
#define MAX_PROCESSES 128
#undef VERBOSE_LOGGING	/* This really fills your nessusd.messages */

#undef DEBUG_CONFLICTS
static void read_running_processes();
static void update_running_processes();
	
static struct running processes[MAX_PROCESSES];
static int    num_running_processes;
static int    max_running_processes;
static int    old_max_running_processes;
static struct arglist * non_simult_ports_list;
static void * old_signal_handler;
 

int 
wait_for_children()
{
 int i, ret;
 for(i=0,ret=1;(i < 100) && (ret > 0);i++)
 	ret = waitpid(0, NULL, WNOHANG);
 return 0;
}
/*
 * Signal management
 */

void
process_mgr_sighand_term(sig)
 int sig;
{
 int i;
 for(i=0;i<MAX_PROCESSES;i++)
 {
  if(processes[i].pid)
        {
  	TERMINATE_THREAD(processes[i].pid);
	num_running_processes--;
	bzero(&(processes[i]), sizeof(struct running));
	}
 }
 EXIT(0);
}

 	
static void
update_running_processes()
{
 int i;
 struct timeval now, tv = {0,30000};
 fd_set rd;
 
 gettimeofday(&now, NULL);
 
 if(!num_running_processes)
  return;
  
 for(i=0;i<MAX_PROCESSES;i++)
 {
  if(processes[i].pid)
  {
   int alive;
  if( (! (alive = process_alive(processes[i].pid))) ||
  	(processes[i].timeout > 0 &&
        ((now.tv_sec - processes[i].start.tv_sec) > processes[i].timeout)))
  {
   if(alive){
   	char *  message;
   	log_write("%s (pid %d) is slow to finish - killing it\n", 
   			processes[i].name, 
			processes[i].pid);
	TERMINATE_THREAD(processes[i].pid);
	 message = emalloc(strlen(processes[i].name) + 255);
	 sprintf(message, "The plugin %s was too slow to finish - the server killed it\n", processes[i].name);
	 proto_post_note(arg_get_value(processes[i].plugin,
	 				"plugin_args"),
			  -1,
			  "tcp",
			  message);
	efree(&message);		     
	}
   else {
     struct timeval old_now = now;
     if(now.tv_usec < processes[i].start.tv_usec)
     {
      processes[i].start.tv_sec ++;
      now.tv_usec += 1000000;
     }
     log_write("%s (process %d) finished its job in %ld.%.2ld seconds\n", 
     			processes[i].name,
	 		processes[i].pid,
	 		now.tv_sec - processes[i].start.tv_sec,
			(now.tv_usec - processes[i].start.tv_usec) / 10000);
     now = old_now;			
     }
   num_running_processes--;
   /*
    * Read remaining data in the buffer
    */
   FD_ZERO(&rd);
   FD_SET(processes[i].soc[0], &rd);
   if(select(processes[i].soc[0]+1, &rd, NULL, NULL, &tv) > 0)
     piic_read_socket(processes[i].globals, processes[i].kb, processes[i].soc[0]);
   close(processes[i].soc[0]);
   close(processes[i].soc[1]);
   bzero(&(processes[i]), sizeof(processes[i]));
   }
  }
 }
}

static int
next_free_process(upcoming)
 struct arglist * upcoming;
{
 int r;
       	
 wait_for_children();
 for(r=0;r<MAX_PROCESSES;r++)
 {
  if(processes[r].pid)
  { 
   struct arglist * common_ports;
   if((common_ports = requirements_common_ports(processes[r].plugin, upcoming)))
   {
    int do_wait = -1;
    if(common(common_ports, non_simult_ports_list))
     do_wait = r;
    arg_free(common_ports);
    if(do_wait >= 0)
     {
#ifdef DEBUG_CONFLICT
      printf("Waiting has been initiated...\n");
#endif      
       log_write("Ports in common - waiting...\n");
      while(process_alive(processes[r].pid))
      	{
	read_running_processes();
	update_running_processes();
	wait_for_children();
	}
#ifdef DEBUG_CONFLICT      
      printf("End of the wait - was that long ?\n");
#endif      
    }
   }
  }
 }
 r = 0;
 while((r < MAX_PROCESSES) &&
       (processes[r].pid))r++;
       
 
 if(r >= MAX_PROCESSES)
  return -1;
 else
  return r;
}

#ifdef DEBUG
# ifndef PRINT_ALIVE_TIME
#  define PRINT_ALIVE_TIME	30
# endif
#endif

static void
read_running_processes()
{
 int i;
 fd_set rd;
 struct timeval tv = {0, 30000};
 int max = 0;
 int	dead_proc_n = 0;
#ifdef PRINT_ALIVE_TIME
 static time_t	last_time = 0;
 time_t		t;
 int		flag = 0;
#endif

 if(!num_running_processes)
  return;
 
#ifdef VERBOSE_LOGGING
 log_write("Waiting for spawned processes (%d)\n", num_running_processes); 
#endif 

#ifdef PRINT_ALIVE_TIME
 t = time(NULL);
 if (t - last_time >= PRINT_ALIVE_TIME)
   {
     flag = 1;
     last_time = t;
   }
#endif

 FD_ZERO(&rd);
 for(i=0;i<MAX_PROCESSES;i++)
 {
  if(processes[i].pid)
  {
    errno = 0;
    if (kill(processes[i].pid, 0) && errno == ESRCH)
      {
#ifdef DEBUG
	fprintf(stderr, "%s (process %d) is dead\n",
		processes[i].name, processes[i].pid);
#endif
	if (++ dead_proc_n  >= num_running_processes)
	  return;
      }
    else
      {
#ifdef PRINT_ALIVE_TIME
	if (flag)
	  {
	    last_time = t;
	    fprintf(stderr, "%s (process %d) is still alive\n",
		    processes[i].name, processes[i].pid);
	  }
#endif
	if(processes[i].soc[0] > max)
	  max = processes[i].soc[0];
	FD_SET(processes[i].soc[0], &rd);
      }
  }
 }

again:
 if(select(max + 1, &rd, NULL, NULL, &tv)  >= 0)
 {
  for(i=0;i<MAX_PROCESSES;i++)
  {
   if((processes[i].pid) && FD_ISSET(processes[i].soc[0], &rd))
   {
    piic_read_socket(processes[i].globals, processes[i].kb, processes[i].soc[0]);
   }
  }
 }
 else 
  {
   if(errno == EINTR)
    {
    tv.tv_sec = 0;
    tv.tv_usec = 30000; 
    goto again;
    }
   else
    perror("select() ");
  }
}


void
pluginlaunch_init(globals)
 struct arglist * globals;
{
 struct arglist * preferences = arg_get_value(globals, "preferences");
 non_simult_ports_list = arg_get_value(preferences, "non_simult_ports_list");
 max_running_processes = get_max_checks_number(preferences);
 old_max_running_processes = max_running_processes;
 
 if(max_running_processes > MAX_PROCESSES)
 {
  log_write("max_checks (%d) > MAX_PROCESSES (%d) - modify nessus-core/nessusd/pluginlaunch.c\n",
  			max_running_processes,
			MAX_PROCESSES);
   max_running_processes = MAX_PROCESSES;
 }
 
 log_write("user %s : will launch %d plugins at the same time against each host\n",
 		(char*)arg_get_value(globals, "user"),
		max_running_processes);
		
 num_running_processes = 0;
 bzero(&(processes), sizeof(processes));
 old_signal_handler = signal(SIGTERM, process_mgr_sighand_term);
}

void
pluginlaunch_disable_parrallel_checks()
{
  max_running_processes = 1;
}

void
pluginlaunch_enable_parrallel_checks()
{
 max_running_processes = old_max_running_processes;
}


void
pluginlaunch_stop()
{
 int i;
 read_running_processes();
 for(i=0;i<MAX_PROCESSES;i++)
 {
  if(processes[i].pid)
  	 {
 	 TERMINATE_THREAD(processes[i].pid);
	 num_running_processes--;
	 bzero(&(processes[i]), sizeof(struct running));
	 }
 }
 signal(SIGTERM, old_signal_handler);
}


int
plugin_launch(globals, plugin, hostinfos, preferences, key, name, launcher)
	struct arglist * globals;
	struct arglist * plugin;
	struct arglist * hostinfos;
	struct arglist * preferences;
	struct arglist * key;
	char * name;
	pl_class_t * launcher;
{ 
 int p;
 struct arglist * args = arg_get_value(plugin->value, "plugin_args");
  
 /*
  * Wait for a free slot and read input
  */ 
 do
 {
  read_running_processes();
  update_running_processes();
 }
 while (num_running_processes >= max_running_processes);
 
 p = next_free_process(plugin->value);
 processes[p].kb = key;
 processes[p].globals = globals;
 processes[p].plugin  = plugin->value;
 processes[p].name    = plugin->name;
 processes[p].timeout =
 preferences_plugin_timeout(preferences,(int)arg_get_value(args, "ID"));
 if(!(processes[p].timeout))
   processes[p].timeout = (int)arg_get_value(args, "TIMEOUT");
 
 
 
 if(!(processes[p].timeout))
 {
  int category = (int)arg_get_value(args, "CATEGORY");
  if(category == ACT_SCANNER)processes[p].timeout = -1;
  else processes[p].timeout = preferences_plugins_timeout(preferences);
 }
 

 socketpair(AF_UNIX, SOCK_STREAM, 0, processes[p].soc);
 gettimeofday(&(processes[p].start), NULL);

#if 0
 fprintf(stderr, "About to launch plugin: name=%s PPID=%d PID=%d\n",
	 name, getppid(), getpid());
 fflush(stderr); /* Just in case... */
#endif

 processes[p].pid = 
 	(*launcher->pl_launch)(globals,
 			        plugin->value,
				hostinfos,
				preferences,
				key,
				name,
				processes[p].soc[1]);
 num_running_processes++;
 return processes[p].pid;
}


void 
pluginlaunch_wait()
{
 do
 {
  wait_for_children();
  read_running_processes();
  update_running_processes();
 }
 while (num_running_processes);
}
