/* crypto-layer.c
 * Copyright (c) mjh-EDV Beratung, 1996-1999
 * 
 * Nessus
 * Copyright (C) 1998-2001 Renaud Deraison
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2,
 * as published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Jordan Hrycaj <jordan@mjh.teddy-net.com>
 *
 * Some stuff extracted from nessusd.c
 *
 * $Id: crypto-layer.c,v 1.35 2002/05/24 10:37:05 renaud Exp $
 */

#include <includes.h>

#ifdef ENABLE_CRYPTO_LAYER
#include <stdarg.h>
#include <pwd.h>
#include "nessus/libnessus.h"
#include "crypto-layer.h"
#include "log.h"
#include <sys/types.h>
#include "sighand.h"

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#ifdef HAVE_SELECT
 /* stdout, implements sleep () using select () */
# define FILE_LOCK_SELECT_FD 1
#else
# ifdef HAVE_POLL
   /* stdout, implements sleep () using poll () */
#  define FILE_LOCK_POLL_FD  1
#  ifdef HAVE_SYS_POLL_H
#   include <sys/poll.h>
#  endif
#  ifdef HAVE_POLL_H
#   include <poll.h>
#  endif
# endif /* HAVE_POLL */
#endif /* HAVE_SELECT */

#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#elif defined (HAVE_STAT_H)
# include <stat.h>
#else
  extern int stat ();
#endif
#ifdef HAVE__STAT
typedef struct _stat struct_stat ;
# ifndef S_ISDIR
# define S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
# endif
# define lstat(x,y) _stat(x,y)
#else
typedef struct stat struct_stat ;
# ifndef HAVE_LSTAT
#  define lstat(x,y) stat(x,y)
# endif
#endif

#ifdef HAVE__ALLOCA
#define alloca _alloca
#define HAVE_ALLOCA
#endif

/* ------------------------------------------------------------------------- *
 *                      private definitions                                  *
 * ------------------------------------------------------------------------- */

/* XMALLOC returns memory initialized to zero */
#define XMALLOC(x)  emalloc(x)
#define XFREE(x)    efree(&(x))

#define NEED_XREALLOC /* locally installed, no mem zero needed */
#define XREALLOC(x,n) xrealloc(x,n)

/* local on-the-stack memory allocation */
#ifdef HAVE_ALLOCA
# define XALLOCA(x)  alloca(x)
# define XDEALLOC(x) /* nothing */
#else
# define XALLOCA(x)  XMALLOC(x)
# define XDEALLOC(x) XFREE(x)
#endif

/* ---------------------------------------------------------------------- *
 *                  private data and structures                           *
 * ---------------------------------------------------------------------- */

static char *setpwd;

/* ---------------------------------------------------------------------- *
 *                     private functions                                  *
 * ---------------------------------------------------------------------- */

static 
char* 
get_pwd 
 (int mode) {
  char *s, *t ;

  switch (mode) {
  case 0:
    /* key activation mode */
    return getpass ("Pass phrase: "); 
  case 2:
    /* created new key mode defaults to no password */
    return "" ;
  }

  /* change pass phrase mode */
  if ((s = getpass ("New pass phrase: ")) == 0)
    return (char*)-1;
  s = estrdup (s);
  if ((t = getpass ("Repeat         : ")) == 0) {
    efree (&s);
    return (char*)-1;
  }
  if (strcmp (s, t) != 0) 
    t = 0 ;
  efree (&s);
  if (t == 0)
    return (char*)-1;
  return t ;
}

static char *
pass_dialog (int s)
{
  return setpwd;
}


static const char *
harg_s2b
  (harglst       *list,
   const char *varname,
   const char  *defval)
{
  char *s ;
  harg_add_default_string (list, varname, defval);
  s    =  harg_get_string (list, varname);
  harg_type_set_blob      (list, varname);
  return s;
}


static int
harg_s2i
  (harglst        *list,
   const char  *varname,
   int           defval)
{
  char *s = harg_get_string (list, varname) ;
  int n ;
  
  if (s == 0) {
    harg_add_int (list, varname, defval);
    return defval;
  }
  harg_add_int (list, varname, n = atoi (s));
  return n;
}


/* sleep some milli secs */
static void
sleep_ms 
  (unsigned ms) 
{
  if (ms < 10)
    ms = 10 ;
  {
#ifdef FILE_LOCK_SELECT_FD
    struct timeval tv;
    tv.tv_sec  =  0;
    tv.tv_usec = ms;
    select (FILE_LOCK_SELECT_FD, 0, 0, 0, &tv);
#else
# ifdef FILE_LOCK_POLL_FD
    struct pollfd pfd;
    memset (&pfd, 0, sizeof (pfd)) ;
    pfd.fd = FILE_LOCK_POLL_FD;
    poll (&pfd, 1, ms) ;
# else
    sleep (1);
# endif /* FILE_LOCK_POLL_FD */
#endif /* FILE_LOCK_SELECT_FD */
  }
}


static int
fake_home_env
  (const char *pfx,
   const char *usr)
{
  struct_stat stbuf ;
  char buf [FILENAME_MAX + 10];

  /* check string lengths */
  if (strlen (pfx) + strlen (usr) + 1 > FILENAME_MAX) {
    print_error ("%s/%s too long - aborting.\n", pfx, usr);
    DO_EXIT (1);
  }
  /* assemble the user profile path */
  strcpy (buf, pfx);
  strcat (buf, "/");
  strcat (buf, usr);

  /* we want a subdirectory: sizeof() == strlen() * 1 */
  if (lstat (buf, &stbuf))
    return -1;
# ifdef HAVE_LSTAT
  /* check for regular file or symlink */
  if (S_ISLNK (stbuf.st_mode)) {
    errno = ENOTDIR ;
    return -1;
  }
# endif
  /* only ckeck directories */
  if (S_ISDIR(stbuf.st_mode) == 0) {
    errno = ENOTDIR ;
    return -1;
  }
  peks_set_homedir (buf);
}


static void
list_users
  (harglst *prefs)
{
  DIR *D ;
  struct dirent *d;
  int per_user_key_list, per_user_pwd_list, did_it = 0;
  char *pfx ;

  char *pf  = harg_get_blob   (prefs, "peks_usrpwds");
  char *kf  = harg_get_blob   (prefs, "peks_usrkeys");

  per_user_key_list = (kf [0] == '~' && kf [1] == '/');
  if (pf != 0 && strcmp (pf, kf) == 0)
    pf = 0 ;
  per_user_pwd_list = (pf == 0 
		       ? per_user_key_list 
		       : pf [0] == '~' && pf [1] == '/');

  if (per_user_key_list == 0) {
    if (peks_list_keyfile (0 /* prints to stdout */, kf) < 0) 
      goto error;
    did_it ++ ;
  }
  if (per_user_pwd_list == 0) {
    if (peks_list_keyfile (0 /* prints to stdout */, pf) < 0) 
      goto error;
    did_it ++ ;
  }
  if (did_it > 1)
    return;
  
  if ((pfx = harg_get_value (prefs, "per_user_base")) == 0)
    pfx = NESSUSD_LOGINS;

  if ((D = opendir (pfx)) == 0) {
    print_error ("%s: %s - aborting.\n", pfx, peks_strerr (errno));
    DO_EXIT (1);
  }
  if ((d = readdir (D)) == 0 || strcmp (d->d_name,  ".") ||
      (d = readdir (D)) == 0 || strcmp (d->d_name, "..")) {
    print_error 
      ("Missing . and .. while searcing %s - aborting.\n", pfx);
    DO_EXIT (1);
  }
  /* read subdirectories */
  while ((d = readdir (D)) != 0) {
    if (fake_home_env (pfx, d->d_name) < 0)
      continue ;
    if (per_user_key_list && 
	peks_list_keyfile (0 /* prints to stdout */, kf) < 0) 
      goto error;
    if (per_user_pwd_list &&
	peks_list_keyfile (0 /* prints to stdout */, pf) < 0) 
      goto error;
  }
  peks_set_homedir (0);
  closedir (D);
  return;

 error:
  print_error("ERROR: %s.\n", peks_strerr (errno));
  DO_EXIT (1);
}


static void
set_user_home_env 
   (harglst  *prefs,
   const char *user)
{
  char *p, *u = XALLOCA (strlen (user)) ;
  char *pfx = harg_get_value (prefs, "per_user_base");

  strcpy (u, user);
  if ((p = strchr (u, '@')) != 0)
    p [0] = '\0';

  /* default if the prefix is missing */
  if (pfx == 0)
    pfx = NESSUSD_LOGINS;
  fake_home_env (pfx, u);
  XDEALLOC (u);
}


/* ---------------------------------------------------------------------- *
 *                    public functions: standard init                     *
 * ---------------------------------------------------------------------- */

void
cipher_stuff_init
  (harglst *prefs)
{
  peks_key *key;
  const char *user, *file, *devrnd, *ukys ;
  int klen ;

  /* suppress path checking */
  int saccess = harg_get_int (prefs, "suppr_acc");
  int do_hint = harg_get_int (prefs, "acc_hint");

  /* overwrite defaults by config */
  klen = harg_s2i (prefs, "peks_keylen",   NESSUSD_KEYLENGTH);
         harg_s2i (prefs, "peks_pwdfail",  NESSUSD_MAXPWDFAIL);
         harg_s2i (prefs, "negot_timeout", NESSUSD_NEGOT_TIMEOUT) ;

  /* Hide strings as pointers and assign default. This former action
     is nesseary as most strings are exported to the nessus client,
     by  default. */

  ukys = harg_s2b (prefs, "peks_usrkeys",  NESSUSD_USERKEYS);  
         harg_s2b (prefs, "peks_usrpwds",  NESSUSD_USERPWDS);
  file = harg_s2b (prefs, "peks_keyfile",  NESSUSD_KEYFILE);
  user = harg_s2b (prefs, "peks_username", NESSUSD_USERNAME);

  if ((devrnd = harg_get_string (prefs, "random_device")) != 0) {
    if (strcasecmp (devrnd, "none") == 0) devrnd = 0 ;
    use_random_device (devrnd) ;
  }

  if ((key = peks_private_key (user, file, get_pwd, klen)) == 0) {
    int prterr = errno ;
    if (saccess) {
      peks_admin_equiv (-1,-1);
      key = peks_private_key (user, file, get_pwd, klen);
    }
    if (key == 0) {
      print_error ("%s - aborting.\n", peks_strerr (errno));
      if (saccess == 0)
	print_error ("\n *** If this is unavoidable, you may try "
		     "restarting with the "
		     "--suppress-cfg-check option!\n\n");
      DO_EXIT (1);
    }
    if (do_hint)
      print_error ("WARNING: %s.\n"
		   "     ... retrying without path check "
		   "as requested!\n\n", peks_strerr (prterr));
  }

  /* store key for global use */
  harg_add_ptr (prefs, "peks_key", key);
}


 
void
do_local_cipher_commands 
  (harglst *prefs)
{
  char *pwd_user =     harg_get_blob (prefs, "pwd_user");
  char *list_upwd =    harg_get_blob (prefs, "list_user_pwd");
  char *pwd_passwd   = harg_get_blob (prefs, "pwd_passwd");
  char *key_expfile  = harg_get_blob (prefs, "key_expfile");
  char *kill_userkey = harg_get_blob (prefs, "kill_userkey");
  int chng_pph =       harg_get_int  (prefs, "chng_pph");
  int list_udb =       harg_get_int  (prefs, "list_udb");
  int saccess  =       harg_get_int  (prefs, "suppr_acc");
  int do_hint  =       harg_get_int  (prefs, "acc_hint");
  peks_key *key      = harg_get_ptr  (prefs, "peks_key");
  char * s;

  /* no connection, but user passwd table etc. request */
  int did_something = 0 ;
  
  /* password table entry list request */
  if (list_upwd != 0) {
    set_user_home_env (prefs, list_upwd);
    if ((s = peks_edit_passwd_file
	 (key, list_upwd, -1,0, harg_get_blob (prefs, "peks_usrpwds"))) == 0) {
      print_error("ERROR: %s.\n", peks_strerr (errno));
      DO_EXIT (1);
    }
    peks_set_homedir (0);
    printf ("passwd specs for %s: %s\n", list_upwd, s);
    did_something ++ ;
  }

  /* password table edit request */
  if (pwd_user != 0) {
    set_user_home_env (prefs, pwd_user);
    /* delete entry ? */
    if (pwd_passwd != 0 && pwd_passwd [0] == 0) {
      pwd_passwd = 0; 
      key = 0;
    }
    if ((s = peks_edit_passwd_file
	 (key, pwd_user, 0, pwd_passwd,
	  harg_get_blob (prefs, "peks_usrpwds"))) == 0) {
      print_error("ERROR: %s.\n", peks_strerr (errno));
      DO_EXIT (1);
    }
    peks_set_homedir (0);

    if (s [0])
      printf ("%s\n", s) ;
    did_something ++ ;
  }
 
  /* change pass phrase */
  if (chng_pph != 0) {
    if (peks_save_private_key
	(harg_get_blob (prefs, "peks_username"), key,
	 harg_get_blob (prefs, "peks_keyfile"), get_pwd) < 0) {
      print_error("ERROR: %s.\n", peks_strerr (errno));
      DO_EXIT (1);
    }
    did_something ++ ;
  }

  /* user key table delete request */
  if (kill_userkey != 0) {
    set_user_home_env (prefs, kill_userkey);
    if (peks_delete_userkey 
	(kill_userkey, harg_get_blob (prefs, "peks_usrkeys"), 0) < 0) {
      print_error("ERROR: %s.\n", peks_strerr (errno));
      DO_EXIT (1);
    }
    peks_set_homedir (0);
    did_something ++ ;
  }
 
  /* user key and passwd table list request */
  if (list_udb != 0) {
    list_users (prefs);
    did_something ++ ;
  }

  /* export public key */
  if (key_expfile != 0) {
    if (peks_save_public_key ("@127.0.0.1", key, key_expfile) < 0) {
      int prterr = errno, n = -1 ;
      if (saccess) {
	peks_admin_equiv (-1,-1);
	n = peks_save_public_key ("@127.0.0.1", key, key_expfile);
      }
      if (n < 0) {
	print_error("ERROR: %s.\n", peks_strerr (errno));
	if (saccess == 0)
	  print_error ("\n *** If this is unavoidable, you may try "
		       "restarting with the "
		       "--suppress-cfg-check option!\n\n");
	DO_EXIT (1);
      }
      if (do_hint)
	print_error ("WARNING: %s.\n"
		     "     ... retrying without path check "
		     "as requested!\n\n", peks_strerr (prterr));
    }
    did_something ++ ;
  }
 
  if (did_something)
    DO_EXIT (0) ;
}

static int
verify_io_process (unused, pid)
     void         *unused;
     unsigned long    pid;
{
  let_em_die (0);

  if (kill (pid, 0) >= 0)
    return 1;

  /* let that process be killed */
  log_write ("Process %lu has died without closing IO threads.\n", pid);
  return 0 ;
}

/* ---------------------------------------------------------------------- *
 *                    public functions: process tracking                  *
 * ---------------------------------------------------------------------- */

void
init_process_tracker (globs) 
     struct arglist * globs;
{
  struct arglist *prefs = arg_get_value (globs, "preferences") ;
  char               *s = arg_get_value (prefs, "track_iothreads");
  
  if (s == 0) {
    arg_add_value (prefs, "track_iothreads", ARG_STRING, 2, "no") ;
    return ;
  }

  /* no process tracker with pthreads */
# ifndef USE_PTHREADS
  if (strcmp (s, "yes") != 0)  {
# endif
    efree (&s);
    arg_set_value (prefs, "track_iothreads", 2, "no") ;
# ifndef USE_PTHREADS
    return ;
  }

  /* in case you want to track io threads */
  if (s == 0 || strcmp (s, "yes") != 0)
    return;

  if (getuid () != 0) {
    log_write ("Only root may track thread ids - disabled.\n");
    return;
  }
  
  /* install trap */
  log_write ("Installing IO thread id tracker.\n");
  io_thtrp ((int)arg_get_value (globs, "global_socket"), 
	    verify_io_process, globs /*for debugging*/, 1);
#endif
}


#endif /* ENABLE_CRYPTO_LAYER */
