#!/usr/bin/perl -wT

use strict;
use LWP::UserAgent;
use HTTP::Request;
use HTTP::Response;
use Getopt::Long;

my %roots;
@roots{"/cgi-bin/","/htbin/","/cgibin/","/cgis/","/cgi/","/htbins/","/cgi-local/","/scripts/"} = ();
my %found_roots;

# verbosity level
my $verbosity = 0;
my $timeout = 30;
my $severity;
my $quiet = 0;
my $help = 0;
my $sansmode = 0;
my $found_one = 0;
my $version = "1.0";

#
# Given a host, root, path, and method, submit an http request
# and return the response code.
#
my %resp_cache;
sub get_response_code {
    my ($host, $root, $path, $method) = @_;

    my $url = "http://" . $host . $root . $path;
    printf "request is %s of %s\n", $method, $url unless $verbosity < 2;

    my $cache_entry = $method . ":" . $url;
    if (!exists $resp_cache{$cache_entry}) {
        my $ua = LWP::UserAgent->new ();
        $ua->timeout($timeout);
        my $req = HTTP::Request->new ($method => $url);
        my $resp = $ua->request ($req);
        # hack to handle cold fusion, which may return 200 for everything.
        if (($resp->code () eq "200")
            && ($path =~ /\.cfm$/)
            && ($resp->content =~ /<\/TD><\/TD><\/TD><\/TH><\/TH><\/TH>/)
            && ($resp->content =~ /HTTP\/1.0 404 Object Not Found/)) {
            $resp_cache{$cache_entry} = "404";
        } else {
            $resp_cache{$cache_entry} = $resp->code ();
        }
        printf "  added to cache: %s\n", $resp_cache{$cache_entry} unless $verbosity < 2;
    } else {
        printf "  found in cache: %s\n", $resp_cache{$cache_entry} unless $verbosity < 2;
    }
    return $resp_cache{$cache_entry};
}



#
# Given a host, root, path, and method, submit an http request
# and return the response code.
#
sub get_server_info {
    my ($host) = @_;
    my $method = "HEAD";

    my $url = "http://" . $host . "/";
    printf "request is %s of %s\n", $method, $url unless $verbosity < 2;

    my $ua = LWP::UserAgent->new ();
    $ua->timeout($timeout);
    my $req = HTTP::Request->new ($method => $url);
    my $resp = $ua->request ($req);
    my $srv_type;
    printf "Server: %s\n", $resp->headers->header("Server") unless $verbosity < 1;
    if ($resp->code () eq "200") {
        if ($resp->headers->header("Server") =~ /Microsoft|Windows/) {
            $srv_type = "Windows";
        } elsif ($resp->headers->header("Server") =~ /NetWare|Novell/) {
            $srv_type = "Novell";
        } elsif ($resp->headers->header("Server") =~ /Unix/) {
            $srv_type = "Unix";
        } else {
            $srv_type = "Other";
        }
    } else {
        $srv_type = "Unknown Error";
    }
    return ($resp->code (), $srv_type, $resp->status_line);
}

my @checks;
#
# Read the database of cgi checks in
# and store it in the %checks hash
#
sub read_db {
    open (DB, "< cgi.db") or die "Unable to open cgi database";
    $/ = '';
    while (<DB>) {
        my @fields = split /^([^:]+):\s*/m;
        shift @fields;
        push @checks, { map /(.*)/, @fields };
    }
    close (DB);
}

#
# print out information on a vulnerability
#
sub report_vuln {
    my ($host, $root, $record) = @_;

    $found_one = 1;
    unless (!$sansmode) {
        printf "  " . $record->{TITLE};
        printf "\n    (See Desc/" . $record->{DESCRIPTION} . " for details)\n";
        return;
    }
    printf "\n\n" . $record->{SEVERITY} . " vulnerability, " .  $record->{TITLE} . "\n";
    printf "Query used: http://" . $host . $root . $record->{"REAL-QUERY"} . "\n" unless $verbosity < 1;
    unless (open (DESC, "< Desc/" . $record->{DESCRIPTION})) {
        printf "no description found\n";
        return;
    }
    while (<DESC>) {
        print;
    }
    close (DESC);
}


#
# Given a host and a check record, do the check
#
sub check_one {
    my ($host, $record) = @_;
    my @check_roots;
    my $root;

    if ($record->{"QUERY-TYPE"} eq "") {
        #printf "!!empty QUERY-TYPE\n";
        $record->{"QUERY-TYPE"} = "GET";
    }

    if (($record->{ROOT} eq "")
        || (exists $found_roots{$record->{ROOT}})
        || (($record->{ROOT} ne "") && (!exists $roots{$record->{ROOT}}))) {

        if ($record->{ROOT} ne "") {
            @check_roots = ( $record->{ROOT} );
        } else {
            @check_roots = ( keys %found_roots );
        }

        foreach $root (@check_roots) {
            my $real_one = get_response_code $host, $root, $record->{"REAL-QUERY"}, $record->{"QUERY-TYPE"};
            if ($real_one !~ "^40") {
                my $known_bad = get_response_code $host, $root, $record->{"TEST-QUERY"}, $record->{"QUERY-TYPE"};
                if ($known_bad != $real_one) {
                    report_vuln $host, $root, $record;
                }
                return;
            }
        }
    }
}

#
# Given a host, determine which of the root directories are present 
#
sub check_roots {
    my ($host) = @_;
    my $dir;

    %found_roots = ();
    foreach $dir (keys %roots) {
        my $resp_code = get_response_code $host, $dir, "", "GET";
        if (($resp_code == "200") || ($resp_code == "403")) {
            printf "Adding %s to CGI directories\n", $dir unless $verbosity < 1;
            $found_roots{$dir} = 1;
        }
        if ($resp_code == "200") {
            printf "The %s directory may be indexable\n", $dir;
        }
    }
}


#
# Given a host, check it
#
sub check_host {
    my ($host) = @_;
    my ($srv_code, $srv_type, $status_line) = get_server_info $host;
    my $record;

    printf "status_line: %s\n", $status_line unless $verbosity < 2;
    if ($status_line =~ /Can\'t connect to.*\((.*)\)/) {
        printf "Error connecting to %s: %s\n", $host, $1 unless $sansmode;
        return;
    }
    printf "Detected %s server\n", $srv_type unless $verbosity < 1;

    check_roots $host;

    foreach $record (@checks) {
        if (($record->{PLATFORM} eq "All")
            || ($srv_type eq "Other")
            || ($srv_type eq $record->{PLATFORM})) {
            if ((!defined $severity)
                || ($record->{SEVERITY} =~ /$severity/i)) {
                check_one $host, $record;
            } else {
                printf "Skipping %s.  Vuln is %s, need %s\n", $record->{TITLE}, $record->{SEVERITY}, $severity unless $verbosity < 2;
            }
        } else {
            printf "Skipping %s.  Server is %s, need %s.\n", $record->{TITLE}, $srv_type, $record->{PLATFORM} unless $verbosity < 2;
        }
    }
}


sub usage {
    printf "Usage: ./cgi.pl [options] hostname\n";
    printf "options are -v verbose  (use more for more verbosity\n";
    printf "            -t timeout  (timeout (in secs) for connections, defaults to 30)\n";
    printf "            -h          (help)\n";
    printf "            -q          (quiet, suppress version info)\n";
    printf "            -d level    (detect checks High, Medium, Low)\n";
    exit (0);
}


#
# main
#

#parse arguments
GetOptions ("v+" => \$verbosity,
            "t=i" => \$timeout,
            "d=s" => \$severity,
            "q"   => \$quiet,
            "s"   => \$sansmode,
            "h"   => \$help
            );
if ($help) { usage (); }

#if (!$verbosity) { $verbosity = 0; }
#if (!$quiet) { $quiet = 0; }

if ($sansmode) { $quiet = 1; $verbosity = 0; }

printf "RAZOR CGI scanner  - version: %s\n", $version unless $quiet;
printf "#2 and #4 - Vulnerable CGI, app extensions, RDS hole\n" unless (!$sansmode);
if (defined $severity) {
    printf "severity = %s\n", $severity unless $sansmode;
}

my $host = shift    or usage ();
printf "checking %s\n", $host unless $sansmode;

read_db ();

check_host $host;

if ($sansmode)
{
  print "\n  Possible web-related vulnerabilities found. Refer to the documents",
      "\n  listed above, as well as Docs/CGI.html.\n\n" unless !($found_one);
  print "\n  No problems related to #2 and #4\n\n" unless ($found_one);
}
#
# cgi.pl  - cgi scanner for use with specs/meta/cgi-scanner.db
#
# $Id: cgi.pl,v 1.1 2000/11/06 15:32:05 loveless Exp $
# $Log: cgi.pl,v $
# Revision 1.1  2000/11/06 15:32:05  loveless
# Moved to sectools section of CVS
#
# Revision 1.8  2000/10/12 16:54:12  loveless
# Added "cgi" to the list of places to look for CGI programs.
#
# Revision 1.7  2000/07/24 17:14:42  loveless
# Updated Docs references to reflect changes to filenames
#
# Revision 1.6  2000/07/21 22:10:30  loveless
# More cleanup, added proper references to Docs files
#
# Revision 1.5  2000/07/20 20:40:22  loveless
# Made output consistent with other modules.
#
# Revision 1.4  2000/07/19 16:38:39  tsabin
# added Leon's suggestion about only doing the test query if the
# real query succeeds.  should optimize out many requests.
#
# Revision 1.3  2000/07/18 14:39:47  tsabin
# updated cgi.db DESCRIPTION fields not to include Desc/ (added it to cgi.pl)
#    (done to ease using in Hackershield)
# fixed some typos in cgi.db
# added cgi-verify script for doing validity check on cgi.db
# added desc2html.pl script for combining/converting description files to
#    one html file
# added some leading whitespace in some description files so they convert
#    to html better
#
# Revision 1.2  2000/06/29 21:43:42  loveless
# Fixed false positive bug in 40? return codes, added a line to show exact
# request used when using verbose mode.
#
# Revision 1.1  2000/06/28 19:20:03  loveless
# Adding to new archive
#
# Revision 1.11  2000/06/28 17:20:56  tsabin
# added standard options, etc.
#
# Revision 1.10  2000/06/23 23:27:31  tsabin
# put main in its own block
# made the verbosity stuff more readable
# added printing if cgi dirs may be indexable
#
# Revision 1.9  2000/06/23 22:53:45  tsabin
# tried to fix the Id: problem
#
# Revision 1.8  2000/06/23 22:51:24  tsabin
# added --severity option
#
# Revision 1.7  2000/06/23 21:54:35  tsabin
# added printing of status_line from initial host test
#
# Revision 1.6  2000/06/23 21:35:57  tsabin
# detect non-existant hosts
#
# Revision 1.5  2000/06/23 20:56:06  tsabin
# added use strict;
# added -v option, fixed Usage msg
#
# Revision 1.4  2000/06/22 14:03:04  tsabin
# added code to detect false return codes from code fusion
# added severity to report
#
# Revision 1.3  2000/06/19 22:15:32  tsabin
# changed srv_type Microsoft to Windows as in the db
#
# Revision 1.2  2000/06/19 21:47:19  tsabin
# set verbosity to 0 by default
#
#
