/* Linux Prism II Stumbler - Utility Scan for 802_11 networks under Linux
 * 
 * File : analyse.c
 * Project : WifiScanner (c) 2002 Herv Schauer Consultants
 * Usage : This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA).
 * 
 * Base code was from prismstumbler Jan Fernquist <Jan.B.Fernquist@telia.com>
 * and wlanctl from www.linux-wlan.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: analyse.c,v 1.22 2003/07/25 11:23:08 poggij Exp $
 */

#include <include.h>
#include <src/analyse.h>
#include <src/interface.h>
#include <src/crt_io.h>
#include <src/conversion.h>

// Private declaration
void LogPutSSID_byAddMAc (UINT8 AddMac[WLAN_ADDR_LEN], char *SSID);
void LogPutSSID_byBSSID (UINT8 BSSID[WLAN_ADDR_LEN], char *SSID);
void CalculCoef (UINT32 X[HISTORY_SIZE], UINT32 Y[HISTORY_SIZE], float *a,
		 float *b);

static UINT8 ID[] = "$Id: analyse.c,v 1.22 2003/07/25 11:23:08 poggij Exp $";
static UINT16 NumberOfDetectedClient = 0;
static ClientInfo_t ClientInfo[MAX_NUMBER_OF_DETECTED_CLIENT];
const UINT8 BroadcastMAC[WLAN_ADDR_LEN] =
  { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

// External references
extern unsigned int DebugLevel;
extern WINDOW *Panel_WND, *RealTime_WND;
extern ScanResult_t Res;
extern Statistics_t Stats;
extern UINT8 ids_warning;
#ifdef WITH_THREAD
extern pthread_mutex_t screen_mutex;
#endif


// --------------------

/** Compare two MAC address */
UINT8
CompareAddMac (UINT8 AddMac1[WLAN_ADDR_LEN], UINT8 AddMac2[WLAN_ADDR_LEN])
{
/*  Very baaaaad code ...
  int res = 1;
  res = res & (AddMac1[0] == AddMac2[0]) & (AddMac1[1] == AddMac2[1]);
  res = res & (AddMac1[2] == AddMac2[2]) & (AddMac1[3] == AddMac2[3]);
  res = res & (AddMac1[4] == AddMac2[4]) & (AddMac1[5] == AddMac2[5]);
*/
#ifdef WORDS_BIGENDIAN
  return (((UINT64) AddMac1 & 0xFFFFFFFFFFFF0000) ==
	  ((UINT64) AddMac2 & 0xFFFFFFFFFFFF0000));
#else
  return (((UINT64) AddMac1[5] & 0x0000FFFFFFFFFFFF) ==
	  ((UINT64) AddMac2[5] & 0x0000FFFFFFFFFFFF));
#endif
}

void
LogPutSN (UINT8 AddMac[WLAN_ADDR_LEN], UINT32 SN)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  if (ClientInfo[i].SNMax < SN)
	    ClientInfo[i].SNMax = SN;
	  else if ((ClientInfo[i].SNMin > SN) || (ClientInfo[i].SNMin == 0))
	    ClientInfo[i].SNMin = SN;
	  break;
	}
    }
}

void
LogPutRate (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 Rate)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  if (ClientInfo[i].RateMax < Rate)
	    ClientInfo[i].RateMax = Rate;
	  else if ((ClientInfo[i].RateMin > Rate)
		   || (ClientInfo[i].RateMin == 0))
	    ClientInfo[i].RateMin = Rate;
	  break;
	}
    }
}

void
LogPutMaxRate (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 Rate)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].MaxSpeed = Rate;
	}
    }
}
void
LogDetectedClient (UINT8 AddMac[WLAN_ADDR_LEN])
{
  UINT16 i;
  UINT8 Founded = 0;

  if (NumberOfDetectedClient >= MAX_NUMBER_OF_DETECTED_CLIENT)
    return;

  if (CompareAddMac (BroadcastMAC, AddMac))
    Founded = 2;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  Founded = 1;
	  break;
	}
    }
  if (!Founded)
    {
      debug (1, "New client found (%02X:%02X:%02X:%02X:%02X:%02X)\n",
	     AddMac[0], AddMac[1], AddMac[2], AddMac[3], AddMac[4],
	     AddMac[5]);
      memset (&ClientInfo[NumberOfDetectedClient], 0, sizeof (ClientInfo_t));
      memcpy (&ClientInfo[NumberOfDetectedClient].AddMac, AddMac,
	      WLAN_ADDR_LEN);
      NumberOfDetectedClient++;
    }
}

void
LogPutBSSID (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 BSSID[WLAN_ADDR_LEN])
{
  UINT16 i;

  // ignore broadcast BUG #557306
  if (CompareAddMac (BSSID, BroadcastMAC) == 0)
    {
      for (i = 0; i < NumberOfDetectedClient; i++)
	{
	  if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	    {
	      memcpy (&ClientInfo[i].BSSID, BSSID, WLAN_ADDR_LEN);
	      break;
	    }
	}
    }
}

void
LogPutDS (UINT8 AddMac[WLAN_ADDR_LEN])
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].IsDS = 1;
	  break;
	}
    }
}

void
LogPutWep (UINT8 BSSID[WLAN_ADDR_LEN], UINT8 hasWep)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].BSSID, BSSID))
	{
	  ClientInfo[i].hasWep = hasWep;
	}
    }
}

void
LogPutChannel_byAddMac (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 Channel)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].Channel = Channel;
	  break;
	}
    }
}


void
LogPutChannel_byBSSID (UINT8 BSSID[WLAN_ADDR_LEN], UINT8 Channel)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].BSSID, BSSID))
	{
	  ClientInfo[i].Channel = Channel;
	}
    }
}

void
LogPutChannel (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 BSSID[WLAN_ADDR_LEN],
	       UINT8 Channel)
{
  if (CompareAddMac (BSSID, BroadcastMAC))
    LogPutChannel_byAddMac (AddMac, Channel);
  else
    LogPutChannel_byBSSID (BSSID, Channel);
}


void
LogPutSSID_byAddMac (UINT8 AddMac[WLAN_ADDR_LEN], char *SSID)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  //    ClientInfo[i].SSID ""           and SSID not " "
	  //if ((strlen (ClientInfo[i].SSID) == 0) && (strncmp (SSID, " ", 32))) {  // BUG #557123
	  strncpy (ClientInfo[i].SSID, SSID, WLAN_SSID_MAXLEN);
	  //}
	  break;
	}
    }
}


void
LogPutSSID_byBSSID (UINT8 BSSID[WLAN_ADDR_LEN], char *SSID)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].BSSID, BSSID))
	{
	  //    ClientInfo[i].SSID ""           and SSID not " "
	  //if ((strlen (ClientInfo[i].SSID) == 0) && (strncmp (SSID, " ", 32))) {  // BUG #557123
	  strncpy (ClientInfo[i].SSID, SSID, WLAN_SSID_MAXLEN);
	  //}
	}
    }
}

void
LogPutSSID (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 BSSID[WLAN_ADDR_LEN],
	    char *SSID)
{
  // We don't initialyse SSID if it's blank, BUG #557123
  //   SSID not ""         and  SSID not " "
  if ((strlen (SSID) != 0) && (strncmp (SSID, " ", 32)))
    {
      if (CompareAddMac (BSSID, BroadcastMAC))
	LogPutSSID_byAddMac (AddMac, SSID);
      else
	LogPutSSID_byBSSID (BSSID, SSID);
    }
}

void
LogPutIsAP (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 hasWep, UINT16 bcn_int)
{
  UINT16 i, j;
  UINT8 BSSID[WLAN_ADDR_LEN];

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].hasWep = hasWep;
	  // A little rotation of Beacon Interval
	  for (j = (HISTORY_SIZE - 1); j != 0; j--)
	    ClientInfo[i].bcn_int[j] = ClientInfo[i].bcn_int[j - 1];
	  ClientInfo[i].bcn_int[0] = bcn_int;
	  memcpy (&BSSID, ClientInfo[i].BSSID, WLAN_ADDR_LEN);
	  if (!ClientInfo[i].IsP2P)
	    ClientInfo[i].IsAP = 1;
	  break;
	}
    }
  LogPutWep (BSSID, hasWep);
}

void
LogPutIsP2P (UINT8 AddMac[WLAN_ADDR_LEN])
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].IsP2P = 1;
	  ClientInfo[i].IsAP = 0;
	  break;
	}
    }
}

void
LogPutOtherInformation (UINT8 AddMac[WLAN_ADDR_LEN], char *OtherInformation)
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  strncpy (ClientInfo[i].OtherInformation, OtherInformation,
		   (MAXSIZE_OTHERINFORMATION - 1));
	  break;
	}
    }
}

/** Get a timestamp of a BSSID and put it in the fifo of TS */
void
LogPutTimestamp (UINT8 AddMac[WLAN_ADDR_LEN], UINT64 TimeStamp)
{
  UINT16 i;
  UINT8 j;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  for (j = (HISTORY_SIZE - 1); j != 0; j--)
	    {
	      ClientInfo[i].TimeStamp[j] = ClientInfo[i].TimeStamp[j - 1];
	    }
	  ClientInfo[i].TimeStamp[0] = TimeStamp;
	  debug (3, "TS : %016llX %016llX %016llX %016llX\n",
		 ClientInfo[i].TimeStamp[0], ClientInfo[i].TimeStamp[1],
		 ClientInfo[i].TimeStamp[2], ClientInfo[i].TimeStamp[3]);
	  /// \todo TODO Save System Timestamp
	  break;
	}
    }
}

/** Get a Sequence Number of a BSSID and put it in the fifo of SeqNum */
void
LogPutSeqNum (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 AddMacDst[WLAN_ADDR_LEN],
	      UINT16 SeqNum)
{
  UINT16 i, j;

  /** TODO : Analyse also when it's not a broadcast packet
      ad a new array (Mac@ cli, SeqNum) */
  if (CompareAddMac (AddMacDst, BroadcastMAC))
    {
      for (i = 0; i < NumberOfDetectedClient; i++)
	{
	  if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	    {
	      for (j = (HISTORY_SIZE - 1); j != 0; j--)
		{
		  ClientInfo[i].SeqNum[j] = ClientInfo[i].SeqNum[j - 1];
		}
	      ClientInfo[i].SeqNum[0] = WLAN_GET_SEQ_SEQNUM (SeqNum);
	      break;
	    }
	}
    }
}

/** Put the last IV find in packets */
void
LogPutLastIV (UINT8 AddMac[WLAN_ADDR_LEN], UINT8 IV[4])
{
  UINT16 i;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      if (CompareAddMac (ClientInfo[i].AddMac, AddMac))
	{
	  ClientInfo[i].LastIV[0] = IV[0];
	  ClientInfo[i].LastIV[1] = IV[1];
	  ClientInfo[i].LastIV[2] = IV[2];
	  ClientInfo[i].LastIV[3] = IV[3];
	  break;
	}
    }
}

/**
 * Write the final report
 */
void
LogWriteReport (void)
{
  int i;
  UINT8 j;

  printf ("\n\n");
  printf ("Now a summary of the detection :\n");
  printf ("--------------------------------\n");

  if (NumberOfDetectedClient == 0)
    {
      printf ("No 802.11b information has been catched\n\n");
    }

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      printf ("Station (%02X:%02X:%02X:%02X:%02X:%02X)  -  ",
	      ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
	      ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
	      ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
      printf ("BSSID=%02X:%02X:%02X:%02X:%02X:%02X  -  ",
	      ClientInfo[i].BSSID[0], ClientInfo[i].BSSID[1],
	      ClientInfo[i].BSSID[2], ClientInfo[i].BSSID[3],
	      ClientInfo[i].BSSID[4], ClientInfo[i].BSSID[5]);
      if ((strlen (ClientInfo[i].SSID) == 0)
	  || ((strlen (ClientInfo[i].SSID) == 1)
	      && (*ClientInfo[i].SSID == 0x20)))
	printf ("SSID is not broadcasted\n");
      else
	{
	  //printf ("SSID='%s'\n", ClientInfo[i].SSID);
	  printf ("SSID='");
	  for (j = 0; j < strlen (ClientInfo[i].SSID); j++)
	    {
	      if (isprint (ClientInfo[i].SSID[j]))
		printf ("%c", ClientInfo[i].SSID[j]);
	      else
		printf ("/0x%02X", ClientInfo[i].SSID[j]);
	    }
	  printf ("'\n");
	}

      if (ClientInfo[i].SNMax != ClientInfo[i].SNMin)
	printf ("  Signal is between %ld and %ld", ClientInfo[i].SNMin,
		ClientInfo[i].SNMax);
      else
	printf ("  Signal is stable at %ld", ClientInfo[i].SNMax);

      if (ClientInfo[i].RateMax != ClientInfo[i].RateMin)
	printf (" and Data rate is between %s and %s\n",
		RateToString (ClientInfo[i].RateMin),
		RateToString (ClientInfo[i].RateMax));
      else
	printf (" and Data rate is %s\n",
		RateToString (ClientInfo[i].RateMax));

      if (ClientInfo[i].MaxSpeed != 0)
	printf ("  Max speed available is %s\n",
		RateToString (ClientInfo[i].MaxSpeed));

      if (ClientInfo[i].hasWep)
	printf ("  Channel %d with Wep\n", ClientInfo[i].Channel);
      else
	printf ("  Channel %d with no Wep\n", ClientInfo[i].Channel);

      if (ClientInfo[i].bcn_int[0] != 0)
	printf ("  1 beacon every %d ms is sent\n", ClientInfo[i].bcn_int[0]);

      if (ClientInfo[i].IsDS)
	{
	  if (ClientInfo[i].IsAP)
	    {			// #568053
	      printf
		("  This is an AP that do some NAT or is a router (because it's also a DS)\n");
	    }
	  else
	    {
	      printf
		("  This stations is in the Wired network (DS) of the BSSID\n");
	    }
	}
      else
	{
	  if (ClientInfo[i].IsAP)
	    printf ("  This is an AP\n");
	  else if (ClientInfo[i].IsP2P)
	    printf ("  This an Ad-HOC client\n");
	  else
	    printf ("  This is a client\n");
	}

      if (ClientInfo[i].OtherInformation == NULL)
	{
	  printf ("  Other information discovered:\n");
	  printf ("   > \"%s\"\n", ClientInfo[i].OtherInformation);
	}

      printf ("-------------\n");
    }
}

/**
 * Write the .dot file with all informations we have
 */
void
LogWriteDOT (FILE * filename)
{
  int i;

  if (NumberOfDetectedClient == 0)
    {
      printf ("No 802.11b information has catched\n");
      printf ("So no .dot file is created\n\n");
    }
  else
    {
      printf ("802.11b information has catched so I make my .dot file ");
    }

  fprintf
    (filename, "digraph xyz {\n  ratio=compress;\n  size=\"8,8\";\n"
     "  rankdir=LR;\n  node [shape=box,fontsize=8];\n");

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      printf (".");
      fprintf (filename, "\"%02X:%02X:%02X:%02X:%02X:%02X\" -> ",
	       ClientInfo[i].BSSID[0], ClientInfo[i].BSSID[1],
	       ClientInfo[i].BSSID[2], ClientInfo[i].BSSID[3],
	       ClientInfo[i].BSSID[4], ClientInfo[i].BSSID[5]);
      fprintf (filename, "\"%02X:%02X:%02X:%02X:%02X:%02X\";\n",
	       ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
	       ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
	       ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
      if (ClientInfo[i].IsDS)
	fprintf (filename,
		 "\"%02X:%02X:%02X:%02X:%02X:%02X\" [shape=diamond];\n",
		 ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
		 ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
		 ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
      else
	{
	  if (ClientInfo[i].IsAP)
	    {
	      fprintf (filename,
		       "\"%02X:%02X:%02X:%02X:%02X:%02X\" [shape=circle,label=\"",
		       ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
		       ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
		       ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
	      fprintf (filename,
		       "%02X:%02X:%02X:%02X:%02X:%02X\\nSSID='%s'\"];\n",
		       ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
		       ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
		       ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5],
		       ClientInfo[i].SSID);
	    }
	}
    }
  fprintf (filename, "}\n");
  printf ("Done!\n\n");
}

void
LogWriteHisto (void)
{
#define HISTO_SIGNAL_SIZE 32
#define HISTO_SIGNAL_CHAN 14

  UINT8 i, Max = 0;
  float divid = 0;
  int j;
  char histo[HISTO_SIGNAL_SIZE][HISTO_SIGNAL_CHAN];

  printf ("\n Spectral repartition :\n-----------------------\n\n");

  for (i = 0; i < HISTO_SIGNAL_CHAN; i++)
    {
      debug (3, "%d,", Stats.MaxSignal[i]);
      if (Stats.MaxSignal[i] > Max)
	Max = Stats.MaxSignal[i];
    }

  divid = (float) (Max + 1) / (float) HISTO_SIGNAL_SIZE;
  debug (3, "divid=%f\n", divid);

  for (i = 0; i < HISTO_SIGNAL_CHAN; i++)
    {
      for (j = 0; j < HISTO_SIGNAL_SIZE; j++)
	histo[j][i] = '-';
    }

  for (i = 0; i < HISTO_SIGNAL_CHAN; i++)
    {
      for (j = 0; j < (Stats.MaxSignal[i] / divid); j++)
	{
	  histo[j][i] = '*';
	}
      //debug(2, "j=%d\n", j-1);
    }

  printf ("    01 02 03 04 05 06 07 08 09 10 11 12 13     14\n");
  for (j = HISTO_SIGNAL_SIZE; j != 0; j--)
    {
      printf ("%3d ", (UINT8) (j * divid));
      for (i = 0; i < HISTO_SIGNAL_CHAN; i++)
	{
	  if (i == (HISTO_SIGNAL_CHAN - 1))
	    printf ("    ");
	  printf ("%c%c ", histo[j - 1][i], histo[j - 1][i]);
	}
      printf ("\n");
    }
  printf ("   01 02 03 04 05 06 07 08 09 10 11 12 13     14\n\n");
}


void
WritePanel (UINT8 GoodPacket)
{
  UINT8 i;
  char Line[256];
  char Histo[HISTOSIZE + 2];
  char MAC[20];
  UINT8 SNMax = 0;
  static UINT8 MaxFromAll = 1;
  UINT16 HistoSize = 0;
  UINT8 MinAff, Posit = 1;


  if (NumberOfDetectedClient > (ROW_WND_PANEL - 2))
    MinAff = NumberOfDetectedClient - (ROW_WND_PANEL - 2);
  else
    MinAff = 0;

#ifdef WITH_THREAD
  pthread_mutex_lock (&screen_mutex);
#endif
  for (i = MinAff; i < NumberOfDetectedClient; i++)
    {
      snprintf (MAC, 20, "%02X:%02X:%02X:%02X:%02X:%02X",
		ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
		ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
		ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
      strncpy (Histo, "|____________________________________________",
	       HISTOSIZE + 1);

      if ((ClientInfo[i].SNMax > MaxFromAll) || (Res.Signal > MaxFromAll))
	{
	  if (Res.Signal > MaxFromAll)
	    MaxFromAll = Res.Signal;
	  else
	    MaxFromAll = ClientInfo[i].SNMax;
	}

      if (GoodPacket && !strncmp (MAC, Res.SrcMac, 20))
	{
	  HistoSize = (UINT16) ((Res.Signal * HISTOSIZE) / MaxFromAll);
	  strncpy (Histo,
		   "|=======================================================",
		   HistoSize);
	  mvwprintw (Panel_WND, Posit, 1, ">");
	}
      else
	{
	  mvwprintw (Panel_WND, Posit, 1, " ");
	}

      mvwprintw (Panel_WND, Posit, 2, "%s",
		 ClientInfo[i].IsAP ? "AP " : "STA");

      /* if SN/2 > 64 then SNMax = 64 else SNMax = SN/2 */
      SNMax = (ClientInfo[i].SNMax * HISTOSIZE) / MaxFromAll;
      if (SNMax <= HISTOSIZE)
	{
	  Histo[SNMax - 1] = '|';
	  //Histo[HISTOSIZE] = ' ';
	}
      Histo[HISTOSIZE + 1] = 0;
      snprintf (Line, COL_WND_PANEL - POS_HISTO - 1, "%s (%d,%d)    ", Histo,
		((GoodPacket
		  && !strncmp (MAC, Res.SrcMac, 20)) ? Res.Signal : 0),
		(int) ClientInfo[i].SNMax);
      mvwprintw (Panel_WND, Posit, POS_MAC, "%s", MAC);
      mvwprintw (Panel_WND, Posit, POS_SSID,
		 "\"%s\"                          ", ClientInfo[i].SSID);
      mvwprintw (Panel_WND, Posit, POS_HISTO, " %s", Line);
      Posit++;
    }

  wrefresh (Panel_WND);
#ifdef WITH_THREAD
  pthread_mutex_unlock (&screen_mutex);
#endif

}

/** Function to calculate some stats */
void
DoSummary (void)
{
  UINT8 i;
  UINT8 ChanTab[14] = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

  Stats.AP = 0;
  Stats.STA = 0;
  Stats.Channel = 0;

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      ChanTab[ClientInfo[i].Channel] = 1;

      if (ClientInfo[i].IsAP)
	Stats.AP++;
      else
	Stats.STA++;
    }

  for (i = 0; i < 14; i++)
    {
      if (ChanTab[i] != 0)
	Stats.Channel++;
    }
}

/* ******** */
/* IDS ZONE */
/* ******** */

/**
  Function : CalculCoef
             Resolve equation Y = a.X + b
*/
void
CalculCoef (UINT32 X[HISTORY_SIZE], UINT32 Y[HISTORY_SIZE], float *a,
	    float *b)
{
  float aa[HISTORY_SIZE], bb[HISTORY_SIZE];
  UINT8 NbData, i;

  // in case of ...
#if (HISTORY_SIZE < 2)
#error "STOP ?! HISTORY_SIZE MUST be greater than 1. Change it on include.h"
#endif

  // Find number of data to analyse
  NbData = 0;
  for (i = 0; i < HISTORY_SIZE; i++)
    {
      if ((X[i] != 0) && (Y[i] != 0))
	NbData++;
    }

  // Calculate all aa and all bb
  for (i = 0; i < (NbData - 1); i++)
    {
      aa[i] = (Y[i] - Y[i + 1]) / (X[i] - X[i + 1]);
      bb[i] = Y[i] / (aa[i] * X[i]);
    }

  *a = *b = 0;
  for (i = 0; i < NbData; i++)
    {
      *a = *a + aa[i];
      *b = *b + bb[i];
    }
  *a = *a / NbData;
  *b = *b / NbData;
}


/** IDS module : Analysis of Beacons Intervals */
UINT8
IDS_BcnInt (void)
{
  UINT16 i;
  UINT8 ret = 0, j;
  char MAC[20];

  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      for (j = 0; j < (HISTORY_SIZE - 1); j++)
	{
	  if (((ClientInfo[i].bcn_int[j] != 0)
	       && (ClientInfo[i].bcn_int[j + 1] != 0))
	      && (ClientInfo[i].bcn_int[j] != ClientInfo[i].bcn_int[j + 1]))
	    {
	      snprintf (MAC, 20, "%02X:%02X:%02X:%02X:%02X:%02X",
			ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
			ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
			ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
	      warning ("IDS WARNING - beacon interval is changing for %s\n",
		       MAC);
	      ret = 0x01;	/// Beacon interval history is suspect
	      break;
	    }
	}
    }
  return ret;
}

UINT8
IDS_TimeStamp (void)
{
  UINT16 i;
  UINT8 ret = 0, j;
  char MAC[20];

  // Analysis of Time Stamps
  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      for (j = 0; j < (HISTORY_SIZE - 1); j++)
	{
	  if (((ClientInfo[i].TimeStamp[j] != 0)
	       && (ClientInfo[i].TimeStamp[j + 1] != 0))
	      && (ClientInfo[i].TimeStamp[j] <
		  ClientInfo[i].TimeStamp[j + 1]))
	    {
	      snprintf (MAC, 20, "%02X:%02X:%02X:%02X:%02X:%02X",
			ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
			ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
			ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
	      warning ("IDS WARNING - Time Stamp is not correct for %s\n",
		       MAC);
	      warning
		(" Difference between two catched packets is : 0x%llX\n",
		 ClientInfo[i].TimeStamp[j + 1] - ClientInfo[i].TimeStamp[j]);
	      warning ("Timestamps = (0x%016llX,0x%016llX)\n",
		       ClientInfo[i].TimeStamp[j],
		       ClientInfo[i].TimeStamp[j + 1]);
	      ret = 0x02;	/// TS history is suspect
	      break;
	    }
	}
    }
  return ret;
}

/** 
 * Analyse SeqNumber variation to determine if a problem is occur,
 * like an usurpation of MAC ADD
 */
UINT8
IDS_Var_SeqNum (void)
{
  UINT16 i;
  UINT8 ret = 0, j;
  char MAC[20];

  // Analysis of variation of Seqnum
  for (i = 0; i < NumberOfDetectedClient; i++)
    {
      for (j = 0; j < (HISTORY_SIZE - 1); j++)
	{
	  if (((ClientInfo[i].SeqNum[j] != 0)
	       && (ClientInfo[i].SeqNum[j + 1] != 0))
	      && (ClientInfo[i].SeqNum[j] < ClientInfo[i].SeqNum[j + 1]))
	    {
	      snprintf (MAC, 20, "%02X:%02X:%02X:%02X:%02X:%02X",
			ClientInfo[i].AddMac[0], ClientInfo[i].AddMac[1],
			ClientInfo[i].AddMac[2], ClientInfo[i].AddMac[3],
			ClientInfo[i].AddMac[4], ClientInfo[i].AddMac[5]);
	      warning
		("IDS WARNING - Sequence Number is not correct for %s\n",
		 MAC);
	      debug (1, "SeqNum = (0x%04X,0x%04X)\n", ClientInfo[i].SeqNum[j],
		     ClientInfo[i].SeqNum[j + 1]);
	      ret = 0x04;	/// SeqNum history is suspect
	      break;
	    }
	}
    }
  return ret;
}

UINT8
IDS_Detect_FakeAP (void)
{
  return 0;
}

/** 
 * Function to analyse data and try to determine an Intrusion or a DOS
 * Try also to determine a jammer like FakeAP 
 */
UINT8
IDS (void)
{
  /**	ret == 0x01;	Beacon interval history is suspect
	ret == 0x02;	TS history is suspect
	ret == 0x04;	SeqNum history is suspect
  */
  UINT8 ret = 0;

  ret += IDS_BcnInt ();
  ret += IDS_TimeStamp ();
  ret += IDS_Var_SeqNum ();
  ret += IDS_Detect_FakeAP ();

  return (ret);
}
