/* Linux Prism II Stumbler - Utility Scan for 802_11 networks under Linux
 * 
 * File : wlan-ng.c
 * Project : WifiScanner (c) 2002 Herv Schauer Consultants
 * Usage : This utility is written for use with IEEE 802.11 adapters based
 * on Intersil's PRISM II chipset (PCMCIA).
 * 
 * Base code was from prismstumbler Jan Fernquist <Jan.B.Fernquist@telia.com>
 * and wlanctl from www.linux-wlan.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: wlan-ng.c,v 1.18 2003/07/25 11:23:08 poggij Exp $
 */

#include <include.h>
#include <src/interface.h>
#include <src/crt_io.h>
#include <src/wlan-ng.h>
#include <src/driver.h>

static char *ID = "$Id: wlan-ng.c,v 1.18 2003/07/25 11:23:08 poggij Exp $";

extern unsigned int DebugLevel;
extern WINDOW *Sum_WND, *RealTime_WND;
extern ScanResult_t Res;

#ifdef LWNG_14
static CaptureArg ca;
static char errbuf[PCAP_ERRBUF_SIZE];
#endif /* ifdef LWNG_14 */

#ifdef LWNG_15
extern p80211_caphdr_t wlan_header;
//UINT8 wlan_payload[WLAN_A4FR_MAXLEN];
static UINT8 wlan_payload[MAX_BUFFER_SIZE];
#endif /* ifdef LWNG_15 */

/**** GET FROM WLANCTL-NG ****/
/* Version 0.1.13 */

/*----------------------------------------------------------------
* printmsg
*
* Traverse the message items printing each.
*
* Arguments:
*       msg     buffer containing a complete msg
*       msgcode integer identifying the msg
*
* Returns: 
*       0       - success 
*       ~0      - failure
----------------------------------------------------------------*/
void
printmsg (UINT8 * msg, UINT32 msgcode)
{
  UINT8 tmpitem[MSG_BUFF_LEN];
  UINT8 *msgptr;
  UINT8 *start;
  UINT16 i;
  grplistitem_t *grp;
  UINT32 narg;
  UINT32 offset;
  UINT32 tmpdid;
  p80211meta_t *alist;


  msgptr = msg;

  /* acquire the msg argument metadata list */
  if ((grp = p80211_did2grp (msg_catlist, msgcode)) != NULL)
    {
      alist = grp->itemlist;
      narg = GETMETASIZE (alist);
    }
  else
    {
      printf ("Driver communication: Invalid msgcode of %u\n",
	      (unsigned int) msgcode);
      return;
    }

  /* print the message code */
  printf ("message=%s\n", grp->name);

  start = msg + sizeof (p80211msg_t);

  for (i = 1; i < narg; i++)
    {
      tmpdid = msgcode | P80211DID_MKITEM (i) | alist[i].did;
      offset = p80211item_getoffset (msg_catlist, tmpdid);
      msgptr = start + offset;

      /* pass tmpdid since the 'totext' functions */
      /* expect a non-zero did */
      if (((p80211item_t *) msgptr)->status ==
	  P80211ENUM_msgitem_status_data_ok)
	{
	  if (alist[i].totextptr != NULL)
	    {
	      (*(alist[i].totextptr)) (msg_catlist, tmpdid, msgptr, tmpitem);
	      printf ("  %s\n", tmpitem);
	    }
	  else
	    {
	      p80211_error2text (P80211ENUM_msgitem_status_missing_print_func,
				 tmpitem);
	      printf ("  %s=%s\n", alist[i].name, tmpitem);
	    }
	}
      else
	{
	  p80211_error2text (((p80211item_t *) msgptr)->status, tmpitem);
	  printf ("  %s=%s\n", alist[i].name, tmpitem);
	}

    }				/* for each argument in the metadata */
}


/*----------------------------------------------------------------
* cmdline2requestmsg
*
* Default command line to request message converter.  Takes the
* command (request) code and the cmdline arguments, compares them
* to the metadata for the request arguments for the given request
* and if all required arguments are present and valid, builds a
* message structure.  This function handles the general case.
*
* Arguments:
*       msg     buffer to build msg in (assumed to be at least MSG_BUFF_LEN bytes)
*       msgcode partial did containing category and group indices
*       argc    number of command line arguments
*       argv    array of argument strings
*
* Returns: 
*       0       - success 
*       ~0      - failure
----------------------------------------------------------------*/
INT
cmdline2requestmsg (char *devname, UINT8 * msg, UINT32 msgcode, int argc,
		    char **argv)
{
  UINT8 *cmdlinelist;
  UINT8 *msgptr;
  UINT8 *start;
  UINT8 tmpitem[MSG_BUFF_LEN];
  p80211meta_t *alist = NULL;
  grplistitem_t *grp = NULL;
  INT found;
  INT i;
  INT j;
  INT32 narg;
  UINT32 tmpdid;
  size_t itemlen;
  size_t offset;

  /* Create an array of bytes where each byte represents a flag for
     each command line argument.  For each argument on the command line
     following the msg code, the repsective byte will contain either
     a 0 (for not found) or 1 (found) after an attempt to match the 
     command line argument to one of the metadata arguments for the
     user entered 'valid' msg, such as 'scan' or 'powermgmt'. */

  if ((cmdlinelist = (UINT8 *) malloc (argc)) == NULL)
    {
      printf ("Driver communication: cmdlinelist memory allocation failed\n");
      return 1;
    }
  /* initialize all the bytes to 0 for not found */
  memset (cmdlinelist, 0, argc);
  memset (msg, 0, MSG_BUFF_LEN);
  memset (tmpitem, 0, MSG_BUFF_LEN);

  ((p80211msg_t *) msg)->msgcode = msgcode;
  strncpy (((p80211msg_t *) msg)->devname, devname, WLAN_DEVNAMELEN_MAX - 1);
  ((p80211msg_t *) msg)->msglen = sizeof (p80211msg_t);

  start = msg + sizeof (p80211msg_t);
  msgptr = start;

  /* acquire the msg argument metadata list */
  if ((grp = p80211_did2grp (msg_catlist, msgcode)) != NULL)
    {
      alist = grp->itemlist;
      narg = GETMETASIZE (alist);
    }
  else
    {
      printf ("cmdline2requestmsg: Invalid msgcode of %u\n",
	      (unsigned int) msgcode);
      free (cmdlinelist);
      return 1;
    }

  /*
     printf("The cmd %s is valid with a code of 0x%08lx\n", 
     argv[2], msgcode);
     printf("   argc=%d, narg is %lu\n", argc, narg);
   */

  /* Loop through the metadata for all the arguments of the message
     and initialize the did, the len and set the status to
     status code of "no value" (i.e. the data isn't set) */

  for (i = 1; i < narg; i++)
    {
      tmpdid = msgcode | P80211DID_MKITEM (i) | alist[i].did;

      if ((offset = p80211item_getoffset (msg_catlist, tmpdid)) != 0xffffffff)
	{

	  msgptr = start + offset;

	  ((p80211item_t *) msgptr)->did = tmpdid;
	  ((p80211item_t *) msgptr)->status =
	    (short) P80211ENUM_msgitem_status_no_value;
	  if ((((p80211item_t *) msgptr)->len =
	       (short) (p80211item_maxdatalen (msg_catlist,
					       tmpdid))) != 0xffffffffUL)
	    {
	      ((p80211msg_t *) msg)->msglen +=
		(sizeof (p80211item_t) + ((p80211item_t *) msgptr)->len);
	    }
	  else
	    {
	      printf ("Driver communication: invalid data length for %s\n",
		      alist[i].name);
	      free (cmdlinelist);
	      return 1;
	    }
	}
      else
	{
	  printf ("Driver communication: [1] error creating offset for %s\n",
		  alist[i].name);
	  free (cmdlinelist);
	  return 1;
	}
    }

  /* Build message in the same order as the metadata argument list by
     by looping through msg arg metadata, args always start at index 1 */

  msgptr = start;

  for (i = 1; i < narg; i++)
    {
      found = 0;
      tmpdid = msgcode | P80211DID_MKITEM (i) | alist[i].did;

      if ((offset = p80211item_getoffset (msg_catlist, tmpdid)) != 0xffffffff)
	{
	  /*
	     printf("cmdline2request: "
	     "curr meta data item %s: "
	     "offset=%d\n", alist[i].name, offset);
	   */
	  msgptr = start + offset;
	}
      else
	{
	  printf ("Driver communication: [2] error creating offset for %s\n",
		  alist[i].name);
	  free (cmdlinelist);
	  return 1;
	}

      /* loop through msg arguments on cmdline */
      for (j = 3; (j < argc) && (!found); j++)
	{
	  /* does meta match cmdline arg? */
	  if (strncmp (alist[i].name, argv[j], strlen (alist[i].name)) == 0)
	    {

	      if (P80211ITEM_ISREQUEST (alist[i].flags))
		{
		  found = 1;
		  cmdlinelist[j] = (UINT8) 1;

		  if (alist[i].fromtextptr != NULL)
		    {
		      (*(alist[i].fromtextptr))
			(msg_catlist, tmpdid, tmpitem, argv[j]);
		    }

		  itemlen = sizeof (p80211item_t) +
		    p80211item_maxdatalen (msg_catlist, tmpdid);

		  memcpy (msgptr, tmpitem, itemlen);

		}
	      else
		{
		  printf ("non-request argument found on cmdline.\n");
		  free (cmdlinelist);
		  return 1;
		}
	    }			/* if cmdline match */
	}			/* for each cmdline arg */

    }				/* for each msg argument metadata */

  /* Loop through the built message and check the status field.
     For required request arguments, the status must be "data ok"
     or it's an error.  If the status code is "no value", the
     argument can not be a required request argument; otherwise,
     it's an error.  Any other status code is an error. */

  msgptr = start;

  for (i = 1; i < narg; i++)
    {
      if (((p80211item_t *) msgptr)->status >
	  (short) P80211ENUM_msgitem_status_no_value)
	{

	  p80211_error2text (((p80211item_t *) msgptr)->status, tmpitem);
	  printf ("%s=\"%s\"\n", alist[i].name, tmpitem);
	  free (cmdlinelist);
	  return 1;
	}
      else if (((p80211item_t *) msgptr)->status ==
	       (short) P80211ENUM_msgitem_status_no_value)
	{
	  if ((P80211ITEM_ISREQUIRED (alist[i].flags)) &&
	      (P80211ITEM_ISREQUEST (alist[i].flags)))
	    {
	      printf ("The required argument \'%s\' has no value.\n",
		      alist[i].name);
	      free (cmdlinelist);
	      return 1;
	    }
	}

      msgptr += (sizeof (p80211item_t) + ((p80211item_t *) msgptr)->len);
    }

  /* check to see that each message argument on the command line was */
  /* matched to an argument metadata for the message */
  for (j = 3; j < argc; j++)
    {
      if (!(cmdlinelist[j]))
	{
	  printf ("\'%s\' entered on the command line "
		  "was either an invalid\n"
		  "argument to the cmd \'%s\' or an extra "
		  "occurence of a valid argument.\n", argv[j], argv[2]);
	  free (cmdlinelist);
	  return 1;
	}
    }

  free (cmdlinelist);
  return 0;
}

/*----------------------------------------------------------------
* do_ioctl
*
* TODO: describe
*
* Arguments:
*       argc    number of command line arguments
*       argv    array of argument strings
*
* Returns: 
*       0       - success 
*       ~0      - failure
----------------------------------------------------------------*/
int
do_ioctl (char *devname, UINT8 * msg)
{
  int result = -1;
  int fd;
  p80211ioctl_req_t req;

  /* set the magic */
  req.magic = P80211_IOCTL_MAGIC;

  /* get a socket */
  fd = socket (AF_INET, SOCK_STREAM, 0);
  if (fd == -1)
    {
      perror ("ioctl1 error ");
      return result;
    }

  req.len = MSG_BUFF_LEN;	/* TODO: need to fix the length */
  req.data = msg;
  strncpy (req.name, devname, 6);
  req.result = 0;

  result = ioctl (fd, P80211_IFREQ, &req);

  if (result == -1)
    {
      //perror ("ioctl2 error ");
    }
  close (fd);
  return result;
}


void
dump_msg (void *msg)
{
  p80211msgd_t *msgp = msg;
  int i;
  int bodylen;

  fprintf (stderr, "  msgcode=0x%08lx  msglen=%lu  devname=%s\n",
	   msgp->msgcode, msgp->msglen, msgp->devname);
  fprintf (stderr, "body: ");
  bodylen = msgp->msglen -
    (sizeof (msgp->msgcode) + sizeof (msgp->msglen) + sizeof (msgp->devname));
  for (i = 0; i < bodylen; i += 4)
    {
      fprintf (stderr, "%02x%02x%02x%02x ",
	       msgp->args[i], msgp->args[i + 1],
	       msgp->args[i + 2], msgp->args[i + 3]);
    }
  fprintf (stderr, "\n");
}

int
selectChannelWLAN (char *devname, int channel)
{
  INT result = 0;
  UINT8 message[MSG_BUFF_LEN];
  UINT32 msgcode = P80211DID_INVALID;
  char *Cmd[9];
  char strChannel[16];
  int nbParam;

  PrintScaleChannel (channel);

  debug (3, "Define SChannel = %02d\n", channel);
  Res.SChannel = channel;

/***/
#ifdef LWNG_2_0
  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_wlansniff";
  Cmd[3] = "enable=false";
  nbParam = 4;

  msgcode = DIDmsg_lnxreq_wlansniff;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    result = do_ioctl (devname, message);
  else
    {
      warning ("Warning : change to Channel %d failed\n", channel);
      // TODO : be curses friendly
      printmsg (message, msgcode);
    }
#endif

  sprintf (strChannel, "channel=%d", channel);

  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_wlansniff";
  Cmd[3] = strChannel;
  Cmd[4] = "enable=true";
#ifdef LWNG_14
  Cmd[5] = "keepwepflags=true";
  Cmd[6] = "prismheader=true";
  Cmd[7] = "stripfcs=false";
  nbParam = 8;
#else
  nbParam = 5;
#endif
  msgcode = DIDmsg_lnxreq_wlansniff;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    {
      if ((result = do_ioctl (devname, message)) == 0)
	{
	  //printmsg (message, msgcode);
	  //fprintf (stderr, "%d\n", channel);
	}
    }
  else
    {
      // TODO : Must integrate in curses ...
      printmsg (message, msgcode);
      warning ("Warning : change to Channel %d failed\n", channel);
    }

  return result;
}

int
shutCardWLAN (char *devname)
{
  INT result = 0;
  UINT8 message[MSG_BUFF_LEN];
  UINT32 msgcode = P80211DID_INVALID;
  char *Cmd[5];
  int nbParam;

  system ("ifconfig wlan0 down");

/* // Shutdown card
#ifdef LWNG_14
  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_ifstate";
  Cmd[3] = "ifstate=disable";
  nbParam = 4;

  msgcode = DIDmsg_lnxreq_ifstate;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    {
      result = do_ioctl (devname, message);
    }
  else
    {
      printmsg (message, msgcode);
    }
#endif
*/
   /* Stop Sniff function */ 
  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_wlansniff";
  Cmd[3] = "enable=false";
  nbParam = 4;

  msgcode = DIDmsg_lnxreq_wlansniff;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    {

      if (((result = do_ioctl (devname, message)) == 0) && (DebugLevel >= 3))
	{
	  printmsg (message, msgcode);
	}
    }
  else
    {
      printmsg (message, msgcode);
    }

  return result;
}

int
openCardWLAN (char *devname)
{
  INT result = 0;
  UINT8 message[MSG_BUFF_LEN];
  UINT32 msgcode = P80211DID_INVALID;
  char *Cmd[5];
  int nbParam;

  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_wlansniff";
  Cmd[3] = "enable=false";
  nbParam = 4;

  msgcode = DIDmsg_lnxreq_wlansniff;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    {
      if (((result = do_ioctl (devname, message)) == 0) && (DebugLevel >= 3))
	{
	  printmsg (message, msgcode);
	}
    }
  else
    {
      printmsg (message, msgcode);
    }
#ifdef LWNG_14
  Cmd[0] = "XXXX";
  Cmd[1] = devname;
  Cmd[2] = "lnxreq_ifstate";
  Cmd[3] = "ifstate=enable";
  nbParam = 4;

  msgcode = DIDmsg_lnxreq_ifstate;
  result = cmdline2requestmsg (devname, message, msgcode, nbParam, Cmd);
  if (result == 0)
    {
      result = do_ioctl (devname, message);
    }
  else
    {
      printmsg (message, msgcode);
    }
#endif
  system ("ifconfig wlan0 up");
  return result;
}


// Get packet from card
int
getPacketWLAN (unsigned char *buf, int maxlen)
{
  struct pcap_pkthdr pktHdr;
  u_char *ret;
  fd_set rs;
  p80211msg_lnxind_wlansniffrm_t *Sniff_Frame;

  FD_ZERO (&rs);
  FD_SET (0, &rs);

  ret = (u_char *) pcap_next (ca.pcap, &pktHdr);
  // If no problem and packet is enought big (with data)
  if ((ret) && (pktHdr.len >= sizeof (p80211msg_lnxind_wlansniffrm_t)))
    {
      memcpy (buf, ret, pktHdr.len);
      Sniff_Frame = (p80211msg_lnxind_wlansniffrm_t *) buf;
      // Fill Header
      wlan_header.version = 0;	// It's a reduced capture frame format
      wlan_header.length = 0;	// Not used for now
      wlan_header.mactime = Sniff_Frame->mactime.data;
      wlan_header.hosttime = Sniff_Frame->hosttime.data;
      wlan_header.phytype = phytype_dsss_dot11_b;	// Not used for now
      wlan_header.channel = Sniff_Frame->channel.data;
      wlan_header.datarate = Sniff_Frame->rate.data * 5;	// datarate is in units of 100kbps.
      wlan_header.antenna = 0;	// Not used for now
      wlan_header.priority = 0;	// Not used for now
      wlan_header.ssi_type = 0;	// Not used for now
      wlan_header.ssi_signal = Sniff_Frame->signal.data;
      wlan_header.ssi_noise = Sniff_Frame->noise.data;
      wlan_header.preamble = 0;	// Not used for now
      wlan_header.encoding = 0;	// Not used for now
      // Fill data frame
      memcpy (wlan_payload, &buf[sizeof (p80211msg_lnxind_wlansniffrm_t)],
	      pktHdr.len - sizeof (p80211msg_lnxind_wlansniffrm_t));
      memcpy (buf, wlan_payload,
	      maxlen - sizeof (p80211msg_lnxind_wlansniffrm_t));

      if (pktHdr.len <= sizeof (p80211msg_lnxind_wlansniffrm_t))
	// Don't return negative value
	return 0;
      else
	return (pktHdr.len - sizeof (p80211msg_lnxind_wlansniffrm_t));
    }
  else
    {
      return (0);		/* Noting to read */
    }
}

int
openPacketWLAN (char *devname)
{
  int DataLink;

  ca.pcap = pcap_open_live (devname, 3000, 1, 1, errbuf);
  if (ca.pcap)
    {
      pcap_setnonblock (ca.pcap, 1, errbuf);
      DataLink = pcap_datalink (ca.pcap);
      switch (DataLink)
	{
	case DLT_PRISM_HEADER:
	  debug (2, "pcap_datalink(ca.pcap) = %d = DLT_PRISM_HEADER\n",
		 DataLink);
	  ca.offset = 144;
	  break;
	case DLT_IEEE802_11:
	  debug (2, "pcap_datalink(ca.pcap) = %d = DLT_IEEE802_11\n",
		 DataLink);
	  ca.offset = 0;
	  break;
	case DLT_AIRONET_HEADER:
	  fatal ("pcap_datalink(ca.pcap) = %d = DLT_AIRONET_HEADER:\n",
		 DataLink);
	  break;
	default:		//COOKED
	  debug (2, "pcap_datalink(ca.pcap) = %d = COOKED:\n", DataLink);
	  ca.offset = 160;
	}
      return 1;
    }
  return -1;
}

void
closePacketWLAN (void)
{
  pcap_close (ca.pcap);
}
